// PFS-4481 - CSP Monitor Script
(function () {
    'use strict';

    const TAG = '[PFS-4481]';

    // Utility: Print warning with stack trace
    function warnWithStack(message, detail) {
        const error = new Error();
        console.warn(`${TAG} ${message}`, detail);
        console.debug(`${TAG} Stack trace:`);
        console.debug(error.stack);
    }

    // Patch only setTimeout and setInterval with string-check
    const originalSetTimeout = window.setTimeout;
    window.setTimeout = function (handler, timeout, ...args) {
        if (typeof handler === 'string') {
            warnWithStack('⚠️ Detected use of setTimeout() with string', handler);
        }
        return originalSetTimeout.apply(this, arguments);
    };

    const originalSetInterval = window.setInterval;
    window.setInterval = function (handler, timeout, ...args) {
        if (typeof handler === 'string') {
            warnWithStack('⚠️ Detected use of setInterval() with string', handler);
        }
        return originalSetInterval.apply(this, arguments);
    };

    // Monitor truly dangerous functions
    function patchDangerousFunction(fnName) {
        const original = window[fnName];
        if (!original) return;

        window[fnName] = function (...args) {
            warnWithStack(`⚠️ Detected use of ${fnName}()`, args);
            return original.apply(this, args);
        };
    }

    patchDangerousFunction('eval');
    patchDangerousFunction('Function');

    // Mutation observer to detect suspicious inline <script> tags
    const observer = new MutationObserver((mutationsList) => {
        for (const mutation of mutationsList) {
            for (const node of mutation.addedNodes) {
                if (node.nodeType !== Node.ELEMENT_NODE) continue;
                if (node.tagName === 'SCRIPT') {
                    const isExternal = node.hasAttribute('src');
                    const hasNonce = node.hasAttribute('nonce');
                    if (!isExternal && !hasNonce) {
                        warnWithStack('⚠️ Inline <script> tag added without nonce', node);
                    }
                }
            }
        }
    });

    observer.observe(document.documentElement, {
        childList: true,
        subtree: true
    });

    console.info(`${TAG} CSP monitor active.`);
})();
