/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Dialog")) {
    window.Picasso.Dialog = {};
}

/* Private scope */
(function () {

    const cellFormats = [
        "DD/MM/YYYY",
        "D/M/YYYY",
        "DD.MM.YYYY",
        "D.M.YYYY",
        "DD-MM-YYYY",
        "D-M-YYYY",

        "MM/DD/YYYY",
        "M/D/YYYY",
        "MM.DD.YYYY",
        "M.D.YYYY",
        "MM-DD-YYYY",
        "M-D-YYYY",

        "YYYY-MM-DD",
        "YYYY/MM/DD",
        "YYYY.MM.DD"
    ];

    /**
     * Copy data from inputs to link
     *
     * @param {Picasso.Dialog.Link}  dialog  A #{@link Picasso.Dialog.Link}
     * @param {Object}               inputs  Input fields ({ "id" = HTMLElement })
     **/

    var copyData = function (dialog, inputs) {
        dialog._link.setValidTill(Picasso.Helper.getDateTimePickerValue(
            inputs.pica_link_valid_till, -1));
        dialog._link.setMaxDownloads(Picasso.Helper.parseNumber(
            inputs.pica_link_max_downloads.val()));
        dialog._link.setPassword(inputs.pica_link_password.val());

        /* Flags */
        if (inputs.pica_link_read_write && inputs.pica_link_read_write.is(":checked")) {
            dialog._link._flags |= Picasso.Link.Flags.UPLOADS_ALLOWED;
        }
        if (inputs.pica_link_read_only && inputs.pica_link_read_only.is(":checked")) {
            dialog._link._flags &= ~Picasso.Link.Flags.UPLOADS_ALLOWED;
        }
    };

    /**
     * Assign data to inputs
     *
     * @param {Picasso.Dialog.Link}  dialog  A #{@link Picasso.Dialog.Link}
     * @param {Object}               inputs  Input fields ({ "id" = HTMLElement })
     **/

    var assignData = function (dialog, inputs) {
        /* Assign valid till */
        if (-1 !== dialog._link.getValidTill()) {
            Picasso.Helper.setDateTimePickerValue(inputs.pica_link_valid_till,
                dialog._link.getValidTill());
        }

        /* Assign max downloads */
        if (-1 !== dialog._link.getMaxDownloads()) {
            inputs.pica_link_max_downloads.val(dialog._link.getMaxDownloads());
        }

        /* Assign password */
        inputs.pica_link_password.val(dialog._link.getPassword());

        inputs.pica_link_read_write?.closest("div").show()


        if(dialog._link.getFile().is(Picasso.File.Flags.FILE) && !dialog._link.getFile().isOODocument()) {
            inputs.pica_link_read_write?.closest("div").hide()
            inputs.pica_link_max_downloads?.closest("div").show()
            inputs.pica_link_disable_download?.closest("div").hide()
        }
        if((dialog._link.getFile().is(Picasso.File.Flags.DIRECTORY) || dialog._link.getFile().is(Picasso.File.Flags.FOLDER) ) ) {
            if (!dialog._link.getServerAllowsFileLinkUploads()) {
                inputs.pica_link_read_write?.closest("div").hide()
            }
            inputs.pica_link_max_downloads?.closest("div").hide()
            inputs.pica_link_disable_download?.closest("div").hide()
        }
        if (dialog._link.getFile().isOODocument() || !dialog._link.getServerAllowsFileLinkUploads()){
            if (inputs.pica_link_read_write) {
                inputs.pica_link_read_write.parent().find("lang").text(Picasso.Lang.get("permission_can_read_write"))
            }
        }else {
            if (inputs.pica_link_read_write) {
                inputs.pica_link_read_write.parent().find("lang").text(Picasso.Lang.get("permission_can_upload"))
            }
        }
        if (dialog._link.getFile().isOODocument() && !dialog._link.getDocumentPublicEditingEnabled()){
            inputs.pica_link_read_write?.closest("div").hide()
        }

        /* Assign allow uploads */
        if(dialog._link.is(Picasso.Link.Flags.UPLOADS_ALLOWED)) {
            inputs.pica_link_read_write.trigger("click");
        }else {
            inputs.pica_link_read_only.trigger("click");
        }
        if (dialog._link.getFile().is(Picasso.File.Flags.FILE) && (dialog._link.getFile().isInline() || dialog._link.getFile().isImage())) {
            inputs.pica_link_max_downloads?.closest("div").show()
            inputs.pica_link_disable_download?.closest("div").show()
        }
        if (dialog._link.getFile().is(Picasso.File.Flags.FILE) && (dialog._link.getFile().isOODocument() || dialog._link.getFile().isTXT())) {
            inputs.pica_link_max_downloads?.closest("div").hide()
            inputs.pica_link_disable_download?.closest("div").show()
        }

        (function maybeAutoDisable() {
            if (dialog._link.getMaxDownloads() === 0) {
                const file = dialog._link.getFile();
                const isFile = dialog._link.getFile().is(Picasso.File.Flags.FILE)
                const isSpecial = isFile && (file.isInline() || file.isImage() || file.isOODocument() || file.isTXT());
                if (isSpecial) {
                    $('#pica_link_disable_download').trigger('click');
                }
            }
        })();

    };

    /**
     * Save link data
     *
     * @param {Picasso.Dialog.Link}  dialog     A #{@link Picasso.Dialog.Link}
     * @param {Function}             onSuccess  Call on success
     * @param {Function}             onError    Call on error
     **/

    var saveData = function (dialog, onSuccess, onError) {
        dialog._link.save(onSuccess, onError);
    };

    /**
     * Constructor for {@link Picasso.Dialog.Link}
     *
     * @constructor
     **/

    Picasso.Dialog.Link = function () {

        /* Init */
        this._link = null;

        /* Sanity check */
        var dialog = $("#pica_link_dialog");

        if (0 === dialog.length) {
            throw "Dialog not found";
        }

        Picasso.Helper.createDateTimepicker("#pica_link_valid_till");

        var checkboxField = $("#pica_link_password_protected");
        var passwordField = $("#pica_link_password");

        /* Toggle state when input is changed */
        passwordField.on("change", function () {
            checkboxField.prop("checked",
                ("undefined" !== $.type(this) && 0 < $(this).val().length));
            checkboxField.trigger("change");
        });

        /* Reset checked state when input is cleared */
        passwordField.siblings(".pica-input-clear").click(function () {
            checkboxField.prop("checked", false);
            checkboxField.trigger("change");
        });
    };

    /**
     * Edit {@link Picasso.Link} via dialog
     *
     * @param {Picasso.Link}  link       A #{@link Picasso.Link}
     * @param {Function}      onSuccess  Call on success
     * @param {Function}      onError    Call on error
     **/

    Picasso.Dialog.Link.prototype.edit = function (link, onSuccess, onError)
    {
        var self = this;

        this._link = link;

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_link_dialog",
            Picasso.Lang.get("dialog_title_settings"),
            Picasso.Lang.get("dialog_body_settings", Picasso.Lang.get("label_links")),
            (Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE|Picasso.Dialog.Flags.SPINNER));

        dialog.setAssignHandler(function (inputs) {
            $("#pica_link_dialog .pica-hide-on-create").show();

            assignData(self, inputs);
        });

        dialog.setOkHandler(function (inputs) {
            copyData(self, inputs);
            saveData(self,

                /* Success */
                function () {
                    dialog.hide();

                    if (onSuccess) {
                        onSuccess.apply(null, arguments);
                    }
                },

                /* Error */
                function (e, status, response) {
                    dialog.showSpinner(false);

                    if (status === 406) {
                        if (response && response.message) {
                            Picasso.Notification.show(response.message);
                        }
                    }

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
        });

        dialog.show();
    };

    /**
     * Create {@link Picasso.Link} with settings via dialog
     *
     * @param {Picasso.Link}  link       A #{@link Picasso.Link}
     * @param {Function}      onSuccess  Call on success
     * @param {Function}      onError    Call on error
     **/

    Picasso.Dialog.Link.prototype.create = function (file, onSuccess, onError) {
        var self = this;

        this._link = new Picasso.Link();
        this._link.setFile(file);

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_link_dialog",
            Picasso.Lang.get("dialog_title_settings"),
            Picasso.Lang.get("dialog_body_settings", Picasso.Lang.get("label_links")),
            (Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE));

        dialog.setAssignHandler(function (inputs) {
            const serverAllowsFileLinkUploads = Picasso.get("serverAllowsFileLinkUploads") === 'true';

            const fileLinkDefaultValidity = Picasso.get("fileLinkDefaultValidity")
            const fileLinkMaximumValidity = Picasso.get("fileLinkMaximumValidity")

            const defaultValidityDays = parseInt(fileLinkDefaultValidity, 10);
            const maxValidityDays = parseInt(fileLinkMaximumValidity, 10);
            if (!isNaN(defaultValidityDays) && defaultValidityDays > 0) {
                let now = new Date();
                now.setDate(now.getDate() + defaultValidityDays);
                Picasso.Helper.setDateTimePickerValue(inputs.pica_link_valid_till, now.getTime());

                const datepicker = inputs.pica_link_valid_till
                    .datetimepicker()
                    .data("DateTimePicker");

                if (datepicker) {
                    inputs.pica_link_valid_till.on("dp.show dp.update dp.change", function () {
                        highlightDefaultDay(datepicker,now);
                    });
                }
            }
            if (!isNaN(maxValidityDays) && maxValidityDays > 0) {
                let maxDate = new Date();
                maxDate.setDate(maxDate.getDate() + maxValidityDays);

                const datepicker = inputs.pica_link_valid_till
                    .datetimepicker()
                    .data("DateTimePicker");

                if (datepicker) {
                    datepicker.maxDate(moment(maxDate));
                    inputs.pica_link_valid_till.on("dp.show dp.update dp.change", function () {
                        highlightMaxDay(datepicker);
                    });
                }
            }




            inputs.pica_link_read_write?.closest("div").show()
            if(file.is(Picasso.File.Flags.FILE) && !file.isOODocument()) {
                inputs.pica_link_read_write?.closest("div").hide()
                inputs.pica_link_max_downloads?.closest("div").show()
                inputs.pica_link_disable_download?.closest("div").hide()
            }

            if (file.is(Picasso.File.Flags.FILE) && (file.isInline() || file.isImage())) {
                inputs.pica_link_max_downloads?.closest("div").show()
                inputs.pica_link_disable_download?.closest("div").show()
            }

            if (file.is(Picasso.File.Flags.FILE) && (file.isOODocument() || file.isTXT())) {
                inputs.pica_link_max_downloads?.closest("div").hide()
                inputs.pica_link_disable_download?.closest("div").show()
            }
            if((file.is(Picasso.File.Flags.DIRECTORY) || file.is(Picasso.File.Flags.FOLDER) )) {
                if (!serverAllowsFileLinkUploads) {
                    inputs.pica_link_read_write?.closest("div").hide()
                }
                inputs.pica_link_max_downloads?.closest("div").hide()
                inputs.pica_link_disable_download?.closest("div").hide()
            }
            if (file.isOODocument() || !serverAllowsFileLinkUploads){
                if (inputs.pica_link_read_write) {
                    inputs.pica_link_read_write.parent().find("lang").text(Picasso.Lang.get("permission_can_read_write"))
                }
            }else {
                if (inputs.pica_link_read_write) {
                    inputs.pica_link_read_write.parent().find("lang").text(Picasso.Lang.get("permission_can_upload"))
                }
            }
            if (file.isOODocument() && !Picasso.get("documentPublicEditingEnabled")){
                inputs.pica_link_read_write?.closest("div").hide()
            }

        })

        dialog.setOkHandler(function (inputs) {

            var linkId;

            /* Store data */
            var defer_create = function () {
                var deferred = $.Deferred();

                Picasso.Link.create(file,
                    /* Success */
                    function (data, status) {
                        deferred.resolve(data);
                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_created",
                                Picasso.Lang.get("label_links")), "info");
                    }, onError
                );

                return deferred.promise();
            };

            /* Store data */
            var defer_setSettings = function (link) {
                self._link = link;

                if (-1 !== Picasso.Helper.getDateTimePickerValue(inputs.pica_link_valid_till, -1)) {
                    self._link.setValidTill(Picasso.Helper.getDateTimePickerValue(inputs.pica_link_valid_till, -1));

                }
                self._link.setMaxDownloads(Picasso.Helper.parseNumber(inputs.pica_link_max_downloads.val()));
                self._link.setPassword(inputs.pica_link_password.val());

                /* Flags */
                if (inputs.pica_link_read_write && inputs.pica_link_read_write.is(":checked")) {
                    self._link._flags |= Picasso.Link.Flags.UPLOADS_ALLOWED;
                }
                if (inputs.pica_link_read_only && inputs.pica_link_read_only.is(":checked")) {
                    self._link._flags &= ~Picasso.Link.Flags.UPLOADS_ALLOWED;
                }
                saveData(self,
                        /* Success */
                        function () {
                            if (onSuccess) {
                                onSuccess(self._link);
                            }
                            dialog.hide();
                        }
                    ,
                        /* Error */
                        function (e, status, response) {
                            dialog.showSpinner(false);

                            if (status === 406) {
                                if (response && response.message) {
                                    Picasso.Notification.show(response.message);
                                }
                            }

                            if (onError) {
                                onError(e, status);
                            }
                        }
                    );
            };

            defer_create()
                .then(function (link) {
                    return defer_setSettings(link);
                })
                .catch(onError);
        });

        dialog.show();
    };

    function highlightMaxDay(datepicker) {
        if (!datepicker) return;

        const maxDate = datepicker.maxDate();
        if (!maxDate) return;

        const maxDateStr = maxDate.format("DD.MM.YYYY");


        const $widget = $(".bootstrap-datetimepicker-widget:visible");
        if (!$widget.length) return;

        $widget.find('td[data-action="selectDay"]').each(function () {
            const $td = $(this);
            const cellDay = $td.attr("data-day");

            const cellMoment = moment(cellDay, cellFormats, true);
            if (!cellMoment.isValid()) {
                return;
            }

            const cellDayStr = cellMoment.format("DD.MM.YYYY");

            if (cellDayStr === maxDateStr) {
                $td.css("background-color", "dodgerblue");

                $td.attr("data-toggle", "tooltip");
                $td.attr("data-placement", "auto");
                $td.attr("title", Picasso.Lang.get("label_file_link_validity_maximum"));

                $td.tooltip({
                    container: "body"
                });

                $td.on('click', function () {
                    $(this).tooltip('hide')
                })


            }
        });
    }

    function highlightDefaultDay(datepicker, defaultDate) {
        if (!datepicker || !defaultDate) return;
        const defaultDateStr = moment(defaultDate).format("DD.MM.YYYY");

        const $widget = $(".bootstrap-datetimepicker-widget:visible");

        if (!$widget.length) return;


        $widget.find('td[data-action="selectDay"]').each(function () {
            const $td = $(this);
            const cellDay = $td.attr("data-day");

            const cellMoment = moment(cellDay, cellFormats, true);
            if (!cellMoment.isValid()) {
                return;
            }
            const cellDayStr = cellMoment.format("DD.MM.YYYY");


            if (cellDayStr === defaultDateStr) {

                $td.attr("data-toggle", "tooltip");
                $td.attr("data-placement", "auto");
                $td.attr("title", Picasso.Lang.get("label_file_link_validity_default"));
                $td.tooltip({
                    container: "body"
                });

                $td.on('click', function () {
                    $(this).tooltip('hide')
                })
            }
        });
    }

    $(document).ready(function() {
        $('#pica_link_disable_download').on('change', function() {
            if (!$(this).is(':checked')) {
                $('#pica_link_disable_download_label').parent().find("i").removeClass('glyphicons-check').addClass('glyphicons-unchecked');
                $("#pica_link_max_downloads").removeAttr("disabled");
                $("#pica_link_max_downloads").val("");
            } else {
                $('#pica_link_disable_download_label').parent().find("i").removeClass('glyphicons-unchecked').addClass('glyphicons-check');
                $("#pica_link_max_downloads").attr("disabled", true);
                $("#pica_link_max_downloads").val(0);
            }
        });

        $('#pica_link_disable_download_label').on('click', function() {
            $('#pica_link_disable_download').trigger('click');
        });
        $('#disable_icon').on('click', function() {
            $('#pica_link_disable_download_label').trigger('click');
        });


    });
})();
