/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Dialog")) {
    window.Picasso.Dialog = {};
}

/* Private scope */
(function () {

    /**
     * Copy data from inputs to link
     *
     * @param {Picasso.Dialog.Settings}  dialog  A #{@link Picasso.Dialog.Settings}
     * @param {Object}               inputs  Input fields ({ "id" = HTMLElement })
     **/

    var copyData = function (dialog, inputs) {
        dialog._folder._versions = Picasso.Helper.parseNumber(inputs.pica_settings_versions.val());
        dialog._folder._basePath = inputs.pica_settings_basePath.val();
    };

    /**
     * Save {@link Picasso.Uploadform}data
     *
     * @param {Picasso.Dialog.Settings}  dialog     A #{@link Picasso.Dialog.Settings}
     * @param {Function}                   onSuccess  Call on success
     * @param {Function}                   onError    Call on error
     **/

    var saveData = function (dialog, onSuccess, onError) {
        var defer_version = function() {
            dialog._folder.setMaxVersions(dialog._folder.getMaxVersions(),onSuccess,onError);
        }

        var defer_path = function() {
            dialog._folder.setBasePath(dialog._folder.getBasePath(),onSuccess,onError);
        }

        $.when(defer_version()).then(function () {
            if (Picasso.get("account").is(Picasso.Account.Flags.ORG_ADMIN) || dialog._folder.getBasePath()) {
                $.when(defer_path()).then(function () {
                    if (onSuccess) {
                        onSuccess(dialog._folder);
                        Picasso.Files.fetch();
                    }
                }, onError)
            }
        }, onError);



    };


    /**
     * Assign data to inputs
     *
     * @param {Picasso.Dialog.Link}  dialog  A #{@link Picasso.Dialog.Link}
     * @param {Object}               inputs  Input fields ({ "id" = HTMLElement })
     **/

    var assignData = function (dialog, inputs) {
        /* Assign versions */
        if (inputs.pica_settings_versions && dialog._folder.getMaxVersions()) {
            inputs.pica_settings_versions.val(dialog._folder.getMaxVersions());
        } else {
            inputs.pica_settings_versions.parent().css("display","none");
            $("#pica_settings_purge_archive").parent().css("display","none");
        }
        /* Assign base Path for admin */
        if (inputs.pica_settings_basePath && dialog._folder.getBasePath()) {
            inputs.pica_settings_basePath.val(dialog._folder.getBasePath());
        } else {
            inputs.pica_settings_basePath.parent().css("display", "none");
            $("#pica_settings_rescan").parent().css("display", "none");
        }

    };

    function scan(folder) {
        /* Handle types */
        var url = Picasso.Helper.joinPaths([
            "/wapi/folders",
            folder.getUrl()
        ]);
        var data = {
            action: "scan"
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax("/" + url, data,
                /* Success */
                function (data, status) {
                    Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_scan"),
                            "info");
                }
        );
    }

    function purgeArchive(folder) {
        /* Handle types */
        var url = Picasso.Helper.joinPaths([
            "/wapi/folders",
            folder.getUrl()
        ]);
        var data = {
            action: "purge"
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax("/" + url, data,
                /* Success */
                function (data, status) {
                    Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_purged"),
                            "info");
                }
        );
    }

    /**
     * Constructor for {@link Picasso.Dialog.Restore}
     *
     * @param {Picasso.File}  folder  A folder of type #{@link Picasso.File}
     * @constructor
     **/

    Picasso.Dialog.Settings = function (folder) {
        var self = this;

        /* Init */
        this._folder = folder;
    };

    /**
     * Show restore dialog
     **/

    Picasso.Dialog.Settings.prototype.show = function () {
        var self = this;

        /* Create dialog */
        var dialog = new Picasso.Dialog("#pica_settings_dialog",
                Picasso.Lang.get("dialog_title_settings"),
                Picasso.Lang.get("dialog_body_settings", this._folder.getName()));

        dialog.setAssignHandler(function (inputs) {
            assignData(self, inputs);
        });

        dialog.setOkHandler(function (inputs) {
            copyData(self, inputs);
            saveData(self,
                    function () {
                        dialog.hide();
                    },
                    function (e, status) {
                        dialog.showSpinner(false);
                    }
            );
        });

        $("#pica_settings_rescan").off("click").click(function () {
            scan(self._folder);
        });

        $("#pica_settings_purge_archive").off("click").click(function () {
            purgeArchive(self._folder);
        });

        /* Finally show dialog */
        if (self._folder.getMaxVersions() != null || self._folder.getBasePath() != null) {
            dialog.show();
        }
        else {
            Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_manage_unavailable"),
                    "warning");
        }
    };

    /* Fix dropdowns in scroller */
    Picasso.Combo.fixDropdownInScroller(".pica-inputlist-scroller");
})();
