/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Public")) {
    window.Picasso.Public = {};
}

/* Private scope */
(function () {

    /* Globals */
    var TABLE = new Picasso.Table("#public_files_table",
        Picasso.Table.Flags.ROWS_SELECTABLE);
    var PAGE = 1;

    /**
     * Download given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@link Picasso.File}
     **/

    var downloadFile = function (file) {
        /* Create a new anchor and simulate click to download */
        var a = $("<a>", {
            href: file.getDownloadUrl(),
            download: file.getName()
        });

        $("body").append(a);
        a[0].click();
        a.remove();
    };

    /**
     * Get selected file
     *
     * @param {jQuery|HTMLElement}  elem  Element to check
     *
     * @returns {*} Either found #{@link Picasso.File}; otherwise null
     **/

    var getFileFromElem = function (elem) {
        var file = null;
        var serial = $(elem).find("[data-serial]");

        if (0 < serial.length) {
            file = new Picasso.File(serial.data("serial"));
        }

        return file;
    };

    /**
     * Handle click on element name
     *
     * @param {Event}  e  Click event
     **/

    var handleClickName = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var file = new Picasso.File($(this).data("serial"));

        /* Check type */
        if (file.is(Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY)) {
            TABLE.deselectAll();

            Picasso.Public.fetch(file, false,

                /* Success */
                function () {
                    var a = Picasso.Actionbar.addCrumb(
                        file.getName(), handleClickCrumb);

                    a.data("serial", file.serialize());
                }
            );
        } else { ///< Open
            /* Create a new anchor and simulate click to download */
            var a = $("<a>", {
                href: file.getOpenUrl(),
                target: "_blank"
            });

            $("body").append(a);
            a[0].click();
            a.remove();
        }
    };

    /**
     * Handle click on bread bar
     **/

    var handleClickCrumb = function () {
        var file = $(this).data("serial");

        if (file) {
            file = new Picasso.File(file);
        }

        Picasso.Public.fetch(file);

        /* Clear search field */
        $("#pica_action_search_input").val("");
    };

    /**
     * Handle click on download buttons
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDownload = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selectedFiles = TABLE.getSelectedRows();

        selectedFiles = $.map(selectedFiles, function (val) {
            return getFileFromElem(val);
        });

        if (0 < selectedFiles.length) {
            $.each(selectedFiles, function () {
                downloadFile(this);
            });
        }
    };

    /**
     * Render given json object
     *
     * @param {Picasso.File}  file  A {@type Picasso.File}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderFile = function (file) {
        var tr = $("<tr>", {
            id: file.getElemId(),
            "data-search-keys": [file.getName()],
            draggable: false
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        var icon = file.getIcon(); ///< FIXME: Thumbnails have been deleted

        td.append(icon);

        /* Add thumbnail if found */
        if (file.isImage()) {
            var img = $("<img>", {
                class: "pica-icon-thumb-list",

                onLoad: function () {
                    icon.hide();
                    icon.after(this);
                },

                src: file.getThumbnailUrl()
            });

            img.on("error", function () {
                icon.show();
                this.remove();

                return true;
            });
        }

        tr.append(td);

        /* 2. column: Name */
        td = $("<td>", { class: "pica-overflow" });

        var a = $("<a>", {
            class: "pica-name",
            text: file.getName(),
            href: file.getDownloadUrl(),
            "data-serial": file.serialize()
        });

        a.click(handleClickName);

        td.append(a);
        tr.append(td);

        /* 3. column: Size/Permissions */
        td  = $("<td>");

        td.append(Picasso.Helper.formatSize(file.getSize()));
        td.data("data-sort", file.getSize());

        tr.append(td);

        return tr;
    };

    /**
     * Update table header for selection mode
     *
     * @param {Integer}  nselected    Number of selected rows
     * @param {Object}   selectedRow  Selected row object
     **/

    var handleOnSelection = function (nselected, selectedRow) {
        var onlyRoot = $(".archive-visible-only-root");
        var onlyOther = $(".archive-visible-only-other");

        /* Toggle visibility for root */
        var cwd = Picasso.get("cwd");

        if (cwd) {
            onlyRoot.hide();
            onlyOther.show();
        } else {
            onlyRoot.show();
            onlyOther.hide();
        }
    };

    /**
     * Fetch JSON data
     *
     * @param {Picasso.File}  file       A #{@link Picasso.File}
     * @param {Boolean}       append     Whether to append data
     * @param {Function}      onSuccess  Success handler
     * @param {Function}      onError    Error handler
     **/

    Picasso.Public.fetch = function (file, append, onSuccess, onError) {
        Picasso.set("cwd", file);

        /* Reset page */
        if (true !== append) {
            PAGE = 1;
        }

        /* Update file list */
        var updateFiles = function (files, json) {

            /* Call renderer before infinite scroll */
            if (true === append) {
                TABLE.append(files, renderFile);
            } else {
                TABLE.update(files, renderFile);
            }

            /* Bind infinite scroll */
            Picasso.Helper.bindInfiniteScroll(json, TABLE, PAGE,
                function () {
                    PAGE++;

                    Picasso.Public.fetch(Picasso.get("cwd"), true);
                }
            );

            handleOnSelection();

            if (onSuccess) onSuccess();
        };

        if (file) {
            Picasso.set("folderId", file.getFolderId());

            file.getFiles(updateFiles, onError, PAGE, null, false);
        } else {
            Picasso.Public.initFolder();
        }
    };

    Picasso.Public.initFolder = function() {
        var search = new URLSearchParams(window.location.search);
        var folderIdEnc = search.get("folderID");

        var onSuccess = function (folder) {
            Picasso.Public.fetch(folder);
        };

        Picasso.File.getSingleFolder(onSuccess, undefined, folderIdEnc);
    };

    /* Register view mode table callbacks */
    TABLE.setSelectionHandler(handleOnSelection);

    /* Bind handlers */
    $("#public_home").click(handleClickCrumb);

    $(".files-ui-download").click(handleClickDownload);

    /* Init */
    Picasso.Public.fetch();
})();
