/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Devices")) {
    window.Picasso.Devices = {};
}

/* Private scope */
(function () {

    /* Globals */
    var TABLE = new Picasso.Table("#devices_table");
    var PAGE = 1;

    /**
     * Handle click on edit button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickEdit = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        if (1 === selected.length) {
            var device = new Picasso.Device($(selected[0]).find(".pica-serial").data("serial"));

            device.edit(

                /* Success */
                function (device) {
                    /* Replace row */
                    var row = $("#" + device.getElemId());
                    var rendered = renderDevice(device);

                    row.replaceWith(rendered);
                    rendered.addClass("pica-highlight");
                }
            );
        }
    };

    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDelete = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Device($(val).find(".pica-serial").data("serial"))
        });

        /* Show dialog */
        var what = (1 === selected.length ? selected[0].getName() :
            Picasso.Lang.get("string_selected_elements", selected.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_delete"),
            Picasso.Lang.get("dialog_body_delete", what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {
                var device = this;

                device.remove(

                    /* Success */
                    function () {
                        Picasso.Helper.fadeOut($("#" + device.getElemId()),
                            function () {
                                this.remove();

                                TABLE.checkIfEmpty();
                            });

                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_deleted",
                                Picasso.Lang.get("label_devices")), "info");
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);
                    }
                );
            });
        });

        dialog.show();
    };

    /**
     * Render given json object
     *
     * @param {Picasso.Device}  device  A {@type Picasso.Device}
     * @returns {*}
     **/

    var renderDevice = function (device) {
        var tr = $("<tr>", {
            id: device.getElemId(),
            "data-search-keys": [device.getName()],
            draggable: false
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        var span = $("<span>", {
            class: "pica-icon pica-glyph glyphicons glyphicons-imac"
        });

        td.append(span);
        tr.append(td);

        /* 2. column: Name */
        var td = $("<td>", {
            class: "pica-overflow pica-serial pica-name",
            text: device.getName(),
        });

        td.data("serial", device.serialize());

        tr.append(td);

        /* 3. column: Folders */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: device.getFoldersCount()
        }));

        /* 4. column: Server */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: device.getServer()
        }));

        /* 5. column: Last seen */
        td = $("<td>", {
            class: "hidden-xs",
        });

        var span = $("<span>", {
            "data-tooltip": "",
            "data-placement": "bottom",
            title: Picasso.Helper.formatAge(device.getLastSeen()),
            text: Picasso.Helper.formatDate(device.getLastSeen())
        });

        span.tooltip({ container: "body" });

        td.append(span);
        tr.append(td);

        return tr;
    };

    /**
     * Fetch JSON data
     **/

    Picasso.Devices.fetch = function (append) {

        /* Reset page */
        if (true !== append) {
            PAGE = 1;
        }

        Picasso.Device.getAll(

            /* Success */
            function (devices, json) {
                if (true === append) {
                    TABLE.append(devices, renderDevice);
                } else {
                    TABLE.update(devices, renderDevice);
                }

                /* Bind infinite scroll */
                Picasso.Helper.bindInfiniteScroll(json, TABLE, PAGE,
                    function () {
                        PAGE++;

                        Picasso.Devices.fetch(true);
                    }
                );
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }, PAGE
        );
    };

    /* Bind handlers */
    $("#devices_edit").click(handleClickEdit);
    $("#devices_delete").click(handleClickDelete);

    /* Init */
    Picasso.Devices.fetch();
})();
