/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/* Private scope */
(function () {

    /**
     * Constructor for {@link Picasso.Configuration}
     *
     * @constructor
     **/

    Picasso.Configuration = function (data) {
        var json = Picasso.Helper.tryJSON(data);

        /* Fetch data: Load from server or restore from stringified object  */
        this._flags = Picasso.Helper.getPropOrDefValue(json, 0, "_flags");
        this._name = Picasso.Helper.getPropOrDefValue(json, "", "name", "_name");
        this._foldersCount = Picasso.Helper.getPropOrDefValue(json, 0, "nFolders", "_foldersCount");
        this._defaultVersions = Picasso.Helper.getPropOrDefValue(json, 0, "default_archive_versions", "_defaultVersions");
        this._quota = Picasso.Helper.getPropOrDefValue(json, 0, "spaceAllowed", "_quota");
        this._assignByIp = Picasso.Helper.getPropOrDefValue(json, "", "name", "_name");
        this._assignByLdap = Picasso.Helper.getPropOrDefValue(json, "", "name", "_name");
        this._lastModified = Picasso.Helper.getPropOrDefValue(json, -1, "lastModified", "_lastModified");

        /* Update flags if unset */
        if (json && !json.hasOwnProperty("_flags")) {
            if (json.hasOwnProperty("defaultConfig") && true === json.defaultConfig) {
                this._flags |= Picasso.Configuration.Flags.DEFAULT_CONFIG;
            }
        }

        /* Create proxy to catch getter/setter calls */
        return Picasso.Helper.createProxy(this);
    };

    /* Prototype methods */

    /**
     * Set folders for {@link Picasso.Configuration}
     *
     * @param {Array}     folders    Array of folders
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Configuration.prototype.setFolders = function (folders,
                                                           onSuccess, onError)
    {
        if ("" === this._name) return;

        var data = $.extend({
            name: this._name ///< Required for create
        }, folders);

        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues(
            "/wapi/configurations", "setFolders", data, onSuccess, onError
        );
    };

    /**
     * Get default folders for {@link Picasso.Configuration}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Configuration.prototype.getFolders = function (onSuccess, onError) {
        if ("" === this._name) return;

        var url = "/wapi/configurations";
        var data = {
            action: "getFolders",
            name: this._name
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                if (onSuccess) {
                    onSuccess(ary);
                }
            }, onError
        );
    };

    /**
     * Set properties for {@link Picasso.Configuration}
     *
     * @param {Array}     properties  Array of folders
     * @param {Function}  onSuccess   Call on success
     * @param {Function}  onError     Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Configuration.prototype.setProperties = function (properties,
                                                              onSuccess, onError)
    {
        if ("" === this._name) return;

        var data = $.extend({
            name: this._name ///< Required for create
        }, properties);

        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues(
            "/wapi/configurations", "setProperties", data, onSuccess, onError
        );
    };

    /**
     * Get properties for {@link Picasso.Configuration}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Configuration.prototype.getProperties = function (onSuccess,
                                                              onError)
    {
        if ("" === this._name) return;

        var url = "/wapi/configurations";
        var data = {
            action: "getProperties",
            name: this._name
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                if (onSuccess) {
                    onSuccess(ary);
                }
            }, onError
        );
    };

    /**
     * Set organizations for {@link Picasso.Configuration}
     *
     * @param {Array}     organizations  Array of organizations
     * @param {Function}  onSuccess      Call on success
     * @param {Function}  onError        Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Configuration.prototype.setOrganizations = function (organizations,
                                                           onSuccess, onError)
    {
        if ("" === this._name) return;

        var data = $.extend({
            name: this._name
        }, organizations);

        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues(
            "/wapi/configurations", "setOrganizations", data,
            onSuccess, onError
        );
    };

    /**
     * Get {@link Organization} for {@link Picasso.Configuration}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Configuration.prototype.getOrganizations = function (onSuccess, onError) {
        if ("" === this._name) return;

        var url = "/wapi/configurations";
        var data = {
            action: "getOrganizations",
            name: this._name
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                /* Create organization objects */
                ary = $.map(ary, function (val) {
                    return new Picasso.Organization(val);
                });

                if (onSuccess) {
                    onSuccess(ary);
                }
            }, onError
        );
    };

    /**
     * Remove {@link Picasso.Configuration}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Configuration.prototype.remove = function (onSuccess, onError) {
        if ("" === this._name) return;

        var url = "/wapi/configurations";
        var data = {
            action: "delete",
            name: this._name
        };

        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Save {@link Picasso.Configuration}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Configuration.prototype.save = function (onSuccess, onError) {
        if ("" === this._name) return;

        var url = "/wapi/configurations";
        var data = {
            action: "store",
            method: "POST",
            name: this.getName(),
            spaceAllowed: (-1 !== this.getQuota() ? this.getQuota() : 0),
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /* Global - must be defined after constructor */

    /**
     * Get all groups
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     * @param {Number}    pageNum    Page number to load (optional)
     * @param {Number}    pageSize   Number of items per page (optional)
     * @param {String}    query      Query to filter results (optional)
     **/

    Picasso.Configuration.getAll = function (onSuccess, onError,
                                             pageNum, pageSize, query)
    {
        var url = "/wapi/configurations";
        var data = {
            action: "getAll",
            page: pageNum,
            size: pageSize
        };

        /* Add query parameter */
        if (query) {
            data.query = query;
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                ary = $.map(ary, function (val) {
                    return new Picasso.Configuration(val);
                });

                if (onSuccess) {
                    onSuccess(ary, json);
                }
            }, onError
        );
    };

    /**
     * Get {@link Organization} for {@link Picasso.Configuration}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     * @param {String}    query      Query to filter results (optional)
     **/

    Picasso.Configuration.getDefaultConfiguration = function (onSuccess,
                                                              onError, query)
    {
        var url = "/wapi/configurations";
        var data = {
            action: "getDefaultConfiguration"
        };

        /* Add query parameter */
        if (query) {
            data.query = query;
        }

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                if (onSuccess) {
                    onSuccess(ary);
                }
            }, onError
        );
    };

    /* Flags - must be defined after constructor */

    if (!window.Picasso.Configuration.hasOwnProperty("Flags")) {
        window.Picasso.Configuration.Flags = {};
    }

    /* Flags */
    Picasso.Configuration.Flags.DEFAULT_CONFIG = (1 << 0); ///< Config is default config
})();