/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/* Private scope */
(function () {

    /**
     * Constructor for {@link Picasso.Link}
     *
     * @constructor
     **/

    Picasso.Link = function (data) {
        var json = Picasso.Helper.tryJSON(data);

        /* Fetch data: Load from server or restore from stringified object  */
        this._flags = Picasso.Helper.getPropOrDefValue(json, 0, "_flags");
        this._oid = Picasso.Helper.getPropOrDefValue(json, "", "id", "ID", "oid", "OID", "_oid");
        this._name = Picasso.Helper.getPropOrDefValue(json, "", "name", "fileName", "_name");
        this._folderName = Picasso.Helper.getPropOrDefValue(json, "", "folderDisplay", "_folderName");
        this._accountOid = Picasso.Helper.getPropOrDefValue(json, "", "accountOid", "_accountOid");
        this._accountName = Picasso.Helper.getPropOrDefValue(json, "", "accountDisplayname", "_accountName");
        this._downloads = Picasso.Helper.getPropOrDefValue(json, 0, "downloads", "_downloads");
        this._maxDownloads = Picasso.Helper.getPropOrDefValue(json, 0, "maxDownloads", "_maxDownloads");
        this._downloadUrl = Picasso.Helper.getPropOrDefValue(json, "", "downloadUrl", "_downloadUrl");
        this._linkUrl = Picasso.Helper.getPropOrDefValue(json, "", "link", "_linkUrl");
        this._relUrl = Picasso.Helper.getPropOrDefValue(json, "", "relUrl", "_relUrl");
        this._password = Picasso.Helper.getPropOrDefValue(json, "", "password", "_password");

        this._validTill = Picasso.Helper.getPropOrDefValue(json, -1, "expirationDate", "_validTill");
        this._createdAt = Picasso.Helper.getPropOrDefValue(json, -1, "creationDate", "_createdAt");

        this._documentPublicEditingEnabled = Picasso.Helper.getPropOrDefValue(json, false, "documentPublicEditingEnabled", "_documentPublicEditingEnabled");

        this._serverAllowsFileLinkUploads = Picasso.Helper.getPropOrDefValue(json, true, "serverAllowsFileLinkUploads", "_serverAllowsFileLinkUploads");

        /* Parse permissions */
        this._permission = Picasso.Permission.fromValue(
            Picasso.Helper.getPropOrDefValue(json, null, "permission", "_permission"));

        /* FIXME+WARNING: relativeName is the name of the uploader here */
        this._anonymousName = Picasso.Helper.getPropOrDefValue(json, "", "anonymousName", "_anonymousma,e");

        /* Update file */
        if (json) {
            /* Create artificial file object.. */
            this._file = new Picasso.File(json._file || {
                _oid: (json.fileID ? json.fileID : ""),
                _name: this._name,
                _resUrl: this._linkUrl
            });

            /* Update flags if unset */
            if (!json.hasOwnProperty("_flags")) {
                if (json.isFile) {
                    this._file.setFlags(Picasso.File.Flags.FILE);
                } else if (json.isRoot) {
                    this._file.setFlags(Picasso.File.Flags.FOLDER);
                } else {
                    this._file.setFlags(Picasso.File.Flags.DIRECTORY);
                }

                if (json.hasOwnProperty("isUploadAllowed") &&
                    true === json.isUploadAllowed)
                {
                    this._flags |= Picasso.Link.Flags.UPLOADS_ALLOWED;
                }

                if (json.hasOwnProperty("isAnonymous") && true === json.isAnonymous) {
                    this._flags |= Picasso.Link.Flags.FROM_ANONYMOUS;
                }

                if (json.hasOwnProperty("isDeleted") && true === json.isDeleted) {
                    this._flags |= Picasso.Link.Flags.DELETED;
                }
            }
        }

        /* Create proxy to catch getter/setter calls */
        return Picasso.Helper.createProxy(this);
    };

    /* Prototype methods */

    /**
     * Save link settings
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Link.prototype.save = function (onSuccess, onError) {
        var url = "/wapi/filelink";
        var data = {
            action: "store",
            method: "POST",
            ID: this.getOid(),
            allowUploads: this.is(Picasso.Link.Flags.UPLOADS_ALLOWED),
            password: this.getPassword()
        };

        /* Check values before sending them */
        if (-1 !== this.getValidTill()) {
            data.validTill = Picasso.Helper.formatJSONDate(
                this.getValidTill());
        } else {
            data.validTill = "";
        }

        if (-1 !== this.getMaxDownloads()) {
            data.maxDownloads = this.getMaxDownloads();
        }

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Remove link
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Link.prototype.remove = function (onSuccess, onError) {
        var url = "/wapi/filelink";
        var data = {
            action: "removeLink",
            ID: this.getOid()
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Get link details by Id
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Link.getLink = function (oid, onSuccess, onError) {
        var url = "/wapi/filelink";
        var data = {
            action: "getLink",
            ID: oid
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json, status) {
                switch(status) {
                    case 200:
                        var ary = Picasso.Helper.getResultArray(json);

                        if (ary && 0 < ary.length) {
                            /* Create link object */
                            ary = new Picasso.Link(ary[0]);
                        }

                        if (onSuccess) {
                            onSuccess(ary, status);
                        }
                }
            }, onError
        );
    };

    /* Global - must be defined after constructor */

    /**
     * Create file link
     *
     * @param {Picasso.File}  file       A {@link Picasso.File}
     * @param {Function}      onSuccess  Call on success
     * @param {Function}      onError    Call on error
     **/

    Picasso.Link.create = function (file, onSuccess, onError) {
        var url = null;
        var data = {
            action: "createLink"
        };

        /* Handle types */
        if (file.is(Picasso.File.Flags.FOLDER)) {
            url = Picasso.Helper.joinPaths(["/wapi/filelink", file.getFolderIdEnc()]);
        } else {
            url = Picasso.Helper.joinPaths([
                "/wapi/filelink",
                file.getFolderIdEnc(),
                encodeURIComponent(file.getPath()).replaceAll("%2F","/")
            ]);
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                if (0 < ary.length) {
                    /* Create link object */
                    ary = new Picasso.Link(ary[0]);

                    ary.setFile(file);
                }

                if (onSuccess) onSuccess(ary);
            }, onError
        );
    };

    /**
     * Find folder/file link if any
     *
     * @param {Picasso.File}  file       A {@link Picasso.File}
     * @param {Function}      onSuccess  Call on success
     * @param {Function}      onError    Call on error
     **/

    Picasso.Link.find = function (file, onSuccess, onError) {
        var url = null;
        var data = {
            action: "getFileLink"
        };

        /* Handle types */
        if (file.is(Picasso.File.Flags.FOLDER)) {
            url = Picasso.Helper.joinPaths(["/wapi/filelink",
                file.getFolderIdEnc()]);
        } else {
            url = Picasso.Helper.joinPaths(["/wapi/filelink",
                file.getFolderIdEnc(), file.getPath()]);
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json, status) {
                switch(status) {
                    case 200:
                        var ary = Picasso.Helper.getResultArray(json);

                        if (ary && 0 < ary.length) {
                            /* Create link object */
                            ary = new Picasso.Link(ary[0]);

                            ary.setFile(file);
                        }

                        if (onSuccess) {
                            onSuccess(ary, status);
                        }
                }
            }, onError
        );
    };

    /**
     * Get all folder/file links for current user account
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     * @param {Number}    pageNum    Page number to load (optional)
     * @param {Number}    pageSize   Number of items per page (optional)
     * @param {String}    query      Query to filter results (optional)
     **/

    Picasso.Link.getAll = function (onSuccess, onError, pageNum,
                                    pageSize, query)
    {
        var url = "/wapi/filelink";
        var data = {
            action: "getAll",
            page: pageNum,
            size: pageSize
        };

        /* Add query parameter */
        if (query) {
            data.query = query;
        }

        //check if there is already a calls on the page and cancel them
        if (Picasso.Helper.xhrPool[url] && Picasso.Helper.xhrPool[url].length > 0) {
            Picasso.Helper.abortAll(url)
            $("#links_table tbody .pica-spinner").hide();
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                ary = $.map(ary, function (val) {
                    return new Picasso.Link(val);
                });

                if (onSuccess) onSuccess(ary, json);
            }, onError
        );
    };

    /* Flags - must be defined after constructor */

    if (!window.Picasso.Link.hasOwnProperty("Flags")) {
        window.Picasso.Link.Flags = {};
    }

    /* Flags */
    Picasso.Link.Flags.UPLOADS_ALLOWED = (1 << 0); ///< Link can be used for uploads
    Picasso.Link.Flags.FROM_ANONYMOUS = (1 << 1); ///< Link is from anonymous user
    Picasso.Link.Flags.DELETED = (1 << 2); ///< Link is deleted
})();
