/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/**
 * Some custom classes for specific behavior are used:
 *
 * .pica-inputlist - Base class for parent div
 * .pica-inputlist-taginput - TagInput for input
 * .pica-inputlist-table - Table for data
 * .pica-inputlist-remove - Remove button
 **/

/* Private scope */
(function () {

    /* Options for dropdown menus */
    var defaultOptionsForDropdowns = [
        { text: Picasso.Lang.get("action_remove"), icon: "bin" }
    ];

    /* Helper */

    /**
     * Handle options
     *
     * @param {Picasso.InputList}  inputList   A #{@link Picasso.InputList}
     * @param {Number}             optIdx      Index of selected option
     * @param {Object}             userdata    Userdata (optional)
     * @param {Function}           handleOpts  Custom options handler (optional)
     *
     * @returns {Boolean}
     **/

    var handleDefaultOpts = function (inputList, optIdx, userdata, handleOpts) {
        var that = $(this);

        /* Find list idx */
        var elem = that.closest("[data-inputlist-idx]");

        var idx = elem.data("inputlist-idx");
        var userdata = elem.data("userdata");

        /* Either call passed handler or handle it */
        if (handleOpts) {
            return handleOpts(inputList, optIdx, idx, userdata);
        } else {
            /* Handle default options */
            switch (optIdx) {
                case 0:
                    inputList.removeItem(idx);

                    return false;

                default:
                    return false;
            }
        }
    };

    /**
     * Handle click on edit button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickEdit = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = this._vmTable.getTable().getSelectedRows();

        if (selected && this._onEdit) {
            var that = $(selected);

            var idx = that.data("inputlist-idx");
            var userdata = that.data("userdata");

            this._onEdit(this, idx, userdata);
        }
    };

    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDelete = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = this._vmTable.getTable().getSelectedRows();

        /* Show dialog */
        var what = (1 === selected.length ? $(selected[0]).find(".pica-name").text() :
            Picasso.Lang.get("string_selected_elements", selected.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_remove"),
            Picasso.Lang.get("dialog_body_remove", what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {

                /* Either trigger button or last dropdown entry */
                var menu = $(this).find(".dropdown-menu");

                if (0 === menu.length) {
                    $(this).find("button").trigger("click");
                } else {
                    menu.find("li:last-child a").trigger("click");
                }
            });
        });

        dialog.show();
    };

    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickManage = function (e) {

        /* Get selected elements */
        var selected = this._vmTable.getTable().getSelectedRows();

        /* Show dialog */
        if (1 === selected.length ) {
            var folder = $(selected).data("userdata").folder;
            var file = new Picasso.File(folder);
            var dialog = new Picasso.Dialog.Settings(file);
            dialog.show();
        }

    };

    /**
     * Render given json object for list
     *
     * @param {Object}  obj  Object to render
     *
     * @returns {HTMLElement} Rendered json object
     **/

    var renderItemList = function (obj) {
        var tr = $("<tr>", {
            id: obj.id,
            "data-inputlist-idx": obj.idx,
            "data-userdata": JSON.stringify(obj.userdata),
            draggable: false
        });

        /* Whether to make item unselectable */
        if (true === obj.isUnselectable) {
            tr.addClass("pica-table-selection-disabled");
        }

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        Picasso.Helper.createGlyphAndAvatar(td, obj.icon, obj.userdata);

        tr.append(td);

        /* 2. column: Name */
        var td = $("<td>", {
            class: "pica-name pica-overflow pica-vertical-middle text-left",
            text: obj.name,
            "data-tooltip": "",
            "data-placement": "bottom",
            title: obj.tooltip,
            colspan: obj.description? 1 : 2
        });

        td.tooltip({ container: "body" });


        tr.append(td);

        /* Add description if any or an empty String to avoid display glitch  */
        /* 3. column: Description */
        if (obj.description) {
            td = $("<td>", {
                class: "pica-overflow pica-vertical-middle text-left text-muted",
                text: obj.description
            });
            tr.append(td);
        }

        /* Handle options */
        if (0 < (obj.self._flags & Picasso.InputList.Flags.IMMUTABLE)) {
            td = $("<td>", {
                class: "hidden-xs"
            });

            var button = $("<button>", {
                text: (obj.options[obj.startIdx]["text"] || ""),
                class: "btn btn-default disabled pull-right"
            });

            td.append(button);

            tr.addClass("pica-table-selection-disabled");
        } else {
            if (obj.hideCombo){
                td = $("<td>", {
                    class: "pica-overflow pica-vertical-middle text-left text-muted",
                    text: ""
                });
            }else {
                td = $("<td>", {
                    class: "hidden-xs",
                    html: Picasso.Combo.createDropdownOrButton(obj.options,
                            $.proxy(obj.handleOpts, this, obj.self),
                            obj.startIdx, "pull-right")
                });
            }

        }

        tr.append(td);

        return tr;
    };

    /**
     * Render given json object for list
     *
     * @param {Object}  obj  Object to render
     *
     * @returns {HTMLElement} Rendered json object
     **/

    var renderItemGrid = function (obj) {
        var div = $("<div>", {
            id: obj.id,
            class: "pica-flex-file pica-borderless text-center",
            "data-inputlist-idx": obj.idx,
            "data-userdata": JSON.stringify(obj.userdata),
            draggable: false
        });

        /* Whether to make item unselectable */
        if (true === obj.isUnselectable) {
            tr.addClass("pica-table-selection-disabled");
        }

        var p = $("<p>", {
            class: "pica-well dropdown"
        });

        /* Add icon link */
        var a = $("<a>", {
            class: "dropdown-toggle",
            "data-toggle": "dropdown"
        });

        /* Add icon, caret and name */
        a.append(Picasso.Helper.createGlyph(obj.icon, "pica-glyph-big"));

        p.append(a);


        /* Add name link */
        a = $("<a>", {
            class: "pica-name pica-line-normal pica-overflow center-block dropdown-toggle",
            "data-toggle": "dropdown",
            text: obj.name
        });

        if (0 === (obj.self._flags & Picasso.InputList.Flags.IMMUTABLE)) {
            a.append($("<span>", { class: "caret" }));
        }

        p.append(a);

        if (0 === (obj.self._flags & Picasso.InputList.Flags.IMMUTABLE)) {
            /* Add dropdown menu */
            var menu = Picasso.Combo.createDropdownMenu(obj.options,
                $.proxy(obj.handleOpts, this, obj.self), obj.startIdx);

            p.append(menu);
        }

        div.append(p);

        return div;
    };

    /**
     * Handle completion
     *
     * @param {Picasso.TagInput}  tagInput  A {@link Picasso.TagInput}
     * @param {String}            val       Value to load completions for
     **/

    var handleCompletion = function (tagInput, val) {
        if (this._onCompletion) {
            this._onCompletion(tagInput, val); ///< Just forward

            if (0 < (this._flags & Picasso.InputList.Flags.COMPLETION_NEW)) {
                this._tagInput.showCompletions();
            }
        }
    };

    /**
     * Map {@link Picasso.InputList} options to {@link Picasso.TagInput} options
     *
     * @param {Number}  options  Given options
     *
     * @returns {Number} Mapped options
     **/

    var mapFlags = function (options) {
        var flags = 0;

        if (options) {
            if (0 < (options & Picasso.InputList.Flags.COMPLETION_ONLY)) {
                flags |= Picasso.TagInput.Flags.COMPLETION_ONLY;
            }
            if (0 < (options & Picasso.InputList.Flags.COMPLETION_FILTER)) {
                flags |= Picasso.TagInput.Flags.COMPLETION_FILTER;
            }
            if (0 < (options & Picasso.InputList.Flags.VALIDATE)) {
                flags |= Picasso.TagInput.Flags.VALIDATE;
            }
        }

        return flags;
    };

    /**
     * Constructor for {@link Picasso.InputList}
     *
     * @param {String}   id       Element Id
     * @param {Number}   options  Options for this element (optional)
     *
     * @constructor
     **/

    Picasso.InputList = function (id, options) {
        var self = this;

        /* Init */
        this._id = id;
        this._container = $(id);
        this._nitems = 0;

        this._onCompletion = null;
        this._onUpdate = null;
        this._onEdit = null;
        this._onComplNew = null;
        this._isLoading = false;

        /* Options */
        this._flags = (options || 0);

        if (options && 0 < (options & Picasso.InputList.Flags.IMMUTABLE)) {
            this._container.find("div.pica-inputlist-taginput").hide();
        } else {
            this._container.find("div.pica-inputlist-taginput").show();

            /* Create tag input */
            this._tagInput = new Picasso.TagInput(
                this._container.find("div.pica-inputlist-taginput"),
                mapFlags(options));

            this._tagInput.setCompletionHandler($.proxy(handleCompletion, this));

            if (0 < (this._flags & Picasso.InputList.Flags.COMPLETION_NEW)) {
                this._tagInput.clearFixed(); ///< Remove previously added fixed items

                var li = this._tagInput.addFixedCompletionEntry(
                    Picasso.Lang.get("label_create_new"), "asterisk",
                    Picasso.Lang.get("label_create_new_description"), null, {});

                /* Overwrite handler and call the new one if any */
                li.find("a").off("click").click(function () {
                    if (self._onComplNew) {
                        self._onComplNew.apply(self);
                    }

                    /* Clear input afterwards */
                    self._tagInput.clearInput();
                });
            }
        }

        /* Create table */
        this._htmlTable = this._container.find("table.pica-inputlist-table");
        this._vmTable = new Picasso.ViewModeTable(this._htmlTable,
            ((options && 0 < (options & Picasso.InputList.Flags.GRID_MODE) ?
                Picasso.ViewModeTable.Flags.GRID_MODE :
                Picasso.ViewModeTable.Flags.LIST_MODE)|
                Picasso.ViewModeTable.Flags.NO_MODE_TOGGLE|
                Picasso.ViewModeTable.Flags.ROWS_SELECTABLE));

        this._vmTable.checkIfEmpty();

        this._vmTable.setListRenderer(renderItemList);
        this._vmTable.setGridRenderer(renderItemGrid);

        /* Bind events */
        this._container.find(".pica-inputlist-edit").off(
            "click").click($.proxy(handleClickEdit, this));
        this._container.find(".pica-inputlist-remove").off(
            "click").click($.proxy(handleClickDelete, this));
        this._container.find(".pica-inputlist-manage").off(
                "click").click($.proxy(handleClickManage, this));
    };

    /**
     * Set userdata of given item
     *
     * @param {Number}  idx       Item index
     * @param {Object}  userdata  Item userdata
     *
     * @returns {Object} Item userdata
     **/

    Picasso.InputList.prototype.setUserdataOfItem = function (idx, userdata) {
        this._container.find("[data-inputlist-idx=" + idx + "]")
            .data("userdata", userdata);

        /* Call change handler of any */
        if (this._onUpdate) {
            this._onUpdate(userdata);
        }
    };

    /**
     * Set list renderer
     *
     * @param {Function}  renderFunc  Render function for elements
     **/

    Picasso.InputList.prototype.setListRenderer = function (renderFunc) {
        if (renderFunc) {
            this._vmTable.setListRenderer(renderFunc);
        }
    };

    /**
     * Add item to list
     *
     * @param {String}    icon            Name of the icon
     * @param {String}    name            Name of the item
     * @param {String}    description     Description of the item
     * @param {Object}    userdata        Additional userdata (optional)
     * @param {Array}     options         Options array (optional)
     * @param {Function}  handleOpts      Option handler (optional)
     * @param {Number}    startIdx        Index of the visible item (optional)
     * @param {Boolean}   isUnselectable  Whether item is not selectable (optional)
     * @param {Boolean}   hideCombo       Whether combo should be hidden (optional)
     *
     * @returns {String}  Id of newly added item
     **/

    Picasso.InputList.prototype.addItem = function (icon, name, description, userdata,
                                                    options, handleOpts, startIdx,
                                                    isUnselectable,tooltip,hideCombo){
        /* Check whether this item already exists */
        if (userdata && userdata.oid){
            var id = "item_" + Picasso.Helper.hashCode(name + userdata.oid + this._id);
        }else {
            var id = "item_" + Picasso.Helper.hashCode(name + this._id);
        }

        if (0 === (this._flags & Picasso.InputList.Flags.ALLOW_DUPLICATE) &&
            0 !== $("#" + id).length)
        {
            return null;
        }

        /* Call table renderer */
        var data = {
            self: this, ///< Keep instance reference here
            id: id,
            idx: this._nitems++,
            icon: icon,
            name: name,
            description: description,
            userdata: userdata,
            options: (options || defaultOptionsForDropdowns),
            handleOpts: function () {
                return handleDefaultOpts.apply(this,
                    $.merge(arguments, [ handleOpts ]));
            },
            startIdx: (startIdx || 0),
            isUnselectable: isUnselectable,
            tooltip: tooltip,
            hideCombo: hideCombo
        };

        this._vmTable.append(data);

        /* Call change handler of any */
        if (this._onUpdate) {
            this._onUpdate(userdata);
        }

        return id;
    };

    /**
     * Remove element from list
     *
     * @param {Number}  idx  Index to remove
     *
     * @returns {Boolean} Either true on success; otherwise false
     **/

    Picasso.InputList.prototype.removeItem = function (idx) {
        var self = this;
        var victim = this._container.find("[data-inputlist-idx=" + idx + "]");

        if (0 < victim.length) {
            /* Update table cache */
            var cache = self._vmTable.getCache();

            cache = $.grep(cache, function (v) {
                return v.idx != idx;
            });

            self._vmTable.setCache(cache);

            /* Now fade out */
            Picasso.Helper.fadeOut(victim,
                function () {
                    this.remove();

                    self._vmTable.checkIfEmpty();

                    /* Call change handler if any */
                    if (self._onUpdate) {
                        self._onUpdate(victim.data("userdata"));
                    }
                });

            return true;
        }

        return false;
    };

    /**
     * Get items from table
     *
     * @returns {Array} Array with userdata
     **/

    Picasso.InputList.prototype.getItems = function () {
        var elems = this._htmlTable.find("[data-userdata]");
        var items = [];

        /* Collect data */
        $.each(elems, function () {
            var userdata = $(this).data("userdata");

            if (userdata) {
                items.push(userdata);
            }
        });

        return items;
    };

    /**
     * Disable list and show message. E.g. used for displaying 'Loading' progress.
     *
     * @param showOption Either a loading spinner should be displayed or not.
     * @returns {boolean}
     */

    Picasso.InputList.prototype.showLoadingSpinner = function (showOption) {

        var self = this;
        var table = $(self._htmlTable[0]);
        var container = self._container;

        var spinner = Picasso.Helper.createGlyph("repeat",
            "pica-spinner pica-inputlist-loading-spinner");
        var message = $("<div>", {
            class: "pica-inputlist-loading-message",
            text: "Loading "
        });

        if (showOption) {
            message.append(spinner);
            container.append(message);
            table.hide();
            self._isLoading = true;
        } else {
            container.find(".pica-inputlist-loading-message").remove();
            table.show();
            self._isLoading = false;
        }
    };

    /**
     * Either this account is completely loaded or not.
     *
     * @returns {boolean}
     */

    Picasso.InputList.prototype.isLoading = function () {

        return this._isLoading;
    };

    /**
     * Check whether input list is empty
     *
     * @returns {Boolean} Either {@code true} when not empty; otherwise {@code false}
     **/

    Picasso.InputList.prototype.isEmpty = function () {
        return (0 === this._htmlTable.find("[data-userdata]").length);
    };

    /**
     * Get underlying table
     *
     * @return {Picasso.Table} Underlying #{@link Picasso.Table}
     **/

    Picasso.InputList.prototype.getTable = function () {
        return this._vmTable.getTable();
    };

    /**
     * Get value of input field
     *
     * @returns {String} Value of input field
     **/

    Picasso.InputList.prototype.getInputValue = function () {
        return (this._tagInput ? this._tagInput.getInputValue() : "");
    };

    /**
     * Clear input list
     **/

    Picasso.InputList.prototype.clear = function () {

        /* Clear table */
        this._vmTable.clear();

        /* Clear taginput */
        if (this._tagInput) {
            this._tagInput.clear();
            this._tagInput.clearFixed();
            this._tagInput.clearCompletions();
        }
    };

    /**
     * Set completion handler
     *
     * @param {Function}  onCompletion  Completion handler
     **/

    Picasso.InputList.prototype.setCompletionHandler = function (onCompletion) {
        if (onCompletion) {
            this._onCompletion = onCompletion;
        }
    };

    /**
     * Set input validation handler
     *
     * @param {Function}  onCompletion  input validation handler
     **/

    Picasso.InputList.prototype.setInputValidationHandler = function (onValidation) {
        if (onValidation) {
            this._tagInput.setValidationHandler(onValidation);
        }
    };

    /**
     * Set completion select handler
     *
     * @param {Function}  onComplSelect  Completion select handler
     **/

    Picasso.InputList.prototype.setCompletionSelectHandler = function (onComplSelect) {
        if (onComplSelect && this._tagInput) {
            this._tagInput.setCompletionSelectHandler(
                $.proxy(onComplSelect, this));
        }
    };

    /**
     * Set completion new handler
     *
     * @param {Function}  onComplNew  Remove handler
     **/

    Picasso.InputList.prototype.setCompletionNewHandler = function (onComplNew) {
        if (onComplNew) {
            this._onComplNew = onComplNew;
        }
    };

    /**
     * Set submit handler
     *
     * @param {Function}  onSubmit  Submit handler
     **/

    Picasso.InputList.prototype.setSubmitHandler = function (onSubmit) {
        if (onSubmit && this._tagInput) {
            this._tagInput.setSubmitHandler(
                $.proxy(onSubmit, this));
        }
    };

    /**
     * Set validation error handler
     *
     * @param {Function}  onValidation  Validation handler
     **/

    Picasso.InputList.prototype.setValidationErrorHandler = function (onValidation) {
        if (onValidation && this._tagInput) {
            this._tagInput.setValidationErrorHandler(onValidation);
        }
    };

    /**
     * Set update handler
     *
     * @param {Function}  onUpdate  Update handler
     **/

    Picasso.InputList.prototype.setUpdateHandler = function (onUpdate) {
        if (onUpdate) {
            this._onUpdate = onUpdate;
        }
    };

    /**
     * Set edit handler
     *
     * @param {Function}  onEdit  Edit handler
     **/

    Picasso.InputList.prototype.setEditHandler = function (onEdit) {
        if (onEdit) {
            this._onEdit = onEdit;
        }
    };

    /* Flags - must be defined after constructor */

    if (!window.Picasso.InputList.hasOwnProperty("Flags")) {
        window.Picasso.InputList.Flags = {};
    }

    /* Flags */
    Picasso.InputList.Flags.LIST_MODE = (1 << 0); ///< Table uses list mode
    Picasso.InputList.Flags.GRID_MODE = (1 << 1); ///< Table uses gallery mode
    Picasso.InputList.Flags.VALIDATE = (1 << 2); ///< Validate input
    Picasso.InputList.Flags.COMPLETION_ONLY = (1 << 3); ///< Accept only completed inputs
    Picasso.InputList.Flags.COMPLETION_NEW = (1 << 4); ///< Ask when a new entry should be created
    Picasso.InputList.Flags.COMPLETION_FILTER = (1 << 5); ///< Reverse logic and filter set
    Picasso.InputList.Flags.IMMUTABLE = (1 << 6); ///< List cannot be changed
    Picasso.InputList.Flags.ALLOW_DUPLICATE = (1 << 7); ///< Allow duplicate entries

    /* Fix dropdowns in scroller */
    Picasso.Combo.fixDropdownInScroller(".pica-inputlist-scroller");
})();
