/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/**
 * Some older browsers like IE11 lack support for CSS vars, so to get it
 * working we:
 *
 * 1) Check the capabilities of the browser
 * 2) Parse the vars
 * 3) Reload all CSS files via AJAX
 * 4) Search/replace each occurrence
 * 5) Re-apply the CSS files as new styyle element
 **/

/* Private scope */
(function () {

    /**
     * Constructor for {@link Picasso.CssPolyfill}
     *
     * @constructor
     */

    Picasso.CssPolyfill = function () {
        /* Check for CSS var support */
        if (window.CSS && window.CSS.supports &&
            window.CSS.supports("(--foo: red)"))
        {
            return;
        }

        /* Init */
        this.updatedVars = {};
        this.varsByBlock = {};
        this.oldCss = {};

        this.findCss();
        this.updateCss();
    };

    /**
     * Find style tags
     **/

    Picasso.CssPolyfill.prototype.findCss = function () {
        var self = this;
        var counter = 1;
        var styleBlocks = document.querySelectorAll('link[rel="stylesheet"]');

        [].forEach.call(styleBlocks, function (block) {
            if (block.nodeName === "LINK") {
                self.loadCss(block.getAttribute("href"), counter,
                    function (counter, request) {
                        self.findVars(request.responseText, counter);

                        self.oldCss[counter] = request.responseText;

                        self.updateCss();
                    }
                );
            }
        });
    };

    /**
     * Find CSS variables
     *
     * @param css
     * @param counter
     */

    Picasso.CssPolyfill.prototype.findVars = function (css, counter) {
        this.varsByBlock[counter] = css.match(/(--.+: .+;)/g) || [];
    };

    /**
     * Update CSS
     **/

    Picasso.CssPolyfill.prototype.updateCss = function () {
        var self = this;
        this.updateVars(this.varsByBlock);

        for (var cur in this.oldCss) {
            var newCss = self.replaceVars(
                self.oldCss[cur], this.updatedVars);

            /* Append new style */
            var style = document.createElement("style");

            style.type = "text/css";
            style.innerHTML = newCss;
            style.classList.add("inserted");
            style.id = "inserted" + cur;

            document.getElementsByTagName("head")[0].appendChild(style);
        }
    };

    /**
     * Replace CSS vars in file
     *
     * @param {String}  css        Content of current CSS file
     * @param {Array}   variables  Variable list
     *
     * @returns {String} Updated CSS file content
     **/

    Picasso.CssPolyfill.prototype.replaceVars = function (css, variables) {
        for (var variable in variables) {
            var regex = new RegExp("var\\(" + variable + "\\)", "g");

            css = css.replace(regex, variables[variable]);
        }

        return css;
    };

    /**
     * Split variable name and value
     *
     * @param {Array}  variables  Variable list
     **/

    Picasso.CssPolyfill.prototype.updateVars = function (variables) {
        var self = this;

        for (var variable in variables) {
            var cur = variables[variable];

            cur.forEach(function (variable) {
                var matches = variable.split(/:\s*/);

                self.updatedVars[matches[0]] = matches[1].replace(/;/, "");
            });
        }
    };

    /**
     * Load CSS file
     *
     * @param {String}    url        Url to load
     * @param {Number}    counter    File counter
     * @param {Function}  onSuccess  Success callback
     **/

    Picasso.CssPolyfill.prototype.loadCss = function (url, counter, onSuccess) {
        var request = new XMLHttpRequest();

        request.open("GET", url, true);
        request.overrideMimeType("text/css;");

        request.onload = function () {
            if (request.status >= 200 && request.status < 400) {
                if (onSuccess) {
                    onSuccess(counter, request);
                }
            }
        };

        request.send();
    };

    /* Init */
    new Picasso.CssPolyfill();
})();