/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Dialog")) {
    window.Picasso.Dialog = {};
}

/* Private scope */
(function () {

    /* Options for dropdown menus */
    var optionsForGroups = [
        {
            text: Picasso.Lang.get("permission_is_member"),
            type: "radio",
            group: "permission"
        },
        {
            text: Picasso.Lang.get("permission_is_member_and_admin"),
            type: "radio",
            group: "permission"
        },
        null,
        { text: Picasso.Lang.get("action_remove"), icon: "bin" }
    ];

    /* Options for dropdown menus */
    var optionsForOrganizations = [
        {
            text: Picasso.Lang.get("permission_is_member"),
            type: "checkbox",
            group: "permission"
        },
        null,
        {
            text: Picasso.Lang.get("permission_is_admin"),
            type: "checkbox",
            group: "permission"
        },
        null,
        { text: Picasso.Lang.get("action_remove"), icon: "bin" }
    ];

    /* Options for dropdown menus */
    var optionsForFolders = [
        {
            text: Picasso.Lang.get("permission_can_read"),
            type: "radio",
            group: "permission"
        },
        {
            text: Picasso.Lang.get("permission_can_read_write"),
            type: "radio",
            group: "permission"
        },
        {
            text: Picasso.Lang.get("permission_can_admin"),
            type: "radio",
            group: "permission"
        },
        null,
        {
            text: Picasso.Lang.get("permission_is_owner"),
            type: "radio",
            group: "permission"
        },
        null,
        { text: Picasso.Lang.get("action_remove"), icon: "bin" }
    ];

    /**
     * Handle options for groups
     *
     * @param {Picasso.InputList}  inputList  A #{@link Picasso.InputList}
     * @param {Number}             optIdx     Selected index
     * @param {Number}             idx        Item index of selected row
     * @param {Object}             userdata   Additional userdata (optional)
     **/

    var handleOptionsForGroups = function (inputList, optIdx, idx, userdata) {
        switch(optIdx) {
            case 0: ///< Member
            /* Falls through */
            case 1: ///< Admin

                inputList.setUserdataOfItem(idx,
                    $.extend(userdata, {
                            isGroupAdmin: (1 === optIdx)
                        }
                    ));
                return true;

            case 3:
                inputList.removeItem(idx);
                break;
        }
    };

    /**
     * Handle options for folders
     *
     * @param {Picasso.InputList}  inputList  A #{@link Picasso.InputList}
     * @param {Number}             optIdx     Selected index
     * @param {Number}             idx        Item index of selected row
     * @param {Object}             userdata   Additional userdata (optional)
     **/

    var handleOptionsForFolders = function (inputList, optIdx, idx, userdata) {
        switch (optIdx) {
            case 0:
                /* Falls through */
            case 1:
                /* Falls through */
            case 2:
                /* Falls through */
            case 4:
                var permission;

                switch(optIdx) {
                    case 0: ///< Read
                        permission = Picasso.Permission.READ;
                        break;

                    case 1: ///< Read/Write
                        permission = Picasso.Permission.READ_WRITE;
                        break;

                    case 2: ///< Admin
                        permission = Picasso.Permission.ADMIN;
                        break;

                    case 4: ///< Owner
                        permission = Picasso.Permission.OWNER;
                        break;
                }

                inputList.setUserdataOfItem(idx,
                    $.extend(userdata, {
                            permission: permission
                        }
                    ));
                return true;

            case 6:
                inputList.removeItem(idx);
                break;
        }
    };

    /**
     * Update storage progress bars
     *
     * @param {Picasso.Dialog.Account}  dialog  A #{@link Picasso.Dialog.Account}
     **/

    var updateProgress = function (dialog) {
        var progress = $("#pica_account_total_storage");

        var quotaOld = 0;
        var quotaNew;
        var quotaTotal = 0;
        var quotaAddedAfterSave = 0;
        var quotaFreeAfterSave = 0;
        var percentQuotaOld;
        var percentQuotaAddedAfterSave = 0;
        var percentQuotaFreeAfterSave = 0;

        var caption;

        /* Show progress when organization list is not empty */
        if (!dialog._organizations_inputList.isEmpty()) {
            progress.show();

            /* Get storage */
            var items = dialog._organizations_inputList.getItems();

            $.each(items, function () {
                if (true === this.isMember) {
                    quotaOld = this.quotaUsed;
                    quotaTotal = this.quotaTotal;
                }
            });
        } else {
            progress.hide();

            return;
        }

        quotaNew = Math.round(Picasso.Helper.parseNumber(($("#pica_account_storage").val() || 0), 0) * 1024 * 1024 * 1024);
        percentQuotaOld = (0 < quotaTotal ? Math.floor(quotaOld * 100 / quotaTotal) : 0);
        if (quotaOld > quotaNew) {
            quotaAddedAfterSave = (quotaNew - quotaOld);
            quotaFreeAfterSave = (quotaOld - quotaNew);
            percentQuotaFreeAfterSave = (0 < quotaTotal ? (quotaFreeAfterSave * 100 / quotaTotal) : 0);
            percentQuotaOld -= percentQuotaFreeAfterSave;
            caption = Picasso.Lang.get("string_storage_used", Math.round(percentQuotaOld));
        } else if (quotaOld < quotaNew) {
            quotaAddedAfterSave = (quotaNew - quotaOld);
            percentQuotaAddedAfterSave = (0 < quotaTotal ? (quotaAddedAfterSave * 100 / quotaTotal) : 0);
            caption = Picasso.Lang.get("string_storage_used", Math.round(percentQuotaOld + percentQuotaAddedAfterSave));
        } else {
            percentQuotaAddedAfterSave = 0;
            percentQuotaFreeAfterSave = 0;
            caption = Picasso.Lang.get("string_storage_used", Math.round(percentQuotaOld));
        }

        /* Update progress */
        Picasso.Helper.createQuotaPopover(progress,
            quotaOld, quotaTotal, quotaAddedAfterSave, quotaFreeAfterSave,
            percentQuotaOld, percentQuotaAddedAfterSave, percentQuotaFreeAfterSave, caption);
    };

    /**
     * Update organisation permissions
     *
     * @param {Picasso.InputList}  inputList  A #{@link Picasso.InputList}
     * @param {string}  memeberOfAny
     **/

    var updatePermissions = function (inputList,memeberOfAny) {


        let items =inputList.getItems();
        inputList.clear()
        $.each(items, function () {
            delete optionsForOrganizations[0]["disabled"] ;
            if (memeberOfAny && !this.isMember ) {
                optionsForOrganizations[0]["disabled"] = true;
            }

            var description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                    "string_remaining_storage",
                    Picasso.Helper.formatSize(this.quotaUsed),
                    Picasso.Helper.formatSize(this.quotaTotal));


            var id = inputList.addItem(
                    "education", this.name, description,
                    this, optionsForOrganizations,
                    handleOptionsForOrganizations
            );

            initOptionsForOrganizations(id, this);
        });
    };

    /**
     * Update password field based on authentication type
     *
     * @param {Object}  inputs    Input fields ({ "id" = HTMLElement })
     * @param {String}  authType  Authentication type
     **/

    var updateInputsForAuthType = function (inputs, authType) {
        if (authType && "database" !== authType) {
            /* Disable input and update placeholder */
            inputs.pica_account_password.attr("disabled", true);
            inputs.pica_account_password.attr("placeholder",
                Picasso.Lang.get("string_authed_by", authType));

            /* Add tooltip */
            var div = inputs.pica_account_password.parent();

            Picasso.Helper.addTooltip(div,
                Picasso.Lang.get("tooltip_not_authed_by_db"));
            div.tooltip("enable");
        } else {
            inputs.pica_account_password.removeAttr("disabled");
            inputs.pica_account_password.attr("placeholder",
                Picasso.Lang.get("placeholder_password"));
            inputs.pica_account_password.parent().tooltip("disable");
        }
    };

    var updateDateInputs = function (inputs) {
        inputs.pica_account_creation_date.prop("disabled",true);
        Picasso.Helper.setDateTimePickerValue(inputs.pica_account_creation_date,new Date());

        inputs.pica_account_last_login.prop("disabled",true);
        inputs.pica_account_last_login.val(Picasso.Lang.get("placeholder_never"))
    }

    /**
     * Get proper dropdown caption
     *
     * @param {Object}  userdata  Userdata of current item
     *
     * @returns {String} Selected caption
     **/

    var getCaption = function (userdata) {
        if (userdata.isMember && userdata.isOrgAdmin) {
            return Picasso.Lang.get("permission_is_member_and_admin");
        } else if (userdata.isMember) {
            return Picasso.Lang.get("permission_is_member");
        } else if (userdata.isOrgAdmin) {
            return Picasso.Lang.get("permission_is_admin");
        }
    };

    /**
     * Init options
     *
     * @param {String}  id        Element Id
     * @param {Object}  userdata  Supplied userdata
     **/

    var initOptionsForOrganizations = function (id, userdata) {
        /* FIXME: Init dropdown */
        var dropdown = $("#" + id + " .dropdown");

        var toggle = dropdown.find(".dropdown-toggle");

        toggle.text(getCaption(userdata));
        toggle.append($("<span>", { class: "caret" }));

        var spans = dropdown.find("li span");

        if (userdata.isMember) {
            $(spans[0]).parent().hide();
            $(spans[2]).parent().hide();
        } else {
            $(spans[0]).removeClass("glyphicons-check")
                .addClass("glyphicons-unchecked");
        }

        if (userdata.isOrgAdmin) {
            $(spans[1]).removeClass("glyphicons-unchecked")
                    .addClass("glyphicons-check");

            /* Prevent (org-)admin from revoking his own admin permission */
            if (Picasso.get("account").is(Picasso.Account.Flags.ORG_ADMIN) && Picasso.get("account").getOid() === userdata.userID) {
                $(spans[2]).parent().parent("li").prev("li").remove();
            }

        } else {
            $(spans[1]).removeClass("glyphicons-check").addClass("glyphicons-unchecked");
        }

        var actionableLinks = dropdown.find("li a").filter(function() {
            return $(this).css("display") !== "none";
        }).length;

        if (actionableLinks === 0) {
            toggle.prop("disabled", true);
        } else {
            toggle.prop("disabled", false);
        }
    };

    /**
     * Handle options for organizations
     *
     * @param {Picasso.InputList}  inputList  A #{@link Picasso.InputList}
     * @param {Number}             optIdx     Selected index
     * @param {Number}             idx        Item index of selected row
     * @param {Object}             userdata   Additional userdata (optional)
     **/

    var handleOptionsForOrganizations = function (inputList, optIdx, idx, userdata) {
        switch(optIdx) {
            case 0: ///< Member
                /* Sanity check: One must be checked */
                if (userdata.isMember && !userdata.isOrgAdmin) {
                    return;
                }

                var newOrgId ;
                if (userdata.memberOfAny && userdata.isMember ) {
                     newOrgId = $.extend(userdata, {
                         memberOfAny: null///< remove

                    });
                }

                var newUserdata = $.extend(userdata, {
                    isMember: !userdata.isMember, ///< Toggle

                });

                if (newUserdata.isMember) {
                    newOrgId = $.extend(newUserdata, {
                        memberOfAny: newUserdata.oid///< remove

                    });
                }
                delete optionsForOrganizations[0]["disabled"];

                inputList.setUserdataOfItem(idx, newUserdata);

                updatePermissions(inputList,newUserdata.memberOfAny);


                return getCaption(newUserdata);

            case 2: ///< Admin
                /* Sanity check: One must be checked */
                if (!userdata.isMember && userdata.isOrgAdmin) {
                    return;
                }

                var newUserdata = $.extend(userdata, {
                    isOrgAdmin: !userdata.isOrgAdmin
                });


                inputList.setUserdataOfItem(idx, newUserdata);


                return getCaption(newUserdata);

            case 4:
                inputList.removeItem(idx);
                break;
        }
    };

    /**
     * Copy data from inputs to account
     *
     * @param {Picasso.Dialog.Account}  dialog  A #{@link Picasso.Dialog.Account}
     * @param {Object}                  inputs  Input fields ({ "id" = HTMLElement })
     **/

    var copyData = function (dialog, inputs) {
        dialog._account.setUserName(inputs.pica_account_username.val());
        dialog._account.setPassword(inputs.pica_account_password.val());
        dialog._account.setFirstName(inputs.pica_account_first_name.val());
        dialog._account.setLastName(inputs.pica_account_last_name.val());
        dialog._account.setPhone(inputs.pica_account_phone.val());
        dialog._account.setNotes(inputs.pica_account_notes.val());

        /* Valid till */
        if (!dialog._account.getOid() || !Picasso.get("account").isMe(dialog._account.getOid())) {
        dialog._account.setValidTill(Picasso.Helper.getDateTimePickerValue(
                inputs.pica_account_valid_till, -1));
         }

        /*Account Language  */
        if ($("#account_language").val()!="") {
            dialog._account["set" + Picasso.Helper.capitalize("language")]($("#account_language").val());
        }
        /* Storage */
        var val = inputs.pica_account_storage.val();

        if (-1 !== val) {
            val = Picasso.Helper.parseNumber(val, 0);

            dialog._account.setQuotaTotal(Math.round(val * 1024 * 1024 * 1024));
        }

        /* Max Folders */
        if(inputs.pica_account_max_folders) {
            var val = inputs.pica_account_max_folders.val();

            if (val>=-1) {
                val = Picasso.Helper.parseNumber(val, -1);
                dialog._account.setMaxFolders(val);
            }
        }

        /* Flags */
        if (inputs.hasOwnProperty("pica_account_server_admin")) { ///< Disabled for orgAdmins
            if (inputs.pica_account_server_admin.is(":checked")) {
                dialog._account._flags |= Picasso.Account.Flags.ADMIN;
            } else {
                dialog._account._flags &= ~Picasso.Account.Flags.ADMIN;
            }
        }

        if (inputs.hasOwnProperty("pica_account_reseller")) { ///< Disabled for orgAdmins
            if (inputs.pica_account_reseller.is(":checked")) {
                dialog._account._flags |= Picasso.Account.Flags.ORG_CREATOR;
            } else {
                dialog._account._flags &= ~Picasso.Account.Flags.ORG_CREATOR;
            }
        }

        if (inputs.hasOwnProperty("pica_account_active")) {
            if (inputs.pica_account_active.is(":checked")) {
                dialog._account._flags |= Picasso.Account.Flags.ACTIVE;
            } else {
                dialog._account._flags &= ~Picasso.Account.Flags.ACTIVE;
            }
        }
    };

    /**
     * Assign data to inputs
     *
     * @param {Picasso.Dialog.Account}  dialog  A #{@link Picasso.Dialog.Account}
     * @param {Object}                  inputs   Input fields ({ "id" = HTMLElement })
     **/

    var assignData = function (dialog, inputs) {

        inputs.pica_account_username.val(dialog._account.getUserName());
        inputs.pica_account_first_name.val(dialog._account.getFirstName());
        inputs.pica_account_last_name.val(dialog._account.getLastName());
        inputs.pica_account_phone.val(dialog._account.getPhone());
        inputs.pica_account_storage.val(
            Math.round(dialog._account.getQuotaTotal() / 1024 / 1024 / 1024));
        if (inputs.pica_account_max_folders) {
            if (dialog._account.getMaxFolders() >-1){
                inputs.pica_account_max_folders.val(dialog._account.getMaxFolders());
            }else{
                inputs.pica_account_max_folders.attr("placeholder",Picasso.Lang.get("placeholder_unlimited"))
            }


        }
        inputs.pica_account_notes.val(dialog._account.getNotes());

        inputs.pica_account_last_login.val(Picasso.Lang.get("placeholder_never"))

        /* Just trigger/hide the checkboxes */
        if (dialog._account.is(Picasso.Account.Flags.ADMIN)) {
            if (inputs.hasOwnProperty("pica_account_server_admin")) { ///< Disabled for orgAdmins
                inputs.pica_account_server_admin.trigger("click");
            }
        }

        if (dialog._account.is(Picasso.Account.Flags.ORG_CREATOR)) {
            if (inputs.hasOwnProperty("pica_account_reseller")) { ///< Disabled for orgAdmins
                inputs.pica_account_reseller.trigger("click");
            }
        }

        if (dialog._account.is(Picasso.Account.Flags.ACTIVE)) {
            if (inputs.hasOwnProperty("pica_account_active")) { ///< Disabled for orgAdmins
                inputs.pica_account_active.trigger("click");
            }
        }

        if (!dialog._account.getTwoFactorAuthenticationEnabled()) {
            $("#pica_2fa_reset").closest(".row").hide();
        }

        /* Prevent logged-in user being able to withdraw certain permissions from himself */
        if (Picasso.get("account").isMe(dialog._account.getOid())) {
            if (inputs.hasOwnProperty("pica_account_active")) {
                inputs.pica_account_active.parent(".pica-button-checkbox").hide();
            }
            if (inputs.hasOwnProperty("pica_account_reseller")) {
                inputs.pica_account_reseller.parent(".pica-button-checkbox").hide();
            }
            if (inputs.hasOwnProperty("pica_account_server_admin")) {
                inputs.pica_account_server_admin.parent(".pica-button-checkbox").hide();
            }
        } else {
            if (inputs.hasOwnProperty("pica_account_active")) {
                inputs.pica_account_active.parent(".pica-button-checkbox").show();
            }
            if (dialog._account.getTwoFactorAuthenticationEnabled()) {
                $("#pica_2fa_reset").closest(".row").show();
            }
            if (inputs.hasOwnProperty("pica_account_reseller")) {
                inputs.pica_account_reseller.parent(".pica-button-checkbox").show();
            }
            if (inputs.hasOwnProperty("pica_account_server_admin")) {
                inputs.pica_account_server_admin.parent(".pica-button-checkbox").show();
            }
        }

        /* Valid till */
        if (-1 !== dialog._account.getValidTill()) {
            Picasso.Helper.setDateTimePickerValue(inputs.pica_account_valid_till,
                dialog._account.getValidTill());
        }

        /* Account Language */
        if (dialog._account.getLanguage() != "") {
            $("#account_language").val(dialog._account["get" + Picasso.Helper.capitalize("language")]);
        }


        /* last login date */
        inputs.pica_account_last_login.prop("disabled",true);
        if (-1 !== dialog._account.getLastLogin()) {
            Picasso.Helper.setDateTimePickerValue(inputs.pica_account_last_login,
                    dialog._account.getLastLogin());
        }

        /* creation date  */
        inputs.pica_account_creation_date.prop("disabled",true);
        if (-1 !== dialog._account.getCreationDate()) {
            Picasso.Helper.setDateTimePickerValue(inputs.pica_account_creation_date,
                    dialog._account.getCreationDate());
        }

        /* Set avatar */
        var img = $("#pica_account_avatar");

        img.off("load").on("load", function () {
            var that = $(this);

            that.siblings().hide();
            that.show();
        });

        img.off("error").on("error", function () {
            var that = $(this);

            that.siblings().show();
            that.hide();
        });

        img.attr("src", dialog._account.getAvatarUrl());

        /* Avatar buttons */
        $("#pica_account_avatar_change").off(
            "click").click(function () {
            Picasso.Dialog.Upload.changeAvatar(dialog._account);
        });

        $("#pica_account_loginas").click(function () {
            document.location = Picasso.VARS.baseUrl + "/login?loginAs=" + dialog._account.getUserName();
        });

        $("#pica_account_avatar_delete").off("click").click(function () {
            Picasso.Dialog.Upload.deleteAvatar(dialog._account);
        });

        /* Disable save button until everything is loaded */
        var accountEditSaveButton = $("#pica-account-edit-save");
        accountEditSaveButton.hide();

        /* Enable the save button for account edit dialog */
        var enableSaveButton = function (state) {
            var accountEditSaveButton = $("#pica-account-edit-save");
            if (state) {
                accountEditSaveButton.show();
            } else {
                accountEditSaveButton.hide();
            }
        };

        /* Check if an account is completely loaded (folders, groups, organizations, emails) */
        var checkAccountCompletelyLoaded = function () {

            if (dialog._organizations_inputList.isLoading() ||
                    dialog._folders_inputList.isLoading() ||
                    dialog._groups_inputList.isLoading() ||
                    dialog._mails_inputList.isLoading()) {
                return false;
            }

            enableSaveButton(true);
        };

        /* Load emails */
        dialog._mails_inputList.showLoadingSpinner(true);

        dialog._account.getEmails(

            /* Success */
            function (mails) {
                $.each(mails, function () {
                    dialog._mails_inputList.addItem(
                        "envelope", this, null, {
                            oid: null,
                            name: this
                        }
                    )
                });
                dialog._mails_inputList.showLoadingSpinner(false);
                checkAccountCompletelyLoaded();
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
                enableSaveButton(false);
            }
        );

        /* Load folders */
        dialog._folders_inputList.showLoadingSpinner(true);

        dialog._account.getFolders(

            /* Success */
            function (files) {

                $.each(files, function () {

                    /* Find index of selected permission */
                    var idx = 0;

                    switch (this.getPermission()) {
                        case Picasso.Permission.READ:
                            idx = 0;
                            break;

                        case Picasso.Permission.READ_WRITE:
                            idx = 1;
                            break;

                        case Picasso.Permission.ADMIN:
                            idx = 2;
                            break;

                        case Picasso.Permission.OWNER:
                            idx = 4;
                            break;
                    }

                    var hideCombo = (Picasso.get("account").isMe(dialog._account.getOid()) || this.getPermission() == Picasso.Permission.OWNER )
                        dialog._folders_inputList.addItem(
                                "folder-closed", this.getName(),
                                this.getDescription(), {
                                    oid: this.getOid(),
                                    name: this.getName(),
                                    permission: this.getPermission(),
                                    folder: this
                                }, optionsForFolders, handleOptionsForFolders, idx,null,null,hideCombo
                        );
                   // }
                });

                /* Handling for complete loaded state of account */
                dialog._folders_inputList.showLoadingSpinner(false);
                checkAccountCompletelyLoaded();
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
                enableSaveButton(false);
            }
        );

        /* Load groups */
        dialog._groups_inputList.showLoadingSpinner(true);

        dialog._account.getGroups(

            /* Success */
            function (groups) {
                $.each(groups, function () {
                    dialog._groups_inputList.addItem(
                        "group", this.getName(), null, {
                            oid: this.getOid(),
                            name: this.getName(),
                            isGroupAdmin: this.is(Picasso.Group.Flags.GROUP_ADMIN)
                        }, optionsForGroups, handleOptionsForGroups,
                        (this.is(Picasso.Group.Flags.GROUP_ADMIN) ? 1 : 0)
                    )
                });

                /* Handling for complete loaded state of account */
                dialog._groups_inputList.showLoadingSpinner(false);
                checkAccountCompletelyLoaded();
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
                enableSaveButton(false);
            }
        );

        /* Load organizations */
        dialog._organizations_inputList.showLoadingSpinner(true);

        dialog._account.getOrganizations(

            /* Success */
            function (organizations) {
                $.each(organizations, function () {
                    delete optionsForOrganizations[0]["disabled"];
                    delete optionsForOrganizations[4]["disabled"];

                    var userdata = {
                        oid: this.getOid(),
                        name: this.getName(),
                        quotaUsed: this.getQuotaUsed(),
                        memberOfAny:dialog._account.getOrgOid(),
                        userID:dialog._account.getOid(),
                        quotaTotal: this.getQuotaTotal(),
                        isMember:
                            (this.getOid() === dialog._account.getOrgOid()),
                        isOrgAdmin: this.is(Picasso.Organization.Flags.ORG_ADMIN)
                    };

                    if (dialog._account.getOrgOid() &&!userdata.isMember ) {
                        optionsForOrganizations[0]["disabled"] = true;
                    }
                    if (userdata.isMember ){
                        optionsForOrganizations[4]["disabled"] = true;
                    }

                    var description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                        "string_remaining_storage",
                        Picasso.Helper.formatSize(this.getQuotaUsed()),
                        Picasso.Helper.formatSize(this.getQuotaTotal()));

                    /* Prevent (org-)admin from revoking his own admin permission */
                    if (!Picasso.get("account").is(Picasso.Account.Flags.ADMIN) && (
                        dialog._account.is(Picasso.Account.Flags.ORG_CREATOR) ||
                        this.is(Picasso.Organization.Flags.ORG_ADMIN))) {
                        delete optionsForOrganizations[0]["disabled"];
                        if (Picasso.get("account").getOid() === dialog._account.getOid()) {
                            optionsForOrganizations[2]["disabled"] = true;
                        }
                    }

                    var id = dialog._organizations_inputList.addItem(
                        this.getIcon(), this.getName(), description,
                        userdata, optionsForOrganizations,
                        handleOptionsForOrganizations,null,userdata.isMember
                    );

                    initOptionsForOrganizations(id, userdata);
                });

                /* Handling for complete loaded state of account */
                dialog._organizations_inputList.showLoadingSpinner(false);
                checkAccountCompletelyLoaded();
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
                enableSaveButton(false);
            }
        );

        if (dialog._account.is(Picasso.Account.Flags.ORG_CREATOR) && !dialog._account.is(Picasso.Account.Flags.ADMIN)) {
            $("#customer_tab").show();
            Picasso.Customer.getCustomer(dialog._account.getOid(),
                function (json) {
                    var customer = new Picasso.Customer(json);
                    Picasso.set("customer",customer);
                    renderCustomer(customer);

                }, function (data, status) {
                    switch (status) {
                        case 412: // Not enabled
                            $("#customer_tab").hide();
                            break;
                        case 404: // Customer data not found - silently ignore
                            break;
                    }
                    var customer =new Picasso.Customer()
                    customer.setAccountOid(dialog._account.getOid());
                    Picasso.set("customer",customer);
                }
            );
        } else {
            $("#customer_tab").hide();
        }

    };

    var renderCustomer = function (customer) {
        $("#pica_customer_id").val(customer.getId())
        $("#pica_customer_email").val(customer.getEmail())
        $("#pica_customer_name").val(customer.getName())
        $("#pica_customer_phone").val(customer.getPhone())
        $("#pica_customer_address_line1").val(customer.getAddressLine1())
        $("#pica_customer_address_line2").val(customer.getAddressLine2())
        $("#pica_customer_address_postal_code").val(customer.getAddressPostalCode())
        $("#pica_customer_address_city").val(customer.getAddressCity())
        $("#pica_customer_address_country").val(customer.getAddressCountry())
        $("#pica_customer_vat_id").val(customer.getVatId())
    }
    var assignCustomerData = function (accountOID, customer){
        customer.setAccountOID(accountOID)
        customer.setId($("#pica_customer_id").val())
        customer.setEmail($("#pica_customer_email").val())
        customer.setName($("#pica_customer_name").val())
        customer.setPhone($("#pica_customer_phone").val())
        customer.setAddressLine1($("#pica_customer_address_line1").val())
        customer.setAddressLine2($("#pica_customer_address_line2").val())
        customer.setAddressPostalCode($("#pica_customer_address_postal_code").val())
        customer.setAddressCity($("#pica_customer_address_city").val())
        customer.setAddressCountry($("#pica_customer_address_country").val())
        customer.setVatId($("#pica_customer_vat_id").val())
    }

    var loadFolders = function (dialog){
        /* Load folders */
        dialog._folders_inputList.clear();

        dialog._folders_inputList.showLoadingSpinner(true);

        dialog._account.getFolders(

                /* Success */
                function (files) {

                    $.each(files, function () {

                        /* Find index of selected permission */
                        var idx = 0;

                        switch (this.getPermission()) {
                            case Picasso.Permission.READ:
                                idx = 0;
                                break;

                            case Picasso.Permission.READ_WRITE:
                                idx = 1;
                                break;

                            case Picasso.Permission.ADMIN:
                                idx = 2;
                                break;

                            case Picasso.Permission.OWNER:
                                idx = 4;
                                break;
                        }

                        var hideCombo = (Picasso.get("account").isMe(dialog._account.getOid()) || this.getPermission() == Picasso.Permission.OWNER )
                        dialog._folders_inputList.addItem(
                                "folder-closed", this.getName(),
                                this.getDescription(), {
                                    oid: this.getOid(),
                                    name: this.getName(),
                                    permission: this.getPermission(),
                                    folder: this
                                }, optionsForFolders, handleOptionsForFolders, idx,null,null,hideCombo
                        );
                        // }
                    });

                    /* Handling for complete loaded state of account */
                    dialog._folders_inputList.showLoadingSpinner(false);
                },

                /* Error */
                function (e) {
                    Picasso.debugLog(e);
                    enableSaveButton(false);
                }
        );
    }

    /**
     * Save {@link Picasso.Account} data
     *
     * @param {Picasso.Dialog.Account}  dialog     A #{@link Picasso.Dialog.Account}
     * @param {Function}                onSuccess  Call on success
     * @param {Function}                onError    Call on error
     **/

    var saveData = function (dialog, onSuccess, onError) {

        /* Check organizations */
        var account = Picasso.get("account");

        var items = dialog._organizations_inputList.getItems();
        var isMemberOfAnyOrg = false;

        $.each(items, function () {
            if (this.isMember) {
                /* Set org oid to trigger server quota check */
                dialog._account.setOrgOid(this.oid);

                isMemberOfAnyOrg = true;
            }
        });

        /* Sanity check for reseller and org admins */
        if (!account.is(Picasso.Account.Flags.ADMIN) && (
            account.is(Picasso.Account.Flags.ORG_CREATOR) ||
            account.is(Picasso.Account.Flags.ORG_ADMIN))) {
            if (0 === items.length || !isMemberOfAnyOrg) {
                Picasso.Notification.show(Picasso.Lang.get(
                    "notification_error_must_be_member_of_any_org"), "warning");

                if (onError) {
                    onError();
                }

                return;
            }
        }

        var usedQuota = dialog._account.getQuotaUsed();
        var newQuota = dialog._account.getQuotaTotal();

        if (newQuota > 0 && newQuota < usedQuota) {
            var confirmDialog = new Picasso.Dialog(
                "#pica_confirmation_dialog",
                Picasso.Lang.get("dialog_title_storage_warning"),
                Picasso.Lang.get("dialog_body_storage_update_warning",
                    Picasso.Helper.formatSize(newQuota),
                    Picasso.Helper.formatSize(usedQuota)
                )
            );

            confirmDialog.setOkHandler(function () {
                confirmDialog.hide();
                runSaveChain();
            });

            confirmDialog.setCancelHandler(function () {
                $("#pica_account_dialog").find(".modal-footer button[data-dismiss].btn-primary .pica-spinner").remove();

            });

            confirmDialog.show();
            return;
        }

        runSaveChain();

        function runSaveChain() {
            /**
             * WARNING: Apparently, we cannot start all these async requests
             *          at once, because some synchronizing timing issue
             *          raises database exceptions.
             *
             *          So we basically wrap all requests in functions (to prevent
             *          starting them immediately) and chain them with promises..
             **/

            /* Store data */
            var defer_save = function () {
                return dialog._account.save(null,

                    /* Error */
                    function (e, status) {
                        /* Handle account errors */
                        switch (status) {
                            case 405:
                                Picasso.Notification.show(
                                    Picasso.Lang.get(
                                        "notification_error_cannot_set_expiration_date"));
                                break;
                            case 406:
                                Picasso.Helper.updatePasswordPolicy();

                                Picasso.Notification.show(
                                    Picasso.Lang.get(
                                        "notification_error_password_policy_mismatch"));
                                break;

                            case 409:
                                Picasso.Notification.show(Picasso.Lang.get(
                                    "notification_error_duplicate",
                                    Picasso.Lang.get("label_accounts")), "warning");
                                break;
                            default:
                                Picasso.Dialog.handleOrgErrors(status);
                        }

                        if (onError) {
                            onError(e, status);
                        }
                    }
                );
            };

            /* Store mails */
            var mails = dialog._mails_inputList.getItems();

            var defer_mails = function () {
                return dialog._account.setEmails($.map(mails, function (val) {
                    return val.name
                }))
            };

            /* Store folders */
            var folders = dialog._folders_inputList.getItems();

            dialog._account._folderscount = folders.length;

            var defer_folders = function () {
                var data = {
                    "folders": []
                }

                /* Collect data */
                $.each(folders, function () {
                    var folder = {}
                    if (this.hasOwnProperty("oid") && this.oid) {
                        folder["folderID"] = (this.oid);
                    } else if (this.hasOwnProperty("name") && this.name) {
                        folder["folderName"] = (this.name);
                    }
                    folder["permission"] = Picasso.Permission.toString(this.permission);
                    data["folders"].push(folder);
                });

                return dialog._account.setFolders(data, null,

                    /* Error */
                    function (e, status) {
                        /* Handle folder errors */
                        switch (status) {
                            case 409:
                                Picasso.Notification.show(Picasso.Lang.get(
                                    "notification_error_duplicate",
                                    Picasso.Lang.get("label_folders")), "warning");
                                break;
                        }

                        if (onError) {
                            onError(e, status);
                        }
                    }
                );
            };

            /* Store groups */
            var groups = dialog._groups_inputList.getItems();

            dialog._account._groupscount = groups.length;

            var defer_groups = function () {
                var data = {
                    "groupID[]": [],
                    "groupName[]": [],
                    "isGroupAdmin[]": []
                };

                /* Collect data */
                $.each(groups, function () {
                    if (this.hasOwnProperty("oid") && this.oid) {
                        data["groupID[]"].push(this.oid);
                    } else {
                        data["groupName[]"].push(this.name);
                    }

                    data["isGroupAdmin[]"].push((this.isGroupAdmin || false));
                });

                return dialog._account.setGroups(data, null, onError);
            };

            /* Store organizations */
            var organizations = dialog._organizations_inputList.getItems();

            var defer_organizations = function () {
                var data = {
                    "organizationID[]": [],
                    "isMember[]": [],
                    "isOrgAdmin[]": [],
                    "organizationNames[]": []
                };

                /* Collect data */
                $.each(organizations, function () {
                    if (this.hasOwnProperty("oid") && this.oid) {
                        data["organizationID[]"].push(this.oid);
                    } else {
                        //new org to be created
                        data["organizationID[]"].push("new");

                    }

                    data["organizationNames[]"].push((this.name));
                    data["isMember[]"].push((this.isMember || false));
                    data["isOrgAdmin[]"].push((this.isOrgAdmin || false));

                    /* Check if we have to update the organization of the account */
                    if (this.isMember) {
                        dialog._account.setOrgOid(this.oid);
                    } else if (dialog._account.getOrgOid() === this.oid) {
                        dialog._account.setOrgOid(null);
                    }
                });

                return dialog._account.setOrganizations(data, null,

                    /* Error */
                    function (e, status) {
                        Picasso.Dialog.handleOrgErrors(status);

                        if (onError) {
                            onError(e, status);
                        }
                    }
                )
            };

            var defer_customer = function () {
                if (dialog._account.is(Picasso.Account.Flags.ORG_CREATOR)) {
                    var customer = Picasso.get("customer");
                    if (!customer) {
                        customer = new Picasso.Customer();
                    }
                    var accountOID = dialog._account.getOid();
                    assignCustomerData(accountOID, customer);
                    customer.setCustomer(
                        /* Success */
                        function (data, status) {

                            switch (status) {
                                case 200:
                                    Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_updated"), "info");
                                    break;
                                case 201:
                                    Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_updated"), "info");
                                    break
                            }

                        },

                        /* Error */
                        function (data, status) {
                            switch (status) {
                                case 420:
                                    Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_update_failed"), "danger");
                                    break
                                case 424:
                                    Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_update_failed"), "danger");
                                    break
                            }
                        }
                    );
                }
            }


            /* Future and broken promises */
            $.when(defer_save()).then(function () {
                $.when(defer_mails()).then(function () {
                    $.when(defer_folders()).then(function () {
                        $.when(defer_groups()).then(function () {
                            $.when(defer_organizations()).then(function () {
                                $.when(defer_customer()).then(function () {
                                    if (onSuccess) {
                                        onSuccess(dialog._account);
                                    }
                                }, onError)
                            }, onError)
                        }, onError)
                    }, onError)
                }, onError)
            });
        };
    };

    /**
     * Constructor for {@link Picasso.Dialog.Account}
     *
     * @param {Picasso.Dialog.Flags}  options  Options for this element (optional)
     * @constructor
     **/

    Picasso.Dialog.Account = function (options) {
        var self = this;

        /* Init */
        this._account = null;
        this._flags = (options || 0);

        /* Sanity check */
        var dialog = $("#pica_account_dialog");

        if (0 === dialog.length) {
            throw "Dialog not found";
        }

        $("#pica_account_storage").attr("max", 9999);


        /* Check immutable flag */
        var inputFlags = 0;

        if (0 < (this._flags & Picasso.Dialog.Flags.IMMUTABLE)) {
            inputFlags = Picasso.InputList.Flags.IMMUTABLE;

            $("#pica_account_avatar_change").hide();
            $("#pica_account_avatar_delete").hide();
        } else {
            $("#pica_account_avatar_change").show();
            $("#pica_account_avatar_delete").show();
        }

        /* Init dialog */
        Picasso.Helper.bindNav(dialog.find(".nav"));

        Picasso.Helper.createDateTimepicker("#pica_account_valid_till");

        /* Add handler for password */
        var input = $("#pica_account_password");

        Picasso.Helper.updatePasswordPolicy(input);

        /* Add change handler to update policy info */
        input.off("keyup").on("keyup", function (e) {
            Picasso.Helper.updatePasswordPolicy(this,
                e.originalEvent.target.value);
        });

        /* Add click handler to clear button to update policy as well */
        input.siblings(".pica-input-clear").click(function () {
            Picasso.Helper.updatePasswordPolicy(input);
        });

        /* Add handler for quota */
        $("#pica_account_storage").off("change").on("change", function (e) {
            updateProgress(self);
        });

        /* Create mails list */
        this._mails_inputList =
            new Picasso.InputList("#pica_account_mails",
                (inputFlags|Picasso.InputList.Flags.LIST_MODE|
                    Picasso.InputList.Flags.VALIDATE));

        this._mails_inputList.setSubmitHandler(
            function (e, userdata) {
                var name = this.getInputValue();

                if(0 < name.length) {
                    this.addItem("message-new", name, null, {
                        oid: null,
                        name: name
                    });
                }
            }
        );

        this._mails_inputList.setValidationErrorHandler(function () {
            Picasso.Notification.show(
                Picasso.Lang.get("notification_error_invalid_email"), "warning");
        });

        /* Create folders list */
        this._folders_inputList =
            new Picasso.InputList("#pica_account_folders",
                (inputFlags|Picasso.InputList.Flags.LIST_MODE|
                    Picasso.InputList.Flags.COMPLETION_ONLY|
                    Picasso.InputList.Flags.COMPLETION_NEW));

        this._folders_inputList.setListRenderer(renderItemFolder);

        this._folders_inputList.setCompletionHandler(
            Picasso.TagInput.handleFolderCompletion
        );

        this._folders_inputList.setCompletionSelectHandler(
            function (e, userdata) {
                this.addItem("folder-new", userdata.getName(), null, {
                        oid: userdata.getOid(),
                        name: userdata.getName(),
                        permission: Picasso.Permission.READ
                    }, optionsForFolders, handleOptionsForFolders, 0
                );
                this._tagInput.hideCompletions();

            }
        );

        this._folders_inputList.setCompletionNewHandler(
            function () {
                var name = this.getInputValue();

                if(0 < name.length) {
                    var account = Picasso.get("account");

                    /* Admins create folders without any own permissions */
                    var permission = Picasso.Permission.READ;
                    var optIdx = 0;

                    if (account.is(Picasso.Account.Flags.ADMIN|
                        Picasso.Account.Flags.ORG_ADMIN))
                    {
                        permission = Picasso.Permission.OWNER;
                        optIdx = 4;
                    }

                    this.addItem("folder-plus", name, null, {
                            oid: null,
                            name: name,
                            permission: permission
                        }, optionsForFolders, handleOptionsForFolders, optIdx
                    );
                }
            }
        );

        this._folders_inputList.setSubmitHandler(
            function (e, userdata) {
                var name = this.getInputValue();

                if(0 < name.length) {
                    this.addItem("folder-new", name, null, {
                            oid: null,
                            name: name,
                            permission: Picasso.Permission.READ
                        }, optionsForFolders, handleOptionsForFolders, 0
                    );
                }
            }
        );

        /* Create groups list */
        this._groups_inputList =
            new Picasso.InputList("#pica_account_groups",
                (inputFlags|Picasso.InputList.Flags.LIST_MODE|
                    Picasso.InputList.Flags.COMPLETION_ONLY|
                    Picasso.InputList.Flags.COMPLETION_NEW));

        this._groups_inputList.setCompletionHandler(
            Picasso.TagInput.handleGroupCompletion
        );

        this._groups_inputList.setCompletionSelectHandler(
            function (e, userdata) {
                this.addItem("group", userdata.getName(), null, {
                        oid: userdata.getOid(),
                        name: userdata.getName(),
                        isGroupAdmin: userdata.isGroupAdmin
                    }, optionsForGroups, handleOptionsForGroups, 0
                );
                this._tagInput.hideCompletions();
            }
        );

        this._groups_inputList.setCompletionNewHandler(
            function () {
                var name = this.getInputValue();

                if(0 < name.length) {
                    this.addItem("asterisk", name, null, {
                            oid: null,
                            name: name,
                            isGroupAdmin: true
                        }, optionsForGroups, handleOptionsForGroups, 1
                    );
                }
            }
        );

        var account = Picasso.get("account");
        var options  = (inputFlags|Picasso.InputList.Flags.LIST_MODE| Picasso.InputList.Flags.COMPLETION_ONLY)
        if (account.is(Picasso.Account.Flags.ADMIN)){
            options = options | Picasso.InputList.Flags.COMPLETION_NEW
        }

        /* Create organizations list */
        this._organizations_inputList =
            new Picasso.InputList("#pica_account_organizations", options);

        this._organizations_inputList.setCompletionHandler(
            Picasso.TagInput.handleOrganizationCompletion
        );

        this._organizations_inputList.setCompletionSelectHandler(
           function (e, userdata) {
               this.addItem(userdata.getIcon(), userdata.getName(),
                    Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                        "string_remaining_storage",
                        Picasso.Helper.formatSize(userdata.getQuotaUsed()),
                        Picasso.Helper.formatSize(userdata.getQuotaTotal())),
                    {
                        oid: userdata.getOid(),
                        name: userdata.getName(),
                        quotaUsed: userdata.getQuotaUsed(),
                        quotaTotal: userdata.getQuotaTotal(),
                        isMember: false,
                        isOrgAdmin: true
                    }, optionsForOrganizations,
                    handleOptionsForOrganizations, 2
                );
               this._tagInput.hideCompletions();
           }
        );

        this._organizations_inputList.setUpdateHandler(function () {
            updateProgress(self);
            var items = self._organizations_inputList.getItems();

            $.each(items, function () {
                if (true === this.isMember) {
                    $("#pica_account_storage").attr("max", Picasso.Helper.parseNumber(
                        Picasso.Helper.parseNumber(($("#pica_account_storage").val() || 0))  + Math.round((this.quotaTotal - this.quotaUsed) / 1024 / 1024 / 1024)
                    ));
                }
            });
        });

        /* Admins only can create new organizations */
        if (account.is(Picasso.Account.Flags.ADMIN))
        {
            this._organizations_inputList.setCompletionNewHandler(
                function () {
                    var name = this.getInputValue();

                    if (0 < name.length) {
                        var userdata = {
                            oid: null,
                            name: name,
                            isMember: false,
                            isOrgAdmin: true
                        };

                        var id = this.addItem("asterisk", name, "", userdata,
                            optionsForOrganizations,
                            handleOptionsForOrganizations);

                        initOptionsForOrganizations(id, userdata);
                    }
                }
            );
        }

        /* Clear on close */
        dialog.bind("hidden.bs.modal", function () {
            self._mails_inputList.clear();
            self._folders_inputList.clear();
            self._groups_inputList.clear();
            self._organizations_inputList.clear();
        });
    };

    /**
     * Create new {@link Picasso.Account} via dialog
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Dialog.Account.prototype.create = function (onSuccess, onError) {
        var self = this;

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_account_dialog",
            Picasso.Lang.get("dialog_title_create"),
            Picasso.Lang.get("dialog_body_create", Picasso.Lang.get("label_accounts")),
            (this._flags|Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE|Picasso.Dialog.Flags.SPINNER));

        dialog.setAssignHandler(function (inputs) {
            $("#pica_account_dialog .pica-hide-on-create").hide();
            $("#pica_account_avatar").attr("src", "");
            $("#pica_account_active").trigger("click");

            /* Add organizations of org admins and reseller */
            var account = Picasso.get("account");
            if (account.is(Picasso.Account.Flags.ORG_CREATOR)) {
                $("#pica_account_organizations_nav").click(function () {
                    $("#pica_account_organizations_input").val(' ').trigger('input');
                    $("#pica_account_organizations_input").trigger($.Event('keyup', {which: 32}));
                });
            }

            if (!account.is(Picasso.Account.Flags.ADMIN) &&
                    (account.is(Picasso.Account.Flags.ORG_ADMIN) ||
                            account.is(Picasso.Account.Flags.ORG_CREATOR)))
            {
                account.getOrganizations(function (organizations) {
                    if (organizations.length === 1){
                        $.each(organizations, function () {
                            if (account.getOrgOid() === this.getOid()) {
                                self._organizations_inputList.addItem("user-asterisk",
                                        this.getName(), "", {
                                            oid: this.getOid(),
                                            name: this.getName(),
                                            isMember: true,
                                            isOrgAdmin: false
                                        }, optionsForOrganizations,
                                        handleOptionsForOrganizations
                                );
                            }
                        });
                    }
                });
            }


            updateDateInputs(inputs);
            updateInputsForAuthType(inputs);
            updateProgress(self);
        });

        dialog.setOkHandler(function (inputs) {
            self._account = new Picasso.Account();

            copyData(self, inputs);
            saveData(self,

                /* Success */
                function () {
                    dialog.hide();

                    if (onSuccess) {
                        onSuccess.apply(null, arguments);
                    }
                },

                /* Error */
                function (e, status) {
                    dialog.showSpinner(false);

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
        });

        dialog.show();
    };

    /**
     * Render given folder json object for list of folders
     *
     * @param {Object}  obj  Object to render
     *
     * @returns {HTMLElement} Rendered json object
     **/

    var renderItemFolder = function (obj) {
        var tr = $("<tr>", {
            id: obj.id,
            "data-inputlist-idx": obj.idx,
            "data-userdata": JSON.stringify(obj.userdata),
            draggable: false
        });

        /* Whether to make item unselectable */
        if (true === obj.isUnselectable) {
            tr.addClass("pica-table-selection-disabled");
        }

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        Picasso.Helper.createGlyphAndAvatar(td, obj.icon, obj.userdata);

        tr.append(td);

        /* 2. column: Name */
        var td = $("<td>", {
            class: "pica-name pica-overflow pica-vertical-middle text-left",
            text: obj.name,
            "data-tooltip": "",
            "data-placement": "bottom",
            title: obj.tooltip,
            colspan: obj.description? 1 : 2
        });

        td.tooltip({ container: "body" });


        tr.append(td);

        /* Add description if any or an empty String to avoid display glitch  */
        /* 3. column: Description */
        if (obj.description) {
            td = $("<td>", {
                class: "pica-overflow pica-vertical-middle text-left text-muted",
                text: obj.description
            });
            tr.append(td);
        }
        /* 4. column: Permission */


        /* Handle options */

        if (obj.userdata.folder && Picasso.Permission.OWNER === obj.userdata.folder.getPermission()) {

            td = $("<td>", {
                class: "pica-overflow pica-vertical-middle text-right text-muted",
                text: Picasso.Lang.get("permission_"+Picasso.Permission.toString(obj.userdata.folder.getPermission()).toLowerCase())
            });

            if (obj.userdata.folder._isBlocked) {
                var span = Picasso.Helper.createGlyph("alert", "pica-glyph")
                Picasso.Helper.addTooltip(span,Picasso.Lang.get("account_folder_blocked"));
                td.append(span)
            }
            tr.append(td);

        }
        else if (0 < (obj.self._flags & Picasso.InputList.Flags.IMMUTABLE)) {
            td = $("<td>", {
                class: "hidden-xs"
            });

            var button = $("<button>", {
                text: (obj.options[obj.startIdx]["text"] || ""),
                class: "btn btn-default disabled pull-right"
            });

            td.append(button);

            tr.addClass("pica-table-selection-disabled");
        } else {
            if (obj.hideCombo){
                td = $("<td>", {
                    class: "pica-overflow pica-vertical-middle text-left text-muted",
                    text: ""
                });
            }else {
                td = $("<td>", {
                    class: "hidden-xs",
                    html: Picasso.Combo.createDropdownOrButton(obj.options,
                            $.proxy(obj.handleOpts, this, obj.self),
                            obj.startIdx, "pull-right")
                });
            }

        }

        tr.append(td);

        return tr;
    };

    /**
     * Edit {@link Picasso.Account} via dialog
     *
     * @param {Picasso.Account}  account    A #{@link Picasso.Account}
     * @param {Function}         onSuccess  Call on success
     * @param {Function}         onError    Call on error
     **/

    Picasso.Dialog.Account.prototype.edit = function (account,
                                                      onSuccess, onError)
    {
        var self = this;

        this._account = account;
        this._origQuota = account.getQuotaTotal(); ///< Keep orignal quota for calculations

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_account_dialog",
            Picasso.Lang.get("dialog_title_edit"),
            Picasso.Lang.get("dialog_body_edit", Picasso.Lang.get("label_accounts")),
            (this._flags|Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE|Picasso.Dialog.Flags.SPINNER));

        dialog.setAssignHandler(function (inputs) {
            $("#pica_account_dialog .pica-hide-on-create").show();

            updateInputsForAuthType(inputs, account.getAuthType());
            updateProgress(self);

            assignData(self, inputs);
        });

        if ($("#pica_accounts_block").length){
            $("#pica_accounts_block").off("click").click(function () {
                account.blockAccess()
                loadFolders(self);
            });
        }
        if ($("#pica_accounts_unblock").length){
            $("#pica_accounts_unblock").off("click").click(function () {
                account.unblockAccess()
                loadFolders(self);;
            });
        }

        if ($("#pica_2fa_reset").length){
            $("#pica_2fa_reset").off("click").click(function () {
                account.reset2fa(function (){
                    $("#pica_2fa_reset").closest(".row").hide();
                })
            });
        }

        if ($("#pica_account_set_max_storage").length){
            $("#pica_account_set_max_storage").off("click").click(function () {
                $("#pica_account_storage").val($("#pica_account_storage").attr("max"));
            });
        }


        dialog.setOkHandler(function (inputs) {
            /* Check organizations of org admins */
            var account = Picasso.get("account");

            if (!account.is(Picasso.Account.Flags.ADMIN) &&
                (account.is(Picasso.Account.Flags.ORG_ADMIN) ||
                    account.is(Picasso.Account.Flags.ORG_CREATOR)))
            {
                if (self._organizations_inputList.isEmpty()) {
                    dialog.showSpinner(false);

                    Picasso.Notification.show(Picasso.Lang.get(
                        "notification_error_must_be_member_of_any_org"), "warning");

                    return;
                }
            }

            copyData(self, inputs);
            saveData(self,

                /* Success */
                function () {
                    dialog.hide();

                    if (onSuccess) {
                        onSuccess.apply(null, arguments);
                    }
                },

                /* Error */
                function (e, status) {
                    dialog.showSpinner(false);

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
        });

        dialog.show();
    };
})();
