/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Dialog")) {
    window.Picasso.Dialog = {};
}

/* Private scope */
(function () {

    /* Options for dropdown menus */
    var optionsForAccounts = [
        {
            text: Picasso.Lang.get("permission_is_member"),
            type: "checkbox",
            group: "permission"
        },
        null,
        {
            text: Picasso.Lang.get("permission_is_admin"),
            type: "checkbox",
            group: "permission"
        },
        null,
        { text: Picasso.Lang.get("action_remove"), icon: "bin" }
    ];

    /* Options for dropdown menus */
    var optionsForCreate = [
        {
            text: Picasso.Lang.get("permission_you_admin"),
            klass: "disabled btn-success"
        }
    ];

    /**
     * Get proper dropdown caption
     *
     * @param {Object}  userdata  Userdata of current item
     *
     * @returns {String} Selected caption
     **/

    var getCaption = function (userdata) {
        if (userdata.isMember && userdata.isOrgAdmin) {
            return Picasso.Lang.get("permission_is_member_and_admin");
        } else if (userdata.isMember) {
            return Picasso.Lang.get("permission_is_member");
        } else if (userdata.isOrgAdmin) {
            return Picasso.Lang.get("permission_is_admin");
        }
    };

    /**
     * Init options
     *
     * @param {String}  id        Element Id
     * @param {Object}  userdata  Supplied userdata
     **/

    var initOptionsForMembers = function (id, userdata) {
        /* FIXME: Init dropdown */
        var dropdown = $("#" + id + " .dropdown");

        var toggle = dropdown.find(".dropdown-toggle");

        toggle.text(getCaption(userdata));
        toggle.append($("<span>", { class: "caret" }));

        var spans = dropdown.find("li span");

        if (userdata.isMember) {
            $(spans[0]).parent().hide();
            $(spans[2]).parent().hide();
            $(spans[0]).parents('tr').prev().off( "click" );
        } else {
            $(spans[0]).removeClass("glyphicons-check")
                .addClass("glyphicons-unchecked");
        }

        if (userdata.isOrgAdmin) {

            if (Picasso.get("account").is(Picasso.Account.Flags.ADMIN)) {
                $(spans[1]).removeClass("glyphicons-unchecked")
                    .addClass("glyphicons-check");
            } else {
                $(spans[1]).removeClass("glyphicons-unchecked")
                    .addClass("glyphicons-check");

                /* Prevent (org-)admin from revoking his own admin permission */
                if (Picasso.get("account").getOid() === userdata.oid){

                    $(spans[2]).parent().parent("li").prev("li").remove();
                }
            }
        } else {
            $(spans[1]).removeClass("glyphicons-check").addClass("glyphicons-unchecked");
        }

        var actionableLinks = dropdown.find("li a").filter(function() {
            return $(this).css("display") !== "none";
        }).length;

        if (actionableLinks === 0) {
            toggle.prop("disabled", true);
        } else {
            toggle.prop("disabled", false);
        }
    };

    /**
     * Handle options for accounts
     *
     * @param {Picasso.InputList}  inputList  A #{@link Picasso.InputList}
     * @param {Number}             optIdx     Selected index
     * @param {Number}             idx        Item index of selected row
     * @param {Object}             userdata   Additional userdata (optional)
     **/

    var handleOptionsForMembers = function (inputList, optIdx, idx, userdata) {
        switch(optIdx) {
            case 0: ///< Member
                /* Sanity check: One must be checked */
                if (userdata.isMember && !userdata.isOrgAdmin) {
                    return;
                }

                var newUserdata = $.extend(userdata, {
                    isMember: !userdata.isMember ///< Toggle
                });

                inputList.setUserdataOfItem(idx, newUserdata);

                return getCaption(newUserdata);

            case 2: ///< Admin
                /* Sanity check: One must be checked */
                if (!userdata.isMember && userdata.isOrgAdmin) {
                    return;
                }

                var newUserdata = $.extend(userdata, {
                    isOrgAdmin: !userdata.isOrgAdmin
                });

                inputList.setUserdataOfItem(idx, newUserdata);

                return getCaption(newUserdata);

            case 4:
                inputList.removeItem(idx);
                break;
        }
    };

    /**
     * Update quota progress bars
     *
     * @param {Picasso.Dialog.Organization}  dialog  A #{@link Picasso.Dialog.Organization}
     **/

    var updateProgress = function (dialog) {
        var quotaOld = 0;
        var quotaNew = 0;
        var quotaAddedAfterSave = 0;
        var quotaFreeAfterSave = 0;
        var percentQuotaOld = 0;
        var percentQuotaAddedAfterSave = 0;
        var percentQuotaFreeAfterSave = 0;

        var caption;

        /* Sum up used quota */
        var items = dialog._accounts_inputList.getItems();

        $.each(items, function () {
            /* Only include members */
            if (this.isMember) {
                quotaNew += this.quota;
            }
        });

        var quotaTotal = Math.round(Picasso.Helper.parseNumber(($("#pica_organization_storage").val() || 0), 0) * 1024 * 1024 * 1024);
        /* Check if organization already exists */
        if (dialog._organization) {
            quotaOld = dialog._organization.getQuotaUsed() ;
            percentQuotaOld = Math.floor(quotaOld * 100 / quotaTotal);
        }
        if (quotaOld > quotaNew) {
            quotaAddedAfterSave = (quotaNew - quotaOld);
            quotaFreeAfterSave = (quotaOld - quotaNew);
            percentQuotaFreeAfterSave = (0 < quotaTotal ? (quotaFreeAfterSave * 100 / quotaTotal) : 0);
            percentQuotaOld -= percentQuotaFreeAfterSave;
            caption = Picasso.Lang.get("string_storage_used", Math.round(percentQuotaOld));
        } else if (quotaOld < quotaNew) {
            quotaAddedAfterSave = (quotaNew - quotaOld);
            percentQuotaAddedAfterSave = (0 < quotaTotal ? (quotaAddedAfterSave * 100 / quotaTotal) : 0);
            caption = Picasso.Lang.get("string_storage_used", Math.round(percentQuotaOld + percentQuotaAddedAfterSave));
        } else {
            percentQuotaAddedAfterSave = 0;
            percentQuotaFreeAfterSave = 0;
            caption = Picasso.Lang.get("string_storage_used", Math.round(percentQuotaOld));
        }

        /* Update progress */
        Picasso.Helper.createQuotaPopover("#pica_organization_total_storage",
            quotaOld, quotaTotal, quotaAddedAfterSave, quotaFreeAfterSave,
            percentQuotaOld, percentQuotaAddedAfterSave, percentQuotaFreeAfterSave, caption);
    };

    /**
     * Copy data from inputs to organization
     *
     * @param {Picasso.Dialog.Organization}  dialog  A #{@link Picasso.Dialog.Organization}
     * @param {Object}                       inputs  Input fields ({ "id" = HTMLElement })
     **/

    var copyData = function (dialog, inputs) {
        dialog._organization.setName(inputs.pica_organization_name.val());
        dialog._organization.setNotes(inputs.pica_organization_notes.val());

        /* Max accounts */
        var val ;
        if ((Picasso.get("account").is(Picasso.Account.Flags.RESTRICTED_ORG_ADMIN)) ||
            (Picasso.get("account").is(Picasso.Account.Flags.RESELLER))) {
            val = $("#pica_organization_max_accounts").val()
        }else {
            val = inputs.pica_organization_max_accounts.val();
        }



        if (-1 !== val) {
            dialog._organization.setAccountsMax(Picasso.Helper.parseNumber(val, 0));
        }

        /* Colors */
        if (inputs.pica_organization_color1) {
            dialog._organization.setColor1(inputs.pica_organization_color1.val());
        }
        if (inputs.pica_organization_color2) {
            dialog._organization.setColor2(inputs.pica_organization_color2.val());
        }
        if (inputs.pica_organization_color3) {
            dialog._organization.setColor3(inputs.pica_organization_color3.val());
        }


        /* Quota */
        var val ;
        if ((Picasso.get("account").is(Picasso.Account.Flags.RESTRICTED_ORG_ADMIN)) ||
            (Picasso.get("account").is(Picasso.Account.Flags.RESELLER))) {
            val = $("#pica_organization_storage").val()  * 1024 //from TB to GB
        }else {
            val =  inputs.pica_organization_storage.val();
        }

        if (-1 !== val) {
            val = Picasso.Helper.parseNumber(val, 0);

            dialog._organization.setQuotaTotal(Math.round(val * 1024 * 1024 * 1024));
        }

        /* Valid till and valid to */
        dialog._organization.setValidFrom(Picasso.Helper.getDateTimePickerValue(
            inputs.pica_organization_valid_from, -1));
        dialog._organization.setValidTill(Picasso.Helper.getDateTimePickerValue(
            inputs.pica_organization_valid_till, -1));
    };

    var handleOptionchange = function (e) {
        e.preventDefault();
        e.stopPropagation();
        var selectedOption = $(this).val();
        $("#pica-organization-edit-save").attr("disabled",false);
        if (selectedOption ==="50+") {
            $("#pica-organization-edit-save").attr("disabled",true);
            var url = "mailto:info@powerfolder.com?subject=PowerFolder Drive Bestellung";
            window.open(url, '_blank');
        }
    };

    var createDropDown = function (elementId, minValue, maxValue, unlimitedValue,selectedValue, label) {
        var select = $("#" + elementId);
        select.empty();
        var valueExists = selectedValue >= minValue && selectedValue <= maxValue;

        var isSpecial  = selectedValue === unlimitedValue ;

        if (!valueExists) {
            var text = isSpecial ? Picasso.Lang.get("label_max_ultimate") : selectedValue
            var newOption = $("<option>").val(selectedValue).text(text);
            newOption.attr("selected", true);
            select.append(newOption);
        }

        for (let i = minValue; i <= maxValue; i++) {
            const optionElement = $('<option>', {
                value: "" + i,
                selected: i === selectedValue,
                text: Picasso.Lang.get(label + i)
            });
            select.append(optionElement);
        }

        var final = $("<option>").val("50+").text(Picasso.Lang.get(label + "50+"));
        select.append(final);
        select.change(handleOptionchange);
        return select;
    }


    /**
     * Assign data to inputs
     *
     * @param {Picasso.Dialog.Organization}  dialog  A #{@link Picasso.Dialog.Organization}
     * @param {Object}                       inputs  Input fields ({ "id" = HTMLElement })
     **/

    var assignData = function (dialog, inputs) {

        // Disable name, quota and max accounts for restricted organization admins and restricted resellers
        if ((Picasso.get("account").is(Picasso.Account.Flags.RESTRICTED_ORG_ADMIN)) ||
            (Picasso.get("account").is(Picasso.Account.Flags.RESELLER))) {
            inputs.pica_organization_name.attr("disabled", true);
            inputs.pica_organization_valid_from.attr("disabled", true);
            inputs.pica_organization_valid_till.attr("disabled", true);
            inputs.pica_organization_notes.attr("disabled", true);
        }

        if (Picasso.get("account").is(Picasso.Account.Flags.RESTRICTED_ORG_ADMIN)) {
            inputs.pica_organization_max_accounts.attr("readonly", true);
            inputs.pica_organization_storage.attr("readonly", true);
        }

        if (Picasso.get("account").is(Picasso.Account.Flags.RESELLER)) {
            var maxAccountsDropDown = createDropDown("pica_organization_max_accounts",2,50,999999999,dialog._organization.getAccountsMax(),"label_max_account_");
            var quotaDropDown = createDropDown("pica_organization_storage",1,50, 9999,Math.round(dialog._organization.getQuotaTotal() / 1024 / 1024 / 1024 / 1024 ),"label_storage_");

            var isResellerOrg = dialog._organization.getAccountsMax() == 999999999;
            maxAccountsDropDown.prop("disabled", isResellerOrg);
            quotaDropDown.prop("disabled", isResellerOrg);
        }

        else {
            if (-1 !== dialog._organization.getAccountsMax()) {
                inputs.pica_organization_max_accounts.val(
                    dialog._organization.getAccountsMax());
            }
            if (-1 !== dialog._organization.getQuotaTotal()) {
                inputs.pica_organization_storage.val(
                    Math.round(dialog._organization.getQuotaTotal() / 1024 / 1024 / 1024));
            }
        }

        inputs.pica_organization_name.val(dialog._organization.getName());
        inputs.pica_organization_notes.val(dialog._organization.getNotes());

        /* Just trigger the checkbox */
        if (dialog._organization.is(Picasso.Organization.Flags.RESTRICT_TO_DOMAINS)) {
            inputs.pica_organization_restrict_to_domains.trigger("click");
        }

        /* Valid from */
        if (-1 !== dialog._organization.getValidFrom()) {
            Picasso.Helper.setDateTimePickerValue(inputs.pica_organization_valid_from,
                dialog._organization.getValidFrom());
        }

        /* Valid till */
        if (-1 !== dialog._organization.getValidTill()) {
            Picasso.Helper.setDateTimePickerValue(inputs.pica_organization_valid_till,
                dialog._organization.getValidTill());
        }

        /* Set avatar */
        var img = $("#pica_organization_avatar");

        img.off("load").on("load", function () {
            var that = $(this);

            that.siblings().hide();
            that.show();
        });

        img.off("error").on("error", function () {
            var that = $(this);

            that.siblings().show();
            that.hide();
        });

        img.attr("src", dialog._organization.getAvatarUrl());

        /* Avatar buttons */
        $("#pica_organization_avatar_change").off(
            "click").click(function () {
            Picasso.Dialog.Upload.changeAvatar(dialog._organization);
        });

        $("#pica_organization_avatar_delete").off("click").click(function () {
            Picasso.Dialog.Upload.deleteAvatar(dialog._organization);
        });

        $("#pica_organization_branding_reset").off("click").click(function () {
            Picasso.Dialog.Organization.resetBranding(dialog._organization,inputs)
        });
        if (-1 === dialog._organization.getValidTill()) {
            $("#button_renew_subscription").hide()
            $("#button_cancel_subscription").show()
            $("#button_cancel_subscription").off("click").click(function () {
                Picasso.Dialog.Organization.cancelSubscription(dialog._organization)
            });
        }
        if (-1 !== dialog._organization.getValidTill() && new Date(dialog._organization.getValidTill()) < new Date() ) {
            $("#button_renew_subscription").show()
            $("#button_cancel_subscription").hide()
            $("#button_renew_subscription").off("click").click(function () {
                Picasso.Dialog.Organization.renewSubscription(dialog._organization)
            });
        }

        /* colors */
        if (NaN !== dialog._organization.getColor1() && inputs.pica_organization_color1) {
            inputs.pica_organization_color1.val(dialog._organization.getColor1());
            inputs.pica_organization_color1.parent().find("i").css("background-color",dialog._organization.getColor1())
        }
        if (NaN !== dialog._organization.getColor2() && inputs.pica_organization_color2) {
            inputs.pica_organization_color2.val(dialog._organization.getColor2());
            inputs.pica_organization_color2.parent().find("i").css("background-color",dialog._organization.getColor2())
        }
        if (NaN !== dialog._organization.getColor3() && inputs.pica_organization_color3) {
            inputs.pica_organization_color3.val(dialog._organization.getColor3());
            inputs.pica_organization_color3.parent().find("i").css("background-color",dialog._organization.getColor3())
        }

        /* Disable save button until everything is loaded */
        var organizationEditSaveButton = $("#pica-organization-edit-save");
        organizationEditSaveButton.hide();

        /* Enable the save button for organization edit dialog */
        var enableSaveButton = function (state) {
            var organizationEditSaveButton = $("#pica-organization-edit-save");
            if (state) {
                organizationEditSaveButton.show();
            } else {
                organizationEditSaveButton.hide();
            }
        };

        /* Check if an organization is completely loaded (members, groups, domains) */
        var checkOrganizationCompletelyLoaded = function () {

            if (dialog._accounts_inputList.isLoading() ||
                    dialog._groups_inputList.isLoading() ||
                    dialog._domains_inputList.isLoading()) {
                return false;
            }

            enableSaveButton(true);
        };

        /* Load domains */
        dialog._domains_inputList.showLoadingSpinner(true);
        dialog._organization.getDomains(

            /* Success */
            function (domains) {
                $.each(domains, function () {
                    dialog._domains_inputList.addItem(
                        "globe", this, null, {
                            oid: null,
                            name: this
                        }
                    );
                });
                dialog._domains_inputList.showLoadingSpinner(false);
                checkOrganizationCompletelyLoaded();
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
                enableSaveButton(false);
            }
        );

        /* Load members */
        dialog._accounts_inputList.showLoadingSpinner(true);
        dialog._organization.getAccounts(

            /* Success */
            function (accounts) {
                $.each(accounts, function () {
                    delete optionsForAccounts[2]["disabled"];
                    delete optionsForAccounts[4]["disabled"];
                    var userdata = {
                        oid: this.getOid(),
                        orgID:this.getOrgOid(),
                        name: this.getName(),
                        quota: this.getQuotaTotal(),
                        isMember: (this.getOrgOid() === dialog._organization.getOid()),
                        isOrgAdmin: this.is(Picasso.Account.Flags.ORG_ADMIN)
                    };

                    /* Prevent (org-)admin from revoking his own admin permission */
                    if (!Picasso.get("account").is(Picasso.Account.Flags.ADMIN) && (
                        this.is(Picasso.Account.Flags.ORG_CREATOR) ||
                        this.is(Picasso.Account.Flags.ORG_ADMIN))) {
                        delete optionsForAccounts[0]["disabled"];
                        if (Picasso.get("account").getOid() === userdata.oid) {
                            optionsForAccounts[2]["disabled"] = true;
                        }
                    }

                    if (this.getOrgOid() && !userdata.isMember ) {
                        optionsForAccounts[0]["disabled"] = true;
                    }

                    if (userdata.isMember ){
                        optionsForAccounts[4]["disabled"] = true;
                    }


                    var id = dialog._accounts_inputList.addItem("user",
                        this.getName(), Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                            "string_storage", Picasso.Helper.formatSize(
                                this.getQuotaTotal())), userdata,
                        optionsForAccounts, handleOptionsForMembers,null,userdata.isMember,this.getUserName()
                    );

                    initOptionsForMembers(id, userdata);
                });
                dialog._accounts_inputList.showLoadingSpinner(false);
                checkOrganizationCompletelyLoaded();
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
                enableSaveButton(false);
            }
        );

        /* Load groups */
        dialog._groups_inputList.showLoadingSpinner(true);
        dialog._organization.getGroups(

            /* Success */
            function (groups) {
                $.each(groups, function () {
                    dialog._groups_inputList.addItem("group", this.getName(),
                        this.getDescription(), {
                            oid: this.getOid(),
                            name: this.getName()
                        }
                    );
                });
                dialog._groups_inputList.showLoadingSpinner(false);
                checkOrganizationCompletelyLoaded();
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
                enableSaveButton(false);
            }
        );

        updateProgress(dialog); ///< Call it initially
    };

    /**
     * Save {@link Picasso.Organization} data
     *
     * @param {Picasso.Dialog.Organization}  dialog     A #{@link Picasso.Dialog.Organization}
     * @param {Function}                     onSuccess  Call on success
     * @param {Function}                     onError    Call on error
     **/

    var saveData = function (dialog, onSuccess, onError) {

        /**
         * WARNING: Apparently, we cannot start all these async requests
         *          at once, because some synchronizing timing issue
         *          raises database exceptions.
         *
         *          So we basically wrap all requests in functions (to prevent
         *          starting them immediately) and chain them with promises..
         **/

        /* Store data */
        var defer_save = function () {
            var account = Picasso.get("account");

            /* Exclude org admins */
            if (account.is(Picasso.Account.Flags.ADMIN|
                Picasso.Account.Flags.ORG_CREATOR|
                Picasso.Account.Flags.ORG_ADMIN|
                Picasso.Account.Flags.RESELLER))
            {
                return dialog._organization.save(null,

                    /* Error */
                    function (e, status) {
                        Picasso.Dialog.handleOrgErrors(status);

                        if (onError) onError(e, status);
                    }
                );
            } else {
                return true;
            }
        };

        /* Store domains */
        var domains = dialog._domains_inputList.getItems();

        var defer_domains = function () {
            var data = $.map(domains, function (val) {
                return val.name
            });

            return dialog._organization.setDomains(data, null,

                /* Error */
                function (e, status) {
                    switch (status) {
                        case 409:
                            Picasso.Notification.show(
                                Picasso.Lang.get(
                                    "notification_error_duplicate_domain"),
                                "warning");
                            break;
                    }
                }
            );
        };

        /* Store members */
        var members = dialog._accounts_inputList.getItems();
        var quotaSum = 0;
        var membersCount = 0;

        var defer_members = function () {
            var data = {
                "accountID[]": [],
                "accountName[]": [],
                "isMember[]": [],
                "isOrgAdmin[]": []
            };

            /* Collect data */
            $.each(members, function () {
                if (this.hasOwnProperty("oid") && this.oid) {
                    data["accountID[]"].push(this.oid);
                } else {
                    data["accountName[]"].push(this.name);
                }

                data["isMember[]"].push((this.isMember || false));
                data["isOrgAdmin[]"].push((this.isOrgAdmin || false));

                /* Sum up member quota for in-place editing */
                if (this.isMember) {
                    quotaSum += this.quota;
                }

                if (this.isMember) membersCount++;
            });

            return dialog._organization.setAccounts(data, null,

                /* Error */
                function (e, status) {
                    Picasso.Dialog.handleOrgErrors(status);

                    if (onError) onError(e, status);
                }
            );
        };

        /* Store groups */
        var groups = dialog._groups_inputList.getItems();

        var defer_groups = function () {
            var data = {
                "groupID[]": [],
                "groupName[]": [],
                "isOrgAdmin[]": []
            };

            /* Collect data */
            $.each(groups, function () {
                if (this.hasOwnProperty("oid") && this.oid) {
                    data["groupID[]"].push(this.oid);
                } else {
                    data["groupName[]"].push(this.name);
                }

                data["isOrgAdmin[]"].push(false); ///< TODO
            });

            return dialog._organization.setGroups(data, null, onError);
        };

        /* Future and broken promises */
        $.when(defer_save()).then(function () {
            $.when(defer_domains()).then(function () {
                $.when(defer_members()).then(function () {
                    $.when(defer_groups()).then(function () {
                        if (onSuccess) {
                            /* Update object */
                            dialog._organization.setMembersCount(membersCount);
                            dialog._organization.setQuotaUsed(quotaSum);

                            onSuccess(dialog._organization);
                        }
                    }, onError)
                }, onError)
            }, onError)
        });
    };

    /**
     * Constructor for {@link Picasso.Dialog.Organization}
     *
     * @param {Picasso.Dialog.Flags}  options  Options for this element (optional)
     * @constructor
     **/

    Picasso.Dialog.Organization = function (options) {
        var self = this;

        /* Init */
        this._organization = null;
        this._flags = (options || 0);

        /* Sanity check */
        var dialog = $("#pica_organization_dialog");

        if (0 === dialog.length) {
            throw "Dialog not found";
        }

        /* Check immutable flag */
        var inputFlags = 0;

        if (0 < (this._flags & Picasso.Dialog.Flags.IMMUTABLE)) {
            inputFlags = Picasso.InputList.Flags.IMMUTABLE;

            $("#pica_organization_avatar_change").hide();
            $("#pica_organization_avatar_delete").hide();
        } else {
            $("#pica_organization_avatar_change").show();
            $("#pica_organization_avatar_delete").show();
        }

        /* Init dialog */
        Picasso.Helper.bindNav(dialog.find(".nav"));

        Picasso.Helper.createFromUntilDateTimePicker("#pica_organization_valid_from","#pica_organization_valid_till");

        // Disable date picker for restricted organization admins and restricted resellers
        if ((Picasso.get("account").is(Picasso.Account.Flags.RESTRICTED_ORG_ADMIN)) || (Picasso.get("account").is(Picasso.Account.Flags.RESELLER))) {
            Picasso.Helper.disableDateTimePicker("#pica_organization_valid_from");
            Picasso.Helper.disableDateTimePicker("#pica_organization_valid_till");
        }

        /* Add handler for quota */
        $("#pica_organization_storage").off("change").on("change", function () {
            updateProgress(self);
        });

        /* Create domains list */
        this._domains_inputList =
            new Picasso.InputList("#pica_organization_domains",
                (inputFlags|Picasso.InputList.Flags.LIST_MODE|
                    Picasso.InputList.Flags.VALIDATE));

        this._domains_inputList.setSubmitHandler(
            function (e, userdata) {
                var name = this.getInputValue();

                if(0 < name.length) {
                    this.addItem("globe", name, null, {
                        oid: null,
                        name: name
                    });
                }
            }
        );

        this._domains_inputList.setValidationErrorHandler(
            function () {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_invalid_domain"),
                    "warning");
            }
        );

        /* Create accounts list */
        this._accounts_inputList =
            new Picasso.InputList("#pica_organization_accounts",
                (inputFlags|Picasso.InputList.Flags.LIST_MODE));

        this._accounts_inputList.setCompletionHandler(
            Picasso.TagInput.handleAccountCompletion
        );

        this._accounts_inputList.setCompletionSelectHandler(
            function (e, userdata) {
                this.addItem("user-asterisk", userdata.getName(),
                    Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                        "string_storage", Picasso.Helper.formatSize(
                            userdata.getQuotaTotal())),
                    {
                        oid: userdata.getOid(),
                        name: userdata.getName(),
                        quota: userdata.getQuotaTotal(),
                        isMember: true,
                        isOrgAdmin: false
                    }, optionsForAccounts, handleOptionsForMembers,null,null,userdata.getUserName()
                );
                this._tagInput.hideCompletions();
            }
        );

        this._accounts_inputList.setSubmitHandler(
            function (e, userdata) {
                var name = this.getInputValue();

                if(0 < name.length) {
                    this.addItem("user-asterisk", name, "", {
                            oid: null,
                            name: name,
                            isMember: true,
                            isOrgAdmin: false
                        }, optionsForAccounts, handleOptionsForMembers
                    );
                }
            }
        );

        this._accounts_inputList.setUpdateHandler(function (userdata) {
            updateProgress(self);
        });

        this._accounts_inputList.setValidationErrorHandler(
                    function () {
                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_error_invalid_email"),
                            "warning");
                    }
                );


        /* Create groups list */
        this._groups_inputList =
            new Picasso.InputList("#pica_organization_groups",
                (inputFlags|Picasso.InputList.Flags.LIST_MODE|
                    Picasso.InputList.Flags.COMPLETION_ONLY|
                    Picasso.InputList.Flags.COMPLETION_NEW));

        this._groups_inputList.setCompletionHandler(
            Picasso.TagInput.handleGroupCompletion
        );

        this._groups_inputList.setCompletionSelectHandler(
            function (e, userdata) {
                this.addItem("group", userdata.getName(),
                    userdata.getDescription(), {
                        oid: userdata.getOid(),
                        name: userdata.getName(),
                        isGroupAdmin: userdata.isGroupAdmin
                    }
                );
                this._tagInput.hideCompletions();
            }
        );

        this._groups_inputList.setCompletionNewHandler(
            function () {
                var name = this.getInputValue();

                this.addItem("asterisk", name, "", {
                    oid: null,
                    name: name
                });
            }
        );

        /* Clear on close */
        dialog.bind("hidden.bs.modal", function () {
            self._domains_inputList.clear();
            self._accounts_inputList.clear();
            self._groups_inputList.clear();
        });
    };

    /**
     * Create new {@link Picasso.Organization} via dialog
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Dialog.Organization.prototype.create = function (onSuccess, onError) {
        var self = this;

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_organization_dialog",
            Picasso.Lang.get("dialog_title_create"),
            Picasso.Lang.get("dialog_body_create", Picasso.Lang.get("label_organizations")),
            (this._flags|Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE|Picasso.Dialog.Flags.SPINNER));

        dialog.setAssignHandler(function (inputs) {
            $("#pica_organization_avatar").attr("src", "");

            /* Add reseller as admin */
            var account = Picasso.get("account");

            if (!account.is(Picasso.Account.Flags.ADMIN) &&
                account.is(Picasso.Account.Flags.ORG_CREATOR))
            {
                self._accounts_inputList.addItem("user",
                    account.getName(), "", {
                        oid: account.getOid(),
                        name: account.getName(),
                        isMember: false,
                        isOrgAdmin: true
                    }, optionsForCreate, null, 0, true
                );
            }

            updateProgress(self);
        });

        dialog.setOkHandler(function (inputs) {
            self._organization = new Picasso.Organization();
            self._organization.setOid(Picasso.Helper.getID());
            copyData(self, inputs);
            saveData(self,

                /* Success */
                function () {
                    dialog.hide();

                    if (onSuccess) {
                        onSuccess.apply(null, arguments);
                    }
                },

                /* Error */
                function (e, status) {
                    dialog.showSpinner(false);

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
        });

        if ($("#organization_manage_tab")) {
            $("#organization_manage_tab").hide()
        }

        dialog.show();
    };


    /**
     * Create new {@link Picasso.Organization} via simplified dialog
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Dialog.Organization.prototype.simpleCreate = function (onSuccess, onError) {
        var self = this;

        var dialog = new Picasso.Dialog("#pica_input_dialog",
            Picasso.Lang.get("dialog_title_create"),
            Picasso.Lang.get("dialog_body_create", Picasso.Lang.get("label_organizations")),
            (this._flags | Picasso.Dialog.Flags.VALIDATE | Picasso.Dialog.Flags.HIDE_ON_ESC |
                Picasso.Dialog.Flags.NO_AUTOCLOSE | Picasso.Dialog.Flags.SPINNER));

        dialog.setAssignHandler(function (inputs) {
            inputs.pica_input_text.attr("placeholder", Picasso.Lang.get("placeholder_input_customer"));
        });

        dialog.setOkHandler(function (inputs) {
            self._organization = new Picasso.Organization();
            self._organization.setName(inputs.pica_input_text.val());
            self._organization.simpleSave(
                /* Success */
                function () {
                    dialog.hide();

                    if (onSuccess) {
                        onSuccess.apply(null, arguments);
                    }
                },

                /* Error */
                function (e, status) {
                    dialog.showSpinner(false);

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
        });

        dialog.show();
    };

    /**
     * Edit {@link Picasso.Organization} via dialog
     *
     * @param {Picasso.Organization}  organization  A #{@link Picasso.Organization}
     * @param {Function}              onSuccess     Call on success
     * @param {Function}              onError       Call on error
     **/

    Picasso.Dialog.Organization.prototype.edit = function (organization,
                                                           onSuccess, onError)
    {
        var self = this;

        this._organization = organization;

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_organization_dialog",
            Picasso.Lang.get("dialog_title_edit"),
            Picasso.Lang.get("dialog_body_edit", Picasso.Lang.get("label_organizations")),
            (this._flags|Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE|Picasso.Dialog.Flags.SPINNER));

        dialog.setAssignHandler(function (inputs) {
            assignData(self, inputs);
        });

        dialog.setOkHandler(function (inputs) {
            copyData(self, inputs);
            saveData(self,

                /* Success */
                function () {
                    dialog.hide();

                    if (onSuccess) {
                        onSuccess.apply(null, arguments);
                    }
                },

                /* Error */
                function (e, status) {
                    dialog.showSpinner(false);

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
        });

        if ($("#organization_manage_tab")) {
            $("#organization_manage_tab").show()
        }

        dialog.show();
    };

    Picasso.Dialog.Organization.resetBranding = function (organization, inputs) {

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
                Picasso.Lang.get("dialog_title_delete"),
                Picasso.Lang.get("dialog_body_delete",
                        Picasso.Lang.get("label_branding")));

        dialog.setOkHandler(function () {

            Picasso.Helper.fireAjax("/avatars/" + organization.getOid(),
                    { method: "DELETE" },

                    /* Success */
                    function () {
                        /* Basically just force a reload of avatars */
                        var imgs = $(".pica-avatar");

                        $.each(imgs, function () {
                            var that = $(this);

                            that.attr("src", that.attr("src") + "?" +
                                    new Date().getTime());
                        });

                    }
            );

            if (inputs.pica_organization_color1) {
                inputs.pica_organization_color1.val(null);
                inputs.pica_organization_color1.parent().find("i").css("background-color","")
            }
            if (inputs.pica_organization_color2) {
                inputs.pica_organization_color2.val(null);
                inputs.pica_organization_color2.parent().find("i").css("background-color","")
            }
            if (inputs.pica_organization_color3) {
                inputs.pica_organization_color3.val(null);
                inputs.pica_organization_color3.parent().find("i").css("background-color","")
            }
        });

        dialog.show();

    };
    Picasso.Dialog.Organization.cancelSubscription = function (organization) {
        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_cancel"),
            Picasso.Lang.get("dialog_body_cancel",organization.getName()));

        dialog.setOkHandler(function () {
            Picasso.Helper.fireAjax("/wapi/organizations",
                {
                    action: "cancelSubscription",
                    id: organization.getOid()
                },
                /* Success */
                function () {
                    Picasso.Notification.show(Picasso.Lang.get(
                        "notification_item_disabled",
                        Picasso.Lang.get("label_organizations")), "info");
                    Picasso.Organizations.fetch();
                }
            );
        });
        dialog.show();
    };
    Picasso.Dialog.Organization.renewSubscription = function (organization) {
        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_renew"),
            Picasso.Lang.get("dialog_body_renew",organization.getName()));

        dialog.setOkHandler(function () {
            Picasso.Helper.fireAjax("/wapi/organizations",
                {
                    action: "renewSubscription",
                    id: organization.getOid()
                },
                /* Success */
                function () {
                    Picasso.Notification.show(Picasso.Lang.get(
                        "notification_item_enabled",
                        Picasso.Lang.get("label_organizations")), "success");
                    Picasso.Organizations.fetch();
                }
            );
        });
        dialog.show();
    };
})();
