/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Dialog")) {
    window.Picasso.Dialog = {};
}

/* Private scope */
(function () {
    /**
     * Handle dropzone drag over event
     *
     * @param {Event}  e  Drag event
     **/

    var handleDragOver = function (e) {
        e.stopPropagation();
        e.preventDefault();

        /* Update markers */
        var body = this._dialog.find(".modal-body");

        body.addClass("pica-drag-yes");

        e.originalEvent.dataTransfer.dropEffect = "copy";
    };

    /**
     * Handle dropzone leave event
     *
     * @param {Event}  e  Drag event
     **/

    var handleDragLeave = function (e) {
        e.stopPropagation();
        e.preventDefault();

        /* Apparently browsers send 0/0 when drag was aborted with Esc */
        if (0 === e.originalEvent.x && 0 === e.originalEvent.y) {
            var body = this._dialog.find(".modal-body");

            body.removeClass("pica-drag-yes");
        }
    };

    /**
     * Handle dropzone drop event
     *
     * @param {Event}  e  Drag event
     **/

    var handleDrop = function (e) {

        e.preventDefault();
        e.stopPropagation();

        /* Add files */
        var list = e.originalEvent.dataTransfer.files;

        /* Check support */
        if (e.originalEvent.dataTransfer.items) {
            list = e.originalEvent.dataTransfer.items;
        }

        for (var i = 0; i < list.length; i++) {
            this.enqueue(list[i]);
        }

        /* Remove markers */
        var body = this._dialog.find(".modal-body");

        body.removeClass("pica-drag-yes");
    };

    /**
     * Handle drag end event
     **/

    var handleDragEnd = function () {

        /* Tidy up */
        var body = this._dialog.find(".modal-body");

        body.removeClass("pica-drag-yes");
    };

    /**
     * Render given file
     *
     * @param {Object}  holder  Holder for file with meta information
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderFile = function (holder) {

        var tr = $("<tr>", {
            id: holder.id,
            class: "pica-table-selection-disabled"
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "pica-icon-column" });

        var glyph = Picasso.Helper.createGlyphForExtension(holder.file.name);

        td.append(glyph);
        tr.append(td);

        /* Load preview when possible */
        let idx = -1;
        if (holder.isFile) {
             idx = holder.file.type.indexOf("image");
        }
        if (-1 !== idx) {
            var reader = new FileReader();

            reader.onload = function (e) {
                var img = $("<img>", {
                    class: "pica-icon",
                    src: e.target.result
                });

                glyph.replaceWith(img);
            };

            reader.readAsDataURL(holder.file);
        }

        /* 2. column: Name / path */
        var media = $("<div>", { class: "media" });
        var body = $("<div>", { class: "media-body" });
        var heading = $("<div>", { class: "media-heading" });
        var footer = $("<div>", { class: "pica-overflow" });

        heading.append(Picasso.Helper.encodeHTMLEntities(holder.file.name));
        footer.append(Picasso.Helper.encodeHTMLEntities(holder.path));

        body.append(heading);
        body.append(footer);
        media.append(body);

        tr.append($("<td>", {
            class: "pica-overflow",
            html: media
        }));

        /* 3. column: Progress */
        let size = holder.file.size ? holder.file.size : 0;
        tr.append($("<td>", {
            class: "pica-vertical-middle",
            html: Picasso.Helper.createProgressbar("progress_" + holder.id,
                Picasso.Helper.formatSize(size))
        }));

        /* 4. column: Cancel button */
        var td = $("<td>", {
            class: "pica-vertical-middle pica-icon-column"
        });

        var button = $("<button>", {
            class: "close",
            type: "button",
            html: "&times;"
        });

        button.click(function () {
            /* Find file in queue */
            for (var i = 0; i < holder.dialog._queue.length; i++) {
                if (holder.dialog._queue[i].file === holder.file) {
                    holder.dialog._queue.splice(i, 1);

                    break;
                }
            }

            tr.remove();

            holder.dialog._table.checkIfEmpty();
            holder.dialog.toggleButtons();
        });

        td.append(button);
        tr.append(td);

        holder.dialog._dialog.find("table tbody").append(tr);

        return tr;
    };

    /**
     * Constructor for {@link Picasso.Dialog.Upload}
     *
     * @param  head  Head message
     * @param  body  Body message
     * @constructor
     **/

    Picasso.Dialog.Upload = function (head, body) {
        var self = this;

        /* Sanity check */
        this._dialog = $("#pica_upload_dialog");

        if (0 === this._dialog.length) {
            throw "Dialog not found";
        }

        /* Upload directories not supported by IE */
        if (Picasso.Helper.isInternetExplorer()) {

            /* Prevent file upload */
            var fileUploadDialog = this._dialog.find("#upload_input_directories");
            fileUploadDialog.get(0).type = "button";

            var uploadDirsElem = this._dialog.find("#upload_directory");
            var uploadDirsButton = $(uploadDirsElem[0]);

            /* Rebind click handler */
            uploadDirsButton.off("click").on("click", function () {

                var warningDialog = new Picasso.Dialog("#ie_warning_dialog",
                        Picasso.Lang.get("dialog_title_ie_warning"),
                        Picasso.Lang.get("dialog_body_ie_warning"));

                /* Bring warning dialog to the top */
                $(warningDialog._dialog[0]).css("z-index", 1100);

                warningDialog.show();
            });
        }

        /* Init */
        this._queue = [];
        this._xhr = null;
        this._url = "";
        this._data = null;
        this._autostart = false;

        this._onDone = null;
        this._onClose = null;

        this._table = new Picasso.Table("#upload_table");
        this._uploadedCount = 0;

        /* Update dialog */
        this._dialog.find(".modal-title").text(head);
        this._dialog.find(".modal-body h5").text(body);

        /* Re-bind handler */
        this._dialog.find("#upload_start_button").off("click").click(function () {
            if (self._xhr) return;

            var totalSize = 0;
            for (var i = 0; i < self._queue.length; i++) {
                totalSize += self._queue[i].file.size;
            }

            self.checkStorageBeforeStart(totalSize, function () {
                if ( !self._data || !self._data.files) {
                    if (!self._data) {
                        self._data = {}
                    }
                    self._data.files = self._queue.length;
                }

                self.startUpload(function () {
                    self.stopUpload();

                    if (self._onDone) {
                        self._onDone();
                    }
                });
            });
        });

        this._dialog.find("#upload_stop_button").off("click").click(function () {
            if (self._xhr) {
                var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
                    Picasso.Lang.get("dialog_title_abort"),
                    Picasso.Lang.get("dialog_body_abort"));

                dialog.setOkHandler(function () {
                    self._xhr.abort();

                    self.stopUpload();
                });

                dialog.show();
            } else {
                self.stopUpload();

                if (self._onClose) {
                    self._onClose();
                }

                self._dialog.modal("hide");
            }
        });

        this._dialog.find("#upload_close_button").off("click").click(function () {
            self.stopUpload();
        });

        this._dialog.find(":file").off("change").on("change", function () {

            /* Enqueue all files and reset input afterwards to allow new files */
            $.each(this.files, function () {
                self.enqueueFile(this);
            });

            /* Start automatically if _autostart is true */
            if (true === self._autostart) {
                if (self._xhr) return;

                var totalSize = 0;
                for (var i = 0; i < self._queue.length; i++) {
                    totalSize += self._queue[i].file.size;
                }

                self.checkStorageBeforeStart(totalSize, function () {
                    if (!self._data || !self._data.files) {
                        if (!self._data) {
                            self._data = {};
                        }
                        self._data.files = self._queue.length;
                    }

                    self.startUpload(function () {
                        self.stopUpload();

                        if (self._onDone) {
                            self._onDone();
                        }
                    });
                });
            }

            self.resetFileInputs();
            self.toggleButtons();
        });

        this._dialog.off("hidden.bs.modal").on("hidden.bs.modal", function () {
            self._dialog.find("table tbody").empty();
            self._dialog.off();
        });

        /* Re-bind dropzone handler */
        var body = this._dialog.find(".modal-body");

        body.off("dragover").on("dragover", $.proxy(handleDragOver, this));
        body.off("dragleave").on("dragleave", $.proxy(handleDragLeave, this));
        body.off("drop").on("drop", $.proxy(handleDrop, this));
        body.off("dragend").on("dragend", $.proxy(handleDragEnd, this));
    };

    /**
     * Check storage before starting upload
     *
     * @param {Number} totalSize Total size of all files to be uploaded
     * @param {Function} onSuccess Callback function to execute if storage is sufficient
     **/
    Picasso.Dialog.Upload.prototype.checkStorageBeforeStart = function (totalSize, onSuccess) {
        var self = this;

        var url = (self._data && self._data.fileLinkId) ? "/wapi/filelink" : "/wapi/folders";

        var data = {
            action: "getAvailableUploadStorage",
            method: "GET",
            folderId: self._data && self._data.folderID ? self._data.folderID : null,
            fileLinkId: self._data && self._data.fileLinkId ? self._data.fileLinkId : null
        };

        Picasso.Helper.fireAjax(
            url,
            data,
            function (response) {
                var availableStorage = response.availableStorageBytes;
                var isFolderOwner = response.isFolderOwner;
                var folderOwnerName = response.folderOwnerName ;

                if (totalSize > availableStorage) {

                var dialogMessage;
                if (isFolderOwner) {
                    dialogMessage = Picasso.Lang.get("dialog_body_insuffisant_storage");
                    $("#upgrade-now-button").show();

                } else {
                    dialogMessage = Picasso.Lang.get("dialog_body_insuffisant_storage_owner_exceeded", folderOwnerName);
                    $("#upgrade-now-button").hide();
                }

                var notEnoughStorageDialog = new Picasso.Dialog("#not_enough_storage_dialog",
                    Picasso.Lang.get("dialog_title_insuffisant_storage"),
                    dialogMessage
                );

                    notEnoughStorageDialog.setOkHandler(function () {
                        onSuccess()
                    });

                    notEnoughStorageDialog.setCancelHandler(function () {
                        self.stopUpload();
                        self._dialog.modal("hide");
                    });

                    notEnoughStorageDialog.show();
                } else {
                    onSuccess();
                }
            },
            function () {
                self.stopUpload();
            }
        );
    };



    /**
     * Reset file inputs
     **/

    Picasso.Dialog.Upload.prototype.resetFileInputs = function () {
        var inputs = this._dialog.find(":file");

        /**
         *  Reset input: Browsers prevent clearing the input manually,
         *               changing the type is the easiest way here
         *               without cloning the element.
         **/
       if (!/safari/i.test(navigator.userAgent)) {
           inputs.attr("type", "");
           inputs.attr("type", "file");
       }


    };

    /**
     * Whether to allow multiple file selects
     *
     * @param {Boolean}  allowMultiple  Whether to allow multiple file selects
     **/

    Picasso.Dialog.Upload.prototype.setAllowMultiple = function (allowMultiple) {
        var inputs = this._dialog.find(":file");

        if (true === allowMultiple) {
            inputs.attr("multiple", true);
        } else {
            inputs.removeAttr("multiple");
        }
    };

    /**
     * Whether to allow upload of directories
     *
     * @param {Boolean}  allowDirectories  Whether to allow directories
     **/

    Picasso.Dialog.Upload.prototype.setAllowDirectories = function (allowDirectories) {
        var button = $("#upload_directory");

        /* Just toggle visibility */
        if (true === allowDirectories) {
            button.show();
        } else {
            button.hide();
        }
    };

    /**
     * Set accepted mime types for upload
     *
     * @param {String|Array}  acceptedTypes  Accepted types
     **/

    Picasso.Dialog.Upload.prototype.setAcceptTypes = function (acceptedTypes) {

        /* Update file input */
        if (acceptedTypes) {
            var array = acceptedTypes;

            if (!Array.isArray(acceptedTypes)) {
                array = [ acceptedTypes ];
            }

            var inputs = this._dialog.find(":file");

            inputs.first().attr("accept", array.join(","));
        }
    };

    /**
     * Set whether upload shall automatically start after adding files
     *
     * @param {Boolean}  autostart  Whether to automatically start upload
     **/

    Picasso.Dialog.Upload.prototype.setStartAutomatically = function (autostart) {
        this._autostart = autostart;
    };

    /**
     * Toggle button captions based on queue state
     **/

    Picasso.Dialog.Upload.prototype.toggleButtons = function () {
        var stopButton = $("#upload_stop_button");

        if (0 < this._queue.length) {
            /* Only remove disabled when no upload is running */
            if (!this._xhr) {
                $("#upload_start_button").removeAttr("disabled");
            }

            /* Update stop button */
            stopButton.find("lang").text(Picasso.Lang.get("button_cancel_all"));
            stopButton.addClass("btn-danger");
            stopButton.removeClass("btn-default");
        } else {
            $("#upload_start_button").attr("disabled", true);
            $("#upload_file").removeAttr("disabled");
            $("#upload_directory").removeAttr("disabled");

            /* Update stop button */
            stopButton.find("lang").text(Picasso.Lang.get("button_close"));
            stopButton.removeClass("btn-danger");
            stopButton.addClass("btn-default");
        }
    };

    /**
     * Upload file
     *
     * @param {Object}    file       File object
     * @param {String}    relPath    Relative path (optional)
     * @param {Function}  onSuccess  Success callback
     * @param {Function}  onError    Error callback
     **/

    Picasso.Dialog.Upload.prototype.transferFile = function (file, relPath,
                                                             onSuccess, onError)
    {
        var self = this;

        /* Create form data */
        var fd = new FormData();

        fd.append("file", file);
        fd.append("ajax", 1);

        $("#upload_start_button").attr("disabled", true);

        /* Decode parameter */
        if (this._data) {
            $.each(this._data, function (k, v) {
                fd.append(k, decodeURIComponent(v));
            });
        }

        /* Update relative path */
        if (relPath) {
            var path = fd["path"] ? fd["path"] : "";
            fd.append("path", Picasso.Helper.joinPaths([path, relPath]));
        }

        var progress = $("#progress_upload_" +
                Picasso.Helper.hashCode(relPath + "/" + file.name) + " > .progress-bar");

        /* Send data */
        this._xhr = $.ajax({

            /* Add CSRF Token header */
            beforeSend: function(request) {
                request.setRequestHeader("X-CSRFToken", Picasso.get("CSRF"));
            },

            url: this._url,
            type: "POST",
            data: fd,

            /* Set this to false to let the browser do the guess work */
            cache: false,
            contentType: false,
            processData: false,

            xhr: function () {
                /* Create custom xhd */
                var xhr = $.ajaxSettings.xhr();

                if (xhr.upload) {
                    xhr.upload.addEventListener("progress", function (e) {
                        if (e.lengthComputable) {

                            /* Update progress */
                            var percent = Math.round(e.loaded * 100 / e.total);
                            var span = progress.siblings("span");

                            progress.css("width", percent + "%");

                            /* Update caption */
                            span.text(Picasso.Lang.get("string_percent", percent));



                            /* Remove id to allow re-upload */
                            progress.parents("tr").removeAttr("id");
                        }
                    }, false);

                    return xhr;
                }
            },

            success: function (data, status) {
                progress.addClass("progress-bar-success");
                progress.parent().find("span").text(Picasso.Lang.get("notification_upload_completed")).append(Picasso.Helper.createGlyph("ok-sign"));
                if (onSuccess) {
                    onSuccess(data);
                }
            },

            error: function (xhr, status, e) {


                progress.addClass("progress-bar-danger");

                progress.parent().find("span").text(Picasso.Lang.get("notification_upload_failed")).append(Picasso.Helper.createGlyph("remove-sign"));

                /* Remove id to allow re-upload */
                progress.parents("tr").removeAttr("id");

                if (xhr.status ===409 ) {
                    Picasso.Notification.show(Picasso.Lang.get("notification_error_storage_exceeded"), "danger",false,"remove");

                    self.stopUpload();
                }
                if (xhr.status === 400) {
                    Picasso.Notification.show(Picasso.Lang.get("notification_error_import"), "danger",false,"remove");

                    var reader = new FileReader();
                    reader.onload = function () {
                        var url = window.URL.createObjectURL(new Blob([reader.result], { type: 'text/csv' }));
                        var a = document.createElement('a');
                        a.style.display = 'none';
                        a.href = url;
                        a.download = 'import_errors.csv'; // The name of the file you want to download
                        document.body.appendChild(a);
                        a.click();
                        window.URL.revokeObjectURL(url);
                    };
                    reader.readAsText(new Blob([xhr.responseText]));
                    self.stopUpload();
                }
                if (xhr.status ===405 ) {
                    Picasso.Notification.show(
                            Picasso.Lang.get("notification_error_file_under_edit"),
                            "warning");

                    self.stopUpload();
                }

                if (onError) {
                    onError(status, xhr.status);
                }
            }
        });
    };

    /**
     * Start upload from queue
     *
     * @param {Function}  onStop  Stop callback
     **/

    Picasso.Dialog.Upload.prototype.startUpload = function (onStop) {

        var self = this;

        /* Toggle before removing the last entry */
        this.toggleButtons();

        var entry = this._queue.shift();

        this.transferFile(entry.file, entry.relPath,

            /* Success */
            function () {
                self._uploadedCount++;

                if (0 < self._queue.length) {
                    self.startUpload(onStop);
                } else if (onStop) {
                    self._xhr = null;
                    if (self._uploadedCount > 0) {
                        Picasso.Notification.show(self._uploadedCount > 1 ? Picasso.Lang.get(
                            "notification_uploads_success", self._uploadedCount) : Picasso.Lang.get("notification_upload_success"), "info", false, "ok");
                    } else {
                        Picasso.Notification.show(Picasso.Lang.get("notification_upload_success"), "info", false, "ok");
                    }
                    if (self._data && self._data.files) {
                        delete self._data.files;
                    }
                    onStop();
                }
            },

            /* Error */
            function (e) {
                if (0 < self._queue.length) {
                    self.startUpload(onStop);
                } else if (onStop) {
                    onStop();
                }
            }
        );
    };

    /**
     * Stop uploads
     **/

    Picasso.Dialog.Upload.prototype.stopUpload = function () {
        this._queue = [];
        this._xhr   = null;

        this.resetFileInputs();
        this.toggleButtons();
    };

    /**
     * Put file into upload queue
     *
     * @param {File|DataTransferItem}  target  Target to enqueue
     **/

    Picasso.Dialog.Upload.prototype.enqueue = function (target) {

        var self = this;

        /* Data processor */
        var processItem = function (entry) {

            if (entry.isFile ) {

                /* Load file object */
                entry.file(

                    /* Success */
                    function (file) {

                        /* Webkit does not support relative paths for drag & drop events -
                        so we have to construct the relative path manually */
                        var relPath = file.webkitRelativePath ?
                            null : entry.fullPath;

                        if (relPath) {
                            var i = relPath.lastIndexOf("/");
                            if (i >= 0) {
                                relPath = relPath.substring(0, i + 1);
                            }
                        }
                        self.enqueueFile(file, relPath);
                    },

                    /* Error */
                    function () {
                        /* We ignore this error */
                    }
                );

            } else if (entry.isDirectory) {

                // enqueue empty directories
                var reader = entry.createReader();
                reader.readEntries(function (entries) {
                    //this is an empty dir just append it
                   if(entries.length ===0){
                       self.enqueueFile(entry, entry.fullPath);
                   }
               });
                //get all files inside the directory to complete what was missing
                readAllFiles(entry, function(files) {
                    files.forEach(function (file) {
                        processItem(file)
                    });
                })
            }
        };

        /* Process item */
        processItem(target.webkitGetAsEntry());
    };

    /**
     * Put file into upload queue
     *
     * @param {File}    file  Target to enqueue
     * @param relPath   Only for drag & drop events. Is constructed manually beforehand.
     **/

    Picasso.Dialog.Upload.prototype.enqueueFile = function (file, relPath) {

        /* Update path */
        if (!relPath) {
            let li =-1;
            if (file.webkitRelativePath) {
                 li = file.webkitRelativePath.lastIndexOf("/");
            }
            if (li && li >= 0) {
                relPath = file.webkitRelativePath.substring(0, li);
            }
        }

        if (null !== this._data && this._data.path != null) {
            relPath = Picasso.Helper.joinPaths([
                this._data.path,
                relPath
            ]);
        }
        if (!relPath) {
            relPath = "";
        }

        if ("/" === relPath[0]) {
            relPath = relPath.slice(1);
        }

        /* Allow same named files in different sub dirs and check if file already exists */
        var id = "upload_" + Picasso.Helper.hashCode(relPath + "/" + file.name);

        if (0 !== $("#" + id).length) {
            return null;
        }

        this._queue.push({ file: file, relPath: relPath });

        /* Finally append to table */
        this._table.append({
                id: id,
                dialog: this,
                file: file,
                path: relPath
            }, renderFile
        );

        this.toggleButtons();
    };

    /**
     * Set handler for done callback
     *
     * @param {Function}  onDone  Done handler
     **/

    Picasso.Dialog.Upload.prototype.setDoneHandler = function (onDone) {
        if (onDone) {
            this._onDone = onDone;
        }
    };

    /**
     * Set handler for close callback
     *
     * @param {Function}  onClose  Close handler
     **/

    Picasso.Dialog.Upload.prototype.setCloseHandler = function (onClose) {
        if (onClose) {
            this._onClose = onClose;
        }
    };

    /**
     * Set upload url
     *
     * @param {String}  url  Url to upload to
     **/

    Picasso.Dialog.Upload.prototype.setUrl = function (url) {
        this._url = url;
    };

    /**
     * Set additional upload data
     *
     * @param {Object}  data  Object with data
     **/

    Picasso.Dialog.Upload.prototype.setData = function (data) {
        this._data = data;
    };

    /**
     * Show upload dialog
     **/

    Picasso.Dialog.Upload.prototype.show = function () {
        this._table.checkIfEmpty();

        this._dialog.modal();
    };

    /**
     * bypass DirectoryReader limitation by reading recursivly all files
     **/

    function readAllFiles(folder, callback) {
        var reading = 0
        var files = []
        function partialRead(reader) {
            reading = reading + 1
            reader.readEntries(function(entries) {
                reading = reading - 1
                for (var i= 0; i<entries.length;i++) {
                    var entry = entries[i];
                    if(entry.isDirectory) {
                        files.push(entry)
                        partialRead(entry.createReader())
                    } else {
                        files.push(entry)
                    }
                }
                if(entries.length) {
                    partialRead(reader)
                } else if(reading == 0) {
                    callback(files)
                }
            })
        }
        partialRead(folder.createReader())
    }


    /* Global - must be defined after constructor */

    /**
     * Change avatar for given object
     *
     * @param {Picasso.Account|Picasso.Group}  obj        Either a #{@link Picasso.Account} or #{@link Picasso.Group}
     * @param {Function}                       onSuccess  Success callback
     **/

    Picasso.Dialog.Upload.changeAvatar = function (obj, onSuccess) {

        /* Show dialog */
        var upload = new Picasso.Dialog.Upload(
            Picasso.Lang.get("dialog_title_avatar"),
            Picasso.Lang.get("dialog_body_avatar"));

        upload.setUrl("/avatars/" + obj.getOid());
        upload.setAllowMultiple(false);
        upload.setAllowDirectories(false);
        upload.setStartAutomatically(true);

        /* Limit types */
        upload.setAcceptTypes([
            "jpg", "jpeg", "png", "gif", "bmp"
        ].map(function (type) { return "image/" + type }));

        upload.setDoneHandler(function () {

            /* Basically just force a reload of avatars */
            var imgs = $(".pica-avatar");

            $.each(imgs, function () {
                var that = $(this);

                that.attr("src", that.attr("src") +
                    "?" + new Date().getTime());
            });


            if (onSuccess) {
                onSuccess();
            }
        });

        upload.toggleButtons();
        upload.show();
    };

    /**
     * Delete avatar for given object
     *
     * @param {Picasso.Account|Picasso.Group}  obj        Either a #{@link Picasso.Account} or #{@link Picasso.Group}
     * @param {Function}                       onSuccess  Success callback
     **/

    Picasso.Dialog.Upload.deleteAvatar = function (obj, onSuccess) {

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_delete"),
            Picasso.Lang.get("dialog_body_delete",
                Picasso.Lang.get("label_avatar")));

        dialog.setOkHandler(function () {
            Picasso.Helper.fireAjax("/avatars/" + obj.getOid(),
                { method: "DELETE" },

                /* Success */
                function () {
                    /* Basically just force a reload of avatars */
                    var imgs = $(".pica-avatar");

                    $.each(imgs, function () {
                        var that = $(this);

                        that.attr("src", that.attr("src") + "?" +
                            new Date().getTime());
                    });

                    if (onSuccess) onSuccess();
                }
            );
        });

        dialog.show();
    };
})();
