/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Dialog")) {
    window.Picasso.Dialog = {};
}

/* Private scope */
(function () {

    /**
     * Copy data from inputs to link
     *
     * @param {Picasso.Dialog.Uploadform}  dialog  A #{@link Picasso.Dialog.Uploadform}
     * @param {Object}                     inputs  Input fields ({ "id" = HTMLElement })
     **/

    var copyData = function (dialog, inputs) {
        dialog._uploadform.setHeading(inputs.pica_uploadform_heading.val());
        dialog._uploadform.setDescription(Picasso.Helper.getEditorData('pica_uploadform_description'));
        dialog._uploadform.setValidTill(Picasso.Helper.getDateTimePickerValue(inputs.pica_uploadform_valid_till, -1));
        dialog._uploadform.setMails(inputs.pica_uploadform_mails.val());
        dialog._uploadform.setPassword(inputs.pica_uploadform_password.val());
        dialog._uploadform.setNotifyOwner(inputs.pica_uploadform_notify_owner.is(":checked"));
        dialog._uploadform.setNotifyAdmins(inputs.pica_uploadform_notify_admins.is(":checked"));
        dialog._uploadform.setNotifyAdditional(inputs.pica_uploadform_notify_additionals.is(":checked"));
    }

    /**
     * Assign data to inputs
     *
     * @param {Picasso.Dialog.Uploadform}  dialog  A #{@link Picasso.Dialog.Uploadform}
     * @param {Object}                     inputs  Input fields ({ "id" = HTMLElement })
     **/

    var assignData = function (dialog, inputs) {
        /* Assign heading and description */
        inputs.pica_uploadform_heading.val(
            dialog._uploadform.getHeading());
        inputs.pica_uploadform_description.val(
            dialog._uploadform.getDescription());

        /* Assign expiration date */
        if (-1 !== dialog._uploadform.getValidTill()) {
            Picasso.Helper.setDateTimePickerValue(
                inputs.pica_uploadform_valid_till,
                dialog._uploadform.getValidTill());
        }

        /* Assign mails */
        inputs.pica_uploadform_mails.val(dialog._uploadform.getMails());

        /* Assign password */
        inputs.pica_uploadform_password.val(dialog._uploadform.getPassword());

        if (dialog._uploadform.getNotifyOwner()) {
            inputs.pica_uploadform_notify_owner.trigger("click");
        }
        if (dialog._uploadform.getNotifyAdmins()) {
            inputs.pica_uploadform_notify_admins.trigger("click");
        }
        if (dialog._uploadform.getNotifyAdditional()) {
            inputs.pica_uploadform_notify_additionals.trigger("click");
        }

    };

    /**
     * Save {@link Picasso.Uploadform}data
     *
     * @param {Picasso.Dialog.Uploadform}  dialog     A #{@link Picasso.Dialog.Uploadform}
     * @param {Function}                   onSuccess  Call on success
     * @param {Function}                   onError    Call on error
     **/

    var saveData = function (dialog, onSuccess, onError) {
        dialog._uploadform.save(

            /* Success */
            function (json, status) {
                switch (status) {
                    case 200:
                    case 201:
                        var ary = Picasso.Helper.getResultArray(json);

                        /* Update object */
                        dialog._uploadform.setUrl(ary[0].url);

                        if (onSuccess) onSuccess(dialog._uploadform);
                        break;
                }
            }, onError
        );
    };

    /**
     * Constructor for {@link Picasso.Dialog.Uploadform}
     *
     * @constructor
     **/

    Picasso.Dialog.Uploadform = function () {

        /* Init */
        this._uploadform = null;

        /* Sanity check */
        var dialog = $("#pica_uploadform_dialog");

        if (0 === dialog.length) {
            throw "Dialog not found";
        }

        Picasso.Helper.createDateTimepicker("#pica_uploadform_valid_till");
    };

    /**
     * Create {@link Picasso.Uploadform} via dialog
     *
     * @param {Picasso.File}  file       A #{@link Picasso.File}
     * @param {Function}      onSuccess  Call on success
     * @param {Function}      onError    Call on error
     **/

    Picasso.Dialog.Uploadform.prototype.create = function (file, onSuccess,
                                                           onError)
    {
        /* Handle types */
        if (file.is(Picasso.File.Flags.FOLDER)) {
            var self = this;


            /* Show dialog */
            var dialog = new Picasso.Dialog("#pica_uploadform_dialog",
                Picasso.Lang.get("dialog_title_create"),
                Picasso.Lang.get("dialog_body_create", Picasso.Lang.get("label_upload_form")),
                (Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.HIDE_ON_ESC|
                    Picasso.Dialog.Flags.NO_AUTOCLOSE|Picasso.Dialog.Flags.SPINNER));

            dialog.setAssignHandler(function (inputs) {
                $("#pica_uploadform_dialog .pica-hide-on-create").hide();
                Picasso.Helper.createEditor("pica_uploadform_description")
            });

            dialog.setCloseHandler(function () {
                Picasso.Helper.destroyEditor("pica_uploadform_description");
            })
            dialog.setCancelHandler(function () {
                Picasso.Helper.destroyEditor("pica_uploadform_description");
            })

            dialog.setOkHandler(function (inputs) {
                self._uploadform = new Picasso.Uploadform();

                self._uploadform.setFile(file);

                copyData(self, inputs);
                saveData(self,

                    /* Success */
                    function () {
                        dialog.hide();

                        if (onSuccess) {
                            onSuccess.apply(null, arguments);
                        }
                    },

                    /* Error */
                    function (e, status) {
                        dialog.showSpinner(false);
                        dialog.hide();
                        if (onError) {
                            onError(e, status);
                        }
                    }
                );
            });

            dialog.show();
        } else if (onError) {
            onError();
        }
    };

    /**
     * Edit {@link Picasso.Uploadform} via dialog
     *
     * @param {Picasso.Uploadform}  uploadform  A #{@link Picasso.Uploadform}
     * @param {Function}            onSuccess   Call on success
     * @param {Function}            onError     Call on error
     **/

    Picasso.Dialog.Uploadform.prototype.edit = function (uploadform,
                                                         onSuccess, onError)
    {
        var self = this;

        this._uploadform = uploadform;

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_uploadform_dialog",
            Picasso.Lang.get("dialog_title_edit"),
            Picasso.Lang.get("dialog_body_edit", Picasso.Lang.get("label_upload_form")),
            (Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE|Picasso.Dialog.Flags.SPINNER));

        dialog.setAssignHandler(function (inputs) {
            $("#pica_uploadform_dialog .pica-hide-on-create").show();

            assignData(self, inputs);
            Picasso.Helper.createEditor("pica_uploadform_description",'200px')
        });

        dialog.setCloseHandler(function () {
            Picasso.Helper.destroyEditor("pica_uploadform_description");
        })
        dialog.setCancelHandler(function () {
            Picasso.Helper.destroyEditor("pica_uploadform_description");
        })

        dialog.setOkHandler(function (inputs) {
            copyData(self, inputs);
            saveData(self,

                /* Success */
                function () {
                    dialog.hide();

                    if (onSuccess) {
                        onSuccess.apply(null, arguments);
                    }
                },

                /* Error */
                function (e, status) {
                    dialog.showSpinner(false);

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
            Picasso.Helper.destroyEditor("pica_uploadform_description");
        });

        dialog.show();
    };
})();
