/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

/**
 * WARNING: Code here MUST work without jquery, since we load it early on
 *          and jQuery might not be ready yet
 **/

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Lang")) {
    window.Picasso.Lang = {};
}

/* Private scope */
(function () {

    /* Globals */
    Picasso.Lang.LANG = null;
    Picasso.Lang.LOCALE = {};

    /**
     * Load language data
     *
     * @param {String}  lang  Language name
     **/

    var load = function (lang) {
        Picasso.Lang.LANG = lang;
        Picasso.Lang.LOCALE = {};

        /* Create xhr */
        var xhr = new XMLHttpRequest();

        xhr.onreadystatechange = function () {
            if (4 === xhr.readyState &&
                (200 === xhr.status || 0 === xhr.status)) {
                /* Read json file */
                try {
                    Picasso.Lang.LOCALE = JSON.parse(xhr.responseText);

                    Picasso.Lang.update();
                } catch (e) {
                    console.debug(e);
                }
            }
        };

        /* Now load our language */
        var url = "/locales/lang_" + lang + ".json";

        /* Load the lang file synchronous to stop other scripts from
         * executing until loading is done */
        xhr.open("GET", url, false);
        xhr.send(null);

        console.debug("Yes, Chrome, this synchronous request is intended!");
    };

    /**
     * Set attribute of given element if found
     *
     * @param {String}  elemId     Element Id
     * @param {String}  attrName   Name of the attribute to set
     * @param {String}  attrValue  New value of the attribute
     **/

    var setAttrIfExist = function (elemId, attrName, attrValue) {
        var elem = document.getElementById(elemId);

        if (null !== elem) {
            elem.setAttribute(attrName, attrValue);
        }
    };

    /**
     * Apply plural and modifiers to locales
     *
     * @param {String}              value  Value to apply changes to
     * @param {Picasso.Lang.Flags}  flags  Whether to use plural if any
     *
     * @returns {String} Formatted string
     **/

    var applyModifiers = function (value, flags) {
        var tag = value;

        /* Check plural */
        if (Array.isArray(value)) {
            if (0 < (flags & Picasso.Lang.Flags.PLURALIZE)) {
                tag = value[1];
            } else {
                tag = value[0];
            }
        }

        /* Apply formats */
        if (0 < (flags & Picasso.Lang.Flags.DOWNCASE)) {
            tag = tag.toLowerCase();
        }

        if (0 < (flags & Picasso.Lang.Flags.UPCASE)) {
            tag = tag.toUpperCase();
        }

        if (0 < (flags & Picasso.Lang.Flags.CAPITALIZE)) {
            tag = Picasso.Helper.capitalize(tag);
        }

        return tag;
    };

    /**
     * Format elements based on attribute
     *
     * @param {Object}  elem  HTML Element
     * @param {String}  attr  Name of the attribute
     **/

    var formatElement = function (elem, attr) {
        var value = Picasso.Lang.find(elem.getAttribute(attr));
        var flags = 0;

        /* Get plural */
        var plural = elem.getAttribute("data-plural");

        /* True in HTML isn't necessarily true in JS.. */
        if ("true" === plural || true === plural) {
            flags |= Picasso.Lang.Flags.PLURALIZE;
        }

        /* Get format */
        var format = elem.getAttribute("data-format");

        switch (format) {
            case "downcase":
                flags |= Picasso.Lang.Flags.DOWNCASE;
                break;

            case "upcase":
                flags |= Picasso.Lang.Flags.UPCASE;
                break;

            case "capitalize":
                flags |= Picasso.Lang.Flags.CAPITALIZE;
                break;

            default:
                break;
        }

        return applyModifiers(value, flags);
    };

    var replaceParams = function (tag, flags, params) {
        for (var paramIdx = 1, i = 0; i <= params.length; i++, paramIdx++) {
            var idx1 = -1;

            var inflectNums = (0 < (flags & Picasso.Lang.Flags.INFLECT) &&
                "number" === typeof params[i]);

            /* Just do this when really required before replacing */
            if (inflectNums) {
                idx1 = tag.indexOf("%" + paramIdx);
            }

            tag = tag.replace("%" + paramIdx, params[i]);

            /* Replace (s) accordingly to count */
            if (inflectNums) {
                var idx2 = tag.indexOf("(s)");

                /* Check there is a (s) before the next % symbol */
                if (idx1 < idx2) {
                    var value = "";

                    switch (params[i]) {
                        case 1:
                            value = "";
                            break;

                        case 0:
                        default:
                            value = "s";
                    }

                    tag = tag.replace("(s)", value);
                }
            }
        }

        return tag;
    };

    /**
     * Try and load language based on various environmental settings
     **/

    Picasso.Lang.detect = function () {
        var lang = Picasso.Lang.LANG;

        if (null === lang) {
            /* Check browser settings */
            if (document.documentElement.lang){
                lang = document.documentElement.lang
            }
            else if (null !== navigator) {

                if (null !== navigator.language) {
                    lang = navigator.language;
                } else if (null !== navigator.browserLanguage) {
                    lang = navigator.browserLanguage;
                } else if (null !== navigator.systemLanguage) {
                    lang = navigator.systemLanguage;
                } else if (null !== navigator.userLanguage) {
                    lang = navigator.userLanguage;
                } else if (null !== navigator.userAgent &&
                    (lang = navigator.userAgent.match(/android.*\W(\w\w)-(\w\w)\W/i))) {
                    lang = lang[1];
                }

                /* Post-processing.. */
                lang = lang.substr(0, 2).toLowerCase();
            } else {
                lang = "en"; ///< Default to EN
            }

            /* Restrict languages, since we just support DE/EN */
            //if ("de" !== lang) lang = "en";

            load(lang);
        }
    };

    /**
     * Get localization string for given tag
     *
     * @param {Picasso.Lang.Flags}  flags  Localize options (optional)
     * @param {String}              tag    Locale tag
     * @param {*}                   ...    Variadic number of parameters (optional)
     *
     * @returns {String} Either found string otherwise tag name
     **/

    Picasso.Lang.get = function () {
        /* Variadic action now */
        var tag = (arguments[0] || "error");
        var flags = 0;
        var argIdx = 1;

        /* Check whether first argument is flags */
        if ("number" === typeof arguments[0]) {
            flags = arguments[0];
            tag = arguments[1];
            argIdx++; ///< arguments: [Flags, Tag, Param1, ..])
        }

        /* Check whether key exists at all */
        if (Picasso.Lang.LOCALE.hasOwnProperty(tag)) {
            tag = applyModifiers(Picasso.Lang.LOCALE[tag], flags);

            /* HINT: arguments is NOT an array, although it almost behaves like
             *       it is. The first slice just converts it to an array */
            tag = replaceParams(tag, flags,
                Array.prototype.slice.call(arguments).slice(argIdx));
        }

        return tag;
    };

    /**
     * Find localization string for given tag
     *
     * @param {String}  tag  Tag name of localization
     * @returns {String} Either found string otherwise tag name
     **/

    Picasso.Lang.find = function (tag) {
        return (Picasso.Lang.LOCALE.hasOwnProperty(tag) ?
            Picasso.Lang.LOCALE[tag] : tag);
    };

    /**
     * Update all lang tags
     **/

    Picasso.Lang.update = function () {

        /* WARNING: We have to call this twice, because timing can be a bit off:
         *          1) When the ajax load of the locale file is ready (here)
         *          2) When the footer is reached (footer) */
        var nkeys = Object.keys(Picasso.Lang.LOCALE).length;

        if (0 === nkeys || null === document.body) return; ///< Skip this when no locales are loaded yet

        /* Find all lang tags and replace content */
        var elems = document.body.getElementsByTagName("lang");

        for (var i = 0; i < elems.length; i++) {
            var elem = elems[i];
            var params = [];

            var formatted = formatElement(elem, "name");

            /* Check elem has children; if so treat them as params */
            if (elem.hasChildNodes()) {
                for (var j = 0; j < elem.childNodes.length; j++) {
                    if ("LANG" === elem.childNodes[j].nodeName) {
                        params.push(formatElement(elem.childNodes[j], "name"));
                    }
                }

                formatted = replaceParams(formatted, 0, params);
            }

            /* Finally replace inner HTML */
            elem.innerHTML = formatted;
        }

        /* Find all tooltips */
        elems = document.body.querySelectorAll("[data-tooltip]");

        for (var i = 0; i < elems.length; i++) {
            elems[i].setAttribute("title",
                formatElement(elems[i], "data-tooltip"));
        }

        /* Find all placeholder */
        elems = document.body.querySelectorAll("[data-placeholder]");

        for (var i = 0; i < elems.length; i++) {
            elems[i].setAttribute("placeholder",
                formatElement(elems[i], "data-placeholder"));
        }

        /* Update various special elements */
        setAttrIfExist("login_submit", "value",
            Picasso.Lang.get("action_log_in"));
        setAttrIfExist("register_submit", "value",
            Picasso.Lang.get("button_register"));
        setAttrIfExist("changepass_submit", "value",
            Picasso.Lang.get("button_change"));
        setAttrIfExist("activation_submit", "value",
            Picasso.Lang.get("button_activate"));
    };

    /* Flags - must be defined after constructor */

    if (!window.Picasso.Lang.hasOwnProperty("Flags")) {
        window.Picasso.Lang.Flags = {};
    }

    /* Flags */
    Picasso.Lang.Flags.SINGULARIZE = (1 << 0); ///< Prefer singular
    Picasso.Lang.Flags.PLURALIZE = (1 << 1); ///< Prefer plural
    Picasso.Lang.Flags.CAPITALIZE = (1 << 2); ///< Capitalize string
    Picasso.Lang.Flags.DOWNCASE = (1 << 3); ///< Convert string to lower case
    Picasso.Lang.Flags.UPCASE = (1 << 4); ///< Convert string to upper case
    Picasso.Lang.Flags.INFLECT = (1 << 5); ///< Inflect singular/plural based on numbers

    /* Detect and set up language when document is ready without jQuery */
    Picasso.Lang.detect();
})();
