/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Dashboard")) {
    window.Picasso.Dashboard = {};
}

/* Private scope */
(function () {

    /* Globals */
    var CACHE = [];

    /* Create charts */
    var colors = Picasso.Helper.getChartColors();

    var CHART1 = Picasso.Helper.createChart("#dashboard_chart1", colors);
    $("#dashboard_chart1").css("height","");
    var CHART2 = Picasso.Helper.createChart("#dashboard_chart2",
        colors, Picasso.Helper.formatSize);
    $("#dashboard_chart2").css("height","");

    var CHART3;

    var account = Picasso.get("account");

    if (account.is(Picasso.Account.Flags.ADMIN)) {
        CHART3 = Picasso.Helper.createChart("#dashboard_chart3", colors);
        $("#dashboard_chart3").css("height","");

    }

    var LICENSE_TABLE = new Picasso.Table("#dashboard_license_table");
    var GROUPS_TABLE = new Picasso.Table("#dashboard_groups_table");
    var ORGANIZATIONS_TABLE = new Picasso.Table("#dashboard_organizations_table");
    var LOGS_TABLE = new Picasso.Table("#dashboard_logs_table");

    /**
     * Handle click on group manage button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickGroupEdit = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var group = new Picasso.Group($(this).data("serial"));

        var dialog = new Picasso.Dialog.Group();

        dialog.edit(group,

            /* Success */
            function (group) {
                /* Replace row */
                var rows = $("." + group.getElemId());
                var rendered = renderGroup(group);

                rows.first().before(rendered);
                rows.remove();

                $.each(rendered, function () {
                    this.addClass("pica-highlight");
                });

                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_updated",
                        Picasso.Lang.get("label_groups")), "info");
            }
        );
    };

    /**
     * Handle click on organization manage button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickOrganizationEdit = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get either selected organization or the one from current tab */
        var organization;
        var serial = $(this).data("serial");

        if (serial) {
            organization = new Picasso.Organization(serial);
        } else {
            organization = CACHE[$("#dashboard_nav li.active a").data("idx")];
        }

        /* Show dialog */
        var dialog = new Picasso.Dialog.Organization();

        dialog.edit(organization,

            /* Success */
            function (organization) {
                /* Replace row */
                var rows = $("." + organization.getElemId());
                var rendered = renderOrganization(organization);

                rows.first().before(rendered);
                rows.remove();

                $.each(rendered, function () {
                    this.addClass("pica-highlight");
                });

                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_updated",
                        Picasso.Lang.get("label_organizations")), "info");
            }
        );
    };

    /**
     * Handle window resize events
     **/

    var handleResize = function () {
        $("canvas").each(function () {
            var that = $(this);

            this.width =  that.parent().width();

            $("#dashboard_nav li.active a").trigger("click");
        });
    };

    /**
     * Render given json object
     *
     * @param {Object}  obj  Given json object
     *
     * @returns {HTMLElement} Rendered json object
     **/

    var renderLicense = function (obj) {
        var rows = [];

        /* 1. row: Mail */
        var tr = $("<tr>");

        tr.append($("<td>", {
            html: Picasso.Helper.createLang("label_emails")
        }));

        tr.append($("<td>", { text: obj.email }));

        rows.push(tr);

        /* 2. row: Valid till */
        tr = $("<tr>");

        tr.append($("<td>", {
            html: Picasso.Helper.createLang("label_valid_till")
        }));

        var td = $("<td>");

        var span = $("<span>", {
            "data-tooltip": "",
            title: Picasso.Helper.formatAge(obj.expires),
            text: Picasso.Helper.formatDate(obj.expires, true)
        });

        span.tooltip({ container: "body" });

        td.append(span);
        tr.append(td);

        rows.push(tr);

        /* 3. row: Users */
        tr = $("<tr>");

        tr.append($("<td>", {
            html: Picasso.Helper.createLang("label_accounts")
        }));

        tr.append($("<td>", {
            text: obj.users
        }));

        rows.push(tr);

        return rows;
    };

    /**
     * Render given json object
     *
     * @param {Object}  obj  Given json object
     *
     * @returns {HTMLElement} Rendered json object
     **/

    var renderGroup = function (group) {
        var rows = [];

        /* 1. row: Name */
        var tr = $("<tr>", {
            class: group.getElemId()
        });

        var td = $("<td>", {
            colspan: 2,
        });

        var a = $("<a>", {
            text: group.getName(),
            "data-serial": group.serialize()
        });

        a.click(handleClickGroupEdit);

        td.append(a);
        tr.append(td);

        rows.push(tr);

        /* 2. row: Accounts */
        tr = $("<tr>", {
            class: group.getElemId()
        });

        tr.append($("<td>", {
            html: Picasso.Helper.createLang(Picasso.Lang.Flags.PLURALIZE,
                "label_accounts")
        }));

        var td = $("<td>");

        td.append($("<span>", {
            text: group.getAccountsCount()
        }));

        tr.append(td);

        rows.push(tr);

        /* 3. row: Folders */
        tr = $("<tr>", {
            class: group.getElemId()
        });

        tr.append($("<td>", {
            html: Picasso.Helper.createLang(Picasso.Lang.Flags.PLURALIZE,
                "label_folders")
        }));

        td = $("<td>");

        td.append($("<span>", {
            text: group.getFoldersCount()
        }));

        tr.append(td);

        rows.push(tr);

        return rows;
    };

    /**
     * Render given json object
     *
     * @returns {*}
     * @param organization
     **/

    var renderOrganization = function (organization) {
        var rows = [];

        /* 1. row: Name */
        var tr = $("<tr>", {
            class: organization.getElemId()
        });

        var td = $("<td>", { class: "text-center pica-icon-column" });


        Picasso.Helper.createGlyphAndAvatar(td, "education", organization);

        tr.append(td);

        var td = $("<td>", {
            colspan: 1
        });

        var a = $("<a>", {
            text: organization.getName(),
            "data-serial": organization.serialize()
        });

        a.click(handleClickOrganizationEdit);

        td.append(a);
        tr.append(td);

        /* 3. column: reseller */
        td = $("<td>", {
            class: "text-right",
        });
        if (organization.getHasReseller()) {
            td.append(Picasso.Helper.createGlyph("crown","pica-glyph"))
        }
        tr.append(td)

        rows.push(tr);

        /* 2. row: Accounts */
        tr = $("<tr>", {
            class: organization.getElemId()
        });

        tr.append($("<td>", {
            html: Picasso.Helper.createLang(Picasso.Lang.Flags.PLURALIZE,
                "label_accounts")
        }));

        var td = $("<td>");

        td.append($("<span>", {
            text: organization.getAccountsCount()
        }));

        if (0 < organization.getAccountsMax()) {
            td.append($("<span>", {
                class: "pica-slash-before",
                text: organization.getAccountsMax()
            }));
        }

        tr.append(td);

        rows.push(tr);

        /* 3. row: Quota */
        tr = $("<tr>", {
            class: organization.getElemId()
        });

        tr.append($("<td>", {
            html: Picasso.Helper.createLang(Picasso.Lang.Flags.PLURALIZE,
                "label_storage")
        }));

        td = $("<td>");

        td.append($("<span>", {
            text: Picasso.Helper.formatSize(organization.getQuotaUsed())
        }));

        if (0 < organization.getQuotaTotal()) {
            td.append($("<span>", {
                class: "pica-slash-before",
                text: Picasso.Helper.formatSize(organization.getQuotaTotal())
            }));
        }

        tr.append(td);

        rows.push(tr);

        /* 3. row: Valid till */
        if (-1 !== organization.getValidTill()) {
            tr = $("<tr>");

            tr.append($("<td>", {
                html: Picasso.Helper.createLang("label_valid_till")
            }));

            td = $("<td>");

            var span = $("<span>", {
                "data-tooltip": "",
                title: Picasso.Helper.formatAge(organization.getValidTill()),
                text: Picasso.Helper.formatDate(organization.getValidTill(), true)
            });

            span.tooltip({ container: "body" });

            td.append(span);
        }

        tr.append(td);

        rows.push(tr);

        return rows;
    };

    /**
     * Add new nav tab
     *
     * @param {String}    name     Name of tab
     * @param {String}    tooltip  Tooltip for tab
     * @param {Object}    data     Chart data
     * @param {Function}  handler  Callback handler to update data (optional)
     **/

    var addNavTab = function (name, tooltip, data, handler) {
        var nav = $("#dashboard_nav");

        /* Create new tab */
        var li = $("<li>");

        var a = $("<a>", {
            "data-idx": CACHE.length,
            text: name
        });

        /* Add click handler */
        a.click(function () {
            var that = $(this);

            /* Toggle active state */
            $("#dashboard_nav li.active").removeClass("active");

            that.parents("li").addClass("active");

            /* Call handler if any */
            if (handler) {
                handler.apply(this, [ CACHE[that.data("idx")] ]);
            }
        });

        if (tooltip ) {
            var div = $("<div>");

            var span = $("<span>", {
                class: "text-info",
                text: tooltip
            });

            a.append(span);
            div.append(a);
            li.append(div);
        } else {
            li.append(a);
        }

        nav.append(li);

        CACHE.push(data);
    };

    /**
     * Fetch JSON data
     **/

    Picasso.Dashboard.fetch = function () {
        var account = Picasso.get("account");
        var foldersMountDynamic = Picasso.get("folders_mount_dynamic");

        /* Fetch data based on role */
        if (account.is(Picasso.Account.Flags.ADMIN)) {
            var overview = {
                nAccounts: 0,
                spaceUsageBytes: 0,
                spaceFreeBytes: 0,
                nFolders: 0
            };

            /* Create update handler */
            var handler = function (server) {
                /* Update charts */
                Picasso.Helper.updateChart(CHART1, {
                    "chart_used": server.nAccounts,
                    "chart_remaining": 0
                });

                Picasso.Helper.updateChart(CHART2, {
                    "chart_used": server.spaceUsageBytes,
                    "chart_remaining": server.spaceFreeBytes,
                });

                Picasso.Helper.updateChart(CHART3, {
                    "chart_used": server.nFolders,
                    "chart_remaining": 0
                });
            };

            /* Add overview tab */
            addNavTab(Picasso.Lang.get("label_overview"), "",
                overview, handler);

            /* Fetch servers */
            Picasso.Helper.fireAjax("/wapi/servers", { action: "getAll" },

                /*  Success */
                function (json) {
                    var ary = Picasso.Helper.getResultArray(json);

                    $.each(ary, function (i, server) {
                        addNavTab(server.name, server.programVersion,
                            server, handler);

                        /* Create sums for overview */
                        overview.nAccounts += server.nAccounts;
                        if (foldersMountDynamic) {
                            if (server.spaceUsageBytes && server.spaceFreeBytes) {
                                overview.spaceUsageBytes = server.spaceUsageBytes;
                                overview.spaceFreeBytes = server.spaceFreeBytes;
                            }
                        } else {
                            overview.spaceUsageBytes += server.spaceUsageBytes;
                            overview.spaceFreeBytes += server.spaceFreeBytes;
                        }
                        overview.nFolders += server.nFolders;
                    });

                    /* Trigger click on first tab when complete */
                    $("#dashboard_nav li:first-child a").trigger("click");
                }
            );

            /* Fetch organizations */
            Picasso.Organization.getAll(

                /* Success */
                function (organizations, json) {
                    ORGANIZATIONS_TABLE.update(organizations, renderOrganization);
                }, null, 1, 2
            );

            /* Fetch groups */
            Picasso.Group.getAll(

                /* Success */
                function (groups, json) {
                    GROUPS_TABLE.update(groups, renderGroup);
                }, null, 1, 2
            );

            LOGS_TABLE.updateFromJSON("/wapi/logs", { action: "getAll" },
                Picasso.Logs.renderLogs);

            /* Fetch license */
            LICENSE_TABLE.updateFromJSON("/wapi/license", { action: "getInfo" },
                renderLicense);
        } else if (account.is(Picasso.Account.Flags.ORG_ADMIN)) {
            /* Create update handler */
            var handler = function (organization) {
                /* Fetch organization groups */
                organization.getGroups(

                    /* Success */
                    function (group, json) {
                        GROUPS_TABLE.update(group, renderGroup);
                    }, null, 1, 5
                );

                /* Fetch organization folders */
                ORGANIZATIONS_TABLE.update(organization, renderOrganization);

                /* Update charts */
                Picasso.Helper.updateChart(CHART1, {
                    "chart_used": organization.getAccountsCount(),
                    "chart_remaining": Math.abs(organization.getAccountsMax() -
                        organization.getAccountsCount()),
                });

                Picasso.Helper.updateChart(CHART2, {
                    "chart_assigned": organization.getQuotaUsed(),
                    "chart_remaining": organization.getQuotaTotal() -
                        organization.getQuotaUsed()
                });
            };

            /* Fetch organizations */
            Picasso.Organization.getAll(
                function (organizations, json) {
                    /* Add tabs for each organization */
                    $.each(organizations, function (i, organization) {
                        addNavTab(organization.getName(), "", organization, handler);

                        /* Trigger click on first tab immediately */
                        if (0 === i) {
                            $("#dashboard_nav li:first-child a").trigger("click");
                        }
                    });

                }, null, 1, 10
            );
        }
    };

    /* Bind handlers */
    $("#dashboard_organization_edit").click(handleClickOrganizationEdit);
    $(window).on("resize", handleResize);

    /* Init */
    Picasso.Dashboard.fetch();
    handleResize();
})();

