/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Groups")) {
    window.Picasso.Groups = {};
}

/* Private scope */
(function () {

    /* Globals */
    var TABLE = new Picasso.Table("#groups_table",
        Picasso.Table.Flags.ROWS_SELECTABLE);
    var PAGE = 1;

    /**
     * Handle click on create button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCreate = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Show dialog */
        var dialog = new Picasso.Dialog.Group();

        dialog.create(

            /* Success */
            function () {
                Picasso.Groups.fetch();

                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_created",
                        Picasso.Lang.get("label_groups")), "info");
                window.location.reload();
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Handle click on edit button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickEdit = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        if (1 === selected.length) {
            var group = new Picasso.Group($(selected[0])
                .find(".pica-serial").data("serial"));

            /* Show dialog */
            var dialog = new Picasso.Dialog.Group();

            dialog.edit(group,

                /* Success */
                function (group) {
                    /* Replace row */
                    var row = $("#" + group.getElemId());

                    var rendered = renderGroup(group);

                    row.replaceWith(rendered);
                    rendered.addClass("pica-highlight");

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_updated",
                            Picasso.Lang.get("label_groups")), "info");
                }
            );
        }
    };

    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDelete = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Group($(val).find(".pica-serial").data("serial"))
        });

        /* Show dialog */
        var what = (1 === selected.length ? selected[0].getName() :
            Picasso.Lang.get("string_selected_elements", selected.length));

        let title = "dialog_title_delete"
        let body = "dialog_body_delete"
        let notification = "notification_item_deleted"



        var dialog = new Picasso.Dialog("#pica_confirmation_dialog", Picasso.Lang.get(title), Picasso.Lang.get(body, what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {
                var group = this;

                group.remove(

                    /* Success */
                    function () {
                        Picasso.Helper.fadeOut($("#" + group.getElemId()),
                            function () {
                                this.remove();

                                TABLE.checkIfEmpty();
                            });

                        Picasso.Notification.show(
                            Picasso.Lang.get(notification,
                                Picasso.Lang.get("label_groups")), "info");
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);
                    }
                );
            });
        });

        dialog.show();
    };

    /**
     * Handle click on leave button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickLeave = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Group($(val).find(".pica-serial").data("serial"))
        });

        /* Show dialog */
        var what = (1 === selected.length ? selected[0].getName() :
                Picasso.Lang.get("string_selected_elements", selected.length));

        let title = "dialog_title_leave"
        let body = "dialog_body_leave"
        let notification = "notification_item_left"


        var dialog = new Picasso.Dialog("#pica_confirmation_dialog", Picasso.Lang.get(title), Picasso.Lang.get(body, what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {
                var group = this;

                group.leave(

                        /* Success */
                        function () {
                            Picasso.Helper.fadeOut($("#" + group.getElemId()),
                                    function () {
                                        this.remove();

                                        TABLE.checkIfEmpty();
                                    });

                            Picasso.Notification.show(
                                    Picasso.Lang.get(notification,
                                            Picasso.Lang.get("label_groups")), "info");
                        },

                        /* Error */
                        function (e) {
                            Picasso.debugLog(e);
                        }
                );
            });
        });

        dialog.show();
    };

    /**
     * Handle filter event
     *
     * @param {String}  filterString  Input string for filter
     * @param {Number}  nmatches      Number of visible filter matches
     **/

    var handleFilter = function (filterString, nmatches) {

        /* Limit search */
        if (0 === nmatches && 1 <= filterString.length) {
            Picasso.Groups.fetch(null,filterString);
        } else if (0 === nmatches && 0 < filterString.length) {
            TABLE.showEmptyMessage();
        }  else if (0 === nmatches && 0 === filterString.length) {
            TABLE.clear();
            TABLE.hideEmptyMessage();
            var tr = $("<tr>", {
                class: "pica-not-striped"
            });

            var td = $("<td>", {class: "text-center", colspan: "6"});
            var span = $("<span>", {class: "pica-glyph pica-spinner glyphicons glyphicons-repeat"});
            td.append(span);
            tr.append(td);
            $("#groups_table").append(tr);
            Picasso.Groups.fetch();

        } else {
            TABLE.hideEmptyMessage();
        }
    };

    /**
     * Render given json object
     *
     * @param {Picasso.Group}  group  A {@type Picasso.Group}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderGroup = function (group) {

        let isSimpleAdmin = (group.is(Picasso.Group.Flags.GROUP_ADMIN) && group.getAdminsCount() === 1 );
        let isMultiAdmin = (group.is(Picasso.Group.Flags.GROUP_ADMIN) && group.getAdminsCount() > 1 );
        let isOrgAdmin = account.is(Picasso.Account.Flags.ORG_ADMIN) ;
        let isGroupMember = account.getAccountGroups().indexOf(group.getOid()) > -1;

        var tr = $("<tr>", {
            id: group.getElemId(),
            "data-search-keys": [group.getName(),group.getOid()],
            draggable: false
        });

        tr.click(function (e) {

            var container = $("#groups_table .pica-table-selection-context");

            let lang ;

            function createDeleteLink() {
                var a = $("<a>", {
                    class: "pica-table-selection-multi",
                });
                var span = $("<a>", {
                    class: "pica-table-selection-multi pica-glyph glyphicons",
                });
                a.addClass("groups_delete");
                span.addClass("glyphicons-bin")
                lang = Picasso.Helper.createLang("action_delete");

                a.on("click",function (e) {
                    handleClickDelete(e);
                });
                a.append(span);
                a.append(lang);
                return a;
            }

            function createEditLink() {
                var a = $("<a>", {
                    class: "pica-table-selection-multi",
                });
                var span = $("<a>", {
                    class: "pica-table-selection-multi pica-glyph glyphicons",
                });
                a.addClass("groups_edit");
                span.addClass("glyphicons-edit")
                lang = Picasso.Helper.createLang("action_edit");

                a.on("click",function (e) {
                    handleClickEdit(e);
                });
                a.append(span);
                a.append(lang);
                return a;
            }

            function createLeaveLink() {
                var a = $("<a>", {
                    class: "pica-table-selection-multi",
                });
                var span = $("<a>", {
                    class: "pica-table-selection-multi pica-glyph glyphicons",
                });
                a.addClass("groups_leave");
                span.addClass("glyphicons-door");
                lang = Picasso.Helper.createLang("action_leave");

                a.on("click",function (e) {
                    handleClickLeave(e);
                });
                a.append(span);
                a.append(lang);
                return a;
            }

            function displayDelete() {
                $(".groups_delete").remove();
                $(".groups_leave").remove();
                $(".groups_edit").remove();
                container.append(createEditLink);
                container.append(createDeleteLink);
            }
            function displayLeave() {
                $(".groups_leave").remove()
                $(".groups_delete").remove();
                $(".groups_edit").remove();
                container.append(createLeaveLink)
            }

            function displayDeleteAndLeave() {
                $(".groups_leave").remove()
                $(".groups_delete").remove();
                $(".groups_edit").remove();
                container.append(createEditLink);
                container.append(createLeaveLink)
                container.append(createDeleteLink);
            }



            let showDelete = isSimpleAdmin || (isOrgAdmin && ! isGroupMember )
            let showDeleteAndLeave =  isMultiAdmin || (isOrgAdmin && isGroupMember)
            if (showDelete) {
                displayDelete();
            } else if (showDeleteAndLeave) {
                displayDeleteAndLeave()
            } else {
                displayLeave();
            }


        });


        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        Picasso.Helper.createGlyphAndAvatar(td, "group", group);

        tr.append(td);

        /* 2. column: Name */
        td = $("<td>", {
            class: "pica-overflow pica-serial",
        });

        var a = $("<a>", {
            class: "pica-name",
            text: group.getName(),
        });

        if (group.is(Picasso.Group.Flags.GROUP_ADMIN)) {
            a.click(function (e) {
                tr.addClass(Picasso.Table.Marker.SELECTED);

                handleClickEdit(e);
            });
        }

        td.data("serial", group.serialize());

        td.append(a);
        tr.append(td);

        /* 3. column: Folders */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: group.getFoldersCount()
        }));

        /* 4. column: Members */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: group.getAccountsCount()
        }));

        /* 5. column: Permissions */
        td = $("<td>", {
            class: "hidden-xs",
        });

        let permission = "permission_you_member"

        if ((isMultiAdmin || isSimpleAdmin || isOrgAdmin) && !isGroupMember) {
            permission = "permission_you_admin"
        }
        if ((isMultiAdmin || isSimpleAdmin || isOrgAdmin) && isGroupMember) {
            permission = "permission_you_admin_and_member"
        }

        var button = $("<button>", {
            class: "btn btn-default disabled",
            text: Picasso.Lang.get(permission)
        });

        td.append(button);
        tr.append(td);

        /* 6. column: Organizations */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: group.getOrgName()
        }));

        return tr;
    };

    /**
     * Fetch JSON data
     *
     * @param {Boolean}  append  Whether to append data
     **/

    Picasso.Groups.fetch = function (append,query) {

        /* Reset page */
        if (true !== append) {
            PAGE = 1;
        }

        Picasso.Group.getAll(

            /* Success */
            function (groups, json) {
                if (true === append) {
                    TABLE.append(groups, renderGroup);
                } else {
                    TABLE.update(groups, renderGroup);
                }

                /* Bind infinite scroll */
                Picasso.Helper.bindInfiniteScroll(json, TABLE, PAGE,
                    function () {
                        PAGE++;

                        Picasso.Groups.fetch(true,query);
                    }
                );
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }, PAGE,null,query
        );
    };

    /* Bind handlers */
    $("#groups_create").click(handleClickCreate);
    $("#groups_edit").click(handleClickEdit);

    Picasso.Actionbar.setFilterHandler(handleFilter);

    /* Init */
    Picasso.Groups.fetch();
})();
