/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Logs")) {
    window.Picasso.Logs = {};
}

/* Private scope */
(function () {

    /* Globals */
    var TABLE = new Picasso.Table("#logs_table",
        Picasso.Table.Flags.ROWS_SELECTABLE);
    var TIMER = null;
    var TIMESTAMP = 0;

    /* Regexps */
    Picasso.Logs.REGEXPS = [
        new RegExp(/Account[ :]*'?([^\]']*)[ '\]@]*[ '\]@]+/),
        new RegExp(/for user (.*)/),
        new RegExp(/called by \[(?:Account : )?(.*)\]/),
        new RegExp(/authenticated (.*) from/)
    ];

    /**
     * Refresh logs
     *
     * @param {Boolean}  update  Whether to update or append
     **/

    var refreshLogs = function (update) {
        var url = "/wapi/logs";
        var data = {
            action: "getAll"
        };

        /* Append timestamp if any */
        if (0 < TIMESTAMP) {
            data.timestamp = TIMESTAMP;
        }

        Picasso.Helper.fireAjax(url, data,

            /*  Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                if (ary) {
                    if (true === update) {
                        TABLE.update(ary, Picasso.Logs.renderLogs);
                    } else {
                        TABLE.append(ary.reverse(),
                            Picasso.Logs.renderLogs, true, true);
                    }
                }
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );

        /* Save timestamp of last request */
        TIMESTAMP = new Date().getTime();
    };

    /**
     * Find account name in message string
     **/

    var findAccount = function (mesg) {
        var result = "";

        if (mesg) {
            $.each(Picasso.Logs.REGEXPS, function (i) {
                var match = mesg.match(this);

                if (match) {
                    result = match[1];

                    return;
                }
            });
        }

        return result;
    };

    /**
     * Handle click on refresh button
     **/

    var handleClickRefresh = function () {
        if (null == TIMER) {
            TIMER = setInterval(refreshLogs, 5000);

            Picasso.Notification.show(
                Picasso.Lang.get("label_refresh_start"), "info");
        } else {
            clearInterval(TIMER);

            TIMER = null;

            Picasso.Notification.show(
                Picasso.Lang.get("label_refresh_stop"), "info");
        }
    };

    /**
     * Handle click on copy button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCopy = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var logs = [];

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        $.each(selected, function () {
            var that = $(this);
            var tds = that.find("td");

            var message = [];

            message.push(tds[1].innerText);
            message.push(tds[2].innerText);
            message.push(tds[4].innerText);

            /* Check whether this entry has a stacktrace */
            if (that.hasClass("logs-has-stacktrace")) {
                message.push(that.next().find("pre").text());
            }

            logs.push(message.join("  "));
        });

        TABLE.deselectAll();

        /* Create invisible textarea */
        var textarea = $("<textarea>");

        $("body").append(textarea);

        textarea.val(logs.join("\n"));

        Picasso.Helper.copyToClipboard($(textarea));

        textarea.remove();
    };

    /**
     * Render given json object
     *
     * @param {Object}  obj  Given json object
     *
     * @returns {HTMLElement} Rendered json object
     **/

    Picasso.Logs.renderLogs = function (obj) {
        /* Handle level */
        var icon = "";
        var klass = "";

        switch (obj.level) {
            case "OFF":
                break;
            case "INFO":
                icon = "info-sign";
                break;
            case "FINE":
            case "FINER":
                icon = "question-sign";
                break;
            case "WARNING":
                icon = "warning-sign";
                klass = "warning";
                break;
            case "SEVERE":
                icon = "exclamation-sign";
                klass = "danger";
                break;
        }

        var tr1 = $("<tr>", {
            id: "log_" + Picasso.Helper.hashCode(obj.message + obj.timestamp),
            class: klass,
            "data-search-keys": [obj.message],
            draggable: false
        });

        /* 1. row, 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        var glyph = Picasso.Helper.createGlyph(icon, "pica-glyph");

        td.append(glyph);
        tr1.append(td);

        /* 1. row, 2. column: Timestamp */
        tr1.append($("<td>", {
            class: "pica-small-column",
            text: Picasso.Helper.formatDate(obj.timestamp)
        }));

        /* 1. row, 3. column: Logger */
        tr1.append($("<td>", {
            class: "pica-name pica-small-column hidden-sm hidden-xs",
            text: obj.logger
        }));

        /* 1. row, 4. column: Account */
        var account = findAccount(obj.message);

        tr1.append($("<td>", {
            class: "pica-small-column pica-overflow hidden-sm hidden-xs",
            text: account,
            "data-search-keys": [account]
        }));

        /* 1.row, 5. column: Message */
        tr1.append($("<td>", {
            class: "pica-overflow",
            text: obj.message
        }));

        /* 2. row, 1. column: Stacktrace */
        td = $("<td>", { class: "text-center pica-icon-column" });

        tr1.append(td);

        if (null != obj.stacktrace) {
            glyph = Picasso.Helper.createGlyph("bug", "pica-glyph");

            glyph.attr("data-tooltip", "");
            glyph.attr("title", Picasso.Lang.get("action_show_stacktrace"));

            glyph.tooltip({ container: "body" });

            glyph.click(function () {
                if (tr2.is(":visible")) {
                    tr2.hide();
                } else {
                    tr2.show();
                }
            });

            td.append(glyph);

            var tr2 = $("<tr>", {
                class: "pica-table-selection-disabled",
                style: "display: none"
            });

            tr2.append($("<td>", {
                colspan: TABLE.getVisibleColsCount(),
                html: $("<pre>", {
                    text: obj.stacktrace
                })
            }));

            tr1.addClass("logs-has-stacktrace");

            tr1 = [tr1, tr2];
        }

        return tr1;
    };

    /* Bind event handlers */
    $("#logs_refresh").click(handleClickRefresh);
    $("#logs_copy").click(handleClickCopy);

    /* Init only when logs table is present (might be included for logs renderer) */
    if (0 < $("#logs_table").length) {
        refreshLogs(true);
        handleClickRefresh();
    }
})();
