/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Preferences")) {
    window.Picasso.Preferences = {};
}

/* Private scope */
(function () {

    /**
     * Main preferences categories.
     *
     * @type {*[]}
     */

    var preferenceCategories = [
        {
            icon: "glyphicons-cogwheels",
            title: Picasso.Lang.get("preferences_title_general"),
            text: Picasso.Lang.get("preferences_text_general"),
            name: "general"
        },
        {
            icon: "glyphicons-user-lock",
            title: Picasso.Lang.get("preferences_title_authentication"),
            text: Picasso.Lang.get("preferences_text_authentication"),
            name: "authentication"
        },
        {
            icon: "glyphicons-hdd",
            title: Picasso.Lang.get("preferences_title_storage"),
            text: Picasso.Lang.get("preferences_text_storage"),
            name: "storage"
        },
        {
            icon: "glyphicons-magic",
            title: Picasso.Lang.get("preferences_title_customization"),
            text: Picasso.Lang.get("preferences_text_customization"),
            name: "customization"
        },
        {
            icon: "glyphicons-multiple-displays",
            title: Picasso.Lang.get("preferences_title_network"),
            text: Picasso.Lang.get("preferences_text_network"),
            name: "network"
        },
        {
            icon: "glyphicons-lock",
            title: Picasso.Lang.get("preferences_title_security"),
            text: Picasso.Lang.get("preferences_text_security"),
            name: "security"
        },
        {
            icon: "glyphicons-envelope",
            title: Picasso.Lang.get("preferences_title_mail"),
            text: Picasso.Lang.get("preferences_text_mail"),
            name: "mail"
        },
        {
            icon: "glyphicons-globe-af",
            title: Picasso.Lang.get("preferences_title_web"),
            text: Picasso.Lang.get("preferences_text_web"),
            name: "web"
        }
    ];

    /**
     * Render main preferences categories.
     */

    var renderStaticPreferencesGrid = function () {

        $.each(preferenceCategories, function (idx, cat) {

            var card = $("<div>", {
                class: "pica-card card"
            });

            var icon = $("<span>", {
                class: "pica-card-glyph pica-glyph show glyphicons " + cat.icon,
                align: "center"
            });

            card.append(icon);

            var body = $("<div>", {
                class: "pica-card-body"
            });

            var title = $("<h5>", {
                class: "pica-card-title",
                text: cat.title
            });

            body.append(title);

            var text = $("<p>", {
                class: "pica-card-text",
                text: cat.text
            });

            body.append(text);

            var button = $("<a>", {
                id: "button-" + cat.name,
                class: "btn btn-primary pica-card-button",
                text: Picasso.Lang.get("button_configure")
            });

            /* Bind click handlers */
            button.on("click", handleClickConfigure);

            body.append(button);
            card.append(body);

            $(".pica-flex").append(card);
        })
    };

    /**
     * Render auth chain sub category.
     *
     * @param cat
     * @param entries
     */

    var renderAuthChainSortable = function (cat, entries) {

        /* Render container and columns*/
        var container = $("<div>", {
            class: "container container-width-auto"
        });
        var row = $("<row>");

        var left = $("<div>", {
            class: "form-group col-xs-5"
        });
        var mid = $("<div>", {
            class: "form-group col-xs-2 text-center"
        });
        var right = $("<div>", {
            class: "form-group col-xs-5"
        });

        /* Render arrows */
        var arrows = $("<table>", {
            class: "pica-inputlist-table table"
        });

        var arrowBody = $("<tbody>");
        var arrowRow = $("<tr>");

        var placeholderRow = $("<tr>", {
            class: "pica-sortable-arrow-placeholder"
        });
        var arrowsIcon = $("<td>", {
            class: "glyphicons glyphicons-move pica-sortable-arrow"
        });

        arrowRow.append(arrowsIcon);
        arrowBody.append(placeholderRow);
        arrowBody.append(arrowRow);

        arrows.append(arrowBody);
        mid.append(arrows);

        /* Render active auth table */
        var divider = $("<hr/>", {
            class: "pica-sortable-table-head-body-divider"
        });

        var active = $("<table>", {
            class: "pica-inputlist-table table"
        });

        var ahead = $("<thead>", {
            class: "pica-sortable-table-head pica-dialog-description text-center",
            text: Picasso.Lang.get("preferences_table_auth_chain_active")
        });
        active.append(ahead);
        active.append(divider);

        var abody = $("<tbody>", {
            id: "auth-active",
            class: "pica-connected-sortable pica-sortable-table-body"
        });

        /* Render inactive auth table */
        var divider = $("<hr/>", {
            class: "pica-sortable-table-head-body-divider"
        });

        var inactive = $("<table>", {
            class: "pica-inputlist-table table"
        });

        var ihead = $("<thead>", {
            class: "pica-sortable-table-head pica-dialog-description text-center",
            text: Picasso.Lang.get("preferences_table_auth_chain_inactive")
        });
        inactive.append(ihead);
        inactive.append(divider);

        var ibody = $("<tbody>", {
            id: "auth-inactive",
            class: "pica-connected-sortable pica-sortable-table-body"
        });

        /* Append entries to table */
        $.each(entries, function (key, arr) {

            /* Active */
            if ("authentication.chain.active" === key) {

                var row = $("<tr>", {id: "auth-active-placeholder"});
                var entry = $("<div>", {
                    text: Picasso.Lang.get("preferences_title_auth_chain_no_methods_active"),
                    class: "text-center pica-sortable-table-placeholder"
                });

                row.append(entry);
                abody.append(row);

                if (arr.length > 0) {
                    $.each(arr, function (idx, v) {
                        var r = $("<tr>", {id: v});
                        var m = $("<div>", {
                            text: Picasso.Lang.get("preferences_title_auth_chain_method_" + v),
                            class: "btn btn-block btn-default pica-sortable-table-cell",
                        });

                        r.append(m);
                        abody.append(r);
                    });

                    row.hide();
                }
                return true;
            }

            /* Inactive */
            if ("authentication.chain.inactive" === key) {

                var row = $("<tr>", {id: "auth-inactive-placeholder"});
                var entry = $("<div>", {
                    text: Picasso.Lang.get("preferences_title_auth_chain_no_methods_inactive"),
                    class: "text-center pica-sortable-table-placeholder"
                });

                row.append(entry);
                ibody.append(row);

                if (arr.length > 0) {
                    $.each(arr, function (idx, v) {
                        var r = $("<tr>", {id: v});
                        var m = $("<div>", {
                            id: v,
                            text: Picasso.Lang.get("preferences_title_auth_chain_method_" + v),
                            class: "btn btn-block btn-default pica-sortable-table-cell",
                            "data-color": "primary"
                        });

                        r.append(m);
                        ibody.append(r);
                    });

                    row.hide();
                }
            }
        });

        active.append(abody);
        inactive.append(ibody);

        left.append(active);
        right.append(inactive);

        row.append(left);
        row.append(mid);
        row.append(right);

        container.append(row);
        cat.append(container);

        /* Assign sortable handler */
        $(function () {
            $("#auth-active, #auth-inactive").sortable({
                connectWith: ".pica-connected-sortable",
                cancel: ".pica-table-empty",

                /* Receive item handler */
                receive: function () {
                    var auth = this.id;
                    if ("auth-active" === auth) {
                        $("#auth-active-placeholder").hide();
                    } else {
                        $("#auth-inactive-placeholder").hide();
                    }
                },

                /* Remove item handler */
                remove: function () {
                    var auth = this.id;
                    if ($(this).children().length === 1) {
                        if ("auth-active" === auth) {
                            $("#auth-active-placeholder").show();
                        } else {
                            $("#auth-inactive-placeholder").show();
                        }
                    }
                }
            });
        });
    };

    /**
     * Render LDAP sub category.
     *
     * @param cat
     * @param entries
     */

    var renderLDAPTable = function (cat, entries) {

        /* Empty table */
        $("#ldap-server-content").empty();

        var list = $("<div>", {
            class: "form-group input-group"
        });

        var addon = $("<span>", {
            class: "input-group-addon"
        });

        var server = $("<i>", {
            class: "glyphicons glyphicons-hdd"
        });

        var input = $("<input>", {
            id: "add_ldap_server",
            type: "text",
            class: "form-control",
            "data-placeholder": "placeholder_add_ldap_server",
            placeholder: Picasso.Lang.get("placeholder_add_ldap_server")
        });

        var add = $("<div>", {
            class: "input-group-btn"
        });

        var button = $("<button>", {
            class: "btn btn-default"
        });

        /* Create dialog */
        var dialog = new Picasso.Dialog("#pica_ldap_dialog", "", "",
            (Picasso.Dialog.Flags.VALIDATE | Picasso.Dialog.Flags.HIDE_ON_ESC | Picasso.Dialog.Flags.NO_AUTOCLOSE));

        /* Assign click handler */
        button.on("click", $.proxy(handleAddLDAP, this, dialog));

        var plus = $("<i>", {
            class: "glyphicons glyphicons-plus"
        });

        addon.append(server);
        list.append(addon);

        list.append(input);

        button.append(plus);
        add.append(button);
        list.append(add);

        /* Render table */
        var table = $("<table>", {
            class: "pica-inputlist-table table table-striped"
        });

        var tbody = $("<tbody>", {
            class: "ldap-table"
        });

        $.each(Picasso.Helper.sortObject(entries), function (key, value) {

            var dummy = key === "ldap-dummy";

            var row = $("<tr>", {
                class: dummy ? "pica-table-empty warning " + key : key
            });

            var icon = $("<td>", {
                class: "text-center pica-icon-column"
            });

            var span = $("<span>", {
                class: "glyphicons glyphicons-server pica-glyph"
            });

            icon.append(span);

            var title = value["ldapname"] ? value["ldapname"] : "LDAP Server";

            var name = $("<td>", {
                class: dummy ? "text-center pica-highlight" :
                    "pica-name pica-overflow pica-vertical-middle text-left",
                text: dummy ? Picasso.Lang.get("string_nothing_found") : title
            });

            var options = $("<td>", {
                class: "hidden-xs"
            });

            var edit = $("<button>", {
                class: "btn btn-default pica_ldap_edit_button pull-right",
                text: Picasso.Lang.get("action_edit")
            });

            var remove = $("<button>", {
                class: "btn btn-danger pica_ldap_remove_button pull-right",
                text: Picasso.Lang.get("action_remove")
            });

            renderLDAPDialog(row, edit, remove, value, dialog, dummy);

            options.append(remove);
            options.append(edit);

            if (!dummy) {
                row.append(icon);
            }

            row.append(name);

            if (!dummy) {
                row.append(options);
            }

            if (dummy) {
                row.hide();
            }

            if (Object.keys(entries).length === 1) {
                row.show();
            }

            tbody.append(row);
        });

        table.append(tbody);

        cat.append(list);
        cat.append(table);

        $("#ldap-server-content > .pica-button-checkbox").each(Picasso.Helper.createButtonCheckbox);
    };

    /**
     * Dialog for a main category with different sub categories (tabs + content)
     *
     * @param cat
     * @param entries
     */

    var renderMainDialog = function (cat, entries) {

        /* Empty dialog */
        $("#pica_preference_dialog > .modal-dialog > .modal-content > .modal-body > #pref-tabs").empty();
        $("#pica_preference_dialog > .modal-dialog > .modal-content > .modal-body > #pref-content").empty();

        /* Create new dialog */
        var dialog = new Picasso.Dialog("#pica_preference_dialog",
            Picasso.Lang.get("dialog_title_edit") + " " +
            Picasso.Lang.get("preferences_title_" + cat) + " " +
            Picasso.Lang.get("label_preferences"),
            "", (Picasso.Dialog.Flags.VALIDATE | Picasso.Dialog.Flags.SPINNER |
                Picasso.Dialog.Flags.HIDE_ON_ESC | Picasso.Dialog.Flags.NO_AUTOCLOSE));

        renderContent(entries);

        /* Init handler */
        dialog.setOkHandler(function () {

            /* Get and validate inputs */
            var inputs = $("#pref-content").find(".content");
            var entries = {};

            $.each(inputs, function (key, value) {

                var cat = $(value).attr("name");

                if ("authentication_order" === cat) {

                    var chain = {};
                    var active = [];

                    $("#auth-active").children().each(function () {
                        if (this.id && "auth-active-placeholder" !== this.id) {
                            active.push(this.id)
                        }
                    });

                    chain["authentication.chain.active"] = active;
                    entries[cat] = chain;

                    return true;
                }

                else if ("ldap" === cat) {

                    var ldap = {};

                    $.each($("#ldap-server-content").find(".ldap-server"), function (key, value) {
                        if ("ldap-dummy" !== value.id) {

                            var entries = {};
                            var id = value.id.substring(value.id.indexOf("-") + 1, value.id.length);

                            $.each(getInputs(value), function (key, value) {
                                key = key.replace("-" + id, "");
                                entries[key] = value;
                            });

                            entries["ldapindex"] = id;
                            ldap[id] = entries;
                        }
                    });

                    entries[cat] = ldap;
                }

                /* Normal cases */
                else {
                    entries[cat] = getInputs(value);
                }
            });

            /* Save inputs */
            var url = "/wapi/preferences";
            var data = {
                action: "save",
                saveBy: "category",
                data: JSON.stringify(entries)
            };

            Picasso.Helper.fireAjax(url, data,

                /* Success */
                function (json, status) {
                    dialog.showSpinner(false);

                    switch (status) {
                        case 200:
                            console.log("Preferences successful saved.");
                            dialog.hide();
                            break;
                        case 201:
                            renderRestartConfirmationDialog();
                            break;
                    }
                },

                /* Error */
                function (e) {
                    dialog.showSpinner(false);

                    Picasso.debugLog(e);
                }
            );
        });

        dialog.show();
    };

    /**
     * Dialog for LDAP sub category
     *
     * @param row The row in the LDAP table that belongs to this LDAP server
     * @param entries The LDAP servers config entries
     * @param dialog The modal dialog in which the entries are rendered
     * @param dummy Boolean to determine whether it is a dummy entry. This dummy is initially created only once
     * @param editButton The LDAP server table rows edit button -> To bind a click handler
     * @param removeButton The LDAP server table rows remove button -> To bind a click handler
     */

    var renderLDAPDialog = function (row, editButton, removeButton, entries, dialog, dummy) {

        /* Idx is not an editable config field */
        var idx = entries["ldapindex"];
        delete entries["ldapindex"];

        if (dummy) {
            entries["ldapindex"] = "";
            entries["ldapname"] = "";
            idx = "";
        }

        /* Render body */
        var dialogBody = $("#ldap-server-content");
        var serverBody = $("<div>", {
            id: dummy ? "ldap-dummy" : "ldap-" + idx,
            class: "ldap-server"
        });

        render(serverBody, entries, dummy ? "" : "-" + idx);

        dialogBody.append(serverBody);
        serverBody.hide();

        /* Edit */
        editButton.on("click", function () {

            var serverName = entries["ldapname"] ? entries["ldapname"] : "LDAP Server";
            $("#pica_ldap_dialog > .modal-dialog > .modal-content > " +
                ".modal-header > .pica-dialog-title").text(Picasso.Lang.get("action_edit") + " " + serverName);

            $(".ldap-server").hide();
            $("#ldap-server-content").find(".ldap-server").each(function (k, v) {
                $(v).removeClass("ldap-active")
            });

            serverBody.addClass("ldap-active");
            dialog.show();
            serverBody.show();
        });

        /* Remove */
        removeButton.on("click",
            $.proxy(handleClickLDAPRemove, this, serverBody, row));

        /* Save */
        dialog.setOkHandler(
            function () {

                var active = $(".ldap-active");
                var id = active.attr("id");

                var idx = id.substring(id.indexOf("-") + 1, id.length);
                var name = active.find("#ldapname-" + idx);

                var row = $("." + id);
                row.find(".pica-name").text($(name).val());

                dialog.hide();
            }
        );
    };

    /**
     * Display this dialog when the server needs to be restarted.
     */

    var renderRestartConfirmationDialog = function () {

        var title = "dialog_title_restart";
        var body = "dialog_body_server_restart";
        var warning = "dialog_body_server_restart_warning";

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get(title), Picasso.Lang.get(body) + " " + Picasso.Lang.get(warning));

        dialog.setCancelHandler(function(){
            $("#pica_preference_dialog").modal("hide");
            dialog.hide();
        });

        dialog.setOkHandler(function () {

            var url = "/wapi/preferences";
            var data = {
                action: "restart"
            };

            /* Fire ajax call */
            Picasso.Helper.fireAjax(url, data,

                /* Success */
                function () {
                    $("#pica_preference_dialog").modal("hide");
                    dialog.hide();
                    console.log("Server successful restarted and preferences saved.");
                },

                /* Error */
                function (e) {
                    $("#pica_preference_dialog").modal("hide");
                    dialog.hide();
                    Picasso.debugLog(e);
                }
            );
        });

        dialog.show();
    };

    /**
     * Render sub categories (tabs + content) for a main category.
     *
     * @param arr
     */

    var renderContent = function (arr) {

        var categories = arr[0];
        var counter = 0;
        var tabs = $(".modal-body > .nav-tabs");

        $.each(categories, function (subCat, subs) {

            /* Render sub category tab */
            var tab = $("<li>", {
                class: counter === 0 ? "active" : "",
                id: subCat
            });

            /* Bind click handler */
            tab.on("click", $.proxy(handleClickTab, this, subCat));

            var a = $("<a>");

            var lang = $("<lang>", {
                text: Picasso.Lang.get("preferences_sub_title_" + subCat),
                "data-plural": "false"
            });

            a.append(lang);
            tab.append(a);
            tabs.append(tab);

            /* Render tab content */
            renderInputFields(subCat, sort(subCat, subs));

            /* Open first tab */
            var content = $("." + subCat + "-content");

            if (counter === 0) {
                content.show();
            } else {
                content.hide();
            }

            counter++;
        });

        /* Init checkboxes */
        $(".pica-button-checkbox").each(Picasso.Helper.createButtonCheckbox);
    };

    /**
     * Render input fields for a sub category.
     *
     * @param cat
     * @param entries
     */

    var renderInputFields = async function (cat, entries) {

        var content = $("#pref-content");

        /* Handle special preferences cases */
        var cat = $("<div>", {
            class: cat + "-content content",
            name: cat
        });

        /* LDAP */
        if ("ldap-dummy" in entries) {
            renderLDAPTable(cat, entries);
        }

        /* Authentication chain */
        else if ("authentication.chain.active" in entries) {
            renderAuthChainSortable(cat, entries);
        }

        /* Render inputs */
        else {
            render(cat, entries);
        }

        content.append(cat);

        /* Add special handler */
        if ("webinterface.old.password" in entries) {

            var oldPW = document.getElementById("webinterface.old.password");
            var newPW = document.getElementById("webinterface.new.password");
            var repeatPW = document.getElementById("webinterface.repeat.password");

            cat.find(":password").each(function (k, v) {
                $(v).focusout($.proxy(validateAdminPasswordFields, this, $(oldPW), $(newPW), $(repeatPW)));
            });
        }

        /* Transform to editor */

        if ("login.message" in entries) {
            Picasso.Helper.createEditor("login.message");
            Picasso.Helper.createEditor("register.message");
            Picasso.Helper.createEditor("login.popup.message");
            Picasso.Helper.createEditor("login.popup.reseller.message");
        }


        $(".pica-input-clear").click(function () {
            $(this).closest(".form-group").find("input:not([disabled])").val("");
        });

        $("#net_d2d_renew").unbind('click').click(function () {
            var url = "/wapi/preferences";
            var data = {
                action: "renewCACertificate"
            };
            Picasso.Helper.fireAjax(url, data,
                /* Success */
                function (json) {

                }
            )
        });
    };

    /**
     * Render input fields from a JSON object using its key/value pairs
     *
     * @param cat The element to which the input fields are attached
     * @param index When this method is called several times with the same JSON object, the index avoids duplicate IDs
     * @param obj The JSON object
     */

    var render = function (cat, obj, index) {

        $.each(obj, function (key, value) {
            var translationKey = key.toLowerCase().replace(/\./g, "_");

            var formGroup = $("<div>", {
                class: "form-group"
            });

            var inputGroup = $("<div>", {
                class: "input-group"
            });

            var inputGroupAddon = $("<span>", {
                class: "input-group-addon pref-input-label"
            });

            var label = $("<b>", {
                text: Picasso.Lang.get("preferences_label_" + translationKey)
            });

            var inputClear = $("<span>", {
                class: "pica-input-clear input-group-addon"
            });

            var inputClearIcon = $("<i>", {
                class: "glyphicons glyphicons-delete"
            });

            var input;
            var enableLabel = true;
            var isEditor =false;
            var enableClearInputButton = true;

            /* Render select box */
            if (value instanceof Object && "select" in value) {

                var optionsForDropdowns = [];
                var idx = 0;
                var actualValue = "";

                /* Extract select values */
                $.each(value, function (subKey, subValue) {

                    if ("select" === subKey) {

                        $.each(Picasso.Helper.sortObject(subValue), function (k, v) {

                            if (Picasso.Helper.parseBoolean(v)) {
                                idx = optionsForDropdowns.length;
                                actualValue = k;
                            }
                            var elem = {
                                text: Picasso.Lang.get(k),
                                name: key,
                                value: k
                            };
                            optionsForDropdowns.push(elem);
                        });
                    }
                });

                if ('archive.cleanup.days' === key) {
                    optionsForDropdowns.forEach(function (option) {
                        option.value = Picasso.Helper.convertToDays(option.value);
                    })
                }

                /* Render dropdown */
                input = Picasso.Combo
                    .createDropdownOrButton(optionsForDropdowns, handleClickSelect, idx, "pref-btn-with-label", null);

                /* Fix dropdowns in scroller */
                Picasso.Combo.fixDropdownInScroller(".pica-inputlist-scroller");

                /* Append hidden input to get selection after save */
                var selected = $("<input>", {
                    id: index ? key + index : key,
                    type: "hidden",
                    value: actualValue
                });
                input.append(selected);

                enableClearInputButton = false;
            }

            /* Render boolean checkbox */
            else if (Picasso.Helper.isBoolean(value) && key !== "server.username.isemail") {

                var checkbox = $("<span>", {
                    class: "pica-button-checkbox pull-left pref-btn-checkbox"
                });

                var button = $("<button>", {
                    class: "btn",
                    type: "button",
                    "data-color": "primary"
                });

                var lang = $("<lang>", {
                    text: Picasso.Lang.get("preferences_label_" + translationKey) + "  "
                });

                input = $("<input>", {
                    id: index ? key + index : key,
                    class: "hidden form-control",
                    type: "checkbox"
                });

                /* Set value */
                input.prop("checked", Picasso.Helper.parseBoolean(value));

                button.append(lang);
                checkbox.append(button);
                checkbox.append(input);

                $(inputGroupAddon).remove();

                input = checkbox;

                enableLabel = false;
                enableClearInputButton = false;
            }

            /* Render number input box */
            else if (Picasso.Helper.parseNumber(value, -2) !== -2) {

                input = $("<input>", {
                    id: index ? key + index : key,
                    class: "form-control",
                    type: "number",
                    text: key,
                    "data-placeholder": key
                });

                /* Set value */
                $(input).val(value);
            }

            else {
                var disabled = false;
                if ("net.d2d.status" === key) {
                    disabled = true;
                }

                if (key.indexOf("password") !== -1) {
                    input = $("<input>", {
                        id: index ? key + index : key,
                        class: "pica-password form-control",
                        type: "text",
                        text: key,
                        disabled: disabled
                    });

                    /* Add password policy */
                    Picasso.Helper.updatePasswordPolicy(input);
                    input.keyup(function () {
                        Picasso.Helper.updatePasswordPolicy(input, $(this).val());
                    });

                } else {
                    if (key.indexOf("message")!==-1) {
                        input = $("<textarea>", {
                            id: index ? key + index : key,
                            rows:"10",
                            class: "form-control",
                            text: key,
                            "data-placeholder": key,
                            disabled: disabled
                        });
                        inputGroupAddon.css("display","block");
                        inputGroupAddon.css("width","100%");
                        inputGroupAddon.css("text-align","center");
                        isEditor = true;
                    }
                    else {
                        input = $("<input>", {
                            id: index ? key + index : key,
                            class: "form-control",
                            type: "text",
                            text: key,
                            "data-placeholder": key,
                            disabled: disabled
                        });
                    }

                }

                input.attr("autocomplete", "off");
                $(input).val(value);
            }

            formGroup.append(inputGroup);

            if (enableLabel) {
                inputGroup.append(inputGroupAddon);
                inputGroupAddon.append(label);
            }
            inputClear.append(inputClearIcon);

            inputGroup.append(input);
            if (enableClearInputButton && !isEditor) {
                inputGroup.append(inputClear);
            }

            if ("net.d2d.status" === key) {
                var row = $("<row>");

                var button = $("<button>", {
                    id: "net_d2d_renew",
                    class: "btn btn-default pull-right"
                });

                var lang = $("<lang>", {
                    text: Picasso.Lang.get("preferences_label_net_d2d_renew")
                });


                row.append(button);
                button.append(lang);
                row.append(formGroup);
                cat.append(row);

                return;
            }

            cat.append(formGroup);
        });
    };

    /**
     * Handle click on add LDAP server button.
     */

    var handleAddLDAP = function (dialog) {

        var ldapTable = $(".ldap-table");

        /* Get idx */
        var lastEntry = ldapTable.children("tr").last();

        if ($(lastEntry).hasClass("ldap-dummy")) {
            lastEntry = lastEntry.prev();
        }

        var className = $(lastEntry).attr("class");
        if (className) {
            var idx = Picasso.Helper.parseNumber(className.substring(className.indexOf("-") + 1, className.length)) + 1;
        } else {
            idx = 0;
        }

        /* New name */
        var nameInput = $("#add_ldap_server");
        var name = nameInput.val() ? nameInput.val() : "LDAP Server";

        /* New row */
        var dummyRow = $(".ldap-dummy");
        var emptyTable = dummyRow.hasClass("pica-table-empty");

        var newServerRow = dummyRow.clone();

        if (emptyTable) {
            newServerRow.removeClass("pica-table-empty");
            newServerRow.removeClass("warning");
            dummyRow.hide();
        }

        newServerRow.removeClass("ldap-dummy");
        newServerRow.addClass("ldap-" + idx);

        /* New title */
        var rowTitle = emptyTable ? newServerRow.children("td").eq(0) :
            newServerRow.children("td").eq(1);

        if (emptyTable) {
            rowTitle.removeClass("text-center pica-highlight");
            rowTitle.addClass("pica-name pica-overflow pica-vertical-middle text-left");
        }

        rowTitle.text(name);

        /* New icon */
        if (emptyTable) {

            var icon = $("<td>", {
                class: "text-left pica-icon-column"
            });

            var span = $("<span>", {
                class: "glyphicons glyphicons-server pica-glyph"
            });

            icon.append(span);

            rowTitle.replaceWith(icon);
            newServerRow.append(rowTitle);
        }

        /* New content */
        var dummyContent = $("#ldap-dummy");
        var newServerContent = dummyContent.clone();

        newServerContent.attr("id", "ldap-" + idx);

        /* Set new input IDs */
        newServerContent.find(":input").each(function (k, v) {
            var id = $(v).attr("id");
            $(v).attr("id", id + "-" + idx);
        });

        newServerContent.find("#ldapname-" + idx).val(name);

        var dialogBody = $("#ldap-server-content");
        dialogBody.append(newServerContent);

        /* New buttons */
        if (emptyTable) {

            var options = $("<td>", {
                class: "hidden-xs"
            });

            var edit = $("<button>", {
                class: "btn btn-default pica_ldap_edit_button pull-right",
                text: Picasso.Lang.get("action_edit")
            });

            var remove = $("<button>", {
                class: "btn btn-danger pica_ldap_remove_button pull-right",
                text: Picasso.Lang.get("action_remove")
            });

            options.append(remove);
            options.append(edit);

            newServerRow.append(options);
        }

        /* New handler */
        newServerRow.find(".btn").each(function (key, value) {

            var button = $(value);

            /* Edit */
            if (button.hasClass("pica_ldap_edit_button")) {

                button.on("click", function () {

                    $("#pica_ldap_dialog > .modal-dialog > .modal-content > " +
                        ".modal-header > .pica-dialog-title").text(Picasso.Lang.get("action_edit") + " " + name);

                    $(".ldap-server").hide();
                    $(".ldap-active").each(function (k, v) {
                        $(v).removeClass("ldap-active");
                    });

                    newServerContent.addClass("ldap-active");
                    newServerContent.show();
                    dialog.show();
                });
            }

            /* Remove */
            if (button.hasClass("pica_ldap_remove_button")) {

                button.on("click",
                    $.proxy(handleClickLDAPRemove, this, newServerContent, newServerRow));
            }

            button.show();
        });

        newServerRow.show();
        ldapTable.append(newServerRow);

        nameInput.val("");
    };

    /**
     * Remove LDAP table row and content.
     *
     * @param body
     * @param row
     */

    var handleClickLDAPRemove = function (body, row) {

        body.remove();
        row.remove();

        var table = $(".ldap-table");

        if (table.children("tr").length === 1) {
            $(".ldap-dummy").show();
        }
    };

    /**
     * Handle click on dropdown select value.
     */

    var handleClickSelect = function () {
        document.getElementById(this.name).value = this.getAttribute("value");
        return true;
    };

    /**
     * Handle click on a sub category tab
     *
     * @param subCat
     */

    var handleClickTab = function (subCat) {

        /* Remove old tab focus */
        var tabs = $("#pref-tabs").children();

        $.each(tabs, function (key, value) {
            $("#" + value.id).removeClass("active");
            $("." + value.id + "-content").hide();
        });

        /* Set new tab focus */
        $("#" + subCat).addClass("active");
        $("." + subCat + "-content").show();
    };

    /**
     * Handle click on a main category configure button
     **/

    var handleClickConfigure = function () {

        var that = $(this);

        /* Get category ID */
        var elemID = $(that).attr("id");
        var catID = elemID.substring(elemID.indexOf("-") + 1, elemID.length);

        /* Fetch configurations */
        Picasso.Preferences.fetch(catID);
    };

    /**
     * Sort entries -> Only for special categories where "manual" sorting is needed.
     *
     * @param cat
     * @param entries
     */

    var sort = function (cat, entries) {

        var sorted = {};

        switch (cat) {

            case "proxy":
                sorted["http.proxy.username"] = entries["http.proxy.username"];
                sorted["http.proxy.password"] = entries["http.proxy.password"];
                sorted["http.proxy.host"] = entries["http.proxy.host"];
                sorted["http.proxy.port"] = entries["http.proxy.port"];
                return sorted;

            case "credentials":
                sorted["webinterface.username"] = entries["webinterface.username"];
                sorted["webinterface.old.password"] = entries["webinterface.old.password"];
                sorted["webinterface.new.password"] = entries["webinterface.new.password"];
                sorted["webinterface.repeat.password"] = entries["webinterface.repeat.password"];
                return sorted;

            case "ldap":

                $.each(entries, function (key, value) {

                    var subSorted = {};

                    subSorted["ldapname"] = value["ldapname"];
                    subSorted["ldapserverURL"] = value["ldapserverURL"];
                    subSorted["ldapsearchUsername"] = value["ldapsearchUsername"];
                    subSorted["ldappassword"] = value["ldappassword"];
                    subSorted["ldapsearchBase"] = value["ldapsearchBase"];
                    subSorted["ldapsearchExpression"] = value["ldapsearchExpression"];
                    subSorted["ldapusernameSuffixes"] = value["ldapusernameSuffixes"];

                    subSorted["ldapmappingUsername"] = value["ldapmappingUsername"];
                    subSorted["ldapmappingTitle"] = value["ldapmappingTitle"];
                    subSorted["ldapmappingDisplayName"] = value["ldapmappingDisplayName"];
                    subSorted["ldapmappingCommonName"] = value["ldapmappingCommonName"];
                    subSorted["ldapmappingGivenName"] = value["ldapmappingGivenName"];
                    subSorted["ldapmappingMiddleName"] = value["ldapmappingMiddleName"];
                    subSorted["ldapmappingSurname"] = value["ldapmappingSurname"];
                    subSorted["ldapmappingTelephone"] = value["ldapmappingTelephone"];
                    subSorted["ldapmappingCustom1"] = value["ldapmappingCustom1"];
                    subSorted["ldapmappingCustom2"] = value["ldapmappingCustom2"];
                    subSorted["ldapmappingCustom3"] = value["ldapmappingCustom3"];
                    subSorted["ldapmappingExpiration"] = value["ldapmappingExpiration"];
                    subSorted["ldapmappingQuota"] = value["ldapmappingQuota"];
                    subSorted["ldapmappingValidFrom"] = value["ldapmappingValidFrom"];
                    subSorted["ldapmappingMail"] = value["ldapmappingMail"];
                    subSorted["ldapmatchEmail"] = value["ldapmatchEmail"];

                    subSorted["ldapsyncGroupsEnabled"] = value["ldapsyncGroupsEnabled"];
                    subSorted["ldapgroupsExpression"] = value["ldapgroupsExpression"];
                    subSorted["ldapgroupsMember"] = value["ldapgroupsMember"];
                    subSorted["ldapgroupsMemberOf"] = value["ldapgroupsMemberOf"];
                    subSorted["ldaporgDepth"] = value["ldaporgDepth"];

                    subSorted["ldapimportExpression"] = value["ldapimportExpression"];
                    subSorted["ldapsyncTime"] = value["ldapsyncTime"];
                    subSorted["ldapsyncType"] = value["ldapsyncType"];
                    subSorted["ldapindex"] = value["ldapindex"];

                    sorted[key] = subSorted;
                });
                return sorted;
        }

        return Picasso.Helper.sortObject(entries);
    };

    var validateAdminPasswordFields = function (oldPW, newPW, repeatPW) {

        if (oldPW.val()) {
            oldPW.attr("required", true);
            newPW.attr("required", true);
            repeatPW.attr("required", true);
        }

        if (newPW.val()) {
            oldPW.attr("required", true);
            newPW.attr("required", true);
            repeatPW.attr("required", true);
        }

        if (repeatPW.val()) {
            oldPW.attr("required", true);
            newPW.attr("required", true);
            repeatPW.attr("required", true);
        }

        if (!oldPW.val() && !newPW.val() && !repeatPW.val()) {
            oldPW.attr("required", false);
            newPW.attr("required", false);
            repeatPW.attr("required", false);
        }
    };

    /**
     * Get different input field values from a main category.
     *
     * @param cat
     */

    var getInputs = function (cat) {

        var prefs = {};
        var pairs = $(cat).find(":input");

        $.each(pairs, function (k, v) {

            var id = v.id;

            if (v.type === "checkbox") {
                prefs[id] = $(v).prop("checked");
            } else if (v.type === "number" || v.type === "text" ) {
                prefs[id] = $(v).val();
            } else if (v.type === "hidden") {
                prefs[id] = $(v).val();
            } else if (v.type === "password") {
                prefs[id] = $(v).val();
            }else if (v.type === "textarea") {
                prefs[id] =Picasso.Helper.getEditorData(id);
            }
        });

        return prefs;
    };

    /**
     * Fetch preferences JSON data.
     **/

    Picasso.Preferences.fetch = function (cat) {

        var url = "/wapi/preferences";
        var data = {
            action: "getByCategory",
            category: cat
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                renderMainDialog(cat, Picasso.Helper.getResultArray(json));
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Render static preferences grid.
     **/

    Picasso.Preferences.init = function () {
        renderStaticPreferencesGrid();
    };

    /* Init */
    Picasso.Preferences.init();
})
();
