/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/* Private scope */
(function () {

    /**
     * Update account if property is set
     *
     * @param {Picasso.Account}  self          A #{@link Picasso.Account}
     * @param {Object}           data          Given data object
     * @param {String}           propertyName  Name of the property
     * @param {String}           fieldName     Name of the field
     **/

    var updateIfSet = function (self, data, propertyName, fieldName) {
        if (data.hasOwnProperty(propertyName)) {
            self[fieldName] = data[propertyName];
        }
    };

    /**
     * Constructor for {@link Picasso.Account}
     *
     * @constructor
     **/

    Picasso.Account = function (data) {
        var json = Picasso.Helper.tryJSON(data);

        /* Fetch data: Load from server or restore from stringified object  */
        this._flags = Picasso.Helper.getPropOrDefValue(json, 0, "_flags");
        this._oid = Picasso.Helper.getPropOrDefValue(json, null, "id", "ID", "oid", "OID", "_oid");

        this._orgOid = Picasso.Helper.getPropOrDefValue(json, null, "organizationID", "_orgOid");

        this._authType = Picasso.Helper.getPropOrDefValue(json, null, "authType", "_authType");

        this._userName = Picasso.Helper.getPropOrDefValue(json, "", "username", "_userName");
        this._title = Picasso.Helper.getPropOrDefValue(json, "", "title", "_title");
        this._displayName = Picasso.Helper.getPropOrDefValue(json, "", "displayName", "_displayName");
        this._firstName = Picasso.Helper.getPropOrDefValue(json, "", "firstname", "_firstName");
        this._lastName = Picasso.Helper.getPropOrDefValue(json, "", "surname", "_lastName");

        this._validTill = Picasso.Helper.getPropOrDefValue(json, -1, "validTill", "_validTill");
        this._lastLogin = Picasso.Helper.getPropOrDefValue(json, -1, "lastLogin", "_lastLogin");
        this._creationDate = Picasso.Helper.getPropOrDefValue(json, -1, "created", "_creationDate");
        this._language = Picasso.Helper.getPropOrDefValue(json, "", "language", "_language");

        this._phone = Picasso.Helper.getPropOrDefValue(json, "", "telephone", "_phone");
        this._notes = Picasso.Helper.getPropOrDefValue(json, "", "notes", "_notes");

        this._foldersCount = Picasso.Helper.getPropOrDefValue(json, 0, "nFolder", "_foldersCount");
        this._groupsCount = Picasso.Helper.getPropOrDefValue(json, 0, "nGroups", "_groupsCount");
        this._accountGroups = Picasso.Helper.getPropOrDefValue(json, [], "groups", "_accountGroups");
        this._orgName = Picasso.Helper.getPropOrDefValue(json, "", "organizationName", "_orgName");

        this._quotaTotal = Picasso.Helper.getPropOrDefValue(json, 0, "spaceAllowed", "storageSize", "_quotaTotal");
        this._quotaUsed = Picasso.Helper.getPropOrDefValue(json, 0, "spaceUsed", "storageAssigned", "_quotaUsed");
        this._quotaVersions = Picasso.Helper.getPropOrDefValue(json, 0, "spaceHistory", "_quotaVersions");

        this._maxFolders = Picasso.Helper.getPropOrDefValue(json, -1, "maxFolders", "_maxFolders");
        this._ethAddress = Picasso.Helper.getPropOrDefValue(json, "", "ethAddress", "_ethAddress");

        this._twoFactorAuthenticationEnabled = Picasso.Helper.getPropOrDefValue(json, false, "twoFactorAuthenticationEnabled", "_twoFactorAuthenticationEnabled");

        this._password = null;

        /* Update flags if unset */
        if (json && !json.hasOwnProperty("_flags")) {
            if (json.hasOwnProperty("isProxy") && true === json.isProxy) {
                this._flags |= Picasso.Account.Flags.IS_PROXY;
            }

            if (json.hasOwnProperty("isAdmin") && true === json.isAdmin) {
                this._flags |= Picasso.Account.Flags.ADMIN;
            }

            if (json.hasOwnProperty("isOrgCreator") && true === json.isOrgCreator) {
                this._flags |= Picasso.Account.Flags.ORG_CREATOR;
            }

            if (json.hasOwnProperty("isActive") && true === json.isActive) {
                this._flags |= Picasso.Account.Flags.ACTIVE;
            }

            if (json.hasOwnProperty("isRestrictedOrgAdmin") && true === json.isRestrictedOrgAdmin) {
                this._flags |= Picasso.Account.Flags.RESTRICTED_ORG_ADMIN;
            }

            if (json.hasOwnProperty("isReseller") && true === json.isReseller) {
                this._flags |= Picasso.Account.Flags.RESELLER;
            }
        }

        /* Create proxy to catch getter/setter calls */
        return Picasso.Helper.createProxy(this);
    };

    /**
     * Check whether given OID is our
     *
     * @param {String|Picasso.Account}  oidOrAccount  Oid or #{@link Picasso.Account} to check
     * @returns {Boolean}
     **/

    Picasso.Account.prototype.isMe = function (oidOrAccount) {
        return (oidOrAccount === this._oid ||
            (oidOrAccount.hasOwnProperty("_oid") &&
                oidOrAccount.getOid() === this._oid));
    };

    /**
     * Get name of account
     *
     * @returns {String} Get any name
     **/

    Picasso.Account.prototype.getName = function () {
        return (this._name || this._displayName || this._userName);
    };

    /**
     * Format folders and quota
     *
     * @returns {String} Formatted description
     **/

    Picasso.Account.prototype.getDescription = function () {
        /* Assemble description */
        var description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
            "string_folders_and_storage", this.getFoldersCount(),
            Picasso.Helper.formatSize(this.getQuotaTotal()));

        return description;
    };

    /**
     * Get avatar URL
     *
     * @returns {String} Url for avatar
     **/

    Picasso.Account.prototype.getAvatarUrl = function () {
        if ("" === this._oid) return "";

        return ("/" + Picasso.Helper.joinPaths(["avatars", this._oid]));
    };

    /**
     * Set emails for {@link Picasso.Account}
     *
     * @param {Array}     mails      Array of mail addresses
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Account.prototype.setEmails = function (mails, onSuccess, onError) {

        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues("/wapi/accounts", "setEmails",
            {
                id: this._oid,
                name: this._userName, ///< Required for create
                "emails[]": mails
            },
            onSuccess, onError
        );
    };

    /**
     * setup two factor authentication  {@link Picasso.Account}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Account.prototype.setup2fa = function (onSuccess, onError) {

        var url = "/wapi/accounts";
        var data = {
            action: "setup2fa",
            id: this._oid
        };
        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * toggle on off two factor authentication  {@link Picasso.Account}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Account.prototype.toggle2fa = function (state, secretKey, totp, encodedRecoveryPhrase, onSuccess, onError) {

        var url = "/wapi/accounts";
        var data = {
            action: "toggle2fa",
            tfaEnabled:state,
            secret:secretKey,
            totp:totp,
            encodedRecoveryPhrase:encodedRecoveryPhrase,
            id: this._oid
        };
        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * requestClose on off two factor authentication  {@link Picasso.Account}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Account.prototype.closeAccount = function (mfaCode,onSuccess, onError) {

        var url = "/wapi/accounts";
        var data = {
            action: "requestClose",
            mfaCode:mfaCode
        };
        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Get emails for {@link Picasso.Account}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Account.prototype.getEmails = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/accounts";
        var data = {
            action: "getEmails",
            id: this._oid
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                if (onSuccess) onSuccess(ary);
            }, onError
        );
    };

    /**
     * Set details for {@link Picasso.Account}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     * @param {Object}    details    Account details (optional)
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Account.prototype.setDetails = function (onSuccess, onError,
                                                     details)
    {
        if ("" === this._oid) return null;

        var url = "/wapi/accounts";
        var data = {
            action: "setDetails",
            id: this._oid,
            name: this._userName, ///< Required for create
            accountTitle: this._title,
            accountFirstName: this._firstName,
            accountLastName: this._lastName,
            accountPhone: this._phone,
            language:this._language,
            ethAddress:this._ethAddress
        };

        /* Append if any */
        if (details) {
            data = $.extend(data, details);
        }

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

   /**
     * Get account details
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Account.prototype.getDetails = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var self = this;
        var url = "/wapi/accounts";
        var data = {
            action: "getDetails",
            id: this._oid
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                updateIfSet(self, ary[0], "username", "_username");
                updateIfSet(self, ary[0], "displayName", "_displayname");
                updateIfSet(self, ary[0], "firstname", "_firstname");
                updateIfSet(self, ary[0], "surname", "_lastname");
                updateIfSet(self, ary[0], "telephone", "_phone");

                updateIfSet(self, ary[0], "nFolder", "_foldersCount");
                updateIfSet(self, ary[0], "spaceAllowed", "_quotaTotal");
                updateIfSet(self, ary[0], "spaceUsed", "_quotaUsed");
                updateIfSet(self, ary[0], "spaceHistory", "_quotaversions");

                updateIfSet(self, ary[0], "validTill", "_validTill");
                updateIfSet(self, ary[0], "ethAddress", "_ethAddress");
                updateIfSet(self, ary[0], "twoFactorAuthenticationEnabled", "_twoFactorAuthenticationEnabled");
                updateIfSet(self, ary[0], "twoFactorAuthenticationUrl", "_twoFactorAuthenticationUrl");
                updateIfSet(self, ary[0], "authType", "_authType");

                if (onSuccess) onSuccess(self, ary);
            }, onError
        );
    };

    /**
     * Set password for {@link Picasso.Account}
     *
     * @param {String}    oldPassword  Old password
     * @param {String}    newPassword  New password
     * @param {Function}  onSuccess    Call on success
     * @param {Function}  onError      Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Account.prototype.changePassword = function (oldPassword, newPassword,
                                                         onSuccess, onError)
    {
        if ("" === this._oid) return null;

        var url = "/wapi/accounts";
        var data = {
            action: "changePassword",
            method: "POST",
            id: this._oid,
            oldPassword: oldPassword,
            newPassword: newPassword
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };



    /**
     * Block access to folders for {@link Picasso.Account}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Account.prototype.blockAccess = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/accounts";
        var data = {
            action: "blockFolders",
            id: this._oid
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,function () {
            Picasso.Notification.show(Picasso.Lang.get(
                    "notification_access_blocked"), "info");
            if (onSuccess){
                onSuccess();
            }
        },onError);
    };

    /**
     * UnBlock access to folders for {@link Picasso.Account}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Account.prototype.unblockAccess = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/accounts";
        var data = {
            action: "unblockFolders",
            id: this._oid
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,function () {
            Picasso.Notification.show(Picasso.Lang.get(
                    "notification_access_unblocked"), "info");
            if (onSuccess){
                onSuccess();
            }
        },onError);
    };

    /**
     * reset two factor authentication for {@link Picasso.Account}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Account.prototype.reset2fa = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/accounts";
        var data = {
            action: "reset2fa",
            id: this._oid
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,function () {
            Picasso.Notification.show(Picasso.Lang.get(
                "notification_2fa_reset"), "info");
            if (onSuccess){
                onSuccess();
            }
        },onError);
    };



    /**
     * Set folders for {@link Picasso.Account}
     *
     * @param {Array}     folders    Array of folders
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Account.prototype.setFolders = function (folders, onSuccess, onError) {
        var data = {
            id: this._oid,
            folders: JSON.stringify(folders.folders),
            name: this._userName ///< Required for create
        };

        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues(
            "/wapi/accounts", "saveFoldersToAccount", data, onSuccess, onError
        );
    };

    /**
     * Get folders for {@link Picasso.Account}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Account.prototype.getFolders = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/accounts";
        var data = {
            action: "getFolders",
            id: this._oid
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                /* Create file objects */
                ary = $.map(ary, function (val) {
                    var file = new Picasso.File(val);

                    file.setOid(val.folderID); ///< folderID is actually OID

                    return file;
                });

                if (onSuccess) onSuccess(ary);
            }, onError
        );
    };

    /**
     * Set groups for {@link Picasso.Account}
     *
     * @param {Array}     groups     Array of groups
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Account.prototype.setGroups = function (groups, onSuccess, onError) {
        var data = $.extend({
            id: this._oid,
            name: this._userName ///< Required for create
        }, groups);

        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues(
            "/wapi/accounts", "saveGroupsToAccount", data, onSuccess, onError
        );
    };

    /**
     * Get groups for {@link Picasso.Account}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Account.prototype.getGroups = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/accounts";
        var data = {
            action: "getGroups",
            id: this._oid
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                /* Create group objects */
                ary = $.map(ary, function (val) {
                    var group = new Picasso.Group(val);

                    /* Check whether account is organization admin */
                    if (val.hasOwnProperty("isGroupAdmin") &&
                        true === val.isGroupAdmin)
                    {
                        group.setFlag(Picasso.Group.Flags.GROUP_ADMIN);
                    }

                    return group;
                });

                if (onSuccess) onSuccess(ary);
            }, onError
        );
    };

    /**
     * Set organizations for {@link Picasso.Account}
     *
     * @param {Array}     organizations  Array of organizations
     * @param {Function}  onSuccess      Call on success
     * @param {Function}  onError        Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Account.prototype.setOrganizations = function (organizations,
                                                           onSuccess, onError)
    {
        var data = $.extend({
            id: this._oid,
            name: this._userName ///< Required for create
        }, organizations);

        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues(
            "/wapi/accounts", "saveOrganizationsToAccount", data,
            onSuccess, onError
        );
    };

    /**
     * Get organizations for {@link Picasso.Account}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Account.prototype.getOrganizations = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/accounts";
        var data = {
            action: "getOrganizations",
            id: this._oid
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                /* Create organization objects */
                ary = $.map(ary, function (val) {
                    var organization = new Picasso.Organization(val);

                    /* Check whether account is organization admin */
                    if (val.hasOwnProperty("isOrgAdmin") &&
                        true === val.isOrgAdmin)
                    {
                        organization.setFlag(Picasso.Organization.Flags.ORG_ADMIN);
                    }


                    return organization;
                });

                if (onSuccess) onSuccess(ary);
            }, onError
        );
    };

    /**
     * Get devices for {@link Picasso.Account}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Account.prototype.getDevices = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/accounts";
        var data = {
            action: "getDeviceList",
            id: this._oid,
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                /* Create device objects */
                ary = $.map(ary, function (val) {
                    return new Picasso.Device(val);
                });

                if (onSuccess) onSuccess(ary);
            }, onError
        );
    };

    /**
     * Grant given permission on given folder
     *
     * @param {Picasso.File}        file        A #{@link Picasso.File}
     * @param {Picasso.Permission}  permission  A #{@link Picasso.Permission}
     * @param {Function}            onSuccess   Call on success
     * @param {Function}            onError     Call on error
     **/

    Picasso.Account.prototype.grant = function (file, permission, onSuccess,
                                                onError)
    {
        if ("" === this._oid) return;

        var url = "/wapi/accounts";
        var data = {
            action: "grant",
            id: this._oid,
            folderID: file.getOid(),
            permission: Picasso.Permission.toString(permission)
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Revoke all permissions from given folder
     *
     * @param {Picasso.File}  file        A #{@link Picasso.File}
     * @param {Function}      onSuccess   Call on success
     * @param {Function}      onError     Call on error
     **/

    Picasso.Account.prototype.revokeAll = function (file, onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/accounts";
        var data = {
            action: "revokeAll",
            id: this._oid,
            folderID: file.getOid()
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Remove account
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Account.prototype.remove = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/accounts";
        var data = {
            action: "delete",
            id: this._oid
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Save account settings
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Account.prototype.save = function (onSuccess, onError) {
        var self = this;
        var url = "/wapi/accounts";
        var data = {
            method: "POST",
            action: "store",
            username: this.getUserName(),
            firstname: this.getFirstName(),
            surname: this.getLastName(),
            telephone: this.getPhone(),
            spaceAllowed: this.getQuotaTotal(),
            notes: this.getNotes(),
            adminPermission: this.is(Picasso.Account.Flags.ADMIN),
            orgCreatePermission: this.is(Picasso.Account.Flags.ORG_CREATOR),
            active: this.is(Picasso.Account.Flags.ACTIVE),
            language:this.getLanguage(),
            maxFolders:this.getMaxFolders(),
            twoFactorAuthenticationEnabled:this.getTwoFactorAuthenticationEnabled()
        };

        /* Existing account or new? */
        if (this._oid) {
            data.id = this._oid;
        } else {
            data.username = this._userName;
        }

        /* Check values before sending them */
        if (-1 !== this.getValidTill()) {
            data.validTill = Picasso.Helper.formatJSONDate(
                this.getValidTill());
        } else {
            data.validTill = "";
        }

        /* Don't overwrite password here */
        if (this._password) {
            data.password = this.getPassword();
        }

        /* Add organization if any */
        if (this._orgOid) {
            data.organizationID = this.getOrgOid();
        }

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (data, status) {

                if (data.hasOwnProperty("id")) {
                    self.setOid(data.id);
                }

                if (onSuccess) onSuccess(self);
            }, onError
        );
    };

    /* Global - must be defined after constructor */

    /**
     * Get all accounts
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     * @param {Number}    pageNum    Page number to load (optional)
     * @param {Number}    pageSize   Number of items per page (optional)
     * @param {String}    query      Query to filter results (optional)
     **/

    Picasso.Account.getAll = function (onSuccess, onError,
                                       pageNum, pageSize, query)
    {
        var url = "/wapi/accounts";
        var data = {
            action: "getAll",
            page: pageNum,
            size: pageSize
        };

        /* Add query parameter */
        if (query) {
            data.query = query;
        }

        //check if there is already a calls on the page and cancel them
        if (Picasso.Helper.xhrPool[url] && Picasso.Helper.xhrPool[url].length > 0) {
            Picasso.Helper.abortAll(url)
            $("#accounts_table tbody .pica-spinner").hide();
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                ary = $.map(ary, function (val) {
                    return new Picasso.Account(val);
                });

                if (onSuccess) onSuccess(ary, json);
            }, onError
        );
    };

    /* Flags - must be defined after constructor */

    if (!window.Picasso.Account.hasOwnProperty("Flags")) {
        window.Picasso.Account.Flags = {};
    }

    /* Flags */
    Picasso.Account.Flags.IS_PROXY = (1 << 0); ///< Account is hosted on a remote federated service
    Picasso.Account.Flags.ADMIN = (1 << 1); ///< Account is admin
    Picasso.Account.Flags.ORG_CREATOR = (1 << 2); ///< Account has org create permession
    Picasso.Account.Flags.ORG_ADMIN = (1 << 3); ///< Account is organization admin
    Picasso.Account.Flags.GROUP_ADMIN = (1 << 4); ///< Account is group admin
    Picasso.Account.Flags.ACTIVE = (1 << 5); ///< Account is active
    Picasso.Account.Flags.RESTRICTED_ORG_ADMIN = (1 << 6); ///< Account is restricted org admin
    Picasso.Account.Flags.RESELLER = (1 << 7); ///< Account is restricted reseller
})();
