/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/* Private scope */
(function () {

    var RESELLER_ICON = "crown";
    var NORMAL_ICON = "education";

    /**
     * Constructor for {@link Picasso.Organization}
     *
     * @constructor
     **/

    Picasso.Organization = function (data) {
        var json = Picasso.Helper.tryJSON(data);

        /* Fetch data: Load from server or restore from stringified object  */
        this._flags = Picasso.Helper.getPropOrDefValue(json, 0, "_flags");
        this._oid = Picasso.Helper.getPropOrDefValue(json, "", "id", "ID", "oid", "OID", "_oid");
        this._name = Picasso.Helper.getPropOrDefValue(json, "", "name", "displayName", "_name");
        this._notes = Picasso.Helper.getPropOrDefValue(json, "", "notes", "_notes");

        this._color1 = Picasso.Helper.getPropOrDefValue(json, NaN, "color1", "_color1");
        this._color2 = Picasso.Helper.getPropOrDefValue(json, NaN, "color2", "_color2");
        this._color3 = Picasso.Helper.getPropOrDefValue(json, NaN, "color3", "_color3");

        this._quotaTotal = Picasso.Helper.getPropOrDefValue(json, 0, "storageSize", "_quotaTotal");
        this._quotaUsed = Picasso.Helper.getPropOrDefValue(json, 0, "storageAssigned", "_quotaUsed");
        this._accountsCount = Picasso.Helper.getPropOrDefValue(json, 0, "nUsers", "_accountsCount");
        this._accountsMax = Picasso.Helper.getPropOrDefValue(json, 0, "maxUsers", "_accountsMax");

        this._validFrom = Picasso.Helper.getPropOrDefValue(json, -1, "validFrom", "_validFrom");
        this._validTill = Picasso.Helper.getPropOrDefValue(json, -1, "validTill", "_validTill");
        this._validTillMS = Picasso.Helper.getPropOrDefValue(json, -1, "validTillMS", "_validTillMS");
        this._hasReseller = Picasso.Helper.getPropOrDefValue(json, false, "hasReseller", "_hasReseller");

        this._migrationPercentage = Picasso.Helper.getPropOrDefValue(json, false, "migrationPercentage", "_migrationPercentage");
        this._migrationStatus = Picasso.Helper.getPropOrDefValue(json, false, "migrationStatus", "_migrationStatus");

         /* Update flags if unset */
        if (json && !json.hasOwnProperty("_flags")) {
            if (json.hasOwnProperty("restrictToDomains") &&
                true === json.restrictToDomains)
            {
                this._flags |= Picasso.Organization.Flags.RESTRICT_TO_DOMAINS;
            }
        }

        /* Create proxy to catch getter/setter calls */
        return Picasso.Helper.createProxy(this);
    };

    /**
     * Format accounts and quota
     *
     * @returns {String} Formatted description
     **/

    Picasso.Organization.prototype.getDescription = function () {
        /* Assemble description */
        var description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
            "string_members_and_storage",
            this.getAccountsCount(), this.getAccountsMax(),
            Picasso.Helper.formatSize(this.getQuotaUsed()),
            Picasso.Helper.formatSize(this.getQuotaTotal()));

        return description;
    };

    /**
     * Get avatar URL
     *
     * @returns {String} Url for avatar
     **/

    Picasso.Organization.prototype.getAvatarUrl = function () {
        if ("" === this._oid) return "";

        return ("/" + Picasso.Helper.joinPaths(["avatars", this._oid]));
    };

    /**
     * Get glyph icon to be used
     *
     * @returns {String} Url for avatar
     **/

    Picasso.Organization.prototype.getIcon = function () {
        return this._hasReseller ? RESELLER_ICON : NORMAL_ICON ;
    };

    /**
     * Set domains of {@link Picasso.Organization}
     *
     * @param {Array}     domains    Array of domains
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Organization.prototype.setDomains = function (domains, onSuccess,
                                                          onError)
    {
        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues("/wapi/organizations",
            "saveDomainsToOrganization",
            {
                id: this._oid,
                name: this._name, ///< Required for create
                "domains[]": domains
            },
            onSuccess, onError
        );
    };

    /**
     * Get domains of {@link Picasso.Organization}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Organization.prototype.getDomains = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/organizations";
        var data = {
            action: "getDomains",
            id: this._oid
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                if (onSuccess) onSuccess(ary);
            }, onError
        );
    };

    /**
     * Get folders of {@link Picasso.Organization}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Organization.prototype.getFolders = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/organizations";
        var data = {
            action: "getFolders",
            id: this._oid
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                /* Create folder objects */
                ary = $.map(ary, function (val) {
                    var file = new Picasso.File(val);

                    file.setOid(val.folderID); ///< folderID is actually OID

                    return file;
                });

                if (onSuccess) onSuccess(ary);
            }, onError
        );
    };

    /**
     * Set members of {@link Picasso.Organization}
     *
     * @param {Array}     accounts    Array of accounts
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Organization.prototype.setAccounts = function (accounts, onSuccess,
                                                           onError)
    {
        var data = $.extend({
            id: this._oid,
            name: this._name ///< Required for create
        }, accounts);

        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues(
            "/wapi/organizations", "saveAccountsToOrganization",
            data, onSuccess, onError
        );
    };

    /**
     * Get accounts of {@link Picasso.Organization}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Organization.prototype.getAccounts = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/organizations";
        var data = {
            action: "getAccounts",
            id: this._oid
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                /* Create account objects */
                ary = $.map(ary, function (val) {
                    var account = new Picasso.Account(val);

                    /* Check whether account is organization admin */
                    if (val.hasOwnProperty("isOrgAdmin") &&
                        true === val.isOrgAdmin)
                    {
                        account.setFlag(Picasso.Account.Flags.ORG_ADMIN);
                    }

                    return account;
                });

                if (onSuccess) onSuccess(ary, json);
            }, onError
        );
    };

    /**
     * Set groups for {@link Picasso.Organization}
     *
     * @param {Array}     groups     Array of groups
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Organization.prototype.setGroups = function (groups, onSuccess,
                                                         onError)
    {
        var data = $.extend({
            id: this._oid,
            name: this._name ///< Required for create
        }, groups);

        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues(
            "/wapi/organizations", "saveGroupsToOrganization",
            data, onSuccess, onError
        );
    };

    /**
     * Get groups for {@link Picasso.Organization}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Organization.prototype.getGroups = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/organizations";
        var data = {
            action: "getGroups",
            id: this._oid
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                /* Create group objects */
                ary = $.map(ary, function (val) {
                    return new Picasso.Group(val);
                });

                if (onSuccess) onSuccess(ary, json);
            }, onError
        );
    };

    /**
     * Remove {@link Picasso.Organization}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Organization.prototype.remove = function (onSuccess, onError) {
        if ("" === this._oid) return null;

        var url = "/wapi/organizations";
        var data = {
            action: "delete",
            id: this.getOid(),
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Migration copy {@link Picasso.Organization}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Organization.prototype.migrationCopy = function (onSuccess, onError) {
        if ("" === this._oid) return null;

        var url = "/wapi/organizations";
        var data = {
            action: "migrationCopy",
            id: this.getOid(),
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Migration set active {@link Picasso.Organization}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Organization.prototype.migrationSetActive = function (onSuccess, onError) {
        if ("" === this._oid) return null;

        var url = "/wapi/organizations";
        var data = {
            action: "migrationSetActive",
            id: this.getOid(),
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Save {@link Picasso.Organization}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Organization.prototype.save = function (onSuccess, onError) {
        var self = this;
        var url = "/wapi/organizations";
        var data = {
            action: "store",
            method: "POST",
            name: this.getName(),
            maxUsers: (-1 !== this.getAccountsMax() ? this.getAccountsMax() : 0),
            spaceAllowed: (-1 !== this.getQuotaTotal() ? this.getQuotaTotal() : 0),
            notes: this.getNotes()
        };

        /* Add oid if any */
        if (this._oid) {
            data.id = this._oid;
        }

        /* Check values before sending them */
        if (-1 !== this.getValidFrom()) {
            data.validFrom = Picasso.Helper.formatJSONDate(
                this.getValidFrom());
        } else {
            data.validFrom = "";
        }

        /* Check values before sending them */
        if (-1 !== this.getValidTill()) {
            data.validTill = Picasso.Helper.formatJSONDate(
                this.getValidTill());
        } else {
            data.validTill = "";
        }

        /* colors */

        if (this.getColor1()){
            data.color1 = this.getColor1();
        }
        if (this.getColor2()){
            data.color2 = this.getColor2();
        }if (this.getColor3()){
            data.color3 = this.getColor3();
        }

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (data, status) {
                if (data.hasOwnProperty("ID")) {
                    self.setOid(data.ID);
                }

                if (onSuccess) {
                    onSuccess(self);
                }
            },
            onError
        );
    };

    /**
     * Save {@link Picasso.Organization}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Organization.prototype.simpleSave = function (onSuccess, onError) {
        var self = this;
        var url = "/wapi/organizations";
        var data = {
            action: "simplifiedCreate",
            method: "POST",
            name: this.getName()
        };

        /* Add oid if any */
        if (this._oid) {
            data.id = this._oid;
        }

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (data, status) {
                if (data.hasOwnProperty("ID")) {
                    self.setOid(data.ID);
                }
                if (onSuccess) {
                    onSuccess(self);
                }
            },
            onError
        );
    };

    /* Global - must be defined after constructor */

    /**
     * Get all {@link Picasso.Organization}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     * @param {Number}    pageNum    Page number to load (optional)
     * @param {Number}    pageSize   Number of items per page (optional)
     * @param {String}    query      Query to filter results (optional)
     **/

    Picasso.Organization.getAll = function (onSuccess, onError,
                                            pageNum, pageSize, query)
    {
        var url = "/wapi/organizations";
        var data = {
            action: "getAll",
            page: pageNum,
            size: pageSize
        };

        /* Add query parameter */
        if (query) {
            data.query = query;

            // PF-1812: Show organizations of remote service
            if (query.indexOf("copyAll") >= 0) {
                data.action = "copyAll";
                data.query = query.replaceAll("copyAll", "").trim();
            }
            if (query.indexOf("migrationStatus") >= 0) {
                query = query.replaceAll("migrationStatus", "").trim();
                var queryArr = query.split(" ");
                var migrationStatus = queryArr[0];
                query = query.replaceAll(migrationStatus, "").trim();
                data.query = query;
                data.migrationStatus = migrationStatus;
            }
        }

        //check if there is already a calls on the page and cancel them
        if (Picasso.Helper.xhrPool[url] && Picasso.Helper.xhrPool[url].length > 0) {
            Picasso.Helper.abortAll(url)
            $("#organizations_table tbody .pica-spinner").hide();
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                ary = $.map(ary, function (val) {
                    return new Picasso.Organization(val);
                });

                if (onSuccess) {
                    onSuccess(ary, json);
                }
            }, onError
        );
    };

    /* Flags - must be defined after constructor */

    if (!window.Picasso.Organization.hasOwnProperty("Flags")) {
        window.Picasso.Organization.Flags = {};
    }

    /* Flags */
    Picasso.Organization.Flags.RESTRICT_TO_DOMAINS = (1 << 0); ///< Organization is restricted
    Picasso.Organization.Flags.ORG_ADMIN = (1 << 1); ///< Account is organization admin
})();
