/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Permission")) {
    window.Picasso.Permission = {};
}

/* Private scope */
(function () {

    /* Permissions */
    Picasso.Permission.NO_ACCESS = (1 << 0);
    Picasso.Permission.READ = (1 << 1);
    Picasso.Permission.READ_WRITE = (1 << 2);
    Picasso.Permission.ADMIN = (1 << 3);
    Picasso.Permission.OWNER = (1 << 4);

    /**
     * Convert object from string
     *
     * @param {String}  name  Permission name
     *
     * @returns {Picasso.Permission} Found permission; otherwise Picasso.Permission.NO_ACCESS
     **/

    Picasso.Permission.fromString = function (name) {
        var ret = Picasso.Permission.NO_ACCESS;

        /* Check value type */
        if (name && isNaN(name)) { ///< String?
            var key = name.toUpperCase();

            if (Picasso.Permission.hasOwnProperty(key)) {
                ret = Picasso.Permission[key];
            }
        }

        return ret;
    };

    /**
     * Convert object from either integer or string
     *
     * @param {String}  value  Permission value
     *
     * @returns {Picasso.Permission} Found permission; otherwise Picasso.Permission.NO_ACCESS
     **/

    Picasso.Permission.fromValue = function (value) {
        var ret = Picasso.Permission.NO_ACCESS;

        if (isNaN(value)) {
            ret = Picasso.Permission.fromString(value);
        } else {
            ret = value;
        }

        return ret;
    };

    /**
     * Convert object to string
     *
     * @param {Picasso.Permission}  permission  A {@link Picasso.Permission}
     *
     * @returns {String} String representation of permission; otherwise NO_ACCESS
     **/

    Picasso.Permission.toString = function (permission) {
        var ret  = "NO_ACCESS";
        var keys = Object.keys(Picasso.Permission);

        /* Find permission */
        for (var i = 0; i < keys.length; i++) {
            if (Picasso.Permission[keys[i]] === permission) {
                ret = keys[i].toUpperCase();

                break;
            }
        }

        return ret;
    };

    /**
     * Helper to get values (IE11 fix)
     *
     * @param {Object}  obj  Object to get values from
     * @returns {Array} Array with values
     **/

    var getValues = function (obj) {
        return Object.keys(obj).map(function (key) {
            return obj[key];
        });
    };

    /**
     * Filter permissions values by optional func
     *
     * @param {Function}  filterFunc  Function to filter permissions (optional)
     *
     * @returns {Array}  Array of optionally filtered permissions
     **/

    Picasso.Permission.filter = function (filterFunc) {
        return getValues(Picasso.Permission).filter(function (val) {
            if (!isNaN(val)) {
                if (filterFunc) {
                    return filterFunc(val);
                } else return true;
            }
        });
    };
})();