/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/**
 * Some custom classes for specific behavior are used:
 *
 * .pica-dialog.title           - Dialog title
 * .pica-dialog-description     - Description of the dialog
 * .pica-dialog-immutable-close - Temporary close button for immutable dialogs
 **/

/* Private scope */
(function () {

    /**
     * Reset dialogs field fields
     *
     * @param {Array}  inputs  Input fields
     **/

    var resetInputs = function (inputs) {
        $.each(inputs, function () {
            var that = $(this);

            if (that.is("textarea")) that.val("");
            else if (that.is("input:text")) that.val("");
            else if (that.is('input[type="checkbox"]:checked')) {
                that.trigger("click");
            } else if (that.is('input[type="password"]')) that.val("");
            else if (that.is('input[type="number"]')) that.val("");
            else if (that.is('input[type="tel"]')) that.val("");

            /* Remove error marker */
            that.closest(".has-error").removeClass("has-error");
        });
    };

    /**
     * Constructor for {@link Picasso.Dialog}
     *
     * @param {String}                id           Dialog Id
     * @param {String}                title        Title of the dialog
     * @param {String}                description  Description text of the dialog
     * @param {Picasso.Dialog.Flags}  options      Dialog options
     * @constructor
     **/

    Picasso.Dialog = function (id, title, description, options) {
        var self = this;

        /* Sanity check */
        this._dialog = $(id);

        if (0 === this._dialog.length) {
            throw "Dialog " + id + " not found";
        }

        /* Options */
        this._flags = (options || 0);

        /* Callbacks */
        this._onAssign = null;
        this._onOk = null;
        this._onCancel = null;
        this._onClose = null;
        this._onValidate = null;

        /* Update dialog */
        if (title) {
            this._dialog.find(".pica-dialog-title").text(
                Picasso.Helper.capitalize(title));
        }

        if (description) {
            this._dialog.find(".pica-dialog-description").text(
                Picasso.Helper.capitalize(description));
        }

        /* Reset active nav */
        var tabs = this._dialog.find(".nav li a");

        if (0 !== tabs.length) {
            tabs.first().trigger("click");
        }

        /* Reset and assign input fields */
        this._inputs = this._dialog.find("input, textarea");

        resetInputs(this._inputs);

        /* Map inputs to names */
        this._inputMap = {};

        $.each(this._inputs, function () {
            var input = $(this);

            self._inputMap[input.attr("id")] = input;

            /* Check if dialog is immutable */
            if (0 < (self._flags & Picasso.Dialog.Flags.IMMUTABLE)) {
                input.attr("disabled", true);
            } else {
                input.attr("disabled", false);
            }
        });

        /* Bind handler */
        var okButton = this._dialog.find(
            ".modal-footer button[data-dismiss].btn-primary");

        okButton.off("click").click(function () {
            var valid = true;

            if (0 < (self._flags & Picasso.Dialog.Flags.IMMUTABLE)) {
                return true;
            }

            if (self._onValidate) {
                valid = self._onValidate(self._inputMap, self);
                if (!valid) {
                    return false;
                }
            }

            /* Validate inputs if necessary */
            if (0 < (self._flags & Picasso.Dialog.Flags.VALIDATE)) {
                valid = Picasso.Helper.validateInputs(self._dialog);

                if (!valid) {
                    self.showSpinner(false);

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_error_missing_fields"),
                        "warning");

                    return false;
                }
            }

            /* Show spinner if necessary */
            if (0 < (self._flags & Picasso.Dialog.Flags.SPINNER)) {
                self.showSpinner(true);
            }

            if (valid && self._onOk) {
                self._onOk(self._inputMap, self);
            }

            /* Prevent auto-close; there might be an error */
            if (0 < (self._flags & Picasso.Dialog.Flags.NO_AUTOCLOSE)) {
                return false;
            }
        });

        var cancelButton = this._dialog.find(
            ".modal-footer button[data-dismiss].btn-default");

        cancelButton.off("click").click(
            function () {
                if (self._onCancel) {
                    self._onCancel(self);
                }
            }
        );

        /* Focus first element if any */
        this._dialog.on("shown.bs.modal", function () {
            $(this).find('input[type="text"][autofocus]').focus();
        });

        this._dialog.find(".modal-header button").off("click").click(function () {
            if (self._onClose) {
                self._onClose(self);
            }
        });

        if (1 <= this._inputs.length &&
            (0 < (self._flags & (Picasso.Dialog.Flags.HIDE_ON_ENTER |
                Picasso.Dialog.Flags.HIDE_ON_ESC)))) {
            /* Handle keyboard input */
            Picasso.Helper.handleEnterEscElse(this._inputs[0],

                /* Enter */
                function (val) {
                    if (0 < (self._flags & Picasso.Dialog.Flags.HIDE_ON_ENTER) &&
                        "" !== val) {
                        self._dialog.find(".btn-primary").trigger("click");
                    }
                },

                /* Esc */
                function () {
                    if (0 < (self._flags & Picasso.Dialog.Flags.HIDE_ON_ESC)) {
                        self._dialog.modal("hide");
                    }
                }
            );
        }

        /* Update buttons */
        if (0 < (self._flags & Picasso.Dialog.Flags.IMMUTABLE)) {
            okButton.hide();
            cancelButton.hide();

            var button = $("<button>", {
                class: "pica-dialog-immutable-close btn btn-primary",
                text: Picasso.Lang.get("button_close"),
                "data-dismiss": "modal"
            });

            cancelButton.after(button);
        } else {
            okButton.show();
            cancelButton.show();

            cancelButton.siblings(".pica-dialog-immutable-close").remove();
        }
    };

    /**
     * Check whether given flag is set
     *
     * @param {Number}  flag  Flag to check
     * @returns {Boolean} Either {@code true} when flag is set; otherwise {@code false}
     **/

    Picasso.Dialog.prototype.is = function (flag) {
        return !!(this._flags & flag);
    };

    /**
     * Set assign handler
     *
     * @param {Function}  onAssign  Set assign handler
     */

    Picasso.Dialog.prototype.setAssignHandler = function (onAssign) {
        if (onAssign) {
            this._onAssign = onAssign;
        }
    };

    /**
     * Set ok handler
     *
     * @param {Function}  onOk  Set Ok handler
     */

    Picasso.Dialog.prototype.setOkHandler = function (onOk) {
        if (onOk) {
            this._onOk = onOk;
        }
    };

    /**
     * Set cancel handler
     *
     * @param {Function}  onCancel  Set cancel handler
     */

    Picasso.Dialog.prototype.setCancelHandler = function (onCancel) {
        if (onCancel) {
            this._onCancel = onCancel;
        }
    };

    /**
     * Set close handler
     *
     * @param {Function}  onClose  Set Ok handler
     */

    Picasso.Dialog.prototype.setCloseHandler = function (onClose) {
        if (onClose) {
            this._onClose = onClose;
        }
    };

    /**
     * Set validate function
     *
     * @param {Function}  validate  Set validation function
     */

    Picasso.Dialog.prototype.setValidateFunction = function (validate) {
        if (validate) {
            this._onValidate = validate;
        }
    };

    /**
     * Show/hide spinner
     *
     * @param showSpinner
     **/

    Picasso.Dialog.prototype.showSpinner = function (showSpinner) {
        var primary = this._dialog.find(
            ".modal-footer button[data-dismiss].btn-primary");

        if (true === showSpinner) {
            var spinner = Picasso.Helper.createGlyph("repeat",
                "pica-spinner");

            primary.prepend(spinner);
        } else if (false === showSpinner) {
            primary.find(".pica-spinner").remove();
        }
    };

    /**
     * Show dialog
     **/

    Picasso.Dialog.prototype.show = function () {
        if (this._onAssign) {
            this._onAssign(this._inputMap, this);
        }

        /* Show dialog */
        this._dialog.modal();
    };

    /**
     * Hide dialog
     **/

    Picasso.Dialog.prototype.hide = function () {
        this.showSpinner(false);

        this._dialog.modal("hide");
    };

    /* Flags - must be defined after constructor */

    if (!window.Picasso.Dialog.hasOwnProperty("Flags")) {
        window.Picasso.Dialog.Flags = {};
    }

    /* Flags */
    Picasso.Dialog.Flags.NONE = (1 << 0); ///< No options
    Picasso.Dialog.Flags.HIDE_ON_ENTER = (1 << 1); ///< Hide dialog on enter press
    Picasso.Dialog.Flags.HIDE_ON_ESC = (1 << 2); ///< Hide dialog on esc press
    Picasso.Dialog.Flags.VALIDATE = (1 << 3); ///< Validate inputs
    Picasso.Dialog.Flags.NO_AUTOCLOSE = (1 << 4); ///< Don't autoclose the dialog
    Picasso.Dialog.Flags.IMMUTABLE = (1 << 5); ///< Dialog is immutable
    Picasso.Dialog.Flags.SPINNER = (1 << 6); ///< Show action spinner

    /* Global - must be defined after constructor */

    /**
     * Handle special organization errors
     *
     * @param {Number}  status  Status code
     **/

    Picasso.Dialog.handleOrgErrors = function (status) {
        switch (status) {
            case 409:
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_account_not_in_domain"),
                    "warning");
                break;

            case 412:
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_org_storage_exceeded"),
                    "warning");
                break;

            case 413:
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_org_max_users_reached"),
                    "warning");
                break;
        }
    };
})();