/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/**
 * Some custom classes for specific behavior are used:
 *
 * .pica-name - Selection handler uses this column to get the name of the selected row
 * .pica-serial - Selection handler uses this column to get the serialized object
 *
 * Selections:
 *
 * .pica-table-selection-single - Show this fields when only one row is selected
 * .pica-table-selection-multi - Show this fields when more than one row is selected
 * .pica-table-selection-hide - Hide this fields when any row is selected
 * .pica-table-selection-disabled - Disable selection of this row
 *
 * .pica-table-selection-header - Header for current selection
 * .pica-table-selection-icon - Icon for current selection
 * .pica-table-selection-title - Title of current selection
 * .pica-table-selection-context - Context menu for current selection
 *
 * .pica-table-empty - Empty table message
 *
 * Sortable:
 *
 * .pica-sortable-column - Add sorting to column
 * .pica-sortable-sorted
 * .pica-sortable-icon
 **/

/* Private scope */
(function () {

    /**
     * Reset sorting and remove marker and icons
     *
     * @param {Object}  table  Table object
     **/

    var resetSort = function (table) {
        var that = $(table);

        /* Remove all sorted marker and hide icons */
        that.find(".pica-sortable-sorted").removeClass("pica-sortable-sorted");
        that.find(".pica-sortable-icon").hide();
    };

    /**
     * Sort column
     *
     * @param {Object}   table    Table object
     * @param {Object}   col      Column object
     * @param {Number}   idx      Column index
     * @param {Boolean}  sortAsc  Whether to sort ascending
     **/

    var sortColumn = function (table, col, idx, sortAsc) {
        var that = $(col);
        var tbody = table.find("tbody");
        var isSorted = that.hasClass("pica-sortable-sorted");
        var icons = that.find(".pica-sortable-icon");
        var reverse = 1;

        resetSort(table);

        /* Check ordering */
        if (true === sortAsc) {
            reverse = 1;
        } else if (false === sortAsc) {
            reverse = -1;
        } else {
            if (isSorted) {
                $(icons[0]).hide();
                $(icons[1]).show();
            } else {
                reverse = -1;
                that.addClass("pica-sortable-sorted");

                $(icons[0]).show();
                $(icons[1]).hide();
            }
        }

        /* Sort rows: We rely on plain JS for performance */
        var trs = tbody.children().sort(function (a, b) {
            var elem_a = a.children[idx];
            var elem_b = b.children[idx];

            /* Carefully either fetch sort value or some text */
            var val_a = (elem_a ? (elem_a.hasAttribute("data-sort") ?
                elem_a.getAttribute("data-sort") : elem_a.innerText) : "");
            var val_b = (elem_b ? (elem_b.hasAttribute("data-sort") ?
                elem_b.getAttribute("data-sort") : elem_b.innerText) : "");

            if (val_a.match(/^\d+/g)!=null && val_b.match(/^\d+/g)===null){
                val_b = "0 " + val_b;
            }

            if (val_b.match(/^\d+/g)!=null && val_a.match(/^\d+/g)===null){
                val_a = "0 " + val_a;
            }


            /* Use localCompare for natural sort if available */
            if (val_a.localeCompare) {
                return (reverse * val_a.localeCompare(val_b, undefined, {
                    numeric: true,
                    sensitivity: "base"
                }));
            } else {
                return (reverse * (val_a < val_b ? -1 : val_a > val_b ? 1 : 0));
            }
        });

        /* Replace table body */
        var fragment = tbody.clone(true).empty(); ///< Copy event handler for eg infinite scroll

        for (var i = 0; i < trs.length; i++) {
            fragment.append(trs[i]);
        }

        tbody.replaceWith(fragment);
    };

    /**
     * Update table header for selection mode
     * (uses .pica-name to find the name of the selected row)
     *
     * @param {Picasso.Table}       table        Our table
     * @param {Number}              nselected    Number of marked rows
     * @param {jQuery|HTMLElement}  selectedRow  Selected row object
     **/

    var handleOnSelection = function (table, nselected, selectedRow) {

        /* Find elements */
        var singleOps = table._htmlTable.find(".pica-table-selection-single");
        var multiOps = table._htmlTable.find(".pica-table-selection-multi");

        /* Toggle visibility of selection header and ops */
        var theads = table._htmlTable.find("thead");

        var selectHead = theads.first();
        var normalHead = theads.last();

        var hideOnSelect = $(".pica-table-selection-hide");

        switch (nselected) {
            case undefined:
            case -1:
            case 0:
                selectHead.hide();
                normalHead.show();
                hideOnSelect.show();

                /* Call handler if any */
                if (table._onSelection) {
                    table._onSelection(nselected, selectedRow);
                }
                return;

            case 1:
                singleOps.show();
                multiOps.show();
                hideOnSelect.hide();
                break;

            default:
                singleOps.hide();
                multiOps.show();
                hideOnSelect.hide();
                break;
        }

        /* Update selection header */
        var iconTh = selectHead.find("th.pica-table-selection-icon");
        var nameTh = selectHead.find("th.pica-table-selection-title");

        iconTh.html(Picasso.Helper.createGlyph("unchecked", "pica-glyph"));

        /* Add selection text */
        if (1 === nselected) {
            nameTh.html($("<span>", {
                text: selectedRow.find(".pica-name").text()
            }));
        } else {
            nameTh.html($("<span>", {
                text: Picasso.Lang.get(
                    "string_selected_elements", nselected)
            }));
        }

        /* Re-add select all handler */
        iconTh.off("click").click(function (e) {
            e.preventDefault();
            e.stopPropagation();

            var that = $(this);
            var icon = that.find("span");

            /* Toggle checkbox */
            if (icon.hasClass("glyphicons-unchecked")) {
                table.selectAll();

                icon.removeClass("glyphicons-unchecked");
                icon.addClass("glyphicons-check");
            } else {
                table.deselectAll();

                icon.removeClass("glyphicons-check");
                icon.addClass("glyphicons-unchecked");
            }
        });

        /* Add deselect button - no handler is required here */
        var a = $("<a>", {
            class: "hidden-xs",
            text: Picasso.Lang.get("action_deselect")
        });

        nameTh.append(a);

        selectHead.show();
        normalHead.hide();

        /* Call handler if any */
        if (table._onSelection) {
            table._onSelection(nselected, selectedRow);
        }
    };

    /**
     * Handle click on table row
     *
     * @param {Picasso.Table}  table  Our table
     * @param {Event}          e      Click event
     **/

    var handleRowClick = function (table, e) {
        var that = $(e.currentTarget);
        var trs = table._htmlTable.find("tbody tr");
        var selected = trs.siblings("." + Picasso.Table.Marker.SELECTED);
        var nselected = selected.length;

        var first = selected.first();

        /* Skip empty message and disabled row */
        if (that.hasClass("pica-table-empty") ||
            that.hasClass("pica-table-selection-disabled"))
        {
            return;
        }

        /* Toggle marker */
        if (that.hasClass(Picasso.Table.Marker.SELECTED)) {
            if (e.shiftKey) {
                /* Select from first selected up to this row */
                nselected = table.selectFromTo(trs, $(first), that,
                    Picasso.Table.Marker.SELECTED);
            } else {
                that.removeClass(Picasso.Table.Marker.SELECTED);
                nselected--;
            }
        } else {
            /* Shift key pressed while clicking */
            if (e.shiftKey) {
                if (0 === selected.length) {
                    /* Just select this row */
                    that.addClass(Picasso.Table.Marker.SELECTED);
                    nselected++;
                } else {
                    /* Select from first selected up to this row */
                    nselected = table.selectFromTo(trs, $(first), that,
                        Picasso.Table.Marker.SELECTED);
                }
                /* Either control pressed or click on the icon */
            } else if (e.ctrlKey || $(e.target).hasClass("pica-glyph")) {
                that.addClass(Picasso.Table.Marker.SELECTED);
                nselected++;
            } else {
                /* Select only this row */
                trs.removeClass(Picasso.Table.Marker.SELECTED);
                that.addClass(Picasso.Table.Marker.SELECTED);

                nselected = 1;
            }
        }

        handleOnSelection(table, nselected, that);
    };

    /**
     * Append or prepend rows to container
     *
     * @param {jquery|HTMLElement}  container    Container to add elements to
     * @param {Array}               rows         Rows to render
     * @param {Function}            renderFunc   Render function for elements
     * @param {Boolean}             prependRows  Whether to prepend instead of append rows (optional)
     * @param {Boolean}             highlight    Whether to highlight added rows (optional)
     *
     * @returns {Number} Number of appended rows
     **/

    var appendRows = function (container, rows, renderFunc,
                               prependRows, highlight)
    {
        var nrows = 0;

        $.each(rows, function () {
            /* Render rows */
            var rendered = renderFunc(this);

            if (null != rendered) {
                if (true === prependRows) {
                    container.prepend(rendered);
                } else {
                    container.append(rendered);
                }

                /* Highlight added rows */
                if (true === highlight) {
                    rendered.addClass("pica-highlight");
                }

                /* Sum up rendered rows; might be arrays */
                nrows += ("array" === $.type(rendered) ? rendered.length : 1);
            }
        });

        return nrows;
    };

    /**
     * Create empty message
     *
     * @param {Picasso.Table}  table  Our table
     *
     * @returns {jQuery|HTMLElement} Assembled message row
     **/

    var createEmptyMessage = function (table) {
        var tr = $("<tr>", { class: "pica-table-empty warning" });
        var td = $("<td>", {
            class: "text-center",
            colspan: table.getVisibleColsCount(),
            html: Picasso.Helper.createLang("string_nothing_found")
        });

        tr.append(td);

        return tr;
    };

    /**
     * Constructor for {@link Picasso.Table}
     *
     * @param {String}               id       Table Id
     * @param {Picasso.Table.Flags}  options  Options for this element (optional)
     * @constructor
     **/

    Picasso.Table = function (id, options) {
        var self = this;

        /* Init */
        this._htmlTable = $(id);
        this._onSelection = null;

        /* Init sorter */
        this._htmlTable.find("thead").last().find("th").each(function (i) {
            var that = $(this);

            if (that.hasClass("pica-sortable-column")) {
                that.click(function () {
                    sortColumn(self._htmlTable, this, i);
                });

                that.append(Picasso.Helper.createGlyph(
                    "sort-by-attributes-alt", "pica-sortable-icon", false));
                that.append(Picasso.Helper.createGlyph(
                    "sort-by-attributes", "pica-sortable-icon", false));
            }
        });

        /* Check whether rows should be selectable */
        if (options && 0 < (options & Picasso.Table.Flags.ROWS_SELECTABLE)) {
            /* Bind handlers */
            $(document).click(function (e) {
                e.stopPropagation();

                self.deselectAll();
            });

            /* Delegate row clicks */
            this._htmlTable.off('click').on( "click", "tbody tr",function (e) {
                var target = $(e.target);

                /* Limit delegation to td elements */
                if (target.is("td") || target.hasClass("pica-glyph")) {
                    e.preventDefault();
                    e.stopPropagation();

                    handleRowClick(self, e);
                }
            });

            /* Ignore text selection */
            this._htmlTable.on("selectstart", function (e) {
                e.preventDefault();
            });
        }
    };

    /**
     * Get the number of visible columns based on table header
     *
     * @return {Number} Number of columns
     **/

    Picasso.Table.prototype.getVisibleColsCount = function () {
        var ncols = 0;

        $.each(this._htmlTable.find("thead").last().find("th"), function () {
            var that = $(this);

            ncols += (that.attr("colspan") ?
                parseInt(that.attr("colspan")) : 1);
        });

        return ncols;
    };

    /**
     * Clear data table and show empty message
     **/

    Picasso.Table.prototype.clear = function () {
        var tbody = this._htmlTable.find("tbody");

        /* Create empty fragment */
        var fragment = $("<tbody>", {
            class: tbody.attr("class")
        });

        fragment.html(createEmptyMessage(this));

        tbody.replaceWith(fragment);
    };

    /**
     * Sort table data
     *
     * @param {Number}   columnIdx  Column Idx (optional)
     * @param {Boolean}  sortAsc    Whether to sort ascending
     **/

    Picasso.Table.prototype.sort = function (columnIdx, sortAsc) {
        var ths = this._htmlTable.find("thead").last().find("th");
        var idx = 1;

        if (columnIdx && columnIdx <= ths.length) {
            idx = columnIdx;
        }

        sortColumn(this._htmlTable, ths[idx], idx, sortAsc);
    };

    /**
     * Show empty message
     **/

    Picasso.Table.prototype.showEmptyMessage = function () {
        var tbody = this._htmlTable.find("tbody");

        if (0 === tbody.find(".pica-table-empty").length) {
            tbody.html(createEmptyMessage(this));
        }
    };

    /**
     * Hide all empty messages
     **/

    Picasso.Table.prototype.hideEmptyMessage = function () {
        this._htmlTable.find("tbody .pica-table-empty").remove();
    };

    /**
     * Check whether table is empty
     *
     * @return {Boolean} Either true when table is empty; otherwise false
     **/

    Picasso.Table.prototype.isEmpty = function () {
        return (0 === this._htmlTable.find("tbody").children().length);
    };

    /**
     * Check whethertable is empty and add empty message
     *
     * @return {Boolean} Either {@code true} when empty; otherwise {@code false}
     **/

    Picasso.Table.prototype.checkIfEmpty = function () {
        if (this.isEmpty()) {
            this.showEmptyMessage();

            return true;
        }

        return false;
    };

    /**
     * Update table with data
     *
     * @param {*}        data        Data for update
     * @param {Function} renderFunc  Render function for elements
     **/

    Picasso.Table.prototype.update = function (data, renderFunc) {
        var ary = Picasso.Helper.getResultArray(data);
        var nrows = 0;

        /* Check if empty */
        if (0 === ary.length) {
            this.clear();
        } else {
            var tbody = this._htmlTable.find("tbody");
            var fragment = $("<tbody>", { class: tbody.attr("class") });

            nrows = appendRows(fragment, ary, renderFunc);

            tbody.replaceWith(fragment);
        }

        return nrows;
    };

    /**
     * Update table with data
     *
     * @param {String}   url         Url to load data from
     * @param {Object}   data        Url parameters
     * @param {Function} renderFunc  Render function for elements
     **/

    Picasso.Table.prototype.updateFromJSON = function (url, data, renderFunc) {
        var self = this;

        Picasso.Helper.fireAjax(url, data,

            /*  Success */
            function (json) {
                try {
                    var ary = Picasso.Helper.getResultArray(json);

                    self.update(ary, renderFunc);
                } catch (e) {
                    Picasso.debugLog(e);

                    return;
                }
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Append data to table
     *
     * @param {*}        data         Data to append
     * @param {Function} renderFunc   Render function for elements
     * @param {Boolean}  prependRows  Whether to prepend instead of append rows (optional)
     * @param {Boolean}  highlight    Whether to highlight added rows (optional)
     **/

    Picasso.Table.prototype.append = function (data, renderFunc,
                                               prependRows, highlight)
    {
        var ary = Picasso.Helper.getResultArray(data);
        var nrows = 0;

        /* Sanity checks */
        if (ary && 0 !== ary.length) {
            var tbody = this._htmlTable.find("tbody");

            this.hideEmptyMessage();

            nrows = appendRows(tbody, ary, renderFunc, prependRows, highlight);
        }

        return nrows;
    };

    /**
     * Select elements between from and to with given marker
     *
     * @param {Array}  elems   Elements array
     * @param {Object} from    Start element
     * @param {Object} to      Stop element
     * @param {String} marker  Marker class name
     *
     * @return {Number} Number of marked elements
     **/

    Picasso.Table.prototype.selectFromTo = function (elems, from, to, marker) {
        var nselected = 0;
        var dir = 0;

        elems.removeClass(marker);

        /* Mark enclosing rows and honor the limit */
        for (var i = 0; i < elems.length; i++) {
            var that = $(elems[i]);
            var id = that.attr("id");

            /* Mark in both ways: top-to-bottom + bottom-to-top */
            if (0 === dir) {
                if (from.attr("id") === id) dir = 1;
                else if (to.attr("id") === id) dir = -1;
            }

            /* Skip entries on reverse selection */
            if (0 !== dir) {
                that.addClass(marker);

                nselected++;

                if (1 === dir && to.attr("id") === id ||
                    -1 === dir && from.attr("id") === id) break;
            }
        }

        return nselected;
    };

    /**
     * Select all rows
     **/

    Picasso.Table.prototype.selectAll = function () {
        this._htmlTable.find("tbody tr").not(".pica-table-selection-disabled")
            .addClass(Picasso.Table.Marker.SELECTED);
    };

    /**
     * Deselect all selected rows
     **/

    Picasso.Table.prototype.deselectAll = function () {
        var marked = this._htmlTable.find("tbody tr." +
            Picasso.Table.Marker.SELECTED);

        if (0 < marked.length) {
            marked.removeClass(Picasso.Table.Marker.SELECTED);

            handleOnSelection(this, 0, null);
        }
    };

    /**
     * Get selected rows
     *
     * @returns {Array} Array of selected rows
     **/

    Picasso.Table.prototype.getSelectedRows = function () {
        return $.makeArray(this._htmlTable.find("tbody tr." +
            Picasso.Table.Marker.SELECTED));
    };

    /**
     * Get all rows
     *
     * @returns {Array} Array of all elements
     **/

    Picasso.Table.prototype.getRows = function () {
        return $.makeArray(this._htmlTable.find(
            "tbody tr:not(.pica-table-empty)"));
    };

    /**
     * Call handler when a row is selected
     *
     * @param {Function}  onSelection  Selection callback
     **/

    Picasso.Table.prototype.setSelectionHandler = function (onSelection) {
        if (onSelection) {
            this._onSelection = onSelection;
        }
    };

    /**
     * Method to disable Sort for table columns
     *
     **/

    Picasso.Table.prototype.disableSort = function () {

        this._htmlTable.find(".pica-sortable-column").each(function () {
            $(this).off("click");
        });

    };
    /**
     * Method to enable Sort for table columns
     *
     **/

    Picasso.Table.prototype.enableSort = function () {

        var table = this._htmlTable;

        table.find("thead").last().find("th").each(function (i) {
            var that = $(this);

            if (that.hasClass("pica-sortable-column")) {
                that.off("click").on("click",function () {
                    sortColumn(table, this, i);
                });

                that.append(Picasso.Helper.createGlyph(
                        "sort-by-attributes-alt", "pica-sortable-icon", false));
                that.append(Picasso.Helper.createGlyph(
                        "sort-by-attributes", "pica-sortable-icon", false));
            }
        });

    };

    /**
     * Bind scroll handler to allow infinite scroll
     *
     * @param {Function}  onScroll  Scrolling callback
     **/

    Picasso.Table.prototype.infiniteScroll = function (onScroll) {
        var that = this;

        this._htmlTable.find("tbody").off("scroll").on("scroll", function () {
            var tbody = $(this);

            /* Check for end of scroller minus line height of a row */
            if((tbody.scrollTop() + tbody.innerHeight()) >=
                tbody[0].scrollHeight - 60)
            {
                /* WARNING: We have to disable the event handler here;
                 *          otherwise it gets triggered multiple times
                 *          due to height increase after fetch */
                tbody.off("scroll");

                resetSort(that._htmlTable);

                if (onScroll) {
                    onScroll();
                }
            }
        });
    };

    /* Global - must be defined after constructor */

    Picasso.Table.MAX_SELECTABLE = 5;

    /* Flags - must be defined after constructor */

    if (!window.Picasso.Table.hasOwnProperty("Flags")) {
        window.Picasso.Table.Flags = {};
    }

    Picasso.Table.Flags.ROWS_SELECTABLE = (1 << 0); ///< Table rows are selectable

    /* Marker - must be defined after constructor */

    if (!window.Picasso.Table.hasOwnProperty("Marker")) {
        window.Picasso.Table.Marker = {};
    }

    /* Globals */
    Picasso.Table.Marker.SELECTED = "info";
    Picasso.Table.Marker.CREATED = "success";
    Picasso.Table.Marker.REMOVED = "danger";
    Picasso.Table.Marker.DRAG_YES = "success";
    Picasso.Table.Marker.DRAG_NO = "danger";
})();
