#!/bin/bash
#
# ------------------------------------------------------
# Server start/stop script for Linux/Unix
# ------------------------------------------------------

#
# SERVER_INSTALL needs to point to the directory
# where the .jar file is located.
#
SERVER_INSTALL=.

# SERVER_HOME sets the location where the server
# will store data like the configuration file,
# internal database, keys, etc. A hidden sub-directory
# with the binary name will be created in that directory.
# PLEASE NOTE: By default this is the user's home
# directory. When you change that, you also need to
# change the symlink to your .config file, which
# resides in your SERVER_INSTALL directory!
#
SERVER_HOME=`echo ~`

# JAVA_MEM specifies some memory settings vor the Java
# virtual machine
#
# If running on a 64-bit system with more than 4GB ram
# or if facing an out-of-memory error the value for the
# -Xmx parameter can be raised (example: -Xmx4g,-Xmx8g,
# -Xmx16g,...), so the virtual machine will be allowed
# to use more memory during runtime.
JAVA_MEM="-Xms256m -Xmx2g -XX:NewRatio=8 -XX:MinHeapFreeRatio=10 -XX:MaxHeapFreeRatio=20"

# JAVA_TMP specifies where the server will store
# temporary files.
JAVA_TMP="-Djava.io.tmpdir=/tmp"

# Support for KERBEROS Database authentication goes here.
JAVA_KERBEROS=""
# Example:
# JAVA_KERBEROS="-Djava.security.krb5.conf=/etc/krb5.conf -Djava.security.auth.login.config=/home/powerfolderdbadm/jaas.conf -Djavax.security.auth.useSubjectCredsOnly=false"

# Remote connection/console port (accepts only commands from localhost)
SERVER_RCON=1338


############################################
### Don't change anything below this line! #
############################################

BINARY_NAME="PowerFolder-Server"
DIST_NAME="PowerFolder Server"

# Export variables
export SERVER_INSTALL
export SERVER_HOME
export LC_ALL=en_US.UTF-8
export LANG=en_US.UTF-8

# Unset DISPLAY variable
unset DISPLAY


#
# Check if the user is not root
#
if [ "$(id -u)" = "0" ]; then
	echo "This script must NOT be run as root!" 1>&2
	exit 1
fi

#
# Set correct Java version
#
if [ `getconf LONG_BIT` = "64" ]
then
  RUN_JAVA="$SERVER_INSTALL/jre64/bin/java"
  rm -Rf $SERVER_INSTALL/jre32
else
  echo "Running on 32-bit systems are no longer supported. Please run on a 64-bit architecture."
  exit 1
fi

#
# Check .jar file existance
#
if [ ! -f "$SERVER_INSTALL/PowerFolder-Server.jar" ]
then
    echo "PowerFolder-Server.jar does not exist in the $SERVER_INSTALL directory. Please check if SERVER_INSTALL is set in the PowerFolder-Server.sh file or if PowerFolder-Server.jar exists under the specified path!"
    exit 1
fi


# Set Java classpath
CP="$CLASSPATH:$SERVER_INSTALL/PowerFolder-Server.jar:$SERVER_INSTALL/libs/*"

# Required since open JDK 17
JAVA_ADD="--add-exports=java.base/sun.nio.ch=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-opens=java.base/java.lang=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-opens=java.base/java.lang.reflect=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-opens=java.base/java.io=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-exports=jdk.unsupported/sun.misc=ALL-UNNAMED"

#
# Set command line to invoke program start
#
RUN_CMD="$RUN_JAVA $JAVA_MEM $JAVA_ADD $JAVA_TMP $JAVA_KERBEROS -Duser.home=$SERVER_HOME -Dpf.install.dir=$SERVER_INSTALL -cp $CP -server de.dal33t.Start"


#
# Check PID file
#
if [  -z "$SERVER_PID" ]
then
	if [ ! -d "$SERVER_HOME/.PowerFolder" ]
	then
		mkdir $SERVER_HOME/.PowerFolder
	fi
	SERVER_PID="$SERVER_HOME/.PowerFolder/PowerFolder-Server.pid"
fi


#
# Check for console
#
if [ -z "$SERVER_CONSOLE" ]; then
    if [ -w /dev/console ]; then
        SERVER_CONSOLE="/dev/console"
    elif [ -w /dev/tty ]; then
        SERVER_CONSOLE="/dev/tty"
    else
        # Safe fallback for non-interactive environments
        SERVER_CONSOLE="/dev/stdout"
    fi
fi


#
# Check and invoke action
#
usage()
{
    echo "Usage: $0 {start|stop|restart|status|reload|console}"
    exit 1
}
[ $# -gt 0 ] || usage

ACTION=$1
case "$ACTION" in
	start)
        	echo "Starting PowerFolder Server"

            if [ -f $SERVER_PID ]
            then
                if ps -p `cat $SERVER_PID` >/dev/null ; then
                    echo "Starting PowerFolder Server failed: Already Running!"
                    exit 1
                fi
            fi

        	cd $SERVER_INSTALL
	    	nohup $RUN_CMD > /dev/null 2>&1 &
        	echo $! > $SERVER_PID
			sleep 2
        	echo "STARTED PowerFolder Server on: `date`" >> $SERVER_CONSOLE
        	echo "STARTED PowerFolder Server with: PID" `cat $SERVER_PID`
        	;;

	console)
        	echo "Starting PowerFolder Server"

            if [ -f $SERVER_PID ]
            then
                if ps -p `cat $SERVER_PID` >/dev/null ; then
                    echo "Starting PowerFolder Server failed: Already Running!"
                    exit 1
                fi
            fi

        	echo "STARTED PowerFolder Server on: `date`" >> $SERVER_CONSOLE

        	cd $SERVER_INSTALL
			RUN_CMD="$RUN_CMD -l INFO"
        	eval $RUN_CMD
        	;;
  
	stop)
        	PID=`cat $SERVER_PID 2>/dev/null`
        	eval "$RUN_CMD -k $SERVER_RCON"
        	sleep 8
			if [ -n "$PID" ]
			then
				if ps -p $PID > /dev/null
				then
					echo "Stopping PowerFolder Server with: PID $PID"
					kill -9 $PID 2>/dev/null
				fi
			fi
        	rm -f $SERVER_PID
        	echo "STOPPED PowerFolder Server on: `date`" >>$SERVER_CONSOLE
        	;;

    reload)
            PID=`cat $SERVER_PID 2>/dev/null`
            echo "Reloading config PowerFolder Server with: PID $PID"
            eval "$RUN_CMD -O $SERVER_RCON"
            echo "RELOADED PowerFolder Server on: `date`" >>$SERVER_CONSOLE
            ;;

	restart)
        	$0 stop $*
        	sleep 5
        	$0 start $*
        	;;

	status)
			if [ -f $SERVER_PID ]
				then
					if ps -p `cat $SERVER_PID` >/dev/null ; then
						echo "STATUS PowerFolder Server: Running PID" `cat $SERVER_PID`
						exit 0
					else
						echo "STATUS PowerFolder Server: Currently not running!"
						exit 1
				fi
			else
				echo "STATUS PowerFolder Server: Currently not running!"
				exit 1
			fi
			;;

	*)
        	usage
        	;;

esac
exit 0
