/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Dialog")) {
    window.Picasso.Dialog = {};
}

/* Private scope */
(function () {

    /**
     * Download {@link Picasso.File} version
     *
     * @param {Picasso.File}  file     A #{@link Picasso.File}
     * @param {Number}        version  Version to download
     **/

    var downloadVersion = function (file, version) {
        var downloadUrl = file.getDownloadUrl() + "?v=" + version;

        var a = $("<a>", {
            href: downloadUrl,
            download: file.getName()
        });

        $("body").append(a);
        a[0].click();
        a.remove();

        Picasso.Notification.show(
            Picasso.Lang.get("notification_download_started", file.getName()), "info");
    };

    /**
     * Render given json object
     *
     * @param {Picasso.File}  file  A {@type Picasso.File}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderVersion = function (file) {
        var dialog = this;

        var tr = $("<tr>", {
            class: "pica-table-selection-disabled",
            draggable: false
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "pica-icon-column" });

        var icon = file.getIcon();

        td.append(icon);

        /* Add thumbnail if found */
        if (file.isImage()) {
            var img = $("<img>", {
                class: "pica-icon-thumb-list",

                onLoad: function () {
                    icon.hide();
                    icon.after(this);
                },

                src: file.getThumbnailUrl()
            });

            img.on("error", function () {
                icon.show();
                this.remove();

                return true;
            });
        }

        tr.append(td);

        /* 2. column: Name */
        td = $("<td>", {
            colspan:2,
            class: "pica-name pica-overflow pica-vertical-middle text-left",
            "data-sort": file.getVersion(),
            "data-serial": file.serialize()
        });


        var span = $("<span>", {
            text: file.getName(),
            title: file.getName(),
            "data-tooltip": "",
            "data-placement": "bottom"
        });

        span.tooltip({container: "body"});

        if (file.getVersion() === dialog._file.getVersion()) {
            span.addClass("label label-success");
        }

        td.append(span);
        tr.append(td);

        /* 3. column: Versions */
        td = $("<td>", {
            class: "pica-vertical-middle text-left",
            html: Picasso.Helper.createLang("string_version", file.getVersion())
        });

        tr.append(td);

        /* 4. column: Last modified */

        td = $("<td>", {
            class: "pica-vertical-middle text-left",
        });

        /* Create tooltip */
        var span = $("<span>", {
            text: Picasso.Helper.formatAge(file.getModifiedDateMS()),
            title: Picasso.Helper.formatDate(file.getModifiedDateMS()),
            "data-tooltip": "",
            "data-placement": "bottom"
        });

        if (file.getModifiedDate()) {
            td.attr("data-sort", new Date(file.getModifiedDate()).getTime());
        }

        td.append(span);

        span.tooltip({container: "body"});


        tr.append(td);

        /* 5. column: Size */
        td = $("<td>", {
            class: "pica-vertical-middle text-left",
        });

        if (file.is(Picasso.File.Flags.FILE)) {
            td.append(Picasso.Helper.formatSize(file.getSize()));
        }

        tr.append(td);

        /* 6. column: Download Button */
        td = $("<td>", {
            class: "text-right"
        });

        /* Add download button for files (not directories) */
        if (!file.is(Picasso.File.Flags.DIRECTORY)) {
            if (file.getVersion() === dialog._file.getVersion()) {
                var currentButton = $("<button>", {
                    class: "btn btn-success disabled",
                    text: Picasso.Lang.get("label_current")
                });

                td.append(currentButton);
            } else {
                var downloadButton = $("<button>", {
                    class: "btn btn-default",
                    text: Picasso.Lang.get("action_download")
                });

                downloadButton.click(function () {
                    downloadVersion(file, file.getVersion());
                });

                td.append(downloadButton);
            }
        }

        tr.append(td);

        return tr;
    };

    /**
     * Assign data to inputs
     *
     * @param {Picasso.Dialog.Download}  dialog  A #{@link Picasso.Dialog.Download}
     **/

    var renderVersions = function (dialog) {
        /* Get file versions */
        if (dialog._file.is(Picasso.File.Flags.FILE|
            Picasso.File.Flags.DIRECTORY))
        {

            /* Add version data */
            dialog._file.getVersions(

                /* Success */
                function (versions) {
                    var files = $.merge([ dialog._file ], versions);

                    dialog._table.update(files, $.proxy(renderVersion, dialog));
                    dialog._table.sort(2, false);
                    dialog._table.checkIfEmpty();
                },

                /* Error */
                function (e) {
                    Picasso.debugLog(e);

                    dialog._table.checkIfEmpty();
                }
            );
        }
    };

    /**
     * Constructor for {@link Picasso.Dialog.Download}
     *
     * @param {Picasso.File}  file  A #{@link Picasso.File}
     * @constructor
     **/

    Picasso.Dialog.Download = function (file) {
        var self = this;

        this._file = file;

        var dialog = $("#pica_download_dialog");

        if (0 === dialog.length) {
            throw "Dialog not found";
        }

        this._table = new Picasso.Table("#pica_download_versions");

        dialog.bind("hidden.bs.modal", function () {
            self._table.clear();
        });
    };


    Picasso.Dialog.Download.prototype.show = function () {
        var self = this;

        var dialog = new Picasso.Dialog("#pica_download_dialog",
            Picasso.Lang.get("dialog_title_download"),
            Picasso.Lang.get("dialog_body_download"));

        dialog.setAssignHandler(function () {
            renderVersions(self);
        });

        dialog.show();
    };

    /* Fix dropdowns in scroller */
    Picasso.Combo.fixDropdownInScroller(".pica-inputlist-scroller");
})();
