/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Dialog")) {
    window.Picasso.Dialog = {};
}

/* Private scope */
(function () {

    /* Options for dropdown menus */
    var optionsForMembers = [
        {
            text: Picasso.Lang.get("permission_is_member"),
            type: "radio",
            group: "permission"
        },
        {
            text: Picasso.Lang.get("permission_is_member_and_admin"),
            type: "radio",
            group: "permission"
        },
        null,
        { text: Picasso.Lang.get("action_remove"), icon: "bin" }
    ];

    /* Options for dropdown menus */
    var optionsForFolders = [
        {
            text: Picasso.Lang.get("permission_can_read"),
            type: "radio",
            group: "permission"
        },
        {
            text: Picasso.Lang.get("permission_can_read_write"),
            type: "radio",
            group: "permission"
        },
        {
            text: Picasso.Lang.get("permission_can_admin"),
            type: "radio",
            group: "permission"
        },
        null,
        { text: Picasso.Lang.get("action_remove"), icon: "bin" }
    ];

    /* Options for dropdown menus */
    var optionsForCreate = [
        {
            text: Picasso.Lang.get("permission_you_admin"),
            klass: "disabled btn-success"
        }
    ];

    /**
     * Handle options for members
     *
     * @param {Picasso.InputList}  inputList  A #{@link Picasso.InputList}
     * @param {Number}             optIdx     Selected index
     * @param {Number}             idx        Item index of selected row
     * @param {Object}             userdata   Additional userdata (optional)
     **/

    var handleOptionsForMembers = function (inputList, optIdx, idx, userdata) {
        switch(optIdx) {
            case 0: ///< Member
            /* Falls through */
            case 1: ///< Admin

                inputList.setUserdataOfItem(idx,
                    $.extend(userdata, {
                            isGroupAdmin: (1 === optIdx)
                        }
                    ));
                return true;

            case 3:
                inputList.removeItem(idx);
                break;
        }
    };

    /**
     * Handle options for folders
     *
     * @param {Picasso.InputList}  inputList  A #{@link Picasso.InputList}
     * @param {Number}             optIdx     Selected index
     * @param {Number}             idx        Item index of selected row
     * @param {Object}             userdata   Additional userdata (optional)
     **/

    var handleOptionsForFolders = function (inputList, optIdx, idx, userdata) {
        switch (optIdx) {
            case 0:
                /* Falls through */
            case 1:
                /* Falls through */
            case 2:
                var permission;

                switch(optIdx) {
                    case 0: ///< Read
                        permission = Picasso.Permission.READ;
                        break;

                    case 1: ///< Read/Write
                        permission = Picasso.Permission.READ_WRITE;
                        break;

                    case 2: ///< Admin
                        permission = Picasso.Permission.ADMIN;
                        break;
                }

                inputList.setUserdataOfItem(idx,
                    $.extend(userdata, {
                            permission: permission
                        }
                    ));
                return true;

            case 4:
                inputList.removeItem(idx);
                break;
        }
    };

    /**
     * Copy data from inputs to account
     *
     * @param {Picasso.Dialog.Group}  dialog  A #{@link Picasso.Dialog.Group}
     * @param {Object}                inputs  Input fields ({ "id" = HTMLElement })
     **/

    var copyData = function (dialog, inputs) {
        dialog._group.setName(inputs.pica_group_name.val());
        dialog._group.setNotes(inputs.pica_group_notes.val());
    };

    /**
     * Assign data to inputs
     *
     * @param {Picasso.Dialog.Group}  dialog  A #{@link Picasso.Dialog.Group}
     * @param {Object}                inputs   Input fields ({ "id" = HTMLElement })
     **/

    var assignData = function (dialog, inputs) {
        inputs.pica_group_name.val(dialog._group.getName());
        inputs.pica_group_notes.val(dialog._group.getNotes());

        /* Set avatar */
        var img = $("#pica_group_avatar");

        img.off("load").on("load", function () {
            var that = $(this);

            that.siblings().hide();
            that.show();
        });

        img.off("error").on("error", function () {
            var that = $(this);

            that.siblings().show();
            that.hide();
        });

        img.attr("src", dialog._group.getAvatarUrl());

        /* Avatar buttons */
        $("#pica_group_avatar_change").off("click").click(function () {
            Picasso.Dialog.Upload.changeAvatar(dialog._group);
        });

        $("#pica_group_avatar_delete").off("click").click(function () {
            Picasso.Dialog.Upload.deleteAvatar(dialog._group);
        });


        /* Disable save button until everything is loaded */
        var groupEditSaveButton = $("#pica-group-edit-save");
        groupEditSaveButton.hide();

        /* Enable the save button for group edit dialog */
        var enableSaveButton = function (state) {
            var groupEditSaveButton = $("#pica-group-edit-save");
            if (state) {
                groupEditSaveButton.show();
            } else {
                groupEditSaveButton.hide();
            }
        };

        /* Check if a group is completely loaded (members, folders, organizations) */
        var checkGroupCompletelyLoaded = function () {

            if (dialog._accounts_inputList.isLoading() ||
                    dialog._folders_inputList.isLoading() ||
                    dialog._organizations_inputList.isLoading()) {
                return false;
            }

            enableSaveButton(true);
        };

        /* Load folders */
        dialog._folders_inputList.showLoadingSpinner(true);
        dialog._group.getFolders(

            /* Success */
            function (files) {
                $.each(files, function () {
                    /* Find index of selected permission */
                    var idx = 0;

                    switch (this.getPermission()) {
                        case Picasso.Permission.READ:
                            idx = 0;
                            break;

                        case Picasso.Permission.READ_WRITE:
                            idx = 1;
                            break;

                        case Picasso.Permission.ADMIN:
                            idx = 2;
                            break;
                    }

                    dialog._folders_inputList.addItem(
                        "folder-closed", this.getName(), null, {
                            oid: this.getOid(),
                            name: this.getName(),
                            permission: this.getPermission()
                        }, optionsForFolders, handleOptionsForFolders, idx
                    );
                });
                dialog._folders_inputList.showLoadingSpinner(false);
                checkGroupCompletelyLoaded();
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
                enableSaveButton(false);
            }
        );

        /* Load members */
        dialog._accounts_inputList.showLoadingSpinner(true);
        dialog._group.getAccounts(

            /* Success */
            function (accounts) {
                $.each(accounts, function () {
                    // Create a copy of options for each account to avoid mutation issues
                    var memberOptions = JSON.parse(JSON.stringify(optionsForMembers));

                    // ORG_ADMINs cannot be demoted to just "Member" - disable that option
                    if (this.is(Picasso.Account.Flags.ORG_ADMIN)) {
                        memberOptions[0]["disabled"] = true;
                    }

                    dialog._accounts_inputList.addItem(
                        "user", this.getName(), null, {
                            oid: this.getOid(),
                            name: this.getName(),
                            isGroupAdmin: this.is(Picasso.Account.Flags.GROUP_ADMIN)
                        }, memberOptions, handleOptionsForMembers,
                        (this.is(Picasso.Account.Flags.GROUP_ADMIN) ? 1 : 0)
                    );
                });
                dialog._accounts_inputList.showLoadingSpinner(false);
                checkGroupCompletelyLoaded();
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
                enableSaveButton(false);
            }
        );

        /* Load organizations */
        dialog._organizations_inputList.showLoadingSpinner(true);
        dialog._group.getOrganizations(

            /* Success */
            function (organizations) {
                $.each(organizations, function () {
                    dialog._organizations_inputList.addItem(
                        this.getIcon(), this.getName(), null, {
                            oid: this.getOid(),
                            name: this.getName()
                        }
                    );
                });
                dialog._organizations_inputList.showLoadingSpinner(false);
                checkGroupCompletelyLoaded();
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
                enableSaveButton(false);
            }
        );
    };

    /**
     * Save {@link Picasso.Group} data
     *
     * @param {Picasso.Dialog.Group}  dialog  A #{@link Picasso.Dialog.Group}
     * @param {Function}              onSuccess  Call on success
     * @param {Function}              onError    Call on error
     **/

    var saveData = function (dialog, onSuccess, onError) {

        /**
         * WARNING: Apparently, we cannot start all these async requests
         *          at once, because some synchronizing timing issue
         *          raises database exceptions.
         *
         *          So we basically wrap all requests in functions (to prevent
         *          starting them immediately) and chain them with promises..
         **/

        /* Store data */
        var defer_save = function () {
            return dialog._group.save(null, onError);
        };

        /* Store folders */
        var folders = dialog._folders_inputList.getItems();

        dialog._group.setFoldersCount(folders.length);

        var defer_folders = function () {
            var data = {
                "folderID[]": [],
                "folderName[]": [],
                "permission[]": []
            };

            /* Collect data */
            $.each(folders, function () {
                if (this.hasOwnProperty("oid") && this.oid) {
                    data["folderID[]"].push(this.oid);
                } else {
                    data["folderName[]"].push(this.name);
                }

                data["permission[]"].push(
                    Picasso.Permission.toString(this.permission));
            });

            return dialog._group.setFolders(data, null,

                /* Error */
                function (e, status) {
                    /* Handle folder errors */
                    switch (status) {
                        case 409:
                            Picasso.Notification.show(Picasso.Lang.get(
                                "notification_error_duplicate",
                                Picasso.Lang.get("label_folders")), "warning");
                            break;
                    }

                    if (onError) onError(e, status);
                }
            );
        };

        /* Store members */
        var members = dialog._accounts_inputList.getItems();

        var defer_members = function () {
            var data = {
                "accountID[]": [],
                "accountName[]": [],
                "isGroupAdmin[]": []
            };

            /* Collect data */
            $.each(members, function () {
                if (this.hasOwnProperty("oid") && this.oid) {
                    data["accountID[]"].push(this.oid);
                } else {
                    data["accountName[]"].push(this.name);
                }

                data["isGroupAdmin[]"].push((this.isGroupAdmin || false));
            });

            return dialog._group.setAccounts(data, null, onError);
        };

        dialog._group.setMembersCount(members.length);

        /* Store members */
        var organizations = dialog._organizations_inputList.getItems();

        var defer_organizations = function () {
            var account = Picasso.get("account");

            /* Exclude normal user */
            if (account.is(Picasso.Account.Flags.ADMIN|
                Picasso.Account.Flags.ORG_CREATOR|Picasso.Account.Flags.ORG_ADMIN)|(account.getOrgOid() && account.getOrgOid() != "" )) {
                var data = {
                    "organizationID[]": [],
                    "isOrganizationAdmin[]": []
                };

                /* Collect data */
                $.each(organizations, function () {
                    if (this.hasOwnProperty("oid") && this.oid) {
                        data["organizationID[]"].push(this.oid);
                    }

                    data["isOrganizationAdmin[]"].push(false); ///< TODO
                });

                return dialog._group.setOrganizations(data, null,

                    /* Error */
                    function (e, status) {
                        Picasso.Dialog.handleOrgErrors(status);

                        if (onError) onError(e, status);
                    }
                );
            } else {
                return true;
            }
        };

        /* Future and broken promises */
        $.when(defer_save()).then(function () {
            $.when(defer_folders()).then(function () {
                $.when(defer_members()).then(function () {
                    $.when(defer_organizations()).then(function () {
                        if (onSuccess) {
                            onSuccess(dialog._group);
                        }
                    }, onError)
                }, onError)
            }, onError)
        });
    };

    /**
     * Constructor for {@link Picasso.Dialog.Group}
     *
     * @param {Picasso.Dialog.Flags}  options  Options for this element (optional)
     * @constructor
     **/

    Picasso.Dialog.Group = function (options) {
        var self = this;

        /* Init */
        this._group = null;
        this._flags = (options || 0);

        var dialog = $("#pica_group_dialog");

        if (0 === dialog.length) {
            throw "Dialog not found";
        }

        /* Check immutable flag */
        var inputFlags = 0;

        if (0 < (this._flags & Picasso.Dialog.Flags.IMMUTABLE)) {
            inputFlags = Picasso.InputList.Flags.IMMUTABLE;

            $("#pica_group_avatar_change").hide();
            $("#pica_group_avatar_delete").hide();
        } else {
            $("#pica_group_avatar_change").show();
            $("#pica_group_avatar_delete").show();
        }

        /* Init dialog */
        Picasso.Helper.bindNav(dialog.find(".nav"));

        /* Create folders list */
        this._folders_inputList =
            new Picasso.InputList("#pica_group_folders",
                (inputFlags|Picasso.InputList.Flags.LIST_MODE|
                    Picasso.InputList.Flags.COMPLETION_ONLY|
                    Picasso.InputList.Flags.COMPLETION_NEW));

        this._folders_inputList.setCompletionHandler(
            Picasso.TagInput.handleFolderCompletion
        );

        this._folders_inputList.setCompletionSelectHandler(
            function (e, userdata) {
                this.addItem("folder-new", userdata.getName(), null, {
                        oid: userdata.getOid(),
                        name: userdata.getName(),
                        permission: Picasso.Permission.READ
                    }, optionsForFolders, handleOptionsForFolders, 0
                );
                this._tagInput.hideCompletions();
            }

        );

        this._folders_inputList.setCompletionNewHandler(
            function () {
                var name = this.getInputValue();

                if(0 < name.length) {
                    this.addItem("folder-plus", name, null, {
                            oid: null,
                            name: name,
                            permission: Picasso.Permission.READ
                        }, optionsForFolders, handleOptionsForFolders, 0
                    );
                }
            }
        );

        this._folders_inputList.setSubmitHandler(
            function (e, userdata) {
                var name = this.getInputValue();

                if(0 < name.length) {
                    this.addItem("folder-new", name, null, {
                            oid: null,
                            name: name,
                            permission: Picasso.Permission.READ
                        }, optionsForFolders, handleOptionsForFolders, 0
                    );
                }
            }
        );

        /* Create accounts list */
        this._accounts_inputList =
            new Picasso.InputList("#pica_group_accounts",
                (inputFlags|Picasso.InputList.Flags.LIST_MODE));

        this._accounts_inputList.setCompletionHandler(
            Picasso.TagInput.handleAccountCompletion
        );

        this._accounts_inputList.setCompletionSelectHandler(
            function (e, userdata) {
                // New members are always added as "Member" (not admin) by default
                this.addItem("user-asterisk", userdata.getName(), null, {
                        oid: userdata.getOid(),
                        name: userdata.getName(),
                        isMember: true,
                        isGroupAdmin: false
                    }, optionsForMembers, handleOptionsForMembers, 0
                );
                this._tagInput.hideCompletions();
            }
        );

        this._accounts_inputList.setSubmitHandler(
            function (e, userdata) {
                var name = this.getInputValue();

                // When adding by name (not from completion), we don't know if they're ORG_ADMIN
                // Default to regular member - server will validate
                if(0 < name.length) {
                    this.addItem("user-asterisk", name, null, {
                            oid: "",
                            name: name,
                            isMember: true,
                            isGroupAdmin: false
                        }, optionsForMembers, handleOptionsForMembers, 0
                    );
                }
            }
        );

        this._accounts_inputList.setValidationErrorHandler(
            function () {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_invalid_email"),
                    "warning");
            }
        );

        /* Create organizations list */
        this._organizations_inputList =
            new Picasso.InputList("#pica_group_organizations",
                (inputFlags|Picasso.InputList.Flags.LIST_MODE|
                    Picasso.InputList.Flags.COMPLETION_ONLY));

        this._organizations_inputList.setCompletionHandler(
            Picasso.TagInput.handleOrganizationCompletion
        );

        this._organizations_inputList.setCompletionSelectHandler(
            function (e, userdata) {
                this.addItem(userdata.getIcon(), userdata.getName(), null, {
                    oid: userdata.getOid(),
                    name: userdata.getName()
                });
             this._tagInput.hideCompletions();
            }

    );

        this._organizations_inputList.setSubmitHandler(
            function (e, userdata) {
                var name = this.getInputValue();

                if(0 < name.length) {
                    this.addItem("education", name, null, {
                        oid: "",
                        name: name
                    });
                }
            }
        );

        /* Clear on close */
        dialog.bind("hidden.bs.modal", function () {
            self._folders_inputList.clear();
            self._accounts_inputList.clear();
            self._organizations_inputList.clear();
        });
    };

    /**
     * Create new {@link Picasso.Group} via dialog
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Dialog.Group.prototype.create = function (onSuccess, onError) {
        var self = this;

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_group_dialog",
            Picasso.Lang.get("dialog_title_create"),
            Picasso.Lang.get("dialog_body_create", Picasso.Lang.get("label_groups")),
            (this._flags|Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE|Picasso.Dialog.Flags.SPINNER));

        dialog.setAssignHandler(function (inputs) {
            $("#pica_group_dialog .pica-hide-on-create").hide();
            $("#pica_group_avatar").attr("src", "");

            var account = Picasso.get("account");

            /* Always add caller */
            self._accounts_inputList.addItem("user", account.getName(), null, {
                    oid: account.getOid(),
                    name: account.getName(),
                    isGroupAdmin: true
                }, optionsForCreate, null, 0, true
            );
        });

        dialog.setOkHandler(function (inputs) {
            self._group = new Picasso.Group();
            self._group.setOid(Picasso.Helper.getID());
            copyData(self, inputs);
            saveData(self,

                /* Success */
                function () {
                    dialog.hide();

                    if (onSuccess) {
                        onSuccess.apply(null, arguments);
                    }
                },

                /* Error */
                function (e, status) {
                    dialog.showSpinner(false);

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
        });

        var account = Picasso.get("account");

        /* Always add organization of creator */
        if (account.getOrgOid()) {
            this._organizations_inputList.addItem(
                "education", account.getOrgName(), null, {
                    oid: account.getOrgOid(),
                    name: account.getOrgName()
                });
        }

        dialog.show();
    };

    /**
     * Edit {@link Picasso.Group} via dialog
     *
     * @param {Picasso.Group}  group      A #{@link Picasso.Group}
     * @param {Function}       onSuccess  Call on success
     * @param {Function}       onError    Call on error
     **/

    Picasso.Dialog.Group.prototype.edit = function (group, onSuccess, onError) {
        var self = this;

        this._group = group;

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_group_dialog",
            Picasso.Lang.get("dialog_title_edit"),
            Picasso.Lang.get("dialog_body_edit", Picasso.Lang.get("label_groups")),
            (this._flags|Picasso.Dialog.Flags.VALIDATE|
                Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE|Picasso.Dialog.Flags.SPINNER));

        dialog.setAssignHandler(function (inputs) {
            $("#pica_group_dialog .pica-hide-on-create").show();

            assignData(self, inputs);
        });

        dialog.setOkHandler(function (inputs) {
            copyData(self, inputs);
            saveData(self,

                /* Success */
                function () {
                    dialog.hide();

                    if (onSuccess) {
                        onSuccess.apply(null, arguments);
                    }
                },

                /* Error */
                function (e, status) {
                    dialog.showSpinner(false);

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
        });

        dialog.show();
    };
})();
