/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Dialog")) {
    window.Picasso.Dialog = {};
}

/**
 * Some custom classes for specific behavior are used:
 *
 * .share-min-admin - Requires admin permissions or higher
 * .share-min-owner - Requires owner permissions
 * .share-accounts-only - Elements just for accounts
 **/

/* Private scope */
(function () {

    /* Cache filtered permissions list */
    var permissionList = Picasso.Permission.filter(function (permission) {
        return (Picasso.Permission.OWNER > permission &&
            Picasso.Permission.NO_ACCESS < permission);
    });
    /* Options for dropdown menus */
    var optionsForPermissions = [
        {
            text: Picasso.Lang.get("permission_can_read"),
            type: "radio",
            group: "permissions",
            klass: "share-min-admin"
        },
        {
            text: Picasso.Lang.get("permission_can_read_write"),
            type: "radio",
            group: "permissions",
            klass: "share-min-admin"
        },
        {
            text: Picasso.Lang.get("permission_can_admin"),
            type: "radio",
            group: "permissions",
            klass: "share-min-admin"
        },
        {
            type: "divider",
            klass: "share-min-owner share-accounts-only"
        },
        {
            text: Picasso.Lang.get("permission_can_owner"),
            type: "radio",
            group: "permissions",
            klass: "share-min-owner share-accounts-only"
        },
        null,
        { text: Picasso.Lang.get("action_remove"), icon: "bin" }
    ];

    /* Options for dropdown menus */
    var optionsForUploadforms = [
        {
            text: Picasso.Lang.get("label_settings"),
            icon: "cogwheel"
        },
        null,
        {
            text: Picasso.Lang.get("action_remove"),
            icon: "bin"
        }
    ];

    /* Helper */

    /**
     * Grant {@link Picasso.Permission} on given {@link Picasso.File}
     *
     * @param {Object}              obj         Either a #{@link Picasso.Account} or a #{@link Picasso.Group}
     * @param {Picasso.File}        file        A #{@link Picasso.File}
     * @param {Picasso.Permission}  permission  A #{@link Picasso.Permission}
     * @param {Function}      onSuccess  Call on success
     **/

    var grant = function (obj, file, permission, onSuccess) {
        obj.grant(file, permission,

            /* Success */
            function (json, status) {
                /* Update file stats when user/group gets new access */
                if (201 === status) {
                    if (obj instanceof Picasso.Account) {
                        file.setAccountsCount(file.getAccountsCount() + 1);
                    } else {
                        file.setGroupsCount(file.getGroupsCount() + 1);
                    }
                }

                if (onSuccess) {
                    onSuccess(file);
                }

                /* No need to update UI here; so just a notification */
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_updated",
                        Picasso.Lang.get("label_permissions")), "info");
            },

            /* Error */
            function () {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_update",
                        Picasso.Lang.get("label_permissions")), "danger");

                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Revoke all {@link Picasso.Permission} on given {@link Picasso.File}
     *
     * @param {Object}        obj        Either a #{@link Picasso.Account} or a #{@link Picasso.Group}
     * @param {Picasso.File}  file       A #{@link Picasso.File}
     * @param {Function}      onSuccess  Call on success
     **/

    var revokeAll = function (obj, file, onSuccess) {
        obj.revokeAll(file,

            /* Success */
            function () {

                /* Update file stats */
                if (obj instanceof Picasso.Account) {
                    file.setAccountsCount(file.getAccountsCount() - 1);
                } else {
                    file.setGroupsCount(file.getGroupsCount() - 1);
                }

                if (onSuccess) {
                    onSuccess(file);
                }

                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_removed",
                        Picasso.Lang.get("label_permissions")), "info");
            },

            /* Error */
            function () {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_update",
                        Picasso.Lang.get("label_permissions")), "danger");

                Picasso.debugLog(e);
            }
        )
    };

    /**
     * Reload accounts of given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@link Picasso.File}
     **/

    var reloadAccounts = function (file) {

        /* Reload all accounts data */
        file.getAccountsWithPermission(

            /* Success */
            function (accounts) {
                /* Remove old accounts */
                var rows = Picasso.Dialog.Share.table.getRows();

                $.each(rows, function () {
                    var that = $(this);

                    if (-1 !== that.attr("id").indexOf("share_account")) {
                        that.remove();
                    }
                });

                /* Add new accounts */
                Picasso.Dialog.Share.table.append(
                    accounts, renderWithPermission);

                Picasso.Dialog.Share.table.sort(1, false);
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Handle options for permissions
     *
     * @param {Object}  obj       Object to handle
     * @param {Number}  optIdx    Selected index
     * @param {Object}  userdata  Userdata (optional)
     *
     * @returns {Boolean} Whether selection should be set
     **/

    var handleOptionsWithPermissions = function (obj, optIdx, userdata) {
        var self = $(this);

        /* Get file */
        var file = Picasso.get("selectedFile");

        if (file && obj.holder) {

            /* Handle selected permission */
            switch (optIdx) {
                case 0: /* Can read */
                /* Falls through */
                case 1: /* Can read/write */
                /* Falls through */
                case 2: /* Can administrate */
                    grant(obj.holder, file, permissionList[optIdx]);

                    return true;

                case 4: /* Is owner */
                    var isAccount = (obj.holder instanceof Picasso.Account);

                    /* Allow only accounts to be owner */
                    if (isAccount) {
                        /* Show dialog */
                        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
                            Picasso.Lang.get("dialog_title_change_owner"),
                            Picasso.Lang.get("dialog_body_change_owner",
                                file.getName(), obj.holder.getName()));

                        dialog.setOkHandler(function () {
                            grant(obj.holder, file, Picasso.Permission.OWNER,
                                reloadAccounts);
                        });

                        dialog.show();
                    } else {
                        return false;
                    }
                    break;

                case 6: /* Remove */
                    /* Helper to fade out rows */
                    var fadeOut = function () {
                        Picasso.Helper.fadeOut(self.parents("tr"),
                            function () {
                                this.remove();

                                Picasso.Dialog.Share.table.checkIfEmpty();
                            }
                        );
                    };

                    /* Check whether to show dialog */
                    if (false === userdata) {
                        revokeAll(obj.holder, file, fadeOut);
                    } else {
                        /* Show dialog */
                        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
                            Picasso.Lang.get("dialog_title_remove"),
                            Picasso.Lang.get("dialog_body_remove",
                                obj.holder.getName()));

                        dialog.setOkHandler(function () {
                            revokeAll(obj.holder, file, fadeOut);
                        });

                        dialog.show();
                    }

                    return false;

                default: /* Ignore */
            }
        }

        return false
    };

    /**
     * Handle options for public access
     *
     * @param {Picasso.File}  file    A #{@link Picasso.File}
     * @param {Number}        optIdx  Selected index
     *
     * @returns {Boolean} Whether selection should be set
     **/

    var handleOptionsForPublicAccess = function (file, optIdx) {
        var self = $(this);

        file.setPublicPermission(Picasso.Permission.NO_ACCESS,

            /* Success */
            function () {
                /* Remove rows */
                var tr = self.parents("tr");

                Picasso.Helper.fadeOut(tr,
                    function () {
                        tr.remove();
                        Picasso.Dialog.Share.table.checkIfEmpty();
                    }
                );

                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_deleted",
                        Picasso.Lang.get("label_links")), "info");
            }
        );
    };

    /**
     * Render given json object
     *
     * @param {Object}  obj  Given json object
     *
     * @returns {HTMLElement} Rendered element
     **/

    var renderWithPermission = function (obj) {
        var isAccount = (obj.holder instanceof Picasso.Account);

        var tr = $("<tr>", {
            id: "share_" + (isAccount ? "account_" : "group_") +
                Picasso.Helper.hashCode(obj.holder.getOid()),
            class: "pica-serial",
            draggable: false
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        Picasso.Helper.createGlyphAndAvatar(td,
            (isAccount ? "user" : "group"), obj.holder);

        tr.append(td);

        /* 2. column: Name */
        td = $("<td>", {
            class: "pica-overflow pica-vertical-middle text-left",
            "data-sort": obj.permission,
        });

        var span = $("<span>", {
            class: "pica-name pica-table-selection-title",
            text: (isAccount ? obj.holder.getDisplayName() :
                obj.holder.getName()),
            title: (isAccount ? obj.holder.getUserName() :
                obj.holder.getName()),
            "data-tooltip": "",
            "data-placement": "bottom"
        });

        span.tooltip({ container: "body" });

        td.append(span);
        tr.append(td);

        /* 3. column: Permissions */
        var account = Picasso.get("account");
        var file = Picasso.get("selectedFile");

        /* 3a) Highlight immutable owner and us */
        if (Picasso.Permission.OWNER === obj.permission) {
            td = $("<td>", {
                class: "text-right",
            });

            var button = $("<button>", {
                class: "btn btn-success disabled pull-right",
                text: Picasso.Lang.get(
                    (isAccount && account.isMe(obj.holder) ?
                        "permission_you_owner" : "permission_can_owner"))
            });

            td.append(button);

            tr.addClass("pica-table-selection-disabled");
        } else {
            /* 3b) Add permission checkboxes for other members */
            if (Picasso.Permission.OWNER === file.getPermission() ||
                Picasso.Permission.ADMIN === file.getPermission())
            {
                /* Select correct permission */
                var permissionIdx = permissionList.indexOf(obj.permission);

                permissionIdx = (-1 !== permissionIdx ? permissionIdx : 0);

                var myOptionsForPermissions = optionsForPermissions.slice();
                /* Special case for handling federated permissions */
                if(obj.holder.is(Picasso.Account.Flags.IS_PROXY)) {
                    // Remove administrate + owner.
                    myOptionsForPermissions[2] = null;
                    myOptionsForPermissions[4] = null;
                }

                td = $("<td>", {
                    class: "hidden-xs text-right",
                    html: Picasso.Combo.createDropdownOrButton(myOptionsForPermissions,
                        $.proxy(handleOptionsWithPermissions, this, obj),
                        permissionIdx, "pull-right")
                });

                /* Remove limited options */
                if (!isAccount) {
                    td.find(".share-accounts-only").remove();
                }

                if (Picasso.Permission.ADMIN === file.getPermission() && !account.is(Picasso.Account.Flags.ADMIN)) {
                    td.find(".share-min-owner").remove();
                }

            /* 3c) Add other permissions: file.getPermission() contains our
             *                            permission for this folder */
            } else {
                td = $("<td>", {
                    class: "text-right",
                });

                var button = $("<button>", {
                    class: "btn btn-default disabled pull-right",
                    text: Picasso.Lang.get(
                        (isAccount && account.isMe(obj.holder) ?
                            "permission_you_" : "permission_can_") +
                        Picasso.Permission.toString(obj.permission).toLowerCase())
                });

                td.append(button);

                tr.addClass("pica-table-selection-disabled");
            }
        }

        tr.append(td);

        return tr;
    };

    /**
     * Render given json object
     *
     * @param {Picasso.Invitation}  invitation  A {@type Picasso.Invitation}
     *
     * @returns {HTMLElement} Rendered element
     **/

    var renderInvitation = function (invitation) {
        var tr = $("<tr>", {
            draggable: false
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        td.append(Picasso.Helper.createGlyph("send"));
        tr.append(td);

        var invitationText = invitation.getDisplayName() ? invitation.getDisplayName() : invitation.getRecipientName();
        if (Picasso.Permission.OWNER === invitation.getPermission()) {
            invitationText += " (" + Picasso.Lang.get("permission_owner") + ")";
        }

        /* 2. column: Name */
        td = $("<td>", {
            class: "pica-name pica-overflow pica-vertical-middle text-left",
            text: invitationText,
        });

        tr.append(td);

        /* 3. column: Button */
        td = $("<td>", {
            class: "text-right",
        });

        var button = $("<button>", {
            class: "btn btn-default",
            type: "button",
        });

        button.append(Picasso.Helper.createLang("action_remove"));

        button.click(function () {
            invitation.remove(

                /* Success */
                function () {
                    Picasso.Helper.fadeOut(tr,

                        function () {
                            tr.remove();

                            Picasso.Dialog.Share.table.checkIfEmpty();
                        });

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_deleted",
                        Picasso.Lang.get("label_invitations")), "info");
                }
            );
        });

        td.append(button);
        tr.append(td);

        return tr;
    };

    /**
     * Render given json object
     *
     * @param {Picasso.Link}  link  A {@type Picasso.Link}
     *
     * @returns {HTMLElement} Rendered element
     **/

    var renderLink = function (link) {
        var tr = $("<tr>", {
            id: "share_" + link.getElemId(),
            class: "pica-table-selection-disabled",
            draggable: false
        });

        /* Add icon */
        var td = $("<td>", {
            class: "pica-icon-column",
        });

        var span = $("<span>", {
            class: "pica-icon pica-glyph glyphicons glyphicons-link",
        });

        td.append(span);
        tr.append(td);

        var td = $("<td>", {
            colspan: 2,
            class: "text-right"
        });

        var media = $("<div>", { class: "media" });
        var body = $("<div>", { class: "media-body" });
        var heading = $("<div>", {
            class: "pica-media-header media-heading clearfix"
        });
        var footer = $("<div>", { class: "pica-media-footer" });

        /* Everybody text */
        var div = $("<div>", {
            class: "pica-overflow pica-vertical-middle pull-left",
        });

        div.append(Picasso.Helper.createLang("label_everybody_with_this_link"));

        /* Add valid till */
        if (-1 !== link.getValidTill()) {
            span = $("<span>", {
                class: "pica-in-brackets text-info",
                "data-tooltip": "",
                text: Picasso.Lang.get(Picasso.Lang.Flags.DOWNCASE, "string_until",
                        Picasso.Helper.formatDate(link.getValidTill(), false)),
                title: Picasso.Helper.formatAge(link.getValidTill())
            });
            if (Picasso.Helper.inThePast(link.getValidTill())) {

                span.css('color', '#FF0000');
            }
            span.tooltip({container: "body"});

            span.click(function () {

                var self = $(this);

                /* Show dialog */
                var dialog = new Picasso.Dialog("#pica_input_dialog",
                    Picasso.Lang.get("dialog_title_change"),
                    Picasso.Lang.get("dialog_language_change_valid_till"));

                dialog.setAssignHandler(function () {

                    if (-1 !== link.getValidTill()) {
                        Picasso.Helper.setDateTimePickerValue($("#pica_input_text"),
                            link.getValidTill());
                    }
                });

                dialog.setOkHandler(function (inputs, dialog) {
                    link.setValidTill(Picasso.Helper.getDateTimePickerValue(
                        $("#pica_input_text"), -1))
                    link.save(
                        /* Success */
                        function () {
                            /* Replace rows */
                            var tr = self.parents("tr");

                            var rendered = renderLink(link);

                            $(tr).replaceWith(rendered);

                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_item_updated",
                                    Picasso.Lang.get("label_links")), "info");
                        },
                        /* Error */
                        function (e, status, response) {
                            dialog.showSpinner(false);

                            if (status === 406) {
                                if (response && response.message) {
                                    Picasso.Notification.show(response.message);
                                }
                            }
                        }
                    );
                });

                dialog.show();
            });

            div.append(span);
        }

        heading.append(div);

        var span = $("<span>", {
            class: "cogwheel-icon",
            style: "font-size: 24px; padding: 8px; cursor: pointer;",
        });

        var cogwheelIcon = $("<i>", {
            class: "glyphicons glyphicons-cogwheel",
        });

        span.append(cogwheelIcon);

        span.click(function () {
            var self = $(this);

            var dialog = new Picasso.Dialog.Link();

            dialog.edit(link,

                /* Success */
                function () {
                    /* Replace rows */
                    var tr = self.parents("tr");

                    var rendered = renderLink(link);

                    $(tr).replaceWith(rendered);

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_updated",
                            Picasso.Lang.get("label_links")), "info");
                }
            );
        });

        heading.append(span);

        var deleteIcon = $("<i>", {
            class: "glyphicons glyphicons-bin",
        });

        var deleteSpan = $("<span>", {
            class: "delete-icon",
            style: "font-size: 24px; padding: 8px; cursor: pointer;",
        });

        deleteSpan.append(deleteIcon);

        deleteSpan.click(function () {
            var self = $(this);
            link.remove(

                /* Success */
                function () {
                    /* Remove rows */
                    var tr = self.parents("tr");

                    Picasso.Helper.fadeOut(tr,
                        function () {
                            tr.remove();

                            $("#share_create_link").show();

                            Picasso.Dialog.Share.table.checkIfEmpty();
                        });

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_deleted",
                            Picasso.Lang.get("label_links")), "info");
                }
            );
        });

        heading.append(deleteSpan);

        body.append(heading);

        /* Link input */
        var group = $("<div>", { class: "input-group" });

        var input = $("<input>", {
            type: "text",
            class: "pica-below form-control",
            value: Picasso.Helper.sanitizeUrl(link.getLinkUrl()),
            readonly: true
        });

        group.append(input);

        var groupBtn = $("<div>", { class: "input-group-btn" });
        var divQr = $("<div>", { class: "input-group-btn" });

        var span = $("<span>", {
            class: "pica-icon pica-glyph glyphicons glyphicons-qrcode",
        });

        span.click(function () {
            var dialog = new Picasso.Dialog("#pica_canvas_dialog",
                Picasso.Lang.get("dialog_title_link",link.getName()));

            dialog.setAssignHandler(function () {
                var canvas = document.getElementById("board")
                var qrcode = new QRious({
                    element: canvas,
                    background: '#ffffff',
                    backgroundAlpha: 1,
                    foreground: '#5868bf',
                    foregroundAlpha: 1,
                    level: 'H',
                    padding: 0,
                    size: 300,
                    value: Picasso.Helper.sanitizeUrl(link.getLinkUrl())
                });

            });

            dialog.show()
        });

        divQr.append(span)

        var button = $("<button>", {
            class: "btn btn-default",
            html: $("<i>", { class: "glyphicons glyphicons-copy" }),
            "data-tooltip": "tooltip_copy_to_clip",
            "data-placement": "bottom",
            title: Picasso.Lang.get("tooltip_copy_to_clip")
        });

        button.click(handleClickCopyToClip);
        button.tooltip({ container: "body" });

        groupBtn.append(button);
        group.append(groupBtn);
        footer.append(group);

        body.append(footer);
        media.append(body);

        td.append(media);
        tr.append(td);

        return tr;
    };

    /**
     * Render given json object
     *
     * @param {Picasso.Uploadform}  uploadForm  A {@type Picasso.Uploadform}
     *
     * @returns {HTMLElement} Rendered element
     **/

    var renderUploadForm = function (uploadForm) {
         var tr = $("<tr>", {
            id: "share_" + uploadForm.getElemId(),
            class: "pica-table-selection-disabled",
            draggable: false
        });

        /* Add icon */
        var td = $("<td>", {
            class: "pica-icon-column",
        });

        var span = $("<span>", {
            class: "pica-icon pica-glyph glyphicons glyphicons-wallet",
        });

        td.append(span);
        tr.append(td);

        var td = $("<td>", {
            colspan: 2,
            class: "text-right"
        });

        var media = $("<div>", { class: "media" });
        var body = $("<div>", { class: "media-body" });
        var heading = $("<div>", {
            class: "pica-media-header media-heading clearfix"
        });
        var footer = $("<div>", { class: "pica-media-footer" });

        /* Everybody text */
        var div = $("<div>", {
            class: "pica-overflow pica-vertical-middle pull-left",
        });

        div.append(Picasso.Helper.createLang("upload.form.ready"));

        /* Add valid till */
        if (-1 !== uploadForm.getValidTill()) {
            var span = $("<span>", {
                class: "pica-in-brackets text-info",
                "data-tooltip": "",
                text: Picasso.Lang.get(Picasso.Lang.Flags.DOWNCASE, "string_until",
                        Picasso.Helper.formatDate(uploadForm.getValidTill(), false)),
                title: Picasso.Helper.formatAge(uploadForm.getValidTill())
            });

            if (Picasso.Helper.inThePast(uploadForm.getValidTill())) {
                span.css('color', '#FF0000');
            }

            span.tooltip({container: "body"});

            div.append(span);
        }

        heading.append(div);

        /* Button or dropdown, based on file permissions */
        var file = uploadForm.getFile();

        var span = $("<span>", {
            class: "cogwheel-icon",
            style: "font-size: 24px; padding: 8px; cursor: pointer;",
        });

        var cogwheelIcon = $("<i>", {
            class: "glyphicons glyphicons-cogwheel",
        });

        span.append(cogwheelIcon);

        span.click(function () {
            var self = $(this);

            /* Show dialog */
            var dialog = new Picasso.Dialog.Uploadform();

            dialog.edit(uploadForm,

                /* Success */
                function () {
                    /* Replace rows */
                    var tr = self.parents("tr");

                    var rendered = renderUploadForm(uploadForm);

                    $(tr).replaceWith(rendered);

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_updated",
                            Picasso.Lang.get("label_upload_form")), "info");
                }
            );
        });

        heading.append(span);

        var deleteIcon = $("<i>", {
            class: "glyphicons glyphicons-bin",
        });

        var deleteSpan = $("<span>", {
            class: "delete-icon",
            style: "font-size: 24px; padding: 8px; cursor: pointer;",
        });

        deleteSpan.append(deleteIcon);

        deleteSpan.click(function () {
            var self = $(this);
            uploadForm.remove(

                /* Success */
                function () {
                    /* Remove rows */
                    var tr = self.parents("tr");

                    Picasso.Helper.fadeOut(tr,
                        function () {
                            tr.remove();

                            $("#share_create_form").show();

                            Picasso.Dialog.Share.table.checkIfEmpty();
                        });

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_deleted",
                            Picasso.Lang.get("label_upload_form")), "info");
                }

            );
        });

        heading.append(deleteSpan);

        body.append(heading);

        /* Link input */
        var group = $("<div>", { class: "input-group" });

        var input = $("<input>", {
            type: "text",
            class: "pica-below form-control",
            value: Picasso.Helper.sanitizeUrl(uploadForm.getUrl()),
            readonly: true
        });

        group.append(input);

        var groupBtn = $("<div>", { class: "input-group-btn" });

        var button = $("<button>", {
            class: "btn btn-default",
            html: $("<i>", { class: "glyphicons glyphicons-copy" }),
            "data-tooltip": "tooltip_copy_to_clip",
            "data-placement": "bottom",
            title: Picasso.Lang.get("tooltip_copy_to_clip")
        });

        button.click(handleClickCopyToClip);
        button.tooltip({ container: "body" });

        groupBtn.append(button);
        group.append(groupBtn);
        footer.append(group);

        body.append(footer);
        media.append(body);

        td.append(media);
        tr.append(td);

        return tr;
    };

    /**
     * Render given json object
     *
     * @param {Picasso.File}  file  A {@type Picasso.File}
     *
     * @returns {HTMLElement} Rendered element
     **/

    var renderPublicAccess = function (file) {
         var tr = $("<tr>", {
            id: "share_" + file.getElemId(),
            class: "pica-table-selection-disabled",
            draggable: false
        });

        /* Add icon */
        var td = $("<td>", {
            class: "pica-icon-column",
        });

        var span = $("<span>", {
            class: "pica-icon pica-glyph glyphicons glyphicons-global",
        });

        td.append(span);
        tr.append(td);

        var td = $("<td>", {
            colspan: 2,
            class: "text-right"
        });

        var media = $("<div>", { class: "media" });
        var body = $("<div>", { class: "media-body" });
        var heading = $("<div>", { class: "pica-media-header media-heading clearfix" });
        var footer = $("<div>", { class: "pica-media-footer" });

        /* Everybody text */
        var div = $("<div>", {
            class: "pica-overflow pica-vertical-middle pull-left",
        });

        div.append(Picasso.Helper.createLang("label_everybody_with_this_link"));

        heading.append(div);

        /* Button */
        var button = $("<button>", {
            class: "btn btn-default",
            text: Picasso.Lang.get("action_remove")
        });

        button.click($.proxy(handleOptionsForPublicAccess, this, file));

        heading.append(button);

        body.append(heading);

        /* Link input */
        var group = $("<div>", { class: "input-group" });

        var input = $("<input>", {
            type: "text",
            class: "pica-below form-control",
            value: Picasso.Helper.sanitizeUrl(file.getPublicUrl()),
            readonly: true
        });

        group.append(input);

        var groupBtn = $("<div>", { class: "input-group-btn" });

        button = $("<button>", {
            class: "btn btn-default",
            html: $("<i>", { class: "glyphicons glyphicons-copy" }),
            "data-tooltip": "tooltip_copy_to_clip",
            "data-placement": "bottom",
            title: Picasso.Lang.get("tooltip_copy_to_clip")
        });

        button.click(handleClickCopyToClip);
        button.tooltip({ container: "body" });

        groupBtn.append(button);
        group.append(groupBtn);
        footer.append(group);

        body.append(footer);
        media.append(body);

        td.append(media);
        tr.append(td);

        return tr;
    };

    /**
     * Handle click on taginput submit
     *
     * @return {Boolean}  Either true on success; otherwise false
     **/

    var handleSubmit = function () {

        /* Get file */
        var file = Picasso.get("selectedFile");

        if (file) {
            var tags = Picasso.Dialog.Share.tagInput.getTags();
            var toggle = $("#share_buttons button.dropdown-toggle");
            var permission = toggle.data("permission");

            var tr = $("<tr>", {
                class: "pica-not-striped"
            });

            var td = $("<td>", { class: "text-center" ,colspan:"3"});
            var span = $("<span>", { class: "pica-glyph pica-spinner glyphicons glyphicons-repeat"});
            td.append(span);
            tr.append(td);

            /* Send invitation for each tag */
            $.each(tags, function (idx, tag) {
                var oid = (tag.userdata ? tag.userdata.OID : null);
                var email = (tag.userdata ? tag.userdata.email : null);
                var flags = (tag.userdata && tag.userdata.isGroup ?
                    Picasso.Invitation.Flags.FOR_GROUP : 0);

                if ($("#share_table tbody .pica-spinner").is(":hidden")) {
                    $("#share_table tbody .pica-spinner").remove();
                }

                if ($("#share_table tbody .pica-spinner").length === 0 ) {
                    $("#share_table tbody").prepend(tr);
                }

                /* Handle user and groups differently */
                Picasso.Invitation.create(file, oid,email, tag.name, permission, flags,

                    /* Success */
                    function (invitation) {
                        if (invitation.is(Picasso.Invitation.Flags.PENDING)) {
                            /* Add new invitation to list */
                            Picasso.Dialog.Share.table.append(invitation,
                                renderInvitation, false, true);

                            Picasso.Dialog.Share.table.sort(1, true);

                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_invitation_success",
                                    tag.name), "info");
                        } else {
                            /* Create new local object  */
                            var holder;
                            var isAccount = false;

                            /* Check userdata from completion */
                            if (tag.userdata && tag.userdata.isGroup) {
                                holder = new Picasso.Group();

                                holder.setOid(tag.userdata.OID);
                                holder.setName(tag.name);

                                /* Update file */
                                file.setGroupsCount(file.getGroupsCount() + 1);
                            } else {
                                isAccount = true;

                                holder = new Picasso.Account();

                                if (tag.userdata) {
                                    holder.setOid(tag.userdata.OID);
                                } else {
                                    holder.setOid(invitation.getRecipientOid());
                                }

                                holder.setUserName(tag.name);
                                holder.setDisplayName(tag.name);

                                /* Update file */
                                file.setAccountsCount(file.getAccountsCount() + 1);
                            }

                            var withPermission = {
                                holder: holder,
                                permission: permission
                            };

                            /* Add new member to list */
                            Picasso.Dialog.Share.table.append(withPermission,
                                renderWithPermission, false, true);

                            Picasso.Dialog.Share.table.sort(1, false);



                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_item_created",
                                    Picasso.Lang.get("label_permissions",
                                        { plural: false })), "info");
                        }
                        $("#share_table tbody .pica-spinner").hide();
                    },

                    /* Error */
                    function (e, status) {
                        $("#share_table tbody .pica-spinner").closest("tr").remove();
                        switch (status) {
                            case 400: /* Cannot invite e.g not found/no e-mail*/
                                /* Falls through */
                            case 404: /* User not found */
                                Picasso.Notification.show(Picasso.Lang.get(
                                    "notification_invitation_no_account_error", "danger"));
                                break;

                            case 409: /* Already permission on folder */
                                Picasso.Notification.show(Picasso.Lang.get(
                                    "notification_invitation_already_exists",
                                    "danger"));
                                break;

                            case 412:
                                /* Falls through */
                            case 413:
                                Picasso.Dialog.handleOrgErrors(status);
                                break;

                            default:
                                Picasso.debugLog(e);
                        }
                    }
                );
            });

            return true;
        }

        return false;
    };

    /**
     * Handle click on create link button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCreateLink = function (e) {
        e.preventDefault();
        e.stopPropagation();
        var self = this;

        /* Get file */
        var file = Picasso.get("selectedFile");

        if (file) {

            var dialog = new Picasso.Dialog.Link();
            dialog.create(file,

                /* Success */
                function (link) {
                    var nrows = Picasso.Dialog.Share.table.append(
                        link, renderLink);

                    if (0 < nrows) {
                        $("#share_create_link").hide();
                    }

                    Picasso.Dialog.Share.table.sort(1, false);
                },

                /* Error */
                function (e) {
                    Picasso.debugLog(e);
                }
            );
        }
    };

    /**
     * Handle click on create form button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCreateUploadForm = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get file */
        var file = Picasso.get("selectedFile");

        if (file && file.is(Picasso.File.Flags.FOLDER)) {
            /* Show dialog */
            var dialog = new Picasso.Dialog.Uploadform();

            dialog.create(file,

                /* Success */
                function (uploadform) {
                    var nrows = Picasso.Dialog.Share.table.append(uploadform,
                        renderUploadForm);

                    if (0 < nrows) {
                        $("#share_create_form").hide();
                    }

                    Picasso.Dialog.Share.table.sort(1, false);

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_created",
                        Picasso.Lang.get("label_upload_form")), "info");
                },

                /* Error */
                function (e) {
                    Picasso.debugLog(e);
                }
            );
        }
    };


    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDelete = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = Picasso.Dialog.Share.table.getSelectedRows();

        /* Show dialog */
        var what = (1 === selected.length ? $(selected[0]).find(".pica-name").text() :
            Picasso.Lang.get("string_selected_elements", selected.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_remove"),
            Picasso.Lang.get("dialog_body_remove", what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {
                /* Either trigger button or last dropdown entry */
                var menu = $(this).find(".dropdown-menu");

                if (0 === menu.length) {
                    $(this).find("button").trigger("click", [ false ]);
                } else {
                    menu.find("li:last-child a").trigger("click", [ false ]);
                }
            });
        });

        dialog.show();
    };

    /**
     * Handle click on copy button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCopyToClip = function (e) {
        var that = $(this);

        Picasso.Helper.copyToClipboard.apply(that,
            that.parent().siblings("input"));
    };

    /**
     * Helper to check visibility of elements based on permissions
     *
     * @param {Picasso.Permission}  permission  A {@link Picasso.Permission} to check
     **/

    var checkPermissions = function (permission) {

            /* Check account permissions on file */
            switch (permission) {
                case Picasso.Permission.ADMIN:
                    $(".share-min-admin").show();
                    /* Falls through */

                case Picasso.Permission.OWNER:
                    $(".share-min-owner").show();
                    break;

                default:
                    $(".share-min-admin").hide();
                    $(".share-min-owner").hide();
            }
    };

    /**
     * Constructor for {@link Picasso.Dialog.Share}
     *
     * @param {Picasso.File}  file  File for sharing
     * @constructor
     **/

    Picasso.Dialog.Share = function (file) {
        this._file = file;

        var hideInviteOption = false;
        var hideCreateButtons = false;

        if (this._file.is(Picasso.File.Flags.FILE))  {
            hideInviteOption = true;
        }else if (this._file.is(Picasso.File.Flags.DIRECTORY) && !JSON.parse(Picasso.get("shareSubfolder"))) {
            hideInviteOption = true;
        }
        if (file.getPermission() == Picasso.Permission.READ || file.getPermission() == Picasso.Permission.READ_WRITE){
            hideCreateButtons = true;
        }
        /* Create dialog */
        this._dialog = new Picasso.Dialog("#pica_share_dialog",
            hideInviteOption ? Picasso.Lang.get("dialog_title_share_without_invite", file.getName()) :
                Picasso.Lang.get("dialog_title_share", file.getName()),
            hideInviteOption ? "" : Picasso.Lang.get("dialog_body_share"));

        if (hideInviteOption) {
            document.getElementById("share_taginput").style.visibility = "hidden";
            $("#share_taginput").parent().find(".pica-dialog-description").empty();
            $("#share_create_form").hide();

        } else {
            $("#share_taginput").show();
            $("#share_create_form").show();
            document.getElementById("share_taginput").style.visibility = "visible";

        }

        $("#share_create_link").show();

       if (hideCreateButtons){
            $("#share_create_link").hide();
            $("#share_create_form").hide();
        }

        /* Add click handler to permission dropdown */
        var a = $("#share_buttons .dropdown-menu a");
        var toggle = $("#share_buttons button.dropdown-toggle");
        var i = 0;

        $.each(a, function () {
            var that = $(this);

            that.off("click").click(function () {
                var that = $(this);
                var lang = that.find("lang");

                /* Copy selected lang entry and permission index */
                toggle.find("lang").replaceWith(lang.clone());
                toggle.data("permission", that.data("permission"));

                /* Update checkbox icons */
                that.parents(".dropdown-menu").find("span")
                    .removeClass("glyphicons-check").addClass("glyphicons-unchecked");
                that.find("span").removeClass("glyphicons-unchecked")
                    .addClass("glyphicons-check");
            });

            that.data("permission", permissionList[i++]);
        });

        $(a.get(1)).trigger("click"); ///< Reset by emitting click on first permission

        this._dialog.setAssignHandler(function () {
            checkPermissions(file.getPermission());

            /* Check public permission */
            if (Picasso.Permission.OWNER === file.getPermission()) {
                if (!file.hasFlag(Picasso.File.Flags.FOLDER)) {
                    return;
                }

                /* Check public permissions */
                if (Picasso.Permission.NO_ACCESS !==
                    file.getPublicPermission())
                {
                    Picasso.Dialog.Share.table.append(file,
                        renderPublicAccess, true);


                    Picasso.Dialog.Share.table.sort(1, false);
                }
            }
        });
    };

    /**
     * Set ok handler
     *
     * @param {Function}  onOk  Set Ok handler
     **/

    Picasso.Dialog.Share.prototype.setOkHandler = function (onOk) {
        if (onOk) {
            this._dialog.setOkHandler(onOk);
        }
    };

    /**
     * Show share dialog
     **/

    Picasso.Dialog.Share.prototype.show = function () {
        var self = this;

        checkPermissions(this._file.getPermission());

        var cwd = Picasso.get("cwd");

        if (!cwd) {/* Fetch accounts data */
            this._file.getAccountsWithPermission(
                /* Success */
                function (accounts) {
                    Picasso.Dialog.Share.table.append(
                        accounts, renderWithPermission);

                    Picasso.Dialog.Share.table.sort(1, false);

                    checkPermissions(self._file.getPermission());
                },

                /* Error */
                function (e) {
                    Picasso.debugLog(e);
                }
            );


            /* Fetch groups data */
            this._file.getGroupsWithPermission(
                /* Success */
                function (groups) {
                    Picasso.Dialog.Share.table.append(
                        groups, renderWithPermission);

                    Picasso.Dialog.Share.table.sort(1, false);
                },

                /* Error */
                function (e) {
                    Picasso.debugLog(e);
                }
            );
        }

        /* Check account permissions */
        if (this._file.is(Picasso.File.Flags.FOLDER) && (
            Picasso.Permission.ADMIN === this._file.getPermission() ||
            Picasso.Permission.OWNER === this._file.getPermission())) {
            /* Fetch upload form data */
            Picasso.Uploadform.find(this._file,

                /* Success */
                function (uploadForm) {
                    Picasso.Dialog.Share.table.append(
                        uploadForm, renderUploadForm);

                    $("#share_create_form").hide();

                    Picasso.Dialog.Share.table.sort(1, false);

                    checkPermissions(self._file.getPermission());
                },

                /* Error */
                function (e) {
                    Picasso.debugLog(e);
                }
            );

            /* Fetch invites */
            Picasso.Invitation.find(this._file,

                /* Success */
                function (invitations) {
                    Picasso.Dialog.Share.table.append(
                        invitations, renderInvitation);

                    Picasso.Dialog.Share.table.sort(1, false);

                    checkPermissions(self._file.getPermission());
                },

                /* Error */
                function (e) {
                    Picasso.debugLog(e);
                }
            );
        }


        /* Get link for all types */
        Picasso.Link.find(this._file,

            /* Success */
            function (link) {

                if (link.getFile().is(Picasso.File.Flags.FILE) && link.is(Picasso.Link.Flags.UPLOADS_ALLOWED) && !link.getFile().isOODocument()) {
                    link.setFlags(!Picasso.Link.Flags.UPLOADS_ALLOWED)
                    link.setPermission(2)
                }
                Picasso.Dialog.Share.table.append(
                    link, renderLink, true);

                $("#share_create_link").hide();

                Picasso.Dialog.Share.table.sort(1, false);

                checkPermissions(self._file.getPermission());
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );

        /* Clear table and taginput */
        Picasso.Dialog.Share.table.clear();

        Picasso.Dialog.Share.tagInput.clear();
        Picasso.Dialog.Share.tagInput.clearCompletions();

        /* Finally show dialog */
        this._dialog.show();
    };

    /* Global - must be defined after constructor */

    Picasso.Dialog.Share.table = new Picasso.Table("#share_table",
        Picasso.Table.Flags.ROWS_SELECTABLE);

    Picasso.Dialog.Share.tagInput = new Picasso.TagInput("#share_taginput",
        Picasso.TagInput.Flags.ADD_TAGS);

    Picasso.Dialog.Share.tagInput.setValidationErrorHandler(
        function () {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_invalid_email"),
                    "warning");
            }
     );

    Picasso.Dialog.Share.tagInput.setCompletionHandler(
        Picasso.TagInput.handleAccountAndGroupCompletion);
    Picasso.Dialog.Share.tagInput.setSubmitHandler(handleSubmit);

    /* Bind handlers */
    $("#share_remove").click(handleClickDelete);
    $("#share_create_link").click(handleClickCreateLink);
    $("#share_create_form").click(handleClickCreateUploadForm);

    /* Fix dropdowns in scroller */
    Picasso.Combo.fixDropdownInScroller(".pica-inputlist-scroller");
})();
