/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";


if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Helper")) {
    window.Picasso.Helper = {};
}

/* Private scope */
(function () {

    /* Globals */
    Picasso.Helper.RE_UPPERCASE = new RegExp("[A-Z]+");
    Picasso.Helper.RE_LOWERCASE = new RegExp("[a-z]+");
    Picasso.Helper.RE_DIGIT = new RegExp("\\d+");
    Picasso.Helper.RE_SPECIAL = new RegExp("[\\W_]+");
    Picasso.Helper.RE_HTML_ENTITIES = new RegExp(/[\u00A0-\u9999<>\&]/gim);

    var INTIAL_PAGES_TO_LOAD = 10;
    var PAGE_SIZE = 50;
    var EMAIL_REGEX = /([a-zA-Z0-9!#$%&'*+\/=?^_`{|}~-]+(?:\.[a-zA-Z0-9!#$%&'*+\/=?^_`{|}~-]+)*@(?:[a-zA-Z0-9](?:[a-zA-Z0-9-]*[a-zA-Z0-9])?\.)+[a-zA-Z0-9](?:[a-zA-Z0-9-]*[a-zA-Z0-9])?)/g;

    const editors = {};

    /**
     * Add a zero as prefix if necessary
     *
     * @param {String}  str  String to prefix
     * @returns {String} Prefixed string
     **/

    var prefixZero = function (str) {
        return ("0" + str).slice(-2);
    };

    /**
     * Create popover and update labels of a visible popover
     *
     * @param {String}   label      Label lang key
     * @param {Boolean}  fulfilled  Whether condition for this label is fulfilled
     *
     * @returns {jQuery|HTMLElement}  Constructed label
     **/

    var createAndUpdateLabel = function (label, fulfilled) {
        var h5 = $("<h5>");
        var klass = "profile_" + label;

        /* Clone visible element or create new */
        var lang = $("." + klass);

        if (0 === lang.length) {
            lang = Picasso.Helper.createLang(label);

            lang.addClass(klass);
            lang.addClass("pica-space-right-small label label-danger");

            h5.append(lang);

            lang.add(lang);
        } else {
            /* Just clone visible element */
            var clone = lang.clone();

            h5.append(clone);

            lang.add(clone);
        }

        /* Update state */
        if (fulfilled) {
            lang.removeClass("label-danger");
            lang.addClass("label-success");
        } else {
            lang.removeClass("label-success");
            lang.addClass("label-danger");
        }

        return h5;
    };

    /**
     * Helper to create a hopefully unique id from target
     *
     * @param {Object}  target  Target object
     *
     * @returns {String} String id
     **/

    var getUniqueId = function (target) {
        return Object.prototype.toString.call(target)
            .slice(8, -1) + "_" + Picasso.Helper.hashCode(
            target._oid || target._name || target._heading || target._flags);
    };

    /**
     * Create proxy shim
     *
     * @param {Object}  obj  Object for this proxy
     *
     * @returns {Object} Updated object
     **/

    var createProxyShim = function (obj) {
        /* Create is/hasFlag */
        obj.__proto__.is = function (flag) {
            return !!(this._flags & flag);
        };

        obj.__proto__.hasFlag = obj.__proto__.is;

        /* Create setFlag */
        obj.__proto__.setFlag = function (flag) {
            this._flags |= flag;
        };

        /* Create unsetFlag */
        obj.__proto__.unsetFlag = function (flag) {
            this._flags &= ~flag;
        };

        /* Create serialize */
        obj.__proto__.serialize = function () {
            return JSON.stringify(this);
        };

        /* Create getElemId */
        obj.__proto__.getElemId = function () {
            return getUniqueId(this);
        };

        /* Iterate over every property */
        $.each(Object.keys(obj), function (idx, key) {
            /* Pick properties prefixed with an underscore */
            if ("_" === key[0]) {
                var name = Picasso.Helper.capitalize(key.slice(1));

                /* Create setter */
                obj.__proto__["set" + name] = function (value) {
                    this[key] = value;
                };

                /* Create getter */
                obj.__proto__["get" + name] = function () {
                    return this[key];
                };
            }
        });

        return obj;
    };

    /**
     * Create proxy when supported
     *
     * @param {Object}  obj  Object for this proxy
     *
     * @returns {Proxy} Created {@link Proxy}
     **/

    var createProxy = function (obj) {
        return (new Proxy(obj, {
            get: function (target, property, receiver) {
                if ("string" === typeof property) {
                    /* Flag based functions */
                    if (target.hasOwnProperty("_flags")) {
                        /* 1) Check whether is() has been called */
                        if ("is" === property || "hasFlag" === property) {
                            /* Return a function to checks for flag */
                            return function (flag) {
                                return !!(target._flags & flag);
                            };
                            /* 2) Check whether setFlag() has been called */
                        } else if ("setFlag" === property) {
                            /* Return a function to set flags */
                            return function (flag) {
                                target._flags |= flag;
                            }
                            /* 3) Check whether unsetFlag() has been called */
                        } else if ("unsetFlag" === property) {
                            /* Return a function to unset flags */
                            return function (flag) {
                                target._flags &= ~flag;
                            }
                        }
                    }

                    /* 4) Check whether serialize() has been called */
                    if ("serialize" === property) {
                        /* Return a function to serialize this */
                        return function () {
                            return JSON.stringify(target);
                        };
                    /* 5) Check whether getElemId() has been called */
                    } else if ("getElemId" === property) {
                        /* Return a function to get the elem ID */
                        return function () {
                            return getUniqueId(target);
                        };
                    /* 6) Check whether this property is no defined function */
                    } else if ("function" !== typeof target[property]) {
                        var propName = "_" + property[3].toLowerCase() +
                            property.substr(4);

                        /* 6a) Check whether we have this property and
                               provide transparent getter */
                        if (-1 !== property.indexOf("get") &&
                            target.hasOwnProperty(propName)) {
                            /* Return a function to return the value */
                            return function () {
                                return target[propName];
                            };
                            /* 6b) Or just set it */
                        } else if (-1 !== property.indexOf("set")) {
                            /* Return a function to set the value */
                            return function (val) {
                                target[propName] = val;
                            };
                        }
                    }
                }

                /* 3) Check whether this is a prototype method -
                 *    since we catch all here.. */
                return Reflect.get(target, property, receiver);
            }
        }));
    };

    /**
     * Check whether user agent is inter explorer
     *
     * @returns {Boolean} Either {@code true} when IE; otherwise {@code false}
     **/

    Picasso.Helper.isInternetExplorer = function () {
        var agent = window.navigator.userAgent;
        var idx = agent.indexOf("MSIE");
        var version = -1;

        /* Return version number if IE */
        if (0 < idx) {
            version = parseInt(agent.substring(idx + 5, agent.indexOf(".", idx)));
        }
        /* Check user agent for IE11 */
        else if (!!navigator.userAgent.match(/Trident\/7\./)) {
            version = 11;
        }

        return (-1 !== version);
    };

    /**
     * Helper to create proxy for Picasso objects with setter/getter traps
     *
     * @param {Object}  obj  Object for this proxy
     *
     * @returns {Proxy|Object} Created object proxy
     **/

    Picasso.Helper.createProxy = function (obj) {
        if (!window.hasOwnProperty("Proxy")) {
            return createProxyShim(obj);
        } else {
            return createProxy(obj);
        }
    };

    /**
     * Helper to create a progress bar
     *
     * @param {String}  id       ID of the bar
     * @param {String}  caption  Caption of the progress bar (optional)
     *
     * @returns {jQuery|HTMLElement} Assembled progress bar
     **/

    Picasso.Helper.createProgressbar = function (id, caption) {
        /* Create html */
        var progress = $("<div>", {
            id: id,
            class: "pica-progress progress text-center",
            "data-caption": caption
        });

        var bar = $("<div>", {
            class: "progress-bar progress-bar-striped",
            role: "progressbar",
            "aria-valuemin": 0,
            "aria-valuemax": 100
        });

        var span = $("<span>", {
            class: "pica-progress-caption"
        });

        if (caption) {
            span.text(caption);
        }

        progress.append(bar);
        progress.append(span);

        return progress;
    };

    Picasso.Helper.updateStorageBar = function (account, selector) {

        var progress = $(selector);


        progress.show();

        var label = progress.find("span");
        var progressUsed = progress.find(".progress-bar-info");
        var progressVersions = progress.find(".progress-bar-warning");

        var percentUsed = 100;
        var percentVersions = 0;

        var quotaTotalGB = account.getQuotaTotal() / (1024 * 1024 * 1024);
        if (quotaTotalGB === 0) {
            progress.empty();


            var noStorageText = $("<span>")
                .text(Picasso.Lang.get("string_no_storage"));

            progress.addClass("no-storage-available").append(noStorageText).show();
            return;
        }

        // Ultimate max storage case
        if (quotaTotalGB === 9999) {
            progressUsed.css("width", "100%").removeClass("progress-bar-info progress-bar-danger")
                .addClass("progress-bar-success");
            label.text(Picasso.Lang.get("label_max_ultimate"));

            progress.removeAttr("title data-toggle data-container data-placement data-trigger data-html data-content");

            return;
        }

        // Calculate used and version storage percentages
        if (account.getQuotaTotal() > 0) {
            percentVersions = Math.floor(account.getQuotaversions() * 100 / account.getQuotaTotal());
            if (percentVersions > 100) {
                percentVersions = 100;
            }
            percentUsed = Math.floor(account.getQuotaUsed() * 100 / account.getQuotaTotal());
            if (percentUsed > 100) {
                percentVersions = 100 * percentVersions / percentUsed;
                percentUsed = 100;
            }
        }

        progressUsed.css("width", (percentUsed - percentVersions) + "%");
        progressVersions.css("width", percentVersions + "%");

        // Text for label

        label.text(Picasso.Lang.get("string_storage_used", percentUsed));


        // Handle FULL case (100%)
        if (percentUsed >= 100) {
            progressUsed.removeClass("progress-bar-info progress-bar-warning")
                .addClass("progress-bar-danger");
            label.text(Picasso.Lang.get("label_full")).css("color", "white");
        }
        // Handle almost full (90%+)
        else if (percentUsed >= 90) {
            progressUsed.removeClass("progress-bar-info progress-bar-danger")
                .addClass("progress-bar-warning");
            label.text(Picasso.Lang.get("label_almost_full", percentUsed)).css("color", "black");
        }
        // Handle normal case (below 90%)
        else {
            label.css("color", "black");
            progressUsed.removeClass("progress-bar-danger")
                .addClass("progress-bar-info");
        }

        // Create dynamic popover content
        var div = $("<div>");
        var h5 = $("<h5>");
        var lang = Picasso.Helper.createLang("label_storage");

        lang.addClass("pica-space-right-small label");

        // Apply label styles based on usage percentage
        if (percentUsed >= 90) {
            lang.addClass("label-warning");
        } else if (percentUsed > 80) {
            lang.addClass("label-danger");
        } else {
            lang.addClass("label-info");
        }

        h5.append(lang);

        var span = $("<span>", {class: "pica-slash-after"});
        span.append(Picasso.Helper.formatSize(account.getQuotaUsed()));
        h5.append(span);
        h5.append(Picasso.Helper.formatSize(account.getQuotaTotal()));

        div.append(h5);

        // Add versions information if available
        if (account.getQuotaversions() > 0) {
            h5 = $("<h5>");
            lang = Picasso.Helper.createLang(Picasso.Lang.Flags.PLURALIZE, "label_versions");
            lang.addClass("pica-space-right-small label label-warning");

            h5.append(lang);
            h5.append(Picasso.Helper.formatSize(account.getQuotaversions()));
            div.append(h5);
        }

        // Initialize popover with generated content
        progress.attr({
            "title": Picasso.Lang.get("label_explanation"),
            "data-toggle": "popover",
            "data-container": "body",
            "data-placement": "left",
            "data-trigger": "hover",
            "data-html": "true",
            "data-content": div.html()
        }).popover();
    };


    /**
     * Create quota popover
     *
     * @param {jQuery|HTMLElement}  elem
     * @param {Number}              quotaUsed
     * @param {Number}              quotaTotal
     * @param {Number}              quotaAddedAfterSave
     * @param {Number}              quotaFreedAfterSave
     * @param {Number}              percentUsed           Used total quota in percent
     * @param {Number}              percentAddedAfterSave  Used quota after save in percent
     * @param {Number}              percentFreeAfterSave  Freed quota after save in percent
     * @param {String}              caption               Progressbar caption
     **/

    Picasso.Helper.createQuotaPopover = function (elem, quotaUsed, quotaTotal,
                                                  quotaAddedAfterSave,
                                                  quotaFreedAfterSave,
                                                  percentUsed,
                                                  percentAddedAfterSave,
                                                  percentFreeAfterSave,
                                                  caption)
    {
        var progress = $(elem);

        var label = progress.find("span");
        var progressUsed = progress.find(".progress-bar-info");
        var progressUsedAfterSave = progress.find(".progress-bar-warning");
        var progressFreeAfterSave = progress.find(".progress-bar-success");

        /* Sanitize values */
        if (100 < percentUsed) percentUsed = 100;
        if (100 < percentAddedAfterSave) percentAddedAfterSave = (100 - percentUsed);
        if (100 < percentFreeAfterSave) percentFreeAfterSave = 100;

        /* Finally update progress bars */
        progressUsed.css("width", percentUsed + "%");
        progressUsedAfterSave.css("width", percentAddedAfterSave + "%");
        progressFreeAfterSave.css("width", percentFreeAfterSave + "%");

        label.text(caption);

        /* Create popover */
        var div = $("<div>");
        var h5 = $("<h5>");

        /* Add used label */
        var lang = Picasso.Helper.createLang("label_storage");

        lang.addClass("pica-space-right-small label label-info");

        h5.append(lang);

        var span = $("<span>", { class: "pica-slash-after" });

        span.append(Picasso.Helper.formatSize(quotaUsed));
        h5.append(span);
        h5.append(Picasso.Helper.formatSize(quotaTotal));

        div.append(h5);

        /* Add used after save */
        h5 = $("<h5>");
        lang = Picasso.Helper.createLang("label_used_after_save");

        lang.addClass("pica-space-right-small label label-warning");

        h5.append(lang);
        h5.append(Picasso.Helper.formatSize(quotaUsed + quotaAddedAfterSave));

        div.append(h5);

        /* Add free after save */
        h5 = $("<h5>");
        lang = Picasso.Helper.createLang("label_freed_after_save");

        lang.addClass("pica-space-right-small label label-success");

        h5.append(lang);
        h5.append(Picasso.Helper.formatSize(quotaFreedAfterSave));

        div.append(h5);

        /* And finally add it */
        progress.attr({
            "data-toggle": "popover",
            "data-container": "body",
            "data-placement": "bottom",
            "data-trigger": "hover",
            "title": Picasso.Lang.get("label_explanation"),
            "data-html": "true",
            "data-content": div.html()
        }).popover();
    };
    /**
     * Helper to create a wysiwyg editor
     *
     * @param {String}   id   id of text area
     *
     * @returns {HTMLElement} Assembled editor box
     **/
    Picasso.Helper.createEditor = function (id,height) {
        if (!height){
            height = '150px';
        }

        // If editor already exists, preserve the textarea content before destroying
        var preservedContent = null;
        if (Picasso.Helper.isEditor(id)) {
            var element = document.getElementById(id);
            if (element && element.tagName === 'TEXTAREA') {
                preservedContent = element.value;
            }
            Picasso.Helper.destroyEditor(id);
            if (preservedContent !== null && element) {
                element.value = preservedContent;
            }
        }

        return ClassicEditor
            .create( document.getElementById(id) )
            .then( editor => {
                editors[ id ] = editor;
                const editorElement = editor.ui.view.editable.element;
                editorElement.style.height = height;
            } )
            .catch( error => {
                console.error( error );
            } );
    }

    Picasso.Helper.isEditor = function (id) {
        return id in editors ;
    }

    Picasso.Helper.getEditorData = function (id){
        if (Picasso.Helper.isEditor(id)){
            return editors[id].getData()
        }
        return "";
    }

    Picasso.Helper.destroyEditor = function (id){
        if (Picasso.Helper.isEditor(id)) {
            editors[id].destroy();
            delete editors[id];
            var element = document.getElementById(id);
            if (element) {
                element.innerHTML = '';
                if (element.tagName === 'TEXTAREA') {
                    element.value = '';
                }
            }
        }
    }

    /**
     * Helper to create glyph icon
     *
     * @param {String}   iconName   Name of the icon
     * @param {String}   klass      Additional classes
     * @param {Boolean}  isVisible  Whether icon is visible
     *
     * @returns {HTMLElement} Assembled glyph box
     **/

    Picasso.Helper.createGlyph = function (iconName, klass, isVisible) {

        /* Create html */
        var glyph = $("<span>");

        /* Filetypes are special glyph icons */
        if (iconName && -1 === iconName.indexOf("glyph") &&
            -1 === iconName.indexOf("filetypes")) {
            glyph.addClass("glyphicons glyphicons-" + iconName);
        } else {
            glyph.addClass(iconName);
        }

        if (klass) glyph.addClass(klass);
        if (false === isVisible) glyph.hide();

        return glyph;
    };

    /**
     * Helper to create glyph icon for given file name based on extension
     *
     * @param {String}  fileName  Name of the file
     *
     * @returns {HTMLElement} Assembled glyph box
     **/

    Picasso.Helper.createGlyphForExtension = function (fileName) {
        /* Fall back to file extension */
        var ext = fileName.toLowerCase().substr(fileName.lastIndexOf(".") + 1);

        var glyph = Picasso.Helper.createGlyph(
            "filetypes filetypes-" + ext, "pica-glyph");

        /* Check if glyph was found: Detection here is rather tricky, because
         * we cannot access :after elements, which aren't part of the DOM */
        $("body").append(glyph);

        if (0 === parseInt(glyph.css("width"))) {
            glyph.addClass("filetypes-unknown");
        }

        return glyph;
    };

    /**
     * Helper to create avatar and icon
     *
     * @param {jQuery|HTMLElement}  elem  Element ID
     * @param {String}              icon  Icon name
     * @param {*}                   obj   Object to fetch oid from
     **/

    Picasso.Helper.createGlyphAndAvatar = function (elem, icon, obj) {
        var icon = Picasso.Helper.createGlyph(icon, "pica-glyph");

        $(elem).append(icon);

        /* Add thumbnail if found */
        var img = $("<img>", {
            class: "pica-icon",

            onLoad: function () {
                icon.hide();
                icon.after(this);
            },

            src: Picasso.Helper.getAvatarUrl(obj)
        });

        img.on("error", function (e) {
            icon.show();
            $(this).remove();
        });
    };

    /**
     * Helper to create lang tags
     *
     * @param {String}  name     Lang name
     * @param {Hash}    options  Options (optional)
     *
     * @returns {jQuery|HTMLElement}
     **/

    Picasso.Helper.createLang = function () {
        return $("<lang>", {
            name: (isNaN(arguments[0]) ? arguments[0] : arguments[1]),
            text: Picasso.Lang.get.apply(null, arguments) ///< Forward arguments
        });
    };

    /**
     * Helper to create button checkbox
     **/

    Picasso.Helper.createButtonCheckbox = function () {
        var that = $(this);
        var button = that.find("button");
        var checkbox = that.find("input:checkbox");
        var color = button.data("color");

        button.prepend('<i class="state-icon"></i>');

        var updateDisplay = function () {
            var isChecked = checkbox.prop("checked");

            /* Update icon and color */
            var icon = button.find(".state-icon")

            icon.removeClass();
            icon.addClass("state-icon glyphicons glyphicons-" +
                (isChecked ? "check" : "unchecked"));

            if (isChecked) {
                button.removeClass("btn-default");
                button.addClass("btn-" + color + " active");
            } else {
                button.removeClass("btn-" + color + " active");
                button.addClass("btn-default");
            }
        };

        /* Event handlers */
        button.click(function () {
            if ($(this).hasClass("disabled")) return;

            checkbox.prop("checked", !checkbox.prop("checked"));

            updateDisplay();
        });

        checkbox.on("change", updateDisplay);

        updateDisplay();
    };

    /**
     * Helper to create a dateTimepicker
     *
     * @param {String}  id  Element ID
     **/

    Picasso.Helper.createDateTimepicker = function (id) {
        var dateTimePicker = $(id);
        dateTimePicker.datetimepicker({
            showClear: true,
            showClose: true,
            locale: Picasso.get("language") ? Picasso.get("language") : 'en'
        });

    };

    /**
     * Helper to create two Linked  dateTimepicker from / until
     *
     **/

    Picasso.Helper.createFromUntilDateTimePicker = function (from,until) {
        var fromTimePicker = $(from);
        var untilTimePicker = $(until);
        fromTimePicker.datetimepicker({
            useCurrent: false,
            showClear: true,
            showClose: true,
            locale: Picasso.get("language") ? Picasso.get("language") : 'en'
        });

        untilTimePicker.datetimepicker({
            useCurrent: false,
            showClear: true,
            showClose: true,
            locale: Picasso.get("language") ? Picasso.get("language") : 'en'
        });

        fromTimePicker.on("dp.change", function (e) {
            untilTimePicker.data("DateTimePicker").minDate(e.date);
        });
        untilTimePicker.on("dp.change", function (e) {
            fromTimePicker.data("DateTimePicker").maxDate(e.date);
        });

    };
    /**
     * Helper to disable a datetimePicker
     *
     * @param {String}  id  Element ID
     **/

    Picasso.Helper.disableDateTimePicker = function (id) {
        /* Create picker */
        var picker = $(id);

        picker.datetimepicker().data("DateTimePicker").disable();
    };

    /**
     * Create chart
     *
     * @param {HTMLElement}  elem        Canvas element
     * @param {Array}        colors      Array of chart colors
     * @param {Function}     formatter   Formatter for the dataset (optional)
     *
     * @returns {HTMLElement|null} Either created chart or null when Chart cannot be found
     **/

    Picasso.Helper.createChart = function (elem, colors, formatter) {
        var canvas = $(elem);

        return new Chart(canvas, {
            type: "doughnut",
            data: {
                datasets: [
                    {
                        backgroundColor: colors,
                        formatter: formatter
                    }
                ]
            },
            options: {
                responsive: false,
                legend: {
                    position: "right",
                    labels: {
                        usePointStyles: true,
                        generateLabels: function (chart) {
                            var data = chart.data;
                            if (data.labels.length && data.datasets.length) {
                                return data.labels.map(function (label, i) {
                                    var meta = chart.getDatasetMeta(0);
                                    var ds = data.datasets[0];
                                    var arc = meta.data[i];
                                    var custom = arc && arc.custom || {};
                                    var getValueAtIndexOrDefault = Chart.helpers.getValueAtIndexOrDefault;
                                    var arcOpts = chart.options.elements.arc;
                                    var fill = custom.backgroundColor ?
                                        custom.backgroundColor : getValueAtIndexOrDefault(ds.backgroundColor, i, arcOpts.backgroundColor);
                                    var stroke = custom.borderColor ?
                                        custom.borderColor : getValueAtIndexOrDefault(ds.borderColor, i, arcOpts.borderColor);
                                    var bw = custom.borderWidth ?
                                        custom.borderWidth : getValueAtIndexOrDefault(ds.borderWidth, i, arcOpts.borderWidth);

                                    // We get the value of the current label
                                    var value = chart.config.data.datasets[
                                        arc._datasetIndex].data[arc._index];

                                    return {
                                        text: label + ": " +
                                            (formatter ? formatter(value) : value),
                                        fillStyle: fill,
                                        strokeStyle: stroke,
                                        lineWidth: bw,
                                        hidden: isNaN(ds.data[i]) || meta.data[i].hidden,
                                        index: i
                                    };
                                });
                            } else {
                                return [];
                            }
                        }
                    }
                },
                tooltips: {
                    callbacks: {
                        label: function (tooltipItem, data) {
                            var dataset = data.datasets[tooltipItem.datasetIndex];
                            var index = tooltipItem.index;

                            return " " + data.labels[index] + ": " +
                                (dataset.formatter ? dataset.formatter(
                                    dataset.data[index]) : dataset.data[index]);
                        }
                    }
                }
            }
        }) || null;
    };

    /**
     * Helper to add tooltips to an element
     *
     * @param {Object}  elem     Element to add tooltip
     * @param {String}  tooltip  Tooltip text
     **/

    Picasso.Helper.addTooltip = function (elem, tooltip) {
        var that = $(elem);

        that.attr("data-toggle","tooltip");
        that.attr("data-placement", "auto");

        that.tooltip({
            title: tooltip,
            container: "body",
            placement: "auto",
        });
    };

    /**
     * Fetch color values from CSS for charts
     *
     * @returns {Array} Array with color values
     **/

    Picasso.Helper.getChartColors = function () {
        var colors = [];

        /* We basically create a new div, assign a CSS class, fetch the color
           of the rendered element and remove it afterwards */
        var div = $("<div>", {style: "display: none"});

        $("body").append(div);

        for (var i = 1; i <= 3; i++) {
            div.attr("class", "pica-chart-color" + i);

            colors.push(div.css("color"));
        }

        div.remove();

        return colors;
    };

    /**
     * Get font color for given chart data
     *
     * @param {Object}  data  Chart data
     * @returns {Float|String} Either calculated value or either black or white
     **/

    Picasso.Helper.getChartFontColor = function (data) {
        var rgb = data.dataset.backgroundColor[data.index].match(
            /^rgba?[\s+]?\([\s+]?(\d+)[\s+]?,[\s+]?(\d+)[\s+]?,[\s+]?(\d+)[\s+]?/i);

        var threshold = 140;
        var luminance = 0.299 * rgb[1] + 0.587 * rgb[2] + 0.114 * rgb[3];

        return (luminance > threshold ? "black" : "white");
    };

    /**
     * Update given chart with values
     *
     * @param {Chart}   chart   A #{@link Chart}
     * @param {Object}  values  Either array or object with chart labels and values
     **/

    Picasso.Helper.updateChart = function (chart, values) {
        /* Reset values */
        chart.data.labels = [];

        if (!Array.isArray(values)) {
            values = [values];
        }

        $.each(values, function (idx, obj) {
            chart.data.datasets[idx].data = []

            $.each(obj, function (k, v) {
                chart.data.labels.push(Picasso.Lang.get(k));
                chart.data.datasets[idx].data.push(v);
            });
        });

        chart.update();
    };

    /**
     * Validate input fields
     *
     * @param {String}  elemId  Element ID
     *
     * @returns {Boolean} Returns either {@code true} when all fields are valid;
     *                    otherwise {@code false}
     **/

    Picasso.Helper.validateInputs = function (elemId) {
        var required = $(elemId).find("[required]");
        var isValid = true;
        var hasEmptyFields = false;
        var invalidFields = [];

        $.each(required, function () {
            var that = $(this);
            /* Toggle class */
            let fieldIsValid = that.get(0).checkValidity();// default html validation doesn't support umlaut

            if (!fieldIsValid && Picasso.Helper.isEditor(that.get(0).id)){
                fieldIsValid = editors[that.get(0).id].getData() !== "";
            }

            if (!fieldIsValid) {
                that.closest(".form-group").addClass("has-error");

                if(0 === that.val().length)
                    hasEmptyFields = true;
                 else
                    invalidFields.push(that);

                isValid = false;
            } else {
                that.closest(".form-group").removeClass("has-error");
            }
        });

        if(hasEmptyFields) {
            isValid = false;
        }

        invalidFields.forEach(function (elem) {
            if (elem.attr("type") === "email" && elem.attr("should-validate")) {
                const value = elem.val();
                const unicodeEmailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/u;

                if (unicodeEmailRegex.test(value)) {
                    elem.closest(".form-group").removeClass("has-error");
                } else {
                    Picasso.Notification.show(Picasso.Lang.get("notification_error_invalid_email"));
                    isValid = false;
                }
            }
        });

        var patterned = $(elemId).find("[pattern]");
        $.each(patterned, function () {
            var that = $(this);
            var value = that.val();
            if (value.length > 0 && !that.get(0).checkValidity()) {
                that.closest(".form-group").addClass("has-error");
                var errorKey = that.attr("data-validation-message");
                if (errorKey) {
                    Picasso.Notification.show(Picasso.Lang.get(errorKey));
                }
                isValid = false;
            } else {
                that.closest(".form-group").removeClass("has-error");
            }
        });

        return isValid;
    };

    Picasso.Helper.validateInput = function(elm, errorKey) {
        var isValid = true;
        if(elm.val().length === 0 ) {
            Picasso.Notification.show(
                Picasso.Lang.get("notification_error_missing_fields"));
             elm.closest(".form-group").addClass("has-error");
            isValid =  false;
        } else if(!elm.get(0).checkValidity()) {
            Picasso.Notification.show(Picasso.Lang.get(errorKey));
            elm.closest(".form-group").addClass("has-error");
            isValid =  false;
        } else {
            elm.closest(".form-group").removeClass("has-error");
        }
        return isValid;
    };

    /**
     * Helper to validate email address
     */

    Picasso.Helper.isValidEmail = function (email){
        return email && email.match(EMAIL_REGEX);
    }

    /**
     * Debounce a given function
     *
     * @param {Function}   func       Function to debounce
     * @param {Number}     wait       Time to wait in ms
     * @param {Boolean}    immediate  Whether to call function immediately
     *
     * @returns {Function} A debounced function
     **/

    Picasso.Helper.debounce = function (func, wait, immediate) {
        var timeout;

        return function () {
            var that = this;
            var args = arguments;

            var later = function () {
                timeout = null;

                if (!immediate) {
                    func.apply(that, args);
                }
            };

            var callNow = (immediate && !timeout);

            clearTimeout(timeout);
            timeout = setTimeout(later, wait);

            if (callNow) {
                func.apply(that, args);
            }
        };
    };

    /**
     * Copy text of input field to clipboard
     *
     * @param {Object}  input  HTML element
     **/

    Picasso.Helper.copyToClipboard = function (input) {
        if (true === document.queryCommandSupported("copy")) {
            $(input).select();

            /* Try to copy text */
            try {
                if (document.execCommand("copy")) {
                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_copy_success"), "info");
                } else {
                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_copy_error"), "danger");
                }

            } catch (e) {
                Picasso.debugLog("Oops, unable to copy");
            }
        } else {
            /* Fallback if browser doesn't support .execCommand("copy") */
            Picasso.Notification.show(
                Picasso.Lang.get("notification_copy_unsupported"), "info");
        }
    };

    /**
     * Fade out element and call callback afterwards
     *
     * @param {Object}    elem      Element to fade out
     * @param {Function}  callback  Callback after element is gone
     **/

    Picasso.Helper.fadeOut = function (elem, callback) {
        $(elem).addClass("pica-highlight").delay(1000)
            .fadeOut().promise().done(callback);
    };

    /**
     * Toggle nav tabs
     *
     * Tabs can have extra fields eg for buttons next to the dialog buttons.
     * These extras are referenced via data-nav-tab-id.
     *
     * @param {Event}  e  Click event
     *
     * Special classes:
     *
     * .pica-nav-tab - Nav tabs to display
     * .pica-nav-tab-extra - Extra content to display
     *
     * Special attributes:
     *
     * data-nav-tab-id - Extra nav tab id
     * data-nav-tab-ids - List of ids this extra tab is for
     **/

    Picasso.Helper.toggleNav = function (e) {
        var li = $(this).parent();

        /* Check whether tab is disabled */
        if (li.hasClass("disabled")) return;

        /* Toggle active state */
        li.siblings().removeClass("active");
        li.addClass("active");

        /* Toggle tab */
        var tabs = li.parent().siblings(".pica-nav-tab");
        var curTab = null;

        if (0 < tabs.length) {
            tabs.hide();

            curTab = $(tabs.get(e.data.idx));
            curTab.show();

            /* Toggle tab extra */
            var extras = li.parents("div").find(".pica-nav-tab-extra");

            $.each(extras, function () {
                var extra = $(this);
                var tabIds = extra.attr("data-nav-tab-ids");
                var tabId = curTab.attr("data-nav-tab-id");

                /* Check whether id is included in id list */
                if (tabId && tabIds && -1 !== tabIds.indexOf(tabId)) {
                    extra.show();
                } else {
                    extra.hide();
                }
            });
        }
    };

    /**
     * Bind nav handler and populate overflow nav
     *
     * @param {jQuery|HTMLElement}  nav       Nav element to bind to
     * @param {Function}            callback  Callback handler for clicks (optional)
     **/

    Picasso.Helper.bindNav = function (nav, callback) {
        var menu = $(nav).find(".dropdown-menu");
        var nchildren = (menu.children().length || 0);

        $(nav).find("li:not(.disabled) a").each(function (idx) {
            var that = $(this);

            that.bind("click", {idx: idx}, Picasso.Helper.toggleNav);

            /* Add additional callback if any */
            if (callback) {
                that.click(callback);
            }

            /* Check for overflow element */
            if (0 < menu.length && 0 === nchildren) {
                var clone = that.clone();

                clone.click(function () {
                    that.trigger("click");
                });

                menu.append($("<li>", {html: clone}));
            }
        });
    };

    /**
     * Bind scroll handler for infinite scrolling
     *
     * @param {Objecŧ}         json           JSON data
     * @param {Picasso.Table}  table          A #{@link Picasso.Table}
     * @param {Number}         page           Page to show
     * @param {Function}       scrollHandler  Scroll handler
     **/

    Picasso.Helper.bindInfiniteScroll = function (json, table,
                                                  page, scrollHandler) {
        /* Check for pageable */
        if (null != json && json.hasOwnProperty("ResultSet") &&
            json.ResultSet.hasOwnProperty("Pageable"))
        {
            var pageable = json.ResultSet.Pageable;

            /* load 10 pages first */
            if (scrollHandler && page < INTIAL_PAGES_TO_LOAD  && pageable.curPage < pageable.maxPages ) {
                   scrollHandler();
            }

            /* Enable infinite scrolling */
            else if (page < pageable.maxPages) {
                table.infiniteScroll(scrollHandler);
            }
            /* loaded 500 elemnts but there is more elements left disable sort */
            if (pageable.totalSize > INTIAL_PAGES_TO_LOAD * PAGE_SIZE && pageable.curPage !== pageable.maxPages){
                table.disableSort();
            }
            //everything is loaded we can enable sort
            else if( pageable.curPage === pageable.maxPages){
                table.enableSort();
            }
        }
    };

    /**
     * Get property given by name list or default value if not found
     *
     * @param {Object}  obj       Object to check
     * @param {Object}  defValue  Default value
     * @param {*}       ...       Variadic number of property names
     *
     * @returns {Object} Either found property or default value
     **/

    Picasso.Helper.getPropOrDefValue = function () {
        var obj = arguments[0];
        var ret = arguments[1];

        /* Check if any of the properties can be found */
        if (null != obj) {
            for (var i = 2; i < arguments.length; i++) {
                if (obj.hasOwnProperty(arguments[i])) {
                    ret = obj[arguments[i]];
                    break;
                }
            }
        }

        return ret;
    };

    /**
     * Get result array from result set
     *
     * @param {Object}  data  Data to check
     *
     * @returns {Array|null} Result set of data or null
     **/

    Picasso.Helper.getResultArray = function (data) {
        var ary = data;

        /* Handle types */
        if ("object" === $.type(data)) {
            if (data.hasOwnProperty("ResultSet") &&
                data.ResultSet.hasOwnProperty("Result")) {
                ary = data.ResultSet.Result;

                if ("array" !== $.type(ary)) {
                    ary = [ary];
                }
            } else if (0 !== Object.keys(data).length) { ///< jquery.isEmptyObject()
                ary = [data];
            }
        }

        return ary;
    };

    /**
     * Helper to get dateTimepicker value if any
     *
     * @param {jQuery|HTMLElement}  input     Input element
     * @param {Object}              defValue  Default value if input val is unset
     *
     * @returns {*} Either a {@link Date} object or the default value
     **/

    Picasso.Helper.getDateTimePickerValue = function (input, defValue) {
        if ("" !== input.val()) {
            var datepicker = input.datetimepicker().data("DateTimePicker");
            return moment(datepicker.date()).format("YYYY-MM-DDTHH:mm:ssZ");
        } else {
            return defValue;
        }
    };

    /**
     * Helper to set dateTimePicker value
     *
     * @param {jQuery|HTMLElement}  input  Input element
     * @param {String}              value  Value to set
     */

    Picasso.Helper.setDateTimePickerValue = function (input, value) {
        var datepicker = input.datetimepicker().data("DateTimePicker");
        datepicker.date(moment(new Date(value)))
    };

    /**
     * Helper to properly parse integer
     *
     * @param {String|Number}  val       Value to parse
     * @param {Number}         defValue  Default value (optional)
     *
     * @returns {Number} Either parsed value or defValue/-1
     **/

    Picasso.Helper.parseNumber = function (val, defValue) {
        var retVal = (null != defValue ? defValue :  -1);

        /* Beware: isNaN("") == false */
        if (null !== val && "" !== val && !isNaN(val)) {
            var num = parseInt(val);

            retVal = (0 > num ? -1 : num);
        }

        return retVal;
    };

    /**
     * Helper to properly parse boolean
     *
     * @param {String|Boolean}  val       Value to parse
     *
     * @returns {Boolean} Either parsed value as boolean or undefined
     */

    Picasso.Helper.parseBoolean = function (val) {

        if (typeof val === "boolean") {
            return val;
        }
        if (val.toLowerCase() === "true") {
            return true;
        }
        if (val.toLowerCase() === "false") {
            return false;
        }

        return undefined;
    };

    /**
     * Helper to determine if a given string represents a boolean.
     *
     * @param {String|Boolean}  val       Value to parse
     *
     * @returns {Boolean} Either parsed value is true or false
     */

    Picasso.Helper.isBoolean = function (val) {

        if (typeof val === "boolean") {
            return true;
        }
        if (val.toString().toLowerCase() === "true") {
            return true;
        }
        if (val.toString().toLowerCase() === "false") {
            return true;
        }
        return false;
    };

    /**
     * Check if a given string represent a valid URL.
     *
     * @param str
     * @returns {string | boolean}
     */

    Picasso.Helper.isValidURL = function (str) {

        if (str.indexOf("..") !== -1 || str.indexOf("\\") !== -1) {
            return false;
        }

        var regexp = /(ftp|http|https):\/\/(\w+:{0,1}\w*@)?(\S+)(:[0-9]+)?(\/|\/([\w#!:.?+=&%@!\-\/]))?/;

        return regexp.test(str);
    };

    /**
     * Sort a given JSON object by it's key's.
     *
     * @param obj
     */

    Picasso.Helper.sortObject = function (obj) {
        var sorted = {};

        Object.keys(obj).sort().forEach(function (key) {
            sorted[key] = obj[key];
        });

        return sorted;
    };

    /**
     * Try to convert string of not already JSON
     *
     * @param {String|Object}  data  Data to convert
     *
     * @returns {JSONObject} Converted JSON object
     **/

    Picasso.Helper.tryJSON = function (data) {
        var json = data;

        /* Convert data if necessary data */
        if ("string" === $.type(data)) {
            try {
                json = JSON.parse(data);
            } catch (e) {
                /* We ignore that here */
            }
        }

        return json;
    };

    /**
     * Convert string to hash (like Java's hashCode)
     *
     * @param {String}  str  String to hash
     *
     * @return {Number} Hash code of given string
     **/

    Picasso.Helper.hashCode = function (str) {
        var hash = 0;

        if (!str || 0 === str.length) return hash;

        for (var i = 0; i < str.length; i++) {
            var char = str.charCodeAt(i);

            hash = ((hash << 5) - hash) + char;
            hash = hash & hash; // Convert to 32bit integer
        }

        return Math.abs(hash);
    };

    /**
     * Save multiple values
     *
     * @param {String}    url        Url to send request to
     * @param {String}    action     Action to use
     * @param {Hash}      data       Data hash
     * @param {Function}  onSuccess  Call on success (optional)
     * @param {Function}  onError    Call on error (optional)
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Helper.saveMultiValues = function () {
        if (3 > arguments.length) return null;

        var data = {
            method: "POST",
            action: arguments[1]
        };

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(arguments[0], $.extend(data, arguments[2]),
            arguments[3], arguments[4]);
    };

    /**
     * Get Pop Up message
     *
     *
     *
     *
     *
     */

    Picasso.Helper.getPopupMessage = function (messageKey,onSuccess,onError) {
        var url = "/wapi/messages";
        Picasso.Helper.fireAjax(url,{}, function (data) {
            var ary = Picasso.Helper.getResultArray(data);
            var message = Picasso.Helper.getPropOrDefValue(ary[0], "", messageKey);
            if (onSuccess) onSuccess(message);

        }, onError);

    };

    /**
     * Fire ajax request
     *
     * @param {String}    url        Url to send request to
     * @param {Hash}      data       Data hash
     * @param {Function}  onSuccess  Call on success (optional)
     * @param {Function}  onError    Call on error (optional)
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Helper.fireAjax = function (url, data, onSuccess, onError) {
        /* Fetch method if any */
        var method = (data.method || "GET");

        delete data.method;

        /* Finally send ajax */
        return $.ajax({

            /* Add CSRF Token header */
            beforeSend: function(request) {
                request.setRequestHeader("X-CSRFToken", Picasso.get("CSRF"));
            },

            method: method,
            url: url,
            data: $.extend(data, {
                json: 1
            }),

            /* Error handling */
            statusCode: {
                403: function () {
                    Picasso.Notification.show(
                        Picasso.Lang.get(
                            "notification_error_permission_denied"), "danger");
                },

                404: function () {
                    if (!onError) {
                        Picasso.Notification.show(
                            Picasso.Lang.get(
                                "notification_error_resource_not_found"), "danger");
                    }
                }
            },

            success: function (data, status, xhr) {
                if (onSuccess) {
                    onSuccess(data, xhr.status);
                }
            },

            error: function (xhr, status) {
                if (onError) {
                    onError(status, xhr.status, xhr.responseJSON);
                }
            }
        });
    };

    /**
     * Join path elements
     *
     * @param {Array}  paths  Array of paths elements
     *
     * @returns {String} Joined paths
     **/

    Picasso.Helper.joinPaths = function (paths) {
        var path = (0 < paths.length ? paths[0] : "");
        var ary = path.split("/");

        for (var i = 1; i < paths.length; i++) {
            if (null != paths[i]) {
                ary = ary.concat(paths[i].split("/"));
            }
        }

        return ary.filter(Boolean).join("/");
    };

    /**
     * Get file extension from file name
     *
     * @param {String}  name  Given file name
     *
     * @returns {String} File extension
     **/

    Picasso.Helper.getFileExt = function (name) {
        var idx = name.lastIndexOf(".");

        return name.substr(idx + 1);
    };
    /**
     * Escape invalide chars from regex
     */

    Picasso.Helper.escapeRegExp = function (str) {
        return  str.replace(/[\[\]\/\{\}\(\)\*\+\?\.\\\^\$\|]/g, "\\$&");
    }

    /**
     * Get base name of path
     *
     * @param {String}  path  Path to file
     **/

    Picasso.Helper.getBaseName = function (path) {
        /* Check for trailing slash */
        if ("/" === path[path.length - 1]) {
            path = path.substr(0, path.length - 1);
        }

        var idx = path.lastIndexOf("/");

        return path.substr(0, idx);
    };

    /**
     * Move upwards in path
     *
     * @param {String}  path  Path to move in
     * @return {String}  Updated path
     **/

    Picasso.Helper.movePathUp = function (path) {
        var subPath = path;

        /* Skip trailing slash if any */
        if ("/" === path[path.length - 1]) {
            subPath = path.substr(0, path.length - 1);
        }

        var idx = subPath ? subPath.lastIndexOf("/") : -1;

        return (-1 === idx ? "" : subPath.substr(0, idx));
    };

    /**
     * Handle various keypress
     *
     * @param {Object}    input    Input element
     * @param {Function}  onEnter  Callback for Enter key
     * @param {Function}  onEsc    Callback for ESC key
     * @param {Function}  onElse   Callback for other keys
     **/

    Picasso.Helper.handleEnterEscElse = function (input, onEnter, onEsc, onElse) {
        var that = $(input);

        that.off("keyup").keyup(function (e) {
            switch (e.which) {
                case 13: ///< Enter
                    if (onEnter) {
                        onEnter.apply(that, [that.val(), e]);
                    }
                    break;

                case 27: ///< ESC
                    if (onEsc) {
                        onEsc.apply(that, [that.val(), e]);
                    }
                    break;

                default: ///< Else
                    if (onElse) {
                        onElse.apply(that,
                            [String.fromCharCode(e.which), that.val(), e]);
                    }
            }
        });
    };

    /**
     * Capitalize string
     *
     * @param {String}  string  String to change
     *
     * @returns {String} Capitalized string
     **/

    Picasso.Helper.capitalize = function (string) {
        return string.charAt(0).toUpperCase() + string.slice(1);
    };

    /**
     * Get avatar url
     *
     * @param {*}  obj  Object to use
     *
     * @returns {String} Either avatar url or empty string
     **/

    Picasso.Helper.getAvatarUrl = function (obj) {
        var oid = (obj.hasOwnProperty("_oid") ? obj._oid
            : (obj.hasOwnProperty("oid") ? obj.oid : null));

        return (oid ? "/" +
            Picasso.Helper.joinPaths(["avatars", oid]) : "");
    };

    /**
     * Sanitize url
     *
     * @param {String}  url  Url to sanitize
     *
     * @returns {String} Sanitized url
     **/

    Picasso.Helper.sanitizeUrl = function (url) {
        return (url && "/" === url[0] ? window.location.origin + url : url);
    };

    /**
     * Sanitize date
     *
     * @param {String}  val  Value to sanitize
     *
     * @returns {Number} Parsed integer
     **/

    Picasso.Helper.sanitizeDate = function (val) {
        var ret = -1;

        /* Beware: isNaN("") == false */
        if ("" !== val && !isNaN(val)) {
            var num = parseInt(val);

            ret = (0 >= num ? -1 : num);
        }

        return ret;
    };

    /**
     * Replace special characters as HTML entities
     *
     * @param {String} string   A string that may contain special characters that may be rendered as HTML
     *
     * @returns {String} A sanitized string that does not contain special characters but the corresponding HTML entity
     */

    Picasso.Helper.encodeHTMLEntities = function (string) {
        return string.replace(this.RE_HTML_ENTITIES, function (i) {
            return '&#' + i.charCodeAt(0) + ";";
        });
    };

    /**
     * Update password policy and display it on hover
     *
     * @param {jQuery|HTMLElement}  elem   Elem to attach popover
     * @param {String}              value  Value to check
     **/

    Picasso.Helper.updatePasswordPolicy = function (elem, value) {
        /* Create popover */
        var div = $("<div>");

        div.append(createAndUpdateLabel("password_policy_uppercase",
            Picasso.Helper.RE_UPPERCASE.test(value || "")));
        div.append(createAndUpdateLabel("password_policy_lowercase",
            Picasso.Helper.RE_LOWERCASE.test(value || "")));
        div.append(createAndUpdateLabel("password_policy_digit",
            Picasso.Helper.RE_DIGIT.test(value || "")));
        div.append(createAndUpdateLabel("password_policy_special",
            Picasso.Helper.RE_SPECIAL.test(value || "")));
        div.append(createAndUpdateLabel("password_policy_count",
            (value && 8 <= value.length)));

        /* And finally add it */
        $(elem).attr({
            "title": Picasso.Helper.createLang(
                "password_policy_explanation").html(),
            "data-toggle": "popover",
            "data-container": "body",
            "data-placement": "bottom",
            "data-trigger": "hover",
            "data-html": "true",
            "data-content": div.html()
        }).popover();
    };

    /**
     * check password policy and display it on the page
     *
     * @param {jQuery|HTMLElement}  elem   Elem to attach popover
     * @param {String}              value  Value to check
     **/

    Picasso.Helper.checkRegistrationPolicy = function (elem, value) {
        $("#password_policy").show()

        var div = $("<div>");

        div.append(createAndUpdateLabel("password_policy_uppercase",
                Picasso.Helper.RE_UPPERCASE.test(value || "")));
        div.append(createAndUpdateLabel("password_policy_lowercase",
                Picasso.Helper.RE_LOWERCASE.test(value || "")));
        div.append(createAndUpdateLabel("password_policy_digit",
                Picasso.Helper.RE_DIGIT.test(value || "")));
        div.append(createAndUpdateLabel("password_policy_special",
                Picasso.Helper.RE_SPECIAL.test(value || "")));
        div.append(createAndUpdateLabel("password_policy_count",
                (value && 8 <= value.length)));

        if ($("#password_policy").children().length === 0) {
            $("#password_policy").html(div.html())
        }

        if ($("#password_policy").find(".label-success").length === 5 || !value) {
            $("#password_policy").hide()
        }

    };

    /**
     * Convert size to human readable string
     *
     * @param {Number}  size  File size in bytes
     *
     * @return {String} Size as human readable string
     **/

    Picasso.Helper.formatSize = function (filesize) {
        if (isNaN(filesize)) return "";

        var suffix = ["bytes", "KB", "MB", "GB", "TB", "PB"];
        var tier = 0;

        while (1024 <= filesize) {
            filesize = filesize / 1024;
            tier++;
        }

        return (Math.round(filesize * 10) / 10 + " " + suffix[tier]);
    };

    /**
     * checks if a date is in the past
     * @param date
     * @returns {boolean}
     */
    Picasso.Helper.inThePast = function (date){
        var moddate = new Date(date);
        var now = new Date();
        return now > moddate;
    }

    /**
     * Convert date to human readable string
     *
     * @param {Number}  date  Date of last modification
     *
     * @return {String} Date as human readable string
     **/

    Picasso.Helper.formatAge = function (date) {
        var moddate = new Date(date);

        var now = new Date();
        var future = (now < moddate);
        var diff = ((now.getTime() - moddate.getTime()) / 1000);
        var absSec = Math.abs(diff);
        var day_diff = Math.round(absSec / 86400);

        if (isNaN(day_diff) || 31 < day_diff) {
            return Picasso.Helper.formatDate(date);
        }

        if (day_diff === 0) {
            if (absSec < 60) {
                return Picasso.Lang.get("age_now");
            } else if (absSec < 3600) {
                return Picasso.Lang.get(Picasso.Lang.Flags.INFLECT, future? "age_in_mins" : "age_mins_ago", Math.round(absSec / 60));
            } else if (absSec < 86400) {
                return Picasso.Lang.get(Picasso.Lang.Flags.INFLECT, future? "age_in_hours": "age_hours_ago", Math.round(absSec / 3600));
            }
        } else if (day_diff === 1) {
            return Picasso.Lang.get(future ? "age_tomorrow" : "age_yesterday");
        } else if (day_diff < 7) {
            return Picasso.Lang.get(Picasso.Lang.Flags.INFLECT, future ? "age_in_days" : "age_days_ago", day_diff);
        } else {
            return Picasso.Lang.get(Picasso.Lang.Flags.INFLECT, future ? "age_in_weeks" : "age_weeks_ago", Math.ceil(day_diff / 7));
        }
    };


    /**
     * Convert date to human readable string
     *
     * @param {String}   date      Unix timestamp (in seconds)
     * @param {Boolean}  omitTime  Whether to omit time in date string
     *
     * @return {String} Time as human readable string
     **/

    Picasso.Helper.formatDate = function (date, omitTime) {
        var d = new Date(date);

        var months = Picasso.Lang.find("array_months_long");

        var dateStr = prefixZero(d.getDate()) + " " +
            months[d.getMonth()] + " " + d.getFullYear();

        /* Whether to omit the timestamp */
        if (true !== omitTime) {
            dateStr += " " + prefixZero(d.getHours()) +
                ":" + prefixZero(d.getMinutes());
        }

        return dateStr;
    };

    /**
     * Convert timestamp to canonical date
     *
     * @param {String}  date  Unix timestamp (in seconds)
     *
     * @return {String} Time as canonical (dd MMM yyyy) date  string
     **/

    Picasso.Helper.formatCanonicalDate = function (date) {
        var d = new Date(date);

        /* FIXME: We need this without dependency on user locale */
        var months = [
            "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
            "Aug", "Sep", "Oct", "Nov", "Dec"
        ];

        return (prefixZero(d.getDate()) + " " +
            months[d.getMonth()] + " " + d.getFullYear());
    };

    /**
     * Convert date to JSON formatted date
     *
     * @param {String}  date  Unix timestamp (in seconds)
     *
     * @return {String} Time as JSON date (yyyy-mm-dd hh:mm:ss.0)
     **/

    Picasso.Helper.formatJSONDate = function (date) {
        var d = new Date(date);
        return moment(d).format();
    };


    /**
     * Convert string duration to days count
     *
     * @param {String}  date  duration 1 month , 1 year , etc
     *
     * @return {int} number of days
     **/

    Picasso.Helper.convertToDays = function (duration) {
        var parts = duration.split(" ");
        var factor = parseInt(parts[0]);
        var timeUnit = parts[1];
        switch (timeUnit){
            case "day":
            case "days":
                return factor;
            case "week":
            case "weeks":
                return 7 * factor;
            case "month":
            case "months":
                return 30*factor;
            case "year":
            case "years":
                return 365*factor;
            default:
                return 0;
        }
    }

    /**
     * Extract host part from a given URL.
     *
     * @param url
     * @returns {string}
     */

    Picasso.Helper.getHostFromURL = function (url) {
        var l = document.createElement("a");
        l.href = url;

        return l.hostname;
    };

    /**
     * Extract host with protocol part from a given URL.
     *
     * @param url
     * @returns {string}
     */

    Picasso.Helper.getHostWithProtocolFromURL = function (url) {
        var l = document.createElement("a");
        l.href = url;

        var hostWithProtocol = l.protocol + "//" + l.hostname;
        if (l.port) {
            hostWithProtocol += ":" + l.port
        }

        return hostWithProtocol;
    };

    /**
     * Extract path part from a given URL.
     *
     * @param url
     * @returns {string}
     */

    Picasso.Helper.getPathFromURL = function (url) {
        var l = document.createElement("a");
        l.href = url;

        return l.pathname;
    };

    /**
     * Extract the file name part from content-disposition.
     *
     * @param contentDisposition
     * @returns {string}
     */

    Picasso.Helper.getFileNameByContentDisposition = function (contentDisposition) {
        var regex = /filename[^;=\n]*=(UTF-8(['"]*))?(.*)/;
        var matches = regex.exec(contentDisposition);
        var filename;

        if (matches != null && matches[3]) {
            filename = matches[3].replace(/['"]/g, '');
        }
        return filename;
    }

    Picasso.Helper.makeUUID = function () {
        function s4() {
            return Math.floor((1 + Math.random()) * 0x10000)
                    .toString(16)
                    .substring(1);
        }
        return s4() + s4() + '-' + s4() + '-' + s4() + '-' +
                s4() + '-' + s4() + s4() + s4();
    }

    Picasso.Helper.getID = function () {
        return  Picasso.Base64.encode(this.makeUUID()).substring(0,20);
    }

    Picasso.Helper.xhrPool = {};
    $(document).ajaxSend(function (e, jqXHR, options) {
        var url  = options.url.split("?")[0];
        url in Picasso.Helper.xhrPool ? Picasso.Helper.xhrPool[url].push(jqXHR) : Picasso.Helper.xhrPool[url] = [jqXHR];
    });
    $(document).ajaxComplete(function (e, jqXHR, options) {
        var url  = options.url.split("?")[0];
        if (Picasso.Helper.xhrPool[url]) {
            Picasso.Helper.xhrPool[url] = $.grep(Picasso.Helper.xhrPool[url], function (x) {
                return x != jqXHR
            });
        }
    });
    Picasso.Helper.abortAll = function (url) {
        $.each(Picasso.Helper.xhrPool[url], function (idx, jqXHR) {
            jqXHR.abort();
        });
    };

    Picasso.Helper.isEmptyPool = function () {
        var pool = Picasso.Helper.xhrPool || {};
        return Object.keys(pool).length === 0;
    };

    Picasso.Helper.abortAllPending = function () {
        var pool = Picasso.Helper.xhrPool || {};
        var keys = Object.keys(pool);

        keys.forEach(function (key) {
            var arr = pool[key];
            if (Array.isArray(arr) && arr.length) {
                arr.slice().forEach(function (jqXHR) {
                    try { jqXHR.abort(); } catch (e) {}
                });
            }
            delete pool[key];
        });
    };

})();
