/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

(function () {

    function addErrorLabel(elem, message) {
        var fieldId = elem.attr("id") || elem.attr("name");
        var formGroup = elem.closest(".form-group");
        var inlineRow = elem.closest(".register-inline-row");
        var errorSpan = $("<span>")
            .addClass("pica-field-error")
            .attr("data-field", fieldId)
            .text(message);

        if (inlineRow.length) {
            inlineRow.after(errorSpan);
        } else {
            formGroup.after(errorSpan);
        }
    }

    function setErrorPlaceholder(elem, message) {
        var $elem = $(elem);
        var originalPlaceholder = $elem.attr("placeholder") || $elem.attr("data-placeholder");
        if (originalPlaceholder && !$elem.attr("data-original-placeholder")) {
            $elem.attr("data-original-placeholder", originalPlaceholder);
        }
        $elem.attr("placeholder", message);
        $elem.addClass("pica-placeholder-error");
    }

    function restorePlaceholder(elem) {
        var $elem = $(elem);
        var original = $elem.attr("data-original-placeholder");
        if (original) {
            var translated = Picasso.Lang.find(original);
            $elem.attr("placeholder", translated || original);
        }
        $elem.removeClass("pica-placeholder-error");
    }

    function getErrorTarget(elem) {
        var inlineRow = elem.closest(".register-inline-row");
        if (inlineRow.length) {
            return elem.closest(".input-group");
        }
        return elem.closest(".form-group");
    }

    function validateRegisterForm(elemId) {
        var $form = $(elemId);
        var required = $form.find("[required]");
        var isValid = true;

        $form.find(".pica-field-error").remove();
        $form.find(".form-group").each(function() {
            $(this).next(".pica-field-error").remove();
        });
        $form.find(".input-group").each(function() {
            $(this).next(".pica-field-error").remove();
        });
        $form.find("[data-original-placeholder]").each(function() {
            restorePlaceholder(this);
        });

        $.each(required, function () {
            var that = $(this);
            var errorTarget = getErrorTarget(that);
            var fieldIsValid = that.get(0).checkValidity();

            if (!fieldIsValid) {
                errorTarget.addClass("has-error");

                var isCheckbox = that.attr("type") === "checkbox";
                if (!isCheckbox && 0 === that.val().length) {
                    var placeholderKey = that.attr("data-placeholder");
                    var fieldName = placeholderKey ? Picasso.Lang.get(placeholderKey) : that.attr("name");
                    if (fieldName) {
                        setErrorPlaceholder(that, Picasso.Lang.get("notification_error_field_required", fieldName));
                    }
                } else if (that.attr("should-validate") === "true") {
                    var value = that.val();
                    var unicodeEmailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/u;
                    if (unicodeEmailRegex.test(value)) {
                        errorTarget.removeClass("has-error");
                        return;
                    }
                    addErrorLabel(that, Picasso.Lang.get("notification_error_invalid_email"));
                }

                isValid = false;
            } else {
                errorTarget.removeClass("has-error");
                restorePlaceholder(that);
            }
        });

        var patterned = $form.find("[pattern]");
        $.each(patterned, function () {
            var that = $(this);
            var errorTarget = getErrorTarget(that);
            var value = that.val();
            if (value.length > 0 && !that.get(0).checkValidity()) {
                errorTarget.addClass("has-error");
                var errorKey = that.attr("data-validation-message");
                if (errorKey) {
                    addErrorLabel(that, Picasso.Lang.get(errorKey));
                }
                isValid = false;
            }
        });

        return isValid;
    }

    function clearErrorOnInput(elem) {
        var $elem = $(elem);
        var inputGroup = $elem.closest(".input-group");
        var formGroup = $elem.closest(".form-group");
        var inlineRow = $elem.closest(".register-inline-row");
        var fieldId = $elem.attr("id") || $elem.attr("name");
        var errorTarget = inlineRow.length ? inputGroup : formGroup;
        var container = inlineRow.length ? inlineRow : formGroup;

        container.siblings(".pica-field-error[data-field='" + fieldId + "']").remove();
        restorePlaceholder(elem);
        errorTarget.removeClass("has-error");
    }

    function validateOnBlur(elem) {
        var $elem = $(elem);
        var inputGroup = $elem.closest(".input-group");
        var formGroup = $elem.closest(".form-group");
        var inlineRow = $elem.closest(".register-inline-row");
        var fieldId = $elem.attr("id") || $elem.attr("name");
        var errorTarget = inlineRow.length ? inputGroup : formGroup;
        var container = inlineRow.length ? inlineRow : formGroup;

        container.siblings(".pica-field-error[data-field='" + fieldId + "']").remove();
        restorePlaceholder(elem);

        var isValid = elem.checkValidity();
        var isCheckbox = $elem.attr("type") === "checkbox";

        if (isCheckbox) {
            isValid = $elem.is(":checked");
        }

        if (isValid && $elem.attr("should-validate") === "true") {
            var value = $elem.val();
            var unicodeEmailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/u;
            isValid = unicodeEmailRegex.test(value);
        }

        if (isValid) {
            errorTarget.removeClass("has-error");
        } else {
            errorTarget.addClass("has-error");
            var value = $elem.val();

            if (!isCheckbox && value.length === 0 && $elem.attr("required") !== undefined) {
                var placeholderKey = $elem.attr("data-placeholder");
                var fieldName = placeholderKey ? Picasso.Lang.get(placeholderKey) : $elem.attr("name");
                if (fieldName) {
                    setErrorPlaceholder($elem, Picasso.Lang.get("notification_error_field_required", fieldName));
                }
            } else if ($elem.attr("should-validate") === "true") {
                addErrorLabel($elem, Picasso.Lang.get("notification_error_invalid_email"));
            } else if (value.length > 0 && $elem.attr("pattern")) {
                var errorKey = $elem.attr("data-validation-message");
                if (errorKey) {
                    addErrorLabel($elem, Picasso.Lang.get(errorKey));
                }
            }
        }
    }

    function initLiveValidation(elemId) {
        $(elemId).on("input change", "[required]", function () {
            clearErrorOnInput(this);
        });

        $(elemId).on("input", "[pattern]", function () {
            clearErrorOnInput(this);
        });

        $(elemId).on("blur", "[required], [pattern]", function () {
            validateOnBlur(this);
        });
    }

    $("#register_form").on("submit", function (e) {
        if (!validateRegisterForm(this)) {
            e.preventDefault();
            e.stopPropagation();
            return;
        }
    });

    $(document).ready(function() {
        initLiveValidation("#register_form");
    });

    var input = $("#register_password");

    input.on("keyup", function (e) {
        Picasso.Helper.checkRegistrationPolicy(this, e.originalEvent.target.value);
    });

    input.siblings(".pica-input-clear").click(function () {
        Picasso.Helper.checkRegistrationPolicy(input);
    });

    var streetInput = $("#registerStreet");
    var postalCodeInput = $("#registerPostalCode");
    var locationInput = $("#registerLocation");
    var reasonInput = $("#registerReason");
    var suggestionsDiv = $("#addressSuggestions");
    var debounceTimer = null;
    var selectedIndex = -1;
    var currentStreetNumber = "";

    if (streetInput.length) {
        streetInput.on("keyup", function (e) {
            var keyCode = e.which || e.keyCode;

            if (keyCode === 40) {
                e.preventDefault();
                navigateSuggestions(1);
                return;
            } else if (keyCode === 38) {
                e.preventDefault();
                navigateSuggestions(-1);
                return;
            } else if (keyCode === 13) {
                e.preventDefault();
                selectCurrentSuggestion();
                return;
            } else if (keyCode === 27) {
                suggestionsDiv.hide();
                selectedIndex = -1;
                return;
            }

            var query = $(this).val();
            if (query.length < 2) {
                suggestionsDiv.hide();
                return;
            }
            clearTimeout(debounceTimer);
            debounceTimer = setTimeout(function () {
                fetchStreetSuggestions(query);
            }, 300);
        });

        streetInput.on("keydown", function (e) {
            var keyCode = e.which || e.keyCode;
            if (keyCode === 40 || keyCode === 38 || keyCode === 13) {
                if (suggestionsDiv.is(":visible")) {
                    e.preventDefault();
                }
            }
        });

        streetInput.on("blur", function () {
            setTimeout(function () {
                suggestionsDiv.hide();
                selectedIndex = -1;
            }, 200);
        });
    }

    function navigateSuggestions(direction) {
        var items = suggestionsDiv.find(".address-suggestion-item");
        if (items.length === 0) return;

        items.removeClass("selected");
        selectedIndex += direction;

        if (selectedIndex < 0) {
            selectedIndex = items.length - 1;
        } else if (selectedIndex >= items.length) {
            selectedIndex = 0;
        }

        var selectedItem = $(items[selectedIndex]);
        selectedItem.addClass("selected");

        var container = suggestionsDiv[0];
        var item = selectedItem[0];
        if (item.offsetTop < container.scrollTop) {
            container.scrollTop = item.offsetTop;
        } else if (item.offsetTop + item.offsetHeight > container.scrollTop + container.clientHeight) {
            container.scrollTop = item.offsetTop + item.offsetHeight - container.clientHeight;
        }
    }

    function selectCurrentSuggestion() {
        var items = suggestionsDiv.find(".address-suggestion-item");
        if (selectedIndex >= 0 && selectedIndex < items.length) {
            $(items[selectedIndex]).trigger("click");
        }
    }

    function fetchStreetSuggestions(query) {
        var streetName = query.trim();
        currentStreetNumber = "";

        var match = streetName.match(/^(.+?)\s+(\d+.*)$/);
        if (match) {
            streetName = match[1].trim();
            currentStreetNumber = match[2].trim();
        }

        $.ajax({
            url: "https://openplzapi.org/de/Streets?name=" + encodeURIComponent(streetName) + "&pageSize=10",
            method: "GET",
            headers: { "Accept": "application/json" },
            success: function (data) { displaySuggestions(data); },
            error: function () { suggestionsDiv.hide(); }
        });
    }

    function displaySuggestions(streets) {
        suggestionsDiv.empty();
        selectedIndex = -1;
        if (!streets || streets.length === 0) {
            suggestionsDiv.hide();
            return;
        }
        streets.forEach(function (street) {
            var displayText;
            var streetValue = street.name;
            if (currentStreetNumber) {
                displayText = street.name + " " + currentStreetNumber + ", " + street.postalCode + " " + street.locality;
                streetValue = street.name + " " + currentStreetNumber;
            } else {
                displayText = street.name + ", " + street.postalCode + " " + street.locality;
            }

            var suggestion = $("<div>")
                .addClass("address-suggestion-item")
                .text(displayText)
                .on("click", function () {
                    streetInput.val(streetValue + (currentStreetNumber ? "" : " "));
                    postalCodeInput.val(street.postalCode);
                    locationInput.val(street.locality);
                    clearErrorOnInput(streetInput[0]);
                    clearErrorOnInput(postalCodeInput[0]);
                    clearErrorOnInput(locationInput[0]);
                    suggestionsDiv.hide();
                    selectedIndex = -1;
                    if (currentStreetNumber) {
                        reasonInput.focus();
                    }
                });
            suggestionsDiv.append(suggestion);
        });
        suggestionsDiv.show();
    }

})();
