/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Accounts")) {
    window.Picasso.Accounts = {};
}

/* Private scope */
(function () {

    /* Globals */
    var TABLE = new Picasso.Table("#accounts_table",
        Picasso.Table.Flags.ROWS_SELECTABLE);
    var PAGE = 1;
    var CURRENT_FILTER = "all";
    var CURRENT_SEARCH = "";

    /**
     * Handle click on create button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCreateAccount = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Show dialog */
        var dialog = new Picasso.Dialog.Account();

        dialog.create(

            /* Success */
            function (account) {
                Picasso.Accounts.fetch();

                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_created",
                        Picasso.Lang.get("label_accounts")), "info");
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Handle click on export button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickExportAccounts = function () {

        var url = "account.csv";

        var req = new XMLHttpRequest();

        req.open("GET", url, true);
        req.responseType = "blob";
        req.onload = function (event) {
            if (4 === req.readyState && (200 === req.status || 0 === req.status)) {
                var content_type = req.getResponseHeader("Content-Type");
                if (content_type && content_type.indexOf('text/csv') === 0){
                    var blob = req.response;
                    var fileName = Picasso.Helper.getFileNameByContentDisposition(req.getResponseHeader("content-disposition"))
                    var link = document.createElement('a');
                    link.href = window.URL.createObjectURL(blob);
                    link.download = fileName;
                    link.click();
                    Picasso.Notification.show(Picasso.Lang.get(
                            "notification_export_complete"), "info");
                }else {
                    req.response.text().then(function (text) {
                        let mailRegex = /\S+[a-z0-9]@[a-z0-9\.]+/img;
                        let mail = text.match(mailRegex);
                        Picasso.Notification.show(Picasso.Lang.get("notification_export_mailed",mail), "info");});
                }

            } else {
                Picasso.Notification.show(Picasso.Lang.get(
                        "notification_export_failed"), "danger");
            }
        };
        req.send();

    }

    /**
     * Handle click on import button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickImportAccounts= function () {

        /* Show dialog */
        var upload = new Picasso.Dialog.Upload(
            Picasso.Lang.get("dialog_title_import_accounts"), null);

        upload.setUrl("/wapi/accounts?action=importAccountCSV");
        upload.setAllowMultiple(false);
        upload.setAllowDirectories(false);
        upload.setStartAutomatically(true);

        /* Limit types */
        upload.setAcceptTypes([".csv"]);

        upload.setDoneHandler(function () {

            /* Basically just reload accounts list */
            Picasso.Accounts.fetch();


        });

        upload.toggleButtons();
        upload.show();
    };

     /**
     * Handle click on edit button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickEdit = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        if (1 === selected.length) {
            var account = new Picasso.Account($(selected[0]).find(
                ".pica-serial").data("serial"));

            /* Show dialog */
            var dialog = new Picasso.Dialog.Account();

            dialog.edit(account,

                /* Success */
                function () {
                    /* Replace row */
                    var row = $("#" + account.getElemId());
                    var rendered = renderAccount(account);

                    row.replaceWith(rendered);
                    rendered.addClass("pica-highlight");

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_updated",
                            Picasso.Lang.get("label_accounts")), "info");
                }
            );
        }
    };

    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDelete = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Account($(val).find(".pica-serial").data("serial"))
        });

        /* Show dialog */
        var what = (1 === selected.length ? selected[0].getUserName() :
            Picasso.Lang.get("string_selected_elements", selected.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_delete"),
            Picasso.Lang.get("dialog_body_delete", what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {
                var account = this;

                account.remove(
                    /* Success */
                    function () {
                        Picasso.Helper.fadeOut($("#" + account.getElemId()),
                            function () {
                                this.remove();

                                TABLE.checkIfEmpty();
                            });

                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_deleted",
                                Picasso.Lang.get("label_accounts")), "info");
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);
                    }
                );
            });
        });

        dialog.show();
    };

    var buildQuery = function () {
        var query = "";

        switch (CURRENT_FILTER) {
            case "not_activated":
                query = "notactivated:true sortdown:registerDate";
                break;
            case "recently_added":
                query = "sortdown:registerDate";
                break;
            default:
                query = "";
        }

        if (CURRENT_SEARCH.length > 0) {
            query = query + (query.length > 0 ? " " : "") + CURRENT_SEARCH;
        }

        return query;
    };

    var handleFilterDropdown = function (filterType) {
        CURRENT_FILTER = filterType;
        $("#pica_accounts_filter .filter-label").text(Picasso.Lang.get("filter_" + filterType));
        doFetch();
    };

    var handleFilter = function (filterString) {
        CURRENT_SEARCH = filterString;
        doFetch();
    };

    var doFetch = function () {
        TABLE.clear();
        TABLE.hideEmptyMessage();
        var tr = $("<tr>", { class: "pica-not-striped" });
        var td = $("<td>", { class: "text-center", colspan: "6" });
        var span = $("<span>", { class: "pica-glyph pica-spinner glyphicons glyphicons-repeat" });
        td.append(span);
        tr.append(td);
        $("#accounts_table tbody").append(tr);

        var query = buildQuery();
        if (query.length >= 1) {
            Picasso.Accounts.fetch(false, query);
        } else {
            Picasso.Accounts.fetch();
        }
    };

    /**
     * Render given json object
     *
     * @param {Picasso.Account}  account  A {@type Picasso.Account}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderAccount = function (account) {
        var tr = $("<tr>", {
            id: account.getElemId(),
            "data-search-keys": [account.getUserName(),account.getOid()],
            draggable: false,
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        var isDeactivated = !account.is(Picasso.Account.Flags.ACTIVE);
        var avatarIcon = isDeactivated ? "user-lock" : "user";
        Picasso.Helper.createGlyphAndAvatar(td, avatarIcon, account);

        tr.append(td);

        /* 2. column: Name */
        td = $("<td>", {
            class: "pica-overflow pica-serial",
        });

        var name = decorateName(account);

        var a = $("<a>", {
            class: "pica-name",
            text: name,
            "data-tooltip": "",
            "data-placement": "bottom",
            title: account.getUserName()
        });

        a.tooltip({ container: "body" });
        if (!account.is(Picasso.Account.Flags.IS_PROXY)) {
            a.click(function (e) {
                tr.addClass(Picasso.Table.Marker.SELECTED);

                handleClickEdit(e);
            });
        }

        td.data("serial", account.serialize());

        td.append(a);
        tr.append(td);

        /* 3a. column: Quota */
        td = $("<td>");

    var usedQuota = account.getQuotaUsed();
    var totalQuota = account.getQuotaTotal();

    if (usedQuota === 0 && totalQuota === 0) {
        td.append($("<span>", { text: Picasso.Lang.get("string_storage_none") }));
    } else {

        var formattedUsedQuota = Picasso.Helper.formatSize(usedQuota);
        var quotaTotalGB = account.getQuotaTotal() / (1024 * 1024 * 1024);
        var formattedTotalQuota = quotaTotalGB >= 9999
            ? $("<span>", { text: "∞", class: "custom-infinity-icon" })
            : Picasso.Helper.formatSize(totalQuota);

        td.append($("<span>", { text: formattedUsedQuota }));
        td.append($("<span>", { class: "pica-slash-before" }).append(formattedTotalQuota));
    }

        tr.append(td);

        /* 4. column: Folders */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: account.getFoldersCount()
        }));

        /* 5. column: Groups */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: account.getGroupsCount()
        }));

        /* 6. column: Organization */
        tr.append($("<td>", {
            class: "pica-overflow hidden-xs",
            text: account.getOrgName()
        }));

        /* 7. column: Valid till */
        td = $("<td>", {
            class: "hidden-xs"
        });

        if (-1 !== account.getValidTill()) {
            var date = Date.parse(account.getValidTill());

            var span = $("<span>", {
                "data-tooltip": "",
                title: Picasso.Helper.formatAge(date),
                text: Picasso.Helper.formatDate(date, true)
            });

            span.tooltip({ container: "body" });
            td.attr('data-sort', date);
            td.append(span);
        }

        tr.append(td);

        /* 8. column: reseller */
        td = $("<td>", {
            class: "text-right",
        });
        if (account.is(Picasso.Account.Flags.RESELLER) && !account.is(Picasso.Account.Flags.ADMIN)) {
            td.append(Picasso.Helper.createGlyph("crown","pica-glyph"))
        }
        tr.append(td)

        return tr;
    };

    var decorateName = function (account) {
        var name = account.getName();

        var isDeactivated = !account.is(Picasso.Account.Flags.ACTIVE);
        var isProxy       = account.is(Picasso.Account.Flags.IS_PROXY);

        if (isDeactivated || isProxy) {
            name += " (";
            if (isDeactivated) {
                name += Picasso.Lang.find("label_deactivated");
            }
            if (isProxy) {
                if (isDeactivated) {
                    name += " / ";
                }
                name += Picasso.Lang.find("label_federated")
            }
            name += ")";
        }
        return name;
    };

    /**
     * Fetch JSON data
     *
     * @param {Boolean}  append  Whether to append data
     **/

    Picasso.Accounts.fetch = function (append,query) {

        /* Reset page */
        if (true !== append) {
            PAGE = 1;
        }

        Picasso.Account.getAll(

            /* Success */
            function (accounts, json) {
                if (true === append) {
                    TABLE.append(accounts, renderAccount);
                } else {
                    TABLE.update(accounts, renderAccount);
                }

                /* Bind infinite scroll */
                Picasso.Helper.bindInfiniteScroll(json, TABLE, PAGE,
                    function () {
                        PAGE++;

                        Picasso.Accounts.fetch(true,query);
                    }
                );
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }, PAGE,null,query
        );
    };

    /* Bind handlers */
    $("#create_account").click(handleClickCreateAccount);
    $("#import_accounts").click(handleClickImportAccounts);
    $("#export_accounts").click(handleClickExportAccounts);
    $("#accounts_edit").click(handleClickEdit);
    $("#accounts_delete").click(handleClickDelete);

    Picasso.Actionbar.setFilterHandler(handleFilter);

    $("#pica_accounts_filter a[data-filter]").click(function (e) {
        e.preventDefault();
        handleFilterDropdown($(this).data("filter"));
    });

    $("#pica_action_search").addClass("pica-has-filter");

    Picasso.Accounts.fetch();
})();

