if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Chat")) {
    window.Picasso.Chat = {};
}

(function () {

    var API_URL = "/wapi/wingman";
    var SSE_DONE = "[DONE]";
    var KEYBOARD_THRESHOLD = 100;
    var TEXTAREA_MAX_HEIGHT = 200;

    var messages = [];
    var messageIdCounter = 0;
    var keyboardVisible = false;

    var $input;
    var $messagesContainer;
    var $scrollContainer;
    var $inputContainer;

    var scrollToBottom = function () {
        $scrollContainer.scrollTop($scrollContainer[0].scrollHeight);
    };

    var renderMessage = function (message) {
        var isUser = message.sender === "user";
        var messageDiv = $("<div>", {
            class: "pica-chat-message " + (isUser ? "pica-chat-message-user" : "pica-chat-message-wingman"),
            "data-message-id": message.id
        });

        var content = $("<div>", { class: "pica-chat-message-content" });
        var body = $("<div>", { class: "pica-chat-message-body" }).text(message.text);

        content.append(body);
        messageDiv.append(content);

        return messageDiv;
    };

    var addMessage = function (text, sender) {
        var message = {
            id: messageIdCounter++,
            text: text,
            sender: sender,
            timestamp: new Date()
        };

        messages.push(message);
        $messagesContainer.append(renderMessage(message));
        scrollToBottom();

        return message;
    };

    var createEmptyMessage = function (sender) {
        return addMessage("", sender);
    };

    var appendToMessage = function (messageId, text) {
        var $body = $('[data-message-id="' + messageId + '"]').find(".pica-chat-message-body");
        $body.text($body.text() + text);
        scrollToBottom();
    };

    var showTypingIndicator = function () {
        var typingDiv = $("<div>", { class: "pica-chat-typing-indicator", id: "pica-chat-typing" });
        var content = $("<div>", { class: "pica-chat-message-content" });
        var body = $("<div>", { class: "pica-chat-message-body" });
        var dots = $("<div>", { class: "pica-chat-typing-dots" })
            .append($("<span>"))
            .append($("<span>"))
            .append($("<span>"));

        body.append(dots);
        content.append(body);
        typingDiv.append(content);

        $messagesContainer.append(typingDiv);
        scrollToBottom();
    };

    var hideTypingIndicator = function () {
        $("#pica-chat-typing").remove();
    };

    var buildStreamUrl = function (message) {
        return API_URL + "?action=send&message=" + encodeURIComponent(message) + "&CSRFToken=" + Picasso.get("CSRF");
    };

    var sendMessage = function (text) {
        if (!text || text.trim().length === 0) {
            return;
        }

        addMessage(text, "user");
        showTypingIndicator();

        var source = new EventSource(buildStreamUrl(text));
        var responseMessage = null;

        source.onmessage = function (e) {
            if (e.data === SSE_DONE) {
                source.close();
                return;
            }

            if (!responseMessage) {
                hideTypingIndicator();
                responseMessage = createEmptyMessage("wingman");
            }

            appendToMessage(responseMessage.id, e.data);
        };

        source.onerror = function () {
            source.close();
            hideTypingIndicator();
            if (!responseMessage) {
                addMessage(text, "wingman");
            }
        };
    };

    var handleFormSubmit = function (e) {
        e.preventDefault();

        var text = $input.val().trim();

        if (text.length > 0) {
            sendMessage(text);
            $input.val("");
            $input[0].style.height = "auto";
        }

        $input.focus();
    };

    var autoResizeTextarea = function () {
        var textarea = $input[0];
        textarea.style.height = "auto";
        textarea.style.height = Math.min(textarea.scrollHeight, TEXTAREA_MAX_HEIGHT) + "px";
    };

    var handleKeyDown = function (e) {
        if (e.key === "Enter" && !e.shiftKey) {
            e.preventDefault();
            $("#pica-chat-form").submit();
        }
    };

    var resetLayout = function () {
        keyboardVisible = false;

        $inputContainer.css({
            transition: "transform 0.15s ease-out",
            transform: "translateY(0)"
        });

        $scrollContainer.css("padding-bottom", "100px");
        window.scrollTo(0, 0);
    };

    var applyKeyboardLayout = function (keyboardHeight) {
        if (!keyboardVisible) {
            keyboardVisible = true;
            $inputContainer.css("transition", "none");
        }

        $inputContainer.css("transform", "translateY(-" + keyboardHeight + "px)");
        $scrollContainer.css("padding-bottom", (keyboardHeight + 100) + "px");

        window.scrollTo(0, 0);
        scrollToBottom();
    };

    var handleVisualViewportChange = function () {
        if (!window.visualViewport) {
            return;
        }

        var keyboardHeight = window.innerHeight - window.visualViewport.height;

        if (keyboardHeight > KEYBOARD_THRESHOLD) {
            applyKeyboardLayout(keyboardHeight);
        }
    };

    var preventScroll = function () {
        window.scrollTo(0, 0);
        document.body.scrollTop = 0;
        document.documentElement.scrollTop = 0;
    };

    var initMobileKeyboardHandling = function () {
        if (window.visualViewport) {
            window.visualViewport.addEventListener("resize", handleVisualViewportChange);
            window.visualViewport.addEventListener("scroll", handleVisualViewportChange);
        }

        window.addEventListener("scroll", preventScroll, { passive: false });
        document.body.addEventListener("scroll", preventScroll, { passive: false });

        $input.on("focus", handleVisualViewportChange);

        $input.on("blur", function () {
            setTimeout(function () {
                if (document.activeElement !== $input[0]) {
                    resetLayout();
                }
            }, 50);
        });
    };

    var cacheElements = function () {
        $input = $("#pica-chat-input");
        $messagesContainer = $("#pica-chat-messages");
        $scrollContainer = $(".pica-chat-messages-container");
        $inputContainer = $(".pica-chat-input-container");
    };

    Picasso.Chat.init = function () {
        cacheElements();

        if ($input.attr("data-placeholder")) {
            $input.attr("placeholder", Picasso.Lang.get($input.attr("data-placeholder")));
        }

        $("#pica-chat-form").on("submit", handleFormSubmit);
        $input.on("input", autoResizeTextarea);
        $input.on("keydown", handleKeyDown);

        initMobileKeyboardHandling();

        $input.focus();
        Picasso.debugLog("Chat interface initialized");
    };

    $(document).ready(function () {
        Picasso.Chat.init();
    });

})();
