/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Configurations")) {
    window.Picasso.Configurations = {};
}

/* Private scope */
(function () {

    /* Globals */
    var TABLE = new Picasso.Table("#configurations_table",
        Picasso.Table.Flags.ROWS_SELECTABLE);
    var PAGE = 1;

    /**
     * Handle click on create button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCreate = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Show dialog */
        var dialog = new Picasso.Dialog.Configuration();

        dialog.create(

            /* Success */
            function () {
                Picasso.Configurations.fetch();

                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_created",
                        Picasso.Lang.get("label_profiles")), "info");
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Handle click on edit button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickEdit = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        console.dir(selected);

        if (1 === selected.length) {
            var configuration = new Picasso.Configuration(
                $(selected[0]).find(".pica-serial").data("serial"));

            var dialog = new Picasso.Dialog.Configuration();

            dialog.edit(configuration,

                /* Success */
                function (configuration) {
                    /* Replace row */
                    var row = $("#" + configuration.getElemId());
                    var rendered = renderConfiguration(configuration);

                    row.replaceWith(rendered);
                    rendered.addClass("pica-highlight");

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_updated",
                            Picasso.Lang.get("label_profiles")), "info");
                }
            );
        }
    };

    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDelete = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Configuration(
                $(val).find(".pica-serial").data("serial"))
        });

        /* Show dialog */
        var what = (1 === selected.length ? selected[0].getName() :
            Picasso.Lang.get("string_selected_elements", selected.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_delete"),
            Picasso.Lang.get("dialog_body_delete", what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {
                var configuration = this;

                configuration.remove(

                    /* Success */
                    function () {
                        Picasso.Helper.fadeOut($("#" + configuration.getElemId()),
                            function () {
                                this.remove();

                                TABLE.checkIfEmpty();
                            });

                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_deleted",
                                Picasso.Lang.get("label_profiles")), "info");
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);
                    }
                );
            });
        });

        dialog.show();
    };

    /**
     * Handle filter event
     *
     * @param {String}  filterString  Input string for filter
     **/

    var handleFilter = function (filterString) {
        TABLE.clear();
        TABLE.hideEmptyMessage();
        var tr = $("<tr>", { class: "pica-not-striped" });
        var td = $("<td>", { class: "text-center", colspan: "5" });
        var span = $("<span>", { class: "pica-glyph pica-spinner glyphicons glyphicons-repeat" });
        td.append(span);
        tr.append(td);
        $("#configurations_table tbody").append(tr);

        if (filterString.length >= 1) {
            Picasso.Configurations.fetch(false, filterString);
        } else {
            Picasso.Configurations.fetch();
        }
    };

    /**
     * Render given json object
     *
     * @param {Picasso.Configuration}  configuration  A #{@link Picasso.Configuration}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderConfiguration = function (configuration) {
        var tr = $("<tr>", {
            id: configuration.getElemId(),
            "data-search-keys": [configuration.getName()],
            draggable: false
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        var span = $("<span>", {
            class: "pica-icon pica-glyph glyphicons glyphicons-vcard"
        });

        td.append(span);
        tr.append(td);

        /* 2. column: Name */
        td = $("<td>", {
            class: "pica-overflow pica-serial",
            "data-serial": configuration.serialize()
        });

        var a = $("<a>", {
            class: "pica-name",
            text: configuration.getName()
        });

        a.click(function (e) {
            tr.addClass(Picasso.Table.Marker.SELECTED);

            handleClickEdit(e);
        });

        td.append(a);
        tr.append(td);

        /* 3. column: Quota */
        tr.append($("<td>", {
            "data-sort": configuration.getQuota(),
            text: Picasso.Helper.formatSize(configuration.getQuota()),
        }));

        /* 4. column: Folders */
        tr.append($("<td>", {
            text: configuration.getFoldersCount()
        }));

        /* 5. column: Last modified */
        td = $("<td>");

        span = $("<span>", {
            "data-tooltip": "",
            text: Picasso.Helper.formatAge(configuration.getLastModified()),
            title: Picasso.Helper.formatDate(configuration.getLastModified())
        });

        span.tooltip({ container: "body" });

        td.append(span);
        tr.append(td);

        return tr;
    };

    /**
     * Fetch JSON data
     *
     * @param {Boolean}  append  Whether to append data
     **/

    Picasso.Configurations.fetch = function (append, query) {

        /* Reset page */
        if (true !== append) {
            PAGE = 1;
        }

        Picasso.Configuration.getAll(

            /* Success */
            function (configurations, json) {
                if (true === append) {
                    TABLE.append(configurations, renderConfiguration);
                } else {
                    TABLE.update(configurations, renderConfiguration);
                }

                /* Bind infinite scroll */
                Picasso.Helper.bindInfiniteScroll(json, TABLE, PAGE,
                    function () {
                        PAGE++;

                        Picasso.Configurations.fetch(true, query);
                    }
                );
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }, PAGE, null, query
        );

    };

    /* Bind */
    $("#configurations_create").click(handleClickCreate);
    $("#configurations_edit").click(handleClickEdit);
    $("#configurations_delete").click(handleClickDelete);

    Picasso.Actionbar.setFilterHandler(handleFilter);

    /* Init */
    Picasso.Configurations.fetch();
})();

