/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Files")) {
    window.Picasso.Files = {};
}

/* Private scope */
(function () {

    function showFilesSpinner() {
        var $tbody = $("#files_files_table tbody");
        var cols = $("#files_files_table thead tr:last th:visible").length || 6;

        $tbody.empty().append(
            $("<tr>", { class: "pica-not-striped" }).append(
                $("<td>", { class: "text-center", colspan: cols }).append(
                    $("<span>", { class: "pica-glyph pica-spinner glyphicons glyphicons-repeat" })
                )
            )
        );
    }


    window.onpopstate = (event) => {
        var state = event.state;
        if (state) {
            var fileData, searchQuery;

            if (state.file !== undefined || state.searchQuery !== undefined) {
                fileData = state.file;
                searchQuery = state.searchQuery || "";
            } else {
                fileData = state;
                searchQuery = "";
            }

            var file = fileData ? new Picasso.File(fileData) : null;

            showFilesSpinner();

            $("#pica_action_search_input").val(searchQuery);
            if (searchQuery) {
                sessionStorage.setItem("searchQuery", searchQuery);
            } else {
                sessionStorage.removeItem("searchQuery");
            }

            var currentCrumbs = $(".pica-crumb").not("#files_home");
            var currentDepth = currentCrumbs.length;

            var targetDepth = file ? (state.breadcrumbDepth !== undefined ? state.breadcrumbDepth : currentDepth) : 0;

            if (targetDepth < currentDepth) {
                // Going back - remove crumbs
                for (var i = currentDepth; i > targetDepth; i--) {
                    $(".pica-crumb").not("#files_home").last().remove();
                }
            } else if (targetDepth > currentDepth && file) {
                // Going forward - add crumb
                var a = Picasso.Actionbar.addCrumb(file.getName(), handleClickCrumb);
                a.data("file", file.serialize());
            }

            if (searchQuery) {
                Picasso.Files.fetch(file, false, null, null, searchQuery, true);
            } else {
                Picasso.Files.fetch(file, false, null);
            }
        } else {
            $("#files_home").click()
        }
    }

    $(document).ready(function () {

        var cwd = sessionStorage.getItem("navigationHistory");
        var breadcrumb = sessionStorage.getItem("breadcrumb");
        var searchQuery = sessionStorage.getItem("searchQuery");

        if (searchQuery) {
            $("#pica_action_search_input").val(searchQuery);
        }

        if (null !== cwd) {
            if (breadcrumb!==null) {
                JSON.parse(breadcrumb).forEach(function(crumb){
                    Picasso.Actionbar.addCrumb(new Picasso.File(crumb).getName(), handleClickCrumb).data("file", crumb);
                })
            }
            Picasso.Files.fetch(new Picasso.File(cwd));
        } else {
            //Init
            Picasso.Files.fetch();
        }
    });

    $(window).on("beforeunload", function () {
        var cwd = Picasso.get("cwd");
        if (cwd) {
            var breadcrumb = [];
            $(".pica-crumb").not("#files_home").each(function () {
                breadcrumb.push($(this).data("file"))
            })
            sessionStorage.setItem("navigationHistory", cwd.serialize());
            sessionStorage.setItem("breadcrumb", JSON.stringify(breadcrumb));
        }
        if (document.activeElement.href && !document.activeElement.href.includes("/open/")) {
            sessionStorage.removeItem("navigationHistory");
            sessionStorage.removeItem("breadcrumb");
        }
    });

    window.onload = function() {
        var currentUrl = window.location.href;
        var links = document.querySelectorAll('#folder-links .text-center a');
        links.forEach(function(link) {
            link.classList.remove('active');
        });
        links.forEach(function(link) {
            if (link.href === currentUrl) {
                link.classList.add('active');
            }
        });
    };

    /* Globals */
    var VMTABLE = new Picasso.ViewModeTable("#files_files_table",
        (Picasso.ViewModeTable.Flags.LIST_MODE|
            Picasso.ViewModeTable.Flags.ROWS_SELECTABLE));
    var PAGE = 1;
    var SYNC_STATUS_REFRESH_DELAY = 4000;

    /* Options for dropdown menus */
    var optionsForInvitations = [
        {
            text: Picasso.Lang.get("action_accept"),
            icon: "ok-circle"
        },
        null,
        {
            text: Picasso.Lang.get("action_decline"),
            icon: "remove-circle"
        }
    ];

    var optionsForGallery = [
        {
            text: Picasso.Lang.get("action_download"),
            icon: "download",
            klass: "files-ui-download"
        },
        {
            text: Picasso.Lang.get("action_rename"),
            icon: "edit",
            klass: "files-ui-rename"
        },
        {
            text: Picasso.Lang.get("action_restore"),
            icon: "restart",
            klass: "files-ui-restore"
        },
        {
            text: Picasso.Lang.get("action_delete"),
            icon: "bin",
            klass: "files-ui-delete"
        }
    ];

    /* Helper */

    /**
     * Handle options for gallery
     *
     * @param {Number}  optIdx  Selected index
     **/

    var handleOptionsForGallery = function (optIdx) {
        var file = getFileFromElem($(this).parents("p"));

        switch(optIdx) {
            case 0: ///< Download
                downloadFile(file);
                break;

            case 1: ///< Rename
                renameFile(file);
                break;

            case 2: ///< Restore
                restoreFile(file);
                break;

            case 3: ///< Delete
                DeleteFile([ file ]);
                break;
        }
    };

    /**
     * Download given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@link Picasso.File}
     **/

    var downloadFile = function (file) {
        /* Create a new anchor and simulate click to download */
        var a = $("<a>", {
            href: file.getDownloadUrl(),
            download: file.getName()
        });

        $("body").append(a);
        a[0].click();
        a.remove();
    };

    /**
     * Rename given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@Picasso.File}
     **/

    var renameFile = function (file) {
        var elem = $("#" + file.getElemId());

        var oldName = file.getIntName();

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_input_dialog",
            Picasso.Lang.get("dialog_title_change"),
            Picasso.Lang.get("dialog_body_change", Picasso.Lang.get("label_name")),
            (Picasso.Dialog.Flags.HIDE_ON_ENTER|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.NO_AUTOCLOSE));

        dialog.setAssignHandler(function (inputs) {
            var fullName = file.getIntName() || file.getName();
            if (file.is(Picasso.File.Flags.FILE)) {
                var nameOnly = fullName.replace(/\.[^/.]+$/, "");
                inputs.pica_input_text.val(nameOnly);
                inputs.pica_input_text.data("original-extension", fullName.slice(nameOnly.length));
            } else {
                inputs.pica_input_text.val(fullName);
                inputs.pica_input_text.removeData("original-extension");
            }
        });

        dialog.setOkHandler(function (inputs) {
            var baseName = inputs.pica_input_text.val();
            var newName = baseName;
            if (file.is(Picasso.File.Flags.FILE)) {
                baseName = baseName.replace(/\.[^/.]+$/, "");
                var ext = inputs.pica_input_text.data("original-extension") || "";
                newName = baseName + ext;
            }
            file.rename(newName,

                /* Success */
                function () {
                    var a = elem.find(".pica-name");

                    /* Update for new name */
                    a.text(file.getName());
                    a.attr("data-serial", "");
                    a.data("serial", file.serialize());

                    elem.addClass("pica-highlight");

                    Picasso.Notification.show(Picasso.Lang.get(
                        "notification_item_renamed",
                        oldName, newName), "info");

                    dialog.hide();
                    Picasso.Files.fetch(Picasso.get("cwd"));
                },

                /* Error */
                function (e, status) {
                    switch (status) {
                        case 405:
                            Picasso.Notification.show(
                                    Picasso.Lang.get("notification_error_file_under_edit"),
                                    "warning");
                            break;
                        case 406:
                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_error_illegal_chars"),
                                "warning");
                            break;

                        case 409:
                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_error_duplicate", name),
                                "warning");
                            break;

                        case 400:
                            /* Falls through */
                        case 500:
                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_error_rename", name),
                                "danger");
                            break;
                    }

                    Picasso.debugLog(e);
                }
            );
        });

        dialog.setValidateFunction(function (inputs) {
            var baseName = inputs.pica_input_text.val();
            var newName = baseName;
            if (file.is(Picasso.File.Flags.FILE)) {
                baseName = baseName.replace(/\.[^/.]+$/, "");
                var ext = inputs.pica_input_text.data("original-extension") || "";
                newName = baseName + ext;
            }

            var disallowedCharactersRegex = /[\\\/:\*\?„<>]/;

            if (disallowedCharactersRegex.test(newName)) {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_illegal_chars"), "warning");
                return false;
            }

            if (newName.length > 200) {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_name_too_long"), "warning");
                return false;
            }

            var cwd = Picasso.get("cwd");
            if (cwd) {
                var fullPath = cwd.getPath() + "/" + newName;
                if (fullPath.length > 3500) {
                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_error_path_too_long"), "warning");
                    return false;
                }
            }

            return true;
        });

        dialog.show();
    };

    /**
     * Rename given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@Picasso.File}
     **/

    var warnOveruse = function (numberOfFolders) {

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_overuse_warn"),
            Picasso.Lang.get("dialog_body_overuse_warn", numberOfFolders, account.getMaxFolders()),
            (Picasso.Dialog.Flags.HIDE_ON_ENTER|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.NO_AUTOCLOSE));


        dialog.setOkHandler(function (inputs) {
            window.open('/pricing', '_self');
        });

        dialog.show();
    };

    /**
     * Restore given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@Picasso.File}
     **/

    var restoreFile = function (file) {

        /* Show dialog */
        var dialog = new Picasso.Dialog.Restore(file);

        dialog.show();
    };

    /**
     * delete given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@Picasso.File}
     **/

    var DeleteFile = function (file) {

        var title = "dialog_title_delete";
        var body = "dialog_body_delete";
        var notification = "notification_item_deleted";

        /* Show dialog */
        var what = (1 === file.length ? file[0].getName() :
            Picasso.Lang.get("string_selected_elements", file.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get(title), Picasso.Lang.get(body, what));

        dialog.setOkHandler(function () {
            $.each(file, function () {
                var file = this;

                file.remove(

                    /* Success */
                    function () {
                        Picasso.Helper.fadeOut($("#" + file.getElemId()),
                            function () {
                                this.remove();

                                VMTABLE.checkIfEmpty();
                            });

                        Picasso.Notification.show(
                            Picasso.Helper.capitalize(
                                Picasso.Lang.get(notification, Picasso.Lang.get(
                                    file.is(Picasso.File.Flags.FOLDER) ?
                                        "label_folders" :
                                    (file.is(Picasso.File.Flags.DIRECTORY) ?
                                    "label_directories" : "label_files")))), "info");
                    },

                    /* Error */
                    function (e,status) {

                        if (405===status) {
                            Picasso.Notification.show(
                                    Picasso.Lang.get("notification_error_file_under_edit"),
                                    "warning");
                        }
                        else if (file.is(Picasso.File.Flags.FEDERATED)) {
                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_federated_service_not_reachable"), "warning");
                        } else {
                            Picasso.debugLog(e);
                        }
                    }
                );
            });
        });

        dialog.show();
    };

    /**
     * Leave  given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@Picasso.File}
     **/

    var leaveFolder = function (file) {

        var title = "dialog_title_leave";
        var body = "dialog_body_leave";
        var notification = "notification_item_left";

        /* Show dialog */
        var what = (1 === file.length ? file[0].getName() :
                Picasso.Lang.get("string_selected_elements", file.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
                Picasso.Lang.get(title), Picasso.Lang.get(body, what));

        dialog.setOkHandler(function () {
            $.each(file, function () {
                var file = this;

                file.leave(

                        /* Success */
                        function () {
                            Picasso.Helper.fadeOut($("#" + file.getElemId()),
                                    function () {
                                        this.remove();

                                        VMTABLE.checkIfEmpty();
                                    });

                            Picasso.Notification.show(
                                    Picasso.Helper.capitalize(
                                            Picasso.Lang.get(notification, Picasso.Lang.get(
                                                    file.is(Picasso.File.Flags.FOLDER) ?
                                                            "label_folders" :
                                                            (file.is(Picasso.File.Flags.DIRECTORY) ?
                                                                    "label_directories" : "label_files")))), "info");
                        },

                        /* Error */
                        function (e,status) {

                            if (405===status) {
                                Picasso.Notification.show(
                                        Picasso.Lang.get("notification_error_file_under_edit"),
                                        "warning");
                            }
                            else if (file.is(Picasso.File.Flags.FEDERATED)) {
                                Picasso.Notification.show(
                                        Picasso.Lang.get("notification_federated_service_not_reachable"), "warning");
                            } else {
                                Picasso.debugLog(e);
                            }
                        }
                );
            });
        });

        dialog.show();
    };

    /**
     * Opens the settings of given {@link Picasso.File}
     *
     * @param {Picasso.File}  folder  A #{@Picasso.Folder}
     **/

    var settings = function (folder) {

        /* Show dialog */
        var dialog = new Picasso.Dialog.Settings(folder);

        dialog.show();
    };

    /**
     * Get visibilities for {@link Picasso.Permission}
     *
     * @param {Picasso.Permission}  lowestPermission      A #{@link Picasso.Permission}
     * @param {Picasso.Permission}  collectedPermissions  A #{@link Picasso.Permission}
     * @param {Picasso.File.Flags}  collectedFileTypes    A #{@link Picasso.File.Flags}
     * @param {Number}              nselected             Number of selected rows
     **/

    var getVisibilities = function (lowestPermission, collectedPermissions,
                                    collectedFileTypes, nselected)
    {
        var visibilities = 0;

        /* Handle common permissions */
        switch (lowestPermission) {
            case Picasso.Permission.OWNER:
            /* Falls through */
            case Picasso.Permission.ADMIN:
                visibilities |= (
                    Picasso.Files.Visibility.RENAME |
                    Picasso.Files.Visibility.RESTORE
                );
            /* Falls through */
            case Picasso.Permission.READ_WRITE:
                visibilities |= (
                    Picasso.Files.Visibility.CREATE |
                    Picasso.Files.Visibility.RESTORE |
                    Picasso.Files.Visibility.DOWNLOAD |
                    Picasso.Files.Visibility.COPY |
                    Picasso.Files.Visibility.CUT |
                    Picasso.Files.Visibility.UPLOAD |
                    Picasso.Files.Visibility.DELETE |
                    Picasso.Files.Visibility.LEAVE |
                    Picasso.Files.Visibility.SETTINGS
                );

                /* Check delete and leave */
                if (Picasso.File.Flags.FOLDER === collectedFileTypes) {
                    if (Picasso.Permission.fromValue(Picasso.get("FolderDeletePermission")) > lowestPermission ||  Picasso.Permission.ADMIN > lowestPermission ) {
                        visibilities &= ~Picasso.Files.Visibility.DELETE;
                    }else if (Picasso.Permission.OWNER === lowestPermission) {
                        visibilities &= ~Picasso.Files.Visibility.LEAVE;
                    }
                } else {
                    visibilities |= Picasso.Files.Visibility.RENAME;
                }
                break;

            case Picasso.Permission.READ:
                /* Allow leave of folders */
                if (Picasso.File.Flags.FOLDER === collectedFileTypes) {
                    visibilities |= Picasso.Files.Visibility.LEAVE;
                }

                visibilities |= Picasso.Files.Visibility.DOWNLOAD;
                break;
        }

        /* Limit actions based on type */
        if (0 < (collectedFileTypes &
            (Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY))) {
            /* Prevent actions on folders and directories */
            visibilities = (visibilities & ~(Picasso.Files.Visibility.COPY |Picasso.Files.Visibility.RESTORE));
        }

        if (0 < (collectedFileTypes & (Picasso.File.Flags.FILE | Picasso.File.Flags.DIRECTORY))) {
            /* Prevent leave on files */
            visibilities = (visibilities & ~(Picasso.Files.Visibility.LEAVE));
            /* Prevent settings on directories and files */
            visibilities = (visibilities & ~(Picasso.Files.Visibility.SETTINGS));
        }

        /* Limit actions based on multi selections */
        if (1 < nselected) {
            /* Prevent rename and copy */
            visibilities = (visibilities & ~(Picasso.Files.Visibility.RENAME |
                Picasso.Files.Visibility.COPY|Picasso.Files.Visibility.RESTORE|Picasso.Files.Visibility.SETTINGS));

            /* Prevent mixing of leave and delete; but allow leave or delete all */
            if (0 < (collectedPermissions & (Picasso.Permission.READ |
                Picasso.Permission.READ_WRITE)) && 0 < (collectedPermissions &
                (Picasso.Permission.ADMIN|Picasso.Permission.OWNER))) {
                visibilities = (visibilities & ~(Picasso.Files.Visibility.LEAVE |
                    Picasso.Files.Visibility.DELETE));
            }
        }

        return visibilities;
    };

    /**
     * Toggle element visibilities
     *
     * @param {Number}  visibilities  Element visibilities
     **/

    var toggleVisibility = function (visibilities) {
        $.each([
            ".files-ui-create",
            ".files-ui-download",
            ".files-ui-rename",
            ".files-ui-copy",
            ".files-ui-cut",
            ".files-ui-restore",
            ".files-ui-delete",
            ".files-ui-leave",
            ".files-ui-upload",
            ".files-ui-settings"
        ], function (idx) {
            var elems = $(this);

            if (0 < (visibilities & (1 << idx))) {
                elems.show();
            } else {
                elems.hide();
            }
        });
    };

    /**
     * Update table header for selection mode
     *
     * @param {Number}  nselected    Number of selected rows
     * @param {Object}  selectedRow  Selected row object
     **/

    var handleOnSelection = function (nselected, selectedRow) {
        var cwd = Picasso.get("cwd");
        var lowestPermission = 0;
        var collectedPermissions = 0;
        var collectedFileTypes = 0;

        if ($('.conext-dropdown-menu:visible').length) {
            $('.conext-dropdown-menu:visible').closest("tr").removeClass("pica-table-selection-disabled");

            var visibleRow = $('.conext-dropdown-menu:visible').closest("tr");

            if (selectedRow) {
                $('.conext-dropdown-menu:visible').hide();
            }

        }
        /* Check for root */
        if (cwd) {
            /* Toggle paste header if necessary */
            var headers = $("#files_files_table thead");
            var containsBlockedFolder = false ;


            if (Picasso.hasKey("cutFiles")) {
                headers.hide();
                $(headers[1]).show();
            } else if ($(headers[1]).is(":visible")) {
                headers.hide();
                $(headers[2]).show();
            }

            lowestPermission = cwd.getPermission();
        } else {
            /* Only root folders in this path then */
            lowestPermission = Picasso.Permission.OWNER;
            collectedFileTypes = Picasso.File.Flags.FOLDER;
        }

        if (1 <= nselected) {
            /* Set initial values */
            lowestPermission = Picasso.Permission.OWNER; ///< Highest
            collectedFileTypes = 0; ///< Lowest

            $.each(VMTABLE.getTable().getSelectedRows(), function () {
                var selectedFile = getFileFromElem(this);

                if (selectedFile && selectedFile.is(Picasso.File.Flags.INVITED_TO)) {
                    VMTABLE.getTable().deselectAll();
                    return false;
                }

                if (selectedFile.getIsBlocked()){
                    containsBlockedFolder = true;
                }

                if (selectedFile) {
                    var permission = selectedFile.getPermission();

                    /* Find lowest common permission */
                    if (permission < lowestPermission) {
                        lowestPermission = permission;
                    }

                    collectedPermissions |= permission;

                    /* Keep all found file types */
                    collectedFileTypes |= (selectedFile.getFlags() &
                        (Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY|
                            Picasso.File.Flags.FILE));
                }
            });
        }

        /* Toggle visibility of ui actions */
        var visibilities = getVisibilities(lowestPermission,
            collectedPermissions, collectedFileTypes, nselected);

        /* Show restore only when version > 0 */
        if (1 === nselected) {
            var file = getFileFromElem(selectedRow);

            if (0 === file.getVersion()) {
                visibilities &= ~Picasso.Files.Visibility.RESTORE;
            }
        }

        toggleVisibility(visibilities);

        if (containsBlockedFolder){
            Picasso.Notification.show(Picasso.Lang.get("notification_error_cannot_apply_operation_blocked_folder"), "warning")
            VMTABLE.getTable().deselectAll();
        }


        /* Toggle visibility for root */
        var cwd = Picasso.get("cwd");
        var onlyRoot = $(".files-visible-only-root");
        var onlyOther = $(".files-visible-only-other");
        var limited = $(".limited");
        var folderShare = $("#files_folder_share");

        if (cwd) {
            onlyRoot.hide();
            if (cwd.getPermission() != Picasso.Permission.READ) {
                limited.show();
            } else {
                limited.hide();
            }
            onlyOther.show();
            if (cwd.getPermission() >= Picasso.Permission.ADMIN) {
                folderShare.show();
            } else {
                folderShare.hide();
            }
        } else {
            onlyRoot.show();
            limited.hide()
            onlyOther.hide();
            folderShare.hide();
        }
    };

    /**
     * Filter files for gallery view
     *
     * @param {Array}  files  Array of {@link Picasso.File}
     *
     * @returns {Array} Filtered list; might be empty
     **/

    var filterFileGallery = function (files) {
        var filtered = files.filter(function (file) {
            return (file instanceof Picasso.File &&
                file.is(Picasso.File.Flags.FILE) && file.isImage());
        });

        /* Check whether images can be found */
        if (0 === filtered.length) {
            Picasso.Notification.show(
                Picasso.Lang.get("notification_error_no_images_found"), "warning")
        }

        return filtered;
    };

    /**
     * Filter files for video player
     *
     * @param {Array}  files  Array of {@link Picasso.File}
     *
     * @returns {Array} Filtered list; might be empty
     **/

    var filterFileVideoPlayer = function (files) {
        var filtered = files.filter(function (file) {
            return (file instanceof Picasso.File &&
                    file.is(Picasso.File.Flags.FILE) && file.isVideo());
        });

        /* Check whether videos can be found */
        if (0 === filtered.length) {
            Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_no_videos_found"), "warning")
        }

        return filtered;
    };

    /**
     * Filter files for audio player
     *
     * @param {Array}  files  Array of {@link Picasso.File}
     *
     * @returns {Array} Filtered list; might be empty
     **/

    var filterFileAudioPlayer = function (files) {
        var filtered = files.filter(function (file) {
            return (file instanceof Picasso.File &&
                    file.is(Picasso.File.Flags.FILE) && file.isAudio());
        });

        /* Check whether audios can be found */
        if (0 === filtered.length) {
            Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_no_audios_found"), "warning")
        }

        return filtered;
    };


    function processItem(file) {
        var data = {
            name:file.getName(),
            sources: [
                {
                    src:file.getDownloadUrl(),
                    type: 'video/mp4'
                }
            ]
        }
        return data;
    }

    /**
     * Build playlist compatible with player
     *
     * @param {Array}  files  Array of {@link Picasso.File}
     * @param {File}  file to be a the beggining of the list
     *
     * @returns {Array}  Playlist format
     **/

    var buildplayList = function (files,file) {
        var playlist = [];

        files.forEach(function (file) {
            playlist.push(processItem(file));
        });

        playlist.sort(function (x, y) {
            return x.name == file.getName() ? -1 : y.name == file.getName() ? 1 : 0;
        });

        return playlist;
    };

    /**
     * Get selected file
     *
     * @param {jQuery|HTMLElement}  elem  Element to check
     *
     * @returns {*} Either found #{@link Picasso.File}; otherwise null
     **/

    var getFileFromElem = function (elem) {
        var file = null;
        var serial = $(elem).find("[data-serial]");

        if (0 < serial.length) {
            file = new Picasso.File(serial.data("serial"));
        }

        return file;
    };

    /**
     * Handle click on share button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickShare = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get file */
        var selectedFile = getFileFromElem($(this).closest("tr"));

        if (selectedFile) {
            Picasso.set("selectedFile", selectedFile);

            /* Share dialog */
            var share = new Picasso.Dialog.Share(selectedFile);

            share.setOkHandler(function () {

                /* Update entry */
                $("#" + selectedFile.getElemId()).replaceWith(
                    renderFileList(selectedFile));
            });

            share.show();
        }
    };

    /**
     * Handle click on folder share button (shares the current folder/cwd)
     *
     * @param {Event}  e  Click event
     **/

    var handleClickFolderShare = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var cwd = Picasso.get("cwd");

        if (cwd) {
            Picasso.set("selectedFile", cwd);

            var share = new Picasso.Dialog.Share(cwd);
            share.show();
        }
    };

    /**
     * Handle click on element name
     *
     * @param {Event}  e  Click event
     **/

    var handleClickName = function (e) {
        e.preventDefault();
        e.stopPropagation();

        $("[data-toggle='tooltip']").tooltip("hide");

        var file = new Picasso.File($(this).data("serial"));

        history.pushState($(this).data("serial"), file.getName(),"");

        /* Check if file is locked and is an OO document */
        if (file.getWebSyncStatus() === "locked" && file.isOODocument()) {
            Picasso.Notification.show(Picasso.Lang.get("notification_file_locked"), "warning");
            return;
        }

        /* Check type */
        if (file.is(Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY)) {
            VMTABLE.getTable().deselectAll();

            if (file.getIsBlocked()){
                e.preventDefault();
                Picasso.Notification.show(Picasso.Lang.get("notification_error_access_blocked"), "warning")
                return;
            }

            var currentDepth = $(".pica-crumb").not("#files_home").length;
            history.pushState({file: $(this).data("serial"), searchQuery: "", breadcrumbDepth: currentDepth + 1}, file.getName(),"");

            /* Clear search field when entering folder */
            $("#pica_action_search_input").val("");
            sessionStorage.removeItem("searchQuery");

            showFilesSpinner();

            Picasso.Files.fetch(file, false,

                /* Success */
                function () {
                    var a = Picasso.Actionbar.addCrumb(
                        file.getName(), handleClickCrumb);

                    a.data("file", file.serialize());
                },

                /* Error */
                function () {
                    if (file.is(Picasso.File.Flags.FEDERATED)) {
                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_federated_service_not_reachable"), "warning");
                    } else {
                        Picasso.debugLog(e);
                    }
                }
            );
        } else if (file.isImage()) { ///< Image for gallery
            Picasso.set("selectedFile", file);

            VMTABLE.showGallery();
        } else if (file.isVideo()) {
            var player;
            var dialog = new Picasso.Dialog("#pica_video_dialog",
                    Picasso.Lang.get("dialog_title_video"));

            dialog.setAssignHandler(function () {
                player = videojs('video-player');

                var filtred = filterFileVideoPlayer(VMTABLE.getCache()) ;

                var playlist = buildplayList(filtred,file)
                console.log(playlist);

                player.playlist(playlist);

                player.playlist.autoadvance(0);

                player.playlistUi();


            });

            dialog.setCloseHandler(function () {
                player.reset();
            });

            $('#pica_video_dialog').on('hidden.bs.modal', function () {
                player.reset();
            });

            dialog.show()
        } else if (file.isAudio()) {

            var audioPlayer ;
            var dialog = new Picasso.Dialog("#pica_audio_dialog",
                    Picasso.Lang.get("dialog_title_audio",file.getName()));

            dialog.setAssignHandler(function () {
                audioPlayer = videojs('audio-player');
                audioPlayer.src({type: 'audio/mp3', src: file.getDownloadUrl()});

            });

            dialog.setCloseHandler(function () {
                audioPlayer.reset();
            });

            $('#pica_audio_dialog').on('hidden.bs.modal', function () {
                player.reset();
            });

            dialog.show()
        }

        else { ///< Open
            var url = file.getOpenUrl() ? file.getOpenUrl() : file.getDownloadUrl();

            if (Picasso.get("isFederated")) {
                url += "?CSRFToken=" + Picasso.get("CSRF");
            }

            if (file.isInline() && !file.isOODocument()){
                url = file.getDownloadUrl()
                if (-1 === url.indexOf("?")) {
                    url += "?inline=true";
                } else {
                    url += "&inline=true";
                }
            }

            /* Create a new anchor and simulate click to download */
            var a = $("<a>", {
                href:   url,
                target: "_blank"
            });

            $("body").append(a);
            a[0].click();
            a.remove();

            if (file.isOODocument()) {
                setTimeout(function() {
                    refreshFileSyncStatus(file);
                }, 2 * SYNC_STATUS_REFRESH_DELAY);
            }
        }
    };

    var refreshFileSyncStatus = function(file) {
        var cwd = Picasso.get("cwd");
        if (!cwd || !cwd.getFolderIdEnc()) return;

        var row = $("#" + file.getElemId());
        if (!row.length) return;

        var params = {
            action: "getLockStatus",
            fileName: file.getName()
        };
        var subdir = cwd.getPath();
        if (subdir) {
            params.directory = subdir;
        }

        Picasso.Helper.fireAjax("/wapi/folders/" + cwd.getFolderIdEnc(), params, function(data) {
            if (!data || !data.Result) return;

            var status = null;
            for (var i = 0; i < data.Result.length; i++) {
                if (data.Result[i].relativeName === file.getPath() ||
                    data.Result[i].relativeName.endsWith("/" + file.getName()) ||
                    data.Result[i].relativeName === file.getName()) {
                    status = data.Result[i].webSyncStatus;
                    break;
                }
            }

            updateFileSyncStatusIcon(file, status);

            if (status) {
                setTimeout(function() {
                    refreshFileSyncStatus(file);
                }, SYNC_STATUS_REFRESH_DELAY);
            }
        }, function(data) {});
    };

    var updateFileSyncStatusIcon = function(file, status) {
        var row = $("#" + file.getElemId());
        if (!row.length) return;

        var nameWrapper = row.find(".folder-name-wrapper");
        nameWrapper.find(".file-editing-icon, .file-locked-icon, .file-syncing-icon").remove();

        if (!status) return;

        var iconConfig = {
            edit: { icon: "pencil", css: "file-editing-icon", tooltip: "tooltip_file_editing" },
            locked: { icon: "lock", css: "file-locked-icon", tooltip: "tooltip_file_locked" },
            syncing: { icon: "repeat", css: "file-syncing-icon", tooltip: "tooltip_file_syncing" }
        };

        var config = iconConfig[status];
        if (config) {
            var statusIcon = $("<span>", {
                class: "pica-glyph glyphicons glyphicons-" + config.icon + " " + config.css
            });
            Picasso.Helper.addTooltip(statusIcon, Picasso.Lang.get(config.tooltip));
            nameWrapper.prepend(statusIcon);
        }
    };

    /**
     * Handle click on invitation name
     *
     * @param {Event}  e  Click event
     */

    var handleClickInvitation = function (e) {

        e.preventDefault();
        e.stopPropagation();

        var invitation = new Picasso.Invitation($(this).data("serial"));

        var dialog = new Picasso.Dialog.Invitation(Picasso.Lang.get("dialog_title_invitation"),
            Picasso.Lang.get("dialog_body_invitation"));

        dialog.setInvitation(invitation);
        dialog.show();
    };

     /**
     * Handle invitation options
     *
     * @param {Number}  idx  Option index
     **/

    var handleInvitationOptions = function (idx) {
        /* We invoke the invitation constructor here, because we just
         * need the OID for #accept/#decline */
        var invitation = new Picasso.Invitation(
            $(this).parents("tr").find(".pica-name").data("serial"));

        if (invitation) {
            var cwd = Picasso.get("cwd");

            var onError = function (e) {
                Picasso.debugLog(e);

                Picasso.Notification.show(
                    Picasso.Lang.get("notification_invitation_error"), "warning")
            };

            switch (idx) {
                case 0:
                    invitation.accept(

                        /* Success */
                        function () {
                            Picasso.Notification.show(Picasso.Lang.get(
                                "notification_invitation_accepted"), "info")

                            Picasso.Files.fetch(cwd);
                        }, onError
                    );
                    break;

                case 2:
                    invitation.decline(

                        /* Success */
                        function () {
                            Picasso.Notification.show(Picasso.Lang.get(
                                "notification_invitation_declined"), "info")

                            Picasso.Files.fetch(cwd);
                        }, onError
                    );
                    break;

                default: break;
            }
        }
    };

    /**
     * Handle file drag drop
     *
     * @param {Event}         e       Drag event
     * @param {Picasso.File}  target  A #{@link Picasso.File}
     **/

    var handleFileDrop = function (e, target) {
        /* Check for files */
        if (0 < e.originalEvent.dataTransfer.files.length) {
            var upload = new Picasso.Dialog.Upload(Picasso.Lang.get("dialog_title_upload"),
                Picasso.Lang.get("dialog_body_upload", target.getName()));

            upload.setUrl("/wapi/upload/" + target.getFolderIdEnc());
            upload.setData({
                folderID: target.getFolderId(),
                path: target.getPath()
            });

            upload.setStartAutomatically(true);

            upload.setDoneHandler(function () {


                    /* Reload listing when target is current path */
                    var cwd = Picasso.get("cwd");

                    if (cwd && cwd.equals(target)) {
                        Picasso.Files.fetch(target);

                    /* Only add crumb for sub dirs */
                    } else if (cwd && !cwd.equals(target)) {
                        var onSuccess = function () {
                            var a = Picasso.Actionbar.addCrumb(
                                target.getName(), handleClickCrumb);

                            a.data("file", target.serialize());
                        };

                        Picasso.Files.fetch(target, false, onSuccess, null);
                    }
                }
            );

            /* Enqueue files from event */
            var list = e.originalEvent.dataTransfer.files;

            /* Check support */
            if (e.originalEvent.dataTransfer.items) {
                list = e.originalEvent.dataTransfer.items;
            }

            for (var i = 0; i < list.length; i++) {
                upload.enqueue(list[i]);
            }

            upload.toggleButtons();
            upload.show();

            if (upload._autostart) {
                setTimeout(function() {
                    if (upload._queue && upload._queue.length > 0) {
                        $("#upload_start_button").trigger("click");
                    }
                }, 500);
            }
        }
    };

    /**
     * Handle drag start event
     *
     * @param {Event}  e  Drag event
     **/

    var handleRowDragStart = function (e) {

        /* Get dragged/selected elements */
        var draggedFiles = VMTABLE.getTable().getSelectedRows();

        if (0 === draggedFiles.length) {
            draggedFiles = [ getFileFromElem(this) ];
        } else {
            draggedFiles = $.map(draggedFiles, function (val) {
                return getFileFromElem(val);
            });
        }

        if (0 < draggedFiles.length) {
            Picasso.set("draggedFiles", draggedFiles);

            /* Create ghost */
            var ghost = $("<div>", {
                id: "pica_ghost",
                class: "alert alert-default"
            });

            if (1 === draggedFiles.length) {
                /* Copy data from icon and name column */
                var tds = $(this).find("td");

                ghost.append($(tds[0]).html());
                ghost.append($("<span>", {
                    html: $(tds[1]).html()
                }));
            } else {
                /* Create sum up message */
                ghost.append(Picasso.Helper.createGlyph("more-items", "pica-glyph"));
                ghost.append($("<span>", {
                    text: Picasso.Lang.get(
                        "string_selected_elements", draggedFiles.length)
                }));
            }

            $("body").append(ghost);

            /* Set transfer data */
            if (e.originalEvent.dataTransfer) {
                e.originalEvent.dataTransfer.setDragImage(ghost[0], -10, -10);
                e.originalEvent.dataTransfer.setData("Text", draggedFiles.length);
                e.originalEvent.dataTransfer.effectAllowed = "all";
            }
        }
    };

    /**
     * Handle drag enter event
     *
     * @param {Event}  e  Drag event
     **/

    var handleRowDragEnter = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var selectedFile = getFileFromElem(this);
        var tbody = $("#files_files_table tbody");

        if (selectedFile) {
            /* Either mark folder or complete table */
            if (selectedFile.is(Picasso.File.Flags.FOLDER |
                    Picasso.File.Flags.DIRECTORY))
            {
                $(this).addClass(Picasso.Table.Marker.DRAG_YES);

                tbody.removeClass("pica-drag-yes");
                tbody.removeClass("pica-drag-no");
            } else {
                var cwd = Picasso.get("cwd");

                if (cwd) {
                    tbody.addClass("pica-drag-yes");
                } else {
                    tbody.addClass("pica-drag-no");
                }
            }

        }

        return false;
    };

    /**
     * Handle drag over event
     *
     * @param {Event}  e  Drag event
     **/

    var handleRowDragOver = function (e) {
        e.preventDefault();
        e.stopPropagation();

        e.originalEvent.dataTransfer.dropEffect = "copy";

        return false;
    }

    /**
     * Handle drag leave event
     *
     * @param {Event}  e  Drag event
     **/

    var handleRowDragLeave = function (e) {
        var rect = this.getBoundingClientRect();

        /* Ignore leave events that happen when moving into a child element */
        if (e.clientY < rect.top || e.clientY >= rect.bottom ||
                e.clientX <= rect.left || e.clientX >= rect.right)
        {
            var that = $(this);

            that.removeClass(Picasso.Table.Marker.DRAG_YES);
            that.removeClass(Picasso.Table.Marker.DRAG_NO);
        }
    };

    /**
     * Handle drag leave event
     *
     * @param {Event}  e  Drag event
     **/

    var handleRowDragDrop = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var draggedFiles = Picasso.get("draggedFiles");
        var selectedFile = getFileFromElem(this);

        /* Move file or upload from external */
        if (draggedFiles && 0 < draggedFiles.length && selectedFile &&
                selectedFile.is(Picasso.File.Flags.DIRECTORY))
        {
            $.each(draggedFiles, function (idx, file) {
                file.move(selectedFile, null,

                    /* Success */
                    function () {
                        Picasso.Helper.fadeOut($("#" + file.getElemId()),
                            function () {
                                this.remove();

                                VMTABLE.checkIfEmpty();
                            });

                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_move_success",
                                file.getName(), selectedFile.getName()), "info");
                    },

                    /* Error */
                    function (e, status) {
                        if (file.is(Picasso.File.Flags.FEDERATED)) {
                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_federated_service_not_reachable"), "warning");
                        } else {
                            switch (status) {
                                case 409:
                                    Picasso.Notification.show(
                                        Picasso.Lang.get("notification_move_error",
                                            file.getName(), selectedFile.getName()),
                                        "warning");
                                    break;
                            }

                            Picasso.debugLog(e);
                        }
                    }
                );
            });

            VMTABLE.getTable().deselectAll();
        } else if (selectedFile) { ///< Drop from external
            /* Either upload to selected folder/directory or just to cwd */
            if (selectedFile.is(Picasso.File.Flags.FOLDER |
                Picasso.File.Flags.DIRECTORY))
            {
                handleFileDrop(e, selectedFile);
            } else {
                var cwd = Picasso.get("cwd");

                if (cwd) {
                    handleFileDrop(e, cwd);
                }
            }

            handleDragEnd(); ///< Call this manually here
        }
    };

    /**
     * Handle dropzone drag over event
     *
     * @param {Event}  e  Drag event
     **/

    var handleDropzoneDragOver = function (e) {
        e.stopPropagation();
        e.preventDefault();

        /* Update markers */
        var tbody = $("#files_files_table tbody");

        var cwd = Picasso.get("cwd");

        if (cwd) {
            tbody.addClass("pica-drag-yes");
        } else {
            tbody.addClass("pica-drag-no");
        }

        e.originalEvent.dataTransfer.dropEffect = "copy";
    };

    /**
     * Handle dropzone leave event
     *
     * @param {Event}  e  Drag event
     **/

    var handleDropzoneDragLeave = function (e) {
        e.stopPropagation();
        e.preventDefault();

        /* Apparently browsers send 0/0 when drag was aborted with Esc */
        if (0 === e.originalEvent.x && 0 === e.originalEvent.y) {
            var tbody = $("#files_files_table tbody");

            tbody.removeClass("pica-drag-yes");
            tbody.removeClass("pica-drag-no");
        }
    };

    /**
     * Handle dropzone drop event
     *
     * @param {Event}  e  Drag event
     **/

    var handleDropzoneDrop = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var cwd = Picasso.get("cwd");

        if (cwd) {
            handleFileDrop(e, cwd);
        }

        /* Remove markers */
        var tbody = $("#files_files_table tbody");

        tbody.removeClass("pica-drag-yes");
        tbody.removeClass("pica-drag-no");
    };

    /**
     * Handle drag end event
     **/

    var handleDragEnd = function () {

        /* Tidy up */
        $("#files_files_table tr").removeClass(
            Picasso.Table.Marker.DRAG_YES).removeClass(
                Picasso.Table.Marker.DRAG_NO);

        $("#files_files_table tbody").removeClass("pica-drag-yes")
            .removeClass("pica-drag-no");

        $("#pica_ghost").remove();

        Picasso.unset("draggedFiles");
    };

    /**
     * Handle click on bread bar
     **/

    var handleClickCrumb = function () {

        var file = $(this).data("file");

        // Calculate breadcrumb depth: count crumbs before this one
        var breadcrumbDepth = $(this).prevAll(".pica-crumb").not("#files_home").length;
        history.pushState({file: file, searchQuery: "", breadcrumbDepth: breadcrumbDepth}, file ? new Picasso.File(file).getName() : "Home","");

        if (file) {
            file = new Picasso.File(file);
        }

        showFilesSpinner();

        Picasso.Files.fetch(file);

        /* Clear search field */
        $("#pica_action_search_input").val("");
        sessionStorage.removeItem("searchQuery");
    };

    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDelete = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = VMTABLE.getTable().getSelectedRows();

        selected = $.map(selected, function (val) {
            return getFileFromElem(val);
        });

        DeleteFile(selected);
    };

    /**
     * Handle click on leave button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickLeave = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = VMTABLE.getTable().getSelectedRows();

        selected = $.map(selected, function (val) {
            return getFileFromElem(val);
        });

        leaveFolder(selected);
    };

    /**
     * Handle click on restore button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickRestore = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var selected = VMTABLE.getTable().getSelectedRows();

        if (1 === selected.length) {
            var file = getFileFromElem(selected[0]);

            restoreFile(file);
        }
    };

    /**
     * Handle click on create dir buttons
     **/

    var handleClickCreateDir = function () {
        var cwd = Picasso.get("cwd");
        var itemType = Picasso.Lang.get((cwd ? "label_directories" : "label_folders"));

        if (!cwd) {
            $(".pica-button-encryption").show();
        }else {
            $(".pica-button-encryption").hide();
        }

        var dialog = new Picasso.Dialog("#pica_input_dialog",
            Picasso.Lang.get("dialog_title_create"),
            Picasso.Lang.get("dialog_body_create", itemType),
            (Picasso.Dialog.Flags.HIDE_ON_ENTER|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.NO_AUTOCLOSE|
                Picasso.Dialog.Flags.SPINNER));

        dialog.setOkHandler(function (inputs) {

            var name = inputs.pica_input_text.val();
            var encrypted = false;
            if (inputs.hasOwnProperty("pica_folder_encrypted")) {
                encrypted = inputs.pica_folder_encrypted.is(':checked');
            }

            var cwd = Picasso.get("cwd");

            /* Success handler for top-level folder creation */
            var onSuccessRootDir = function (response) {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_created",
                        itemType), "info");

                dialog.hide();

                if (response && response.folder) {
                    var newFolder = new Picasso.File(response.folder);
                    VMTABLE.append([newFolder]);
                    VMTABLE.getTable()._htmlTable
                        .find("a.pica-name")
                        .filter(function () {
                            const data = $(this).data("serial");
                            return data && data._folderId === response.folder.ID;
                        })
                        .click();
                } else {
                    Picasso.Files.fetch(cwd);
                }
            };

            /* Success handler for subdirectory creation */
            var onSuccessSubDir = function (response) {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_created",
                        itemType), "info");

                dialog.hide();

                if (response && response.file) {
                    var newDir = new Picasso.File(response.file);
                    VMTABLE.append([newDir]);
                    VMTABLE.getTable()._htmlTable.find("a.pica-name[data-serial='" + newDir.serialize() + "']").click();
                } else {
                    Picasso.Files.fetch(cwd);
                }
            };

            /* Error */
            var onError = function (e, status) {
                dialog.showSpinner(false);

                switch (status) {
                    case 406:
                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_error_illegal_chars"),
                            "warning");
                        break;

                    case 409:
                        Picasso.Notification.show(Picasso.Lang.get(
                            "notification_error_duplicate",
                            itemType), "warning");
                        break;

                    case 412:
                        Picasso.Notification.show(Picasso.Lang.get(
                            "notification_error_create_overuse_detected",
                            name), "warning");
                        break;

                    case 400:
                        /* Falls through */
                    case 500:
                        Picasso.Notification.show(Picasso.Lang.get(
                            "notification_error_create",
                            itemType), "danger");
                }

                Picasso.debugLog(e);
            };

            if (null == cwd) {
                Picasso.File.createRootDir(name, encrypted, onSuccessRootDir, onError);
            } else {
                cwd.createSubDir(name, onSuccessSubDir, onError);
            }
        });

        dialog.setValidateFunction(function (inputs) {
            var documentName = inputs.pica_input_text.val();

            var disallowedCharactersRegex = /[\\\/:\*\?„<>]/;

            if (disallowedCharactersRegex.test(documentName)) {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_invalid_chars"), "warning");
                return false;
            }

            if (documentName.length > 200) {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_name_too_long"), "warning");
                return false;
            }

            var cwd = Picasso.get("cwd");
            if (cwd) {
                var fullPath = cwd.getPath() + "/" + documentName;
                if (fullPath.length > 3500) {
                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_error_path_too_long"), "warning");
                    return false;
                }
            }

            return true;
        });

        dialog.show();
    };

    /**
     * Handle click on create doc buttons
     **/

    var handleClickCreateDoc = function (e) {
        e.preventDefault();
        e.stopPropagation();

        $(".pica-button-encryption").hide();

        var a = $(this).find("a");

        var docType = a.data("type");
        var docLabel = a.data("label");

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_input_dialog",
            Picasso.Lang.get("dialog_title_create"),
            Picasso.Lang.get("dialog_body_create",
                Picasso.Lang.get("label_" + docLabel)),
            (Picasso.Dialog.Flags.HIDE_ON_ENTER|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.NO_AUTOCLOSE));

        dialog.setOkHandler(function (inputs) {
            var cwd = Picasso.get("cwd");
            var name = encodeURIComponent(inputs.pica_input_text.val() )+ "." + docType;

            /* Check whether file already exists */
            cwd.exists(name,

                /* Success */
                function (exists) {
                    if (exists) {
                        Picasso.Notification.show(Picasso.Lang.get(
                            "notification_error_duplicate", name), "warning");
                    } else {
                        /* Assemble url */
                        var url = "/open/" + Picasso.Helper.joinPaths([
                            cwd.getFolderIdEnc(), cwd.getPath(), name
                        ]);

                        /* Create a new anchor and simulate click to download */
                        var a = $("<a>", {
                            href: url,
                            target: "_blank"
                        });

                        $("body").append(a);
                        a[0].click();
                        a.remove();

                        dialog.hide();

                        /* Reload files after delay to allow document to be created,
                           then start sync status polling for the new file */
                        var decodedName = decodeURIComponent(name);
                        var cwdFolderId = cwd.getFolderIdEnc();
                        setTimeout(function() {
                            var currentCwd = Picasso.get("cwd");
                            if (!currentCwd || currentCwd.getFolderIdEnc() !== cwdFolderId) {
                                return;
                            }
                            Picasso.Files.fetch(cwd, false, function(files) {
                                /* Find the newly created file and start sync status polling */
                                for (var i = 0; i < files.length; i++) {
                                    if (files[i].getName() === decodedName) {
                                        refreshFileSyncStatus(files[i]);
                                        break;
                                    }
                                }
                            });
                        }, SYNC_STATUS_REFRESH_DELAY);
                    }
                },

                /* Error */
                function (e, status) {
                    Picasso.debugLog(e);
                }
            );

            return;


        });

        dialog.setValidateFunction(function (inputs) {
            var documentName = inputs.pica_input_text.val();

            var disallowedCharactersRegex = /[\\\/:\*\?„<>]/;

            if (disallowedCharactersRegex.test(documentName)) {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_invalid_chars"), "warning");
                return false;
            }


            if (documentName.length > 200) {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_name_too_long"), "warning");
                return false;
            }

            var cwd = Picasso.get("cwd");
            if (cwd) {
                var fullPath = cwd.getPath() + "/" + documentName + "." + docType;
                if (fullPath.length > 3500) {
                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_error_path_too_long"), "warning");
                    return false;
                }
            }

            return true;
        });

        dialog.show();
    };

    /**
     * Handle click on rename buttons
     *
     * @param {Event}  e  Click event
     **/

    var handleClickRename = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var selected = VMTABLE.getTable().getSelectedRows();

        if (1 === selected.length) {
            var file = getFileFromElem(selected[0]);

            renameFile(file);
        }
    };

    /**
     * Handle click on copy buttons
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCopy = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var selected = VMTABLE.getTable().getSelectedRows();

        if (1 === selected.length) {
            var file = getFileFromElem(selected[0]);
            copyFile(file);
        }
    };

    var copyFile = function (file) {
        var oldName = (file.getIntName() || file.getName());

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_input_dialog",
            Picasso.Lang.get("dialog_title_copy"),
            Picasso.Lang.get("dialog_body_copy", Picasso.Lang.get("label_name")),
            Picasso.Dialog.Flags.VALIDATE);

        dialog.setAssignHandler(function (inputs) {
            if (file.is(Picasso.File.Flags.FILE)) {
                var nameOnly = oldName.replace(/\.[^/.]+$/, ""); // remove extension
                var ext = oldName.slice(nameOnly.length);        // get extension
                inputs.pica_input_text.val(Picasso.Lang.get("string_copy", nameOnly));
                inputs.pica_input_text.data("original-extension", ext); // store extension
            } else {
                inputs.pica_input_text.val(Picasso.Lang.get("string_copy", oldName));
                inputs.pica_input_text.removeData("original-extension");
            }
        });



        dialog.setOkHandler(function (inputs) {
            var baseName = inputs.pica_input_text.val();
            var newName = baseName;
            if (file.is(Picasso.File.Flags.FILE)) {
                baseName = baseName.replace(/\.[^/.]+$/, "");
                var ext = inputs.pica_input_text.data("original-extension") || "";
                newName = baseName + ext;
            }

            var cwd = Picasso.get("cwd");

            file.move(cwd, newName,

                /* Success */
                function () {
                    Picasso.Notification.show(Picasso.Lang.get(
                        "notification_move_copied",
                        oldName, newName), "info");

                    Picasso.Files.fetch(cwd);
                },

                /* Error */
                function (e, status) {
                    switch (status) {

                        case 403:
                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_copy_error_invalid_path",
                                    newName, cwd.getName()), "warning");
                            break;

                        case 404:
                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_copy_error_directory_not_found",
                                    newName, cwd.getName()), "warning");
                            break;

                        case 409:
                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_move_error",
                                    newName, cwd.getName()), "warning");
                            break;
                        case 420:
                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_error_storage_exceeded"), "warning");
                        break;
                    }

                    Picasso.debugLog(e);
                }
            );
        });

        dialog.setValidateFunction(function (inputs) {
            var baseName = inputs.pica_input_text.val();
            var newName = baseName;
            if (file.is(Picasso.File.Flags.FILE)) {
                baseName = baseName.replace(/\.[^/.]+$/, "");
                var ext = inputs.pica_input_text.data("original-extension") || "";
                newName = baseName + ext;
            }

            var disallowedCharactersRegex = /[\\\/:\*\?„<>]/;

            if (disallowedCharactersRegex.test(newName)) {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_illegal_chars"), "warning");
                return false;
            }

            if (newName.length > 200) {
                Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_name_too_long"), "warning");
                return false;
            }

            var cwd = Picasso.get("cwd");
            if (cwd) {
                var fullPath = cwd.getPath() + "/" + newName;
                if (fullPath.length > 3500) {
                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_error_path_too_long"), "warning");
                    return false;
                }
            }

            return true;
        });

        dialog.show();
    };

    /**
     * Handle click on paste buttons
     *
     * @param {Event}  e  Click event
     **/

    var handleClickPaste = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var cutFiles = Picasso.get("cutFiles");

        if (0 < cutFiles.length) {
            var cwd = Picasso.get("cwd");

            $.each(cutFiles, function (idx, file) {
                file.move(cwd, null,

                    /* Success */
                    function () {
                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_move_success",
                                file.getName(), cwd.getName()), "info");

                        Picasso.Files.fetch(cwd);
                    },

                    /* Error */
                    function (e, status) {
                        if (file.is(Picasso.File.Flags.FEDERATED)) {
                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_federated_service_not_reachable"), "warning");
                        } else {
                            switch (status) {
                                case 405:
                                    Picasso.Notification.show(
                                            Picasso.Lang.get("notification_error_file_under_edit"),
                                            "warning");
                                    break;
                                case 409:
                                    Picasso.Notification.show(
                                        Picasso.Lang.get("notification_move_error",
                                            file.getName(), cwd.getName()), "warning");
                                    break;
                                case 420:
                                    Picasso.Notification.show(
                                        Picasso.Lang.get("notification_error_storage_exceeded"), "warning");
                                    break;
                            }
                            Picasso.debugLog(e);
                        }
                    }
                );
            });

            Picasso.unset("cutFiles");
            handleOnSelection(0, null);
        }
    };

    /**
     * Handle click on copy buttons
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCut = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var cutFiles = VMTABLE.getTable().getSelectedRows();

        cutFiles = $.map(cutFiles, function (val) {
            return getFileFromElem(val);
        });

        if (cutFiles && 0 < cutFiles.length) {

            //resend folder stable to home
            $("lang[name='"+"label_folders"+"']").parent().on('click', function(event) {
                event.preventDefault();
                $("#files_home").click();
            });

            Picasso.set("cutFiles", cutFiles);

            var header = $("#files_paste_header");
            var iconTh = header.find("th.pica-icon-column");
            var nameTh = header.find("th.pica-table-selection-title");

            if (1 === cutFiles.length) {
                iconTh.html(cutFiles[0].getIcon());
                nameTh.html($("<span>", {
                    text: cutFiles[0].getName()
                }));
            } else {
                iconTh.html(Picasso.Helper.createGlyph(
                    "more-items", "pica-glyph"));
                nameTh.html($("<span>", {
                    text: Picasso.Lang.get(
                        "string_selected_elements", cutFiles.length)
                }));
            }

            /* Add cancel button */
            var a = $("<a>", {
                class: "hidden-xs",
                text: Picasso.Lang.get("action_cancel")
            });

            a.click(function () {
                Picasso.unset("cutFiles");

                handleOnSelection(0, null);

                Picasso.Notification.show(
                    Picasso.Lang.get("notification_cut_stop"), "info");
            });

            nameTh.append(a)

            Picasso.Notification.show(
                Picasso.Lang.get("notification_cut_start",
                    cutFiles[0].getName()), "info");

            /* Stop selection now */
            VMTABLE.getTable().deselectAll();
        }
    };

    var cutFile = function (file) {

        //resend folder stable to home
        $("lang[name='" + "label_folders" + "']").parent().on('click', function (event) {
            event.preventDefault();
            $("#files_home").click();
        });
        var cutFiles = [file]
        Picasso.set("cutFiles", cutFiles);

        var header = $("#files_paste_header");
        var iconTh = header.find("th.pica-icon-column");
        var nameTh = header.find("th.pica-table-selection-title");

        if (1 === cutFiles.length) {
            iconTh.html(cutFiles[0].getIcon());
            nameTh.html($("<span>", {
                text: cutFiles[0].getName()
            }));
        } else {
            iconTh.html(Picasso.Helper.createGlyph(
                "more-items", "pica-glyph"));
            nameTh.html($("<span>", {
                text: Picasso.Lang.get(
                    "string_selected_elements", cutFiles.length)
            }));
        }

        /* Add cancel button */
        var a = $("<a>", {
            class: "hidden-xs",
            text: Picasso.Lang.get("action_cancel")
        });

        a.click(function () {
            Picasso.unset("cutFiles");

            handleOnSelection(0, null);

            Picasso.Notification.show(
                Picasso.Lang.get("notification_cut_stop"), "info");
        });

        nameTh.append(a)

        Picasso.Notification.show(
            Picasso.Lang.get("notification_cut_start",
                cutFiles[0].getName()), "info");

    };

    /**
     * Handle click on download buttons
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDownload = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selectedFiles = VMTABLE.getTable().getSelectedRows();

        selectedFiles = $.map(selectedFiles, function (val) {
            return getFileFromElem(val);
        });

        if (0 < selectedFiles.length) {
            downloadFilesInBatches(selectedFiles, 5);
        }
    };

    function downloadFilesInBatches(files, batchSize) {
        var index = 0;

        function downloadBatch() {
            var batch = files.slice(index, index + batchSize);
            index += batchSize;

            if (batch.length > 0) {
                $.each(batch, function () {
                    downloadFile(this);
                });

                setTimeout(downloadBatch, 1000); // Wait 1 second before downloading the next batch
            }
        }

        downloadBatch();
    }

    /**
     * Handle click on upload buttons
     *
     * @param {Event}  e  Click event
     **/

    var handleClickUpload = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Check for current path */
        var cwd = Picasso.get("cwd");

        if (cwd) {
            var upload = new Picasso.Dialog.Upload(
                Picasso.Lang.get("dialog_title_upload"),
                Picasso.Lang.get("dialog_body_upload", cwd.getName()));

            upload.setUrl("/wapi/upload/" + cwd.getFolderIdEnc());
            upload.setData({
                folderID: cwd.getFolderId(),
                path: cwd.getPath()
            });

            upload.setAllowMultiple(true);
            upload.setAllowDirectories(true);
            upload.setStartAutomatically(true);

            upload.setDoneHandler(function () {
                    Picasso.Files.fetch(cwd);
                }
            );

            upload.show();
        }
    };

    /**
     * Handle click on settings button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickSettings = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var selected = VMTABLE.getTable().getSelectedRows();

        if (1 === selected.length) {
            var folder = getFileFromElem(selected[0]);

            settings(folder);
        }
    };

    /**
     * Handle filter event
     *
     * @param {String}  filterString  Input string for filter
     **/

    var handleFilter = function (filterString) {
        var cwd = Picasso.get("cwd");
        var currentDepth = $(".pica-crumb").not("#files_home").length;

        if (!filterString || filterString.length === 0) {
            sessionStorage.removeItem("searchQuery");
            history.replaceState({file: cwd ? cwd.serialize() : null, searchQuery: "", breadcrumbDepth: currentDepth}, "", "");
            showFilesSpinner();
            Picasso.Files.fetch(cwd);
            return;
        }

        sessionStorage.setItem("searchQuery", filterString);
        history.replaceState({file: cwd ? cwd.serialize() : null, searchQuery: filterString, breadcrumbDepth: currentDepth}, "", "");
        showFilesSpinner();
        Picasso.Files.fetch(cwd, null, null, null, filterString, true);

        VMTABLE.getTable().hideEmptyMessage();
    };

    /**
     * Render given json object for list
     *
     * @param {Picasso.File}  file  A {@link Picasso.File}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderFileList = function (file) {
        var tr = $("<tr>", {
            css:{
                position: "relative"
            },
            id: file.getElemId(),
            "data-search-keys": [file.getName(),file.getOid()],
            draggable: (!file.is(Picasso.File.Flags.FOLDER) &&
                !file.is(Picasso.File.Flags.INVITED_TO))
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "pica-icon-column" });

        var icon = file.getIcon();

        td.append(icon);

        /* Add thumbnail if found */
        if (file.isImage()) {
            var img = $("<img>", {
                class: "pica-icon-thumb-list",

                onLoad: function () {
                    icon.hide();
                    icon.after(this);
                },

                src: file.getThumbnailUrl()
            });

            img.on("error", function () {
                icon.show();
                this.remove();

                return true;
            });
        }

        tr.append(td);

        /* 2. column: Name */
        td = $("<td>", { class: "pica-overflow",colspan:2 });

        var nameWrapper = $("<span>", {
            class: "folder-name-wrapper",
        });

        var a = $("<a>", {
            class: "pica-name",
            text: file.getName(),
            href: file.getOpenUrl() ? file.getOpenUrl() : file.getDownloadUrl(),
            "data-serial": file.serialize()
        });

        var isSearching = $("#pica_action_search_input").val().trim().length > 0;
        var tooltipText = isSearching && file.getPath() ? file.getPath() : file.getName();
        Picasso.Helper.addTooltip(a, tooltipText);

        /* Bind listener */
        if (!file.is(Picasso.File.Flags.INVITED_TO)) {
            a.click(handleClickName);
        }

        if (file.is(Picasso.File.Flags.INVITED_TO)) {
            a.click(handleClickInvitation);
        }

        if (file.getIsStorageExceeded()) {
            var warningIcon = $("<span>", {
                class: "pica-glyph glyphicons glyphicons-warning-sign folder-warning-icon",
                title: file.getPermission() === Picasso.Permission.OWNER ? Picasso.Lang.get("tooltip_storage_warning_owner")
                    : Picasso.Lang.get("tooltip_storage_warning_others"),
                "data-placement": "auto"
            });

            warningIcon.tooltip({
                container: "body",
                placement: "auto",
            });

            nameWrapper.append(warningIcon);
        }

        if (file.getWebSyncStatus()) {
            var statusIcon;
            var statusTitle;

            if (file.getWebSyncStatus() === "edit") {
                statusIcon = $("<span>", {
                    class: "pica-glyph glyphicons glyphicons-pencil file-editing-icon"
                });
                statusTitle = Picasso.Lang.get("tooltip_file_editing");
            } else if (file.getWebSyncStatus() === "locked") {
                statusIcon = $("<span>", {
                    class: "pica-glyph glyphicons glyphicons-lock file-locked-icon"
                });
                statusTitle = Picasso.Lang.get("tooltip_file_locked");
            } else if (file.getWebSyncStatus() === "syncing") {
                statusIcon = $("<span>", {
                    class: "pica-glyph glyphicons glyphicons-repeat file-syncing-icon"
                });
                statusTitle = Picasso.Lang.get("tooltip_file_syncing");
            }

            if (statusIcon) {
                statusIcon.attr("title", statusTitle);
                statusIcon.tooltip({
                    container: "body",
                    placement: "auto",
                });
                nameWrapper.append(statusIcon);
            }
        }

        nameWrapper.append(a);

        td.append(nameWrapper);

        tr.append(td);

        /* 3. column: Size */
        var td3 = $("<td>");

        tr.append(td3);

        /* 4. column: Permission, Modified by */
        var td4 = $("<td>", { class: "hidden-xs" });

        tr.append(td4);

        /* 5. column: Members, Modified at */
        var td5 = $("<td>", { class: "hidden-xs" });

        tr.append(td5);

        /* 6. context menu */

        var td7 = $("<td>", {class: "pica-icon-column"});

        if (!file.is(Picasso.File.Flags.INVITED_TO)) {
            var menuButton = $("<a>", {
                class: "dropdown-toggle",
                "data-toggle": "dropdown",
            });

            var menuSpan = $("<a>", {
                class: "pica-glyph glyphicons glyphicons-more",
            });

            menuButton.append(menuSpan)


            var menuDropdownContainer = $("<div>", {
                class: "pica-inline text-right",
            });
            var menuDropdown = $("<ul>", {
                class: "pica-ontop conext-dropdown-menu dropdown-menu pull-right",
            });

            menuButton.click(function () {
                $('.conext-dropdown-menu').not(menuDropdown).each(function () {
                    $(this).closest("tr").removeClass("pica-table-selection-disabled");
                    $(this).hide();
                });

                if (menuDropdown.is(':visible')) {
                    menuDropdown.closest("tr").removeClass("pica-table-selection-disabled");
                } else {
                    VMTABLE.getTable().deselectAll();
                    menuDropdown.closest("tr").addClass("pica-table-selection-disabled");
                    menuDropdown.show();

                }

                menuDropdown.empty();

                populateDropdownMenu(file, menuDropdown);

            });


            menuDropdownContainer.append(menuButton);
            menuDropdownContainer.append(menuDropdown);
            td7.append(menuDropdownContainer);
        }
        tr.append(td7);

        /* 8. column: Share, Accept/Decline */
        var td8 = $("<td>", { class: "pica-icon-column" });

        tr.append(td8);

        /* Fill data based on flags */
        if (file.is(Picasso.File.Flags.INVITED_TO|Picasso.File.Flags.FOLDER)) {
            td4.append(Picasso.Lang.get("permission_you_" +
                Picasso.Permission.toString(file.getPermission()).toLowerCase()));
        }

        if (!file.is(Picasso.File.Flags.INVITED_TO)) { ///< All but..
            if (file.is(Picasso.File.Flags.FOLDER)) { ///< Root

                if (!file.is(Picasso.File.Flags.FEDERATED) && file.getSize() > 0) {
                    td3.append(Picasso.Helper.formatSize(file.getSize()));
                    td3.attr("data-sort", file.getSize());
                }

                var a = $("<a>", {
                    text: file.getDescription()
                });

                a.click(handleClickShare);

                td5.append(a);
            } else { ///< (Sub-) Directories and files
                if (file.is(Picasso.File.Flags.FILE)) {
                    td3.append(Picasso.Helper.formatSize(file.getSize()));
                    td3.attr("data-sort", file.getSize());
                }

                if (file.getModifiedBy() || file.getModifiedByAccount()) {
                    let modifiedBy = file.getModifiedByAccount() ? file.getModifiedByAccount() : file.getModifiedBy() ? file.getModifiedBy() : ""
                    let modifiedOn = file.getModifiedOn() ? Picasso.Lang.get("string_on", file.getModifiedOn()) : "";
                    var span = $("<span>", {
                        text: modifiedBy,
                        title: modifiedOn,
                        "data-tooltip": "tooltip_share",
                        "data-placement": "bottom"
                    });

                    span.tooltip({ container: "body" });

                    td4.append(span);
                } else {
                    td4.append(file.getModifiedOn());
                }

                /* Create tooltip */
                var span = $("<span>", {
                    text: Picasso.Helper.formatAge(file.getModifiedDateMS()),
                    title: Picasso.Helper.formatDate(file.getModifiedDateMS()),
                    "data-tooltip": "",
                    "data-placement": "bottom"
                });

                if (file.getModifiedDate()) {
                    td5.attr("data-sort",
                        new Date(file.getModifiedDate()).getTime()); ///< We need that for sorting
                }

                td5.append(span);

                span.tooltip({ container: "body" });
            }

            /* Add share button if allowed */
            var folderPermission = (Picasso.get("folderPermission") || 0);

            if (0 < ((file.getPermission()|folderPermission) &
                (Picasso.Permission.OWNER|Picasso.Permission.ADMIN)))
            {
                var a = $("<a>", {
                    class: "pica-action-link"
                });

                a.click(handleClickShare);

                if (!file.getIsBlocked()) {
                    /* Create tooltip */
                    var span = $("<span>", {
                        class: "pica-glyph glyphicons glyphicons-share-alt",
                        title: Picasso.Lang.get("tooltip_share"),
                        "data-tooltip": "tooltip_share",
                        "data-placement": "bottom"
                    });

                    span.tooltip({container: "body"});
                }else {
                    var span = $("<span>", {
                        class: "pica-glyph glyphicons glyphicons-alert",
                    });
                    Picasso.Helper.addTooltip(span,Picasso.Lang.get("account_folder_blocked"));
                }
                a.append(span);
                td8.append(a);
            }
        } else { ///< Invitation
            tr.addClass("pica-table-selection-disabled");

            var div = $("<div>", { class: "pica-inline dropdown" });

            var a = $("<a>", {
                class: "dropdown-toggle",
                "data-toggle": "dropdown",
                "data-placement": "bottom",
                "data-tooltip": "",
                "data-serial": file.serialize(),
                "title": Picasso.Lang.get("tooltip_manage_invitation")
            });

            a.tooltip({ container: "body" });
            a.click(handleClickInvitation);

            /* Add icon and caret */
            a.append(Picasso.Helper.createGlyph("directions", "pica-glyph"));
            a.append($("<span>", { class: "caret" }));

            div.append(a);

            /* Add dropdown menu */
            var menu = Picasso.Combo.createDropdownMenu(optionsForInvitations,
                handleInvitationOptions);

            menu.addClass("dropdown-menu-right");

            div.append(menu);
            td8.append(div);
        }

        /* Bind handler */
        if (!file.is(Picasso.File.Flags.INVITED_TO)) {
            tr.on("dragstart", handleRowDragStart);
            tr.on("dragenter", handleRowDragEnter);
            tr.on("dragover", handleRowDragOver);
            tr.on("dragleave", handleRowDragLeave);
            tr.on("drop", handleRowDragDrop);
            tr.on("dragend", handleDragEnd);
        }

        return tr;
    };
    function populateDropdownMenu(file, dropdownMenu) {
        var cwd = Picasso.get("cwd");
        var lowestPermission = 0;
        var collectedPermissions = 0;
        var collectedFileTypes = 0;

        /* Check for root */
        if (cwd) {
            lowestPermission = cwd.getPermission();
        } else {
            /* Only root folders in this path then */
            lowestPermission = Picasso.Permission.OWNER;
            collectedFileTypes = Picasso.File.Flags.FOLDER;
        }

        if (file) {
            /* Set initial values */
            lowestPermission = Picasso.Permission.OWNER; ///< Highest
            collectedFileTypes = 0; ///< Lowest

                var selectedFile = file;

                if (selectedFile) {
                    var permission = selectedFile.getPermission();

                    /* Find lowest common permission */
                    if (permission < lowestPermission) {
                        lowestPermission = permission;
                    }

                    collectedPermissions |= permission;

                    /* Keep all found file types */
                    collectedFileTypes |= (selectedFile.getFlags() &
                        (Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY|
                            Picasso.File.Flags.FILE));
                }
        }

        // Toggle visibility of UI actions
        var visibilities = getVisibilities(lowestPermission, collectedPermissions, collectedFileTypes, 1);

        if (visibilities & Picasso.Files.Visibility.DOWNLOAD) {
            dropdownMenu.append(createMenuItem(dropdownMenu, "files-ui-download", function () { downloadFile(file); }, "download", "action_download"));
        }
        if (visibilities & Picasso.Files.Visibility.RENAME) {
            dropdownMenu.append(createMenuItem(dropdownMenu, "files-ui-rename", function () { renameFile(file); }, "edit", "action_rename"));
        }
        if (visibilities & Picasso.Files.Visibility.COPY) {
            dropdownMenu.append(createMenuItem(dropdownMenu, "files-ui-copy", function () { copyFile(file); }, "copy", "action_copy"));
        }
        if (visibilities & Picasso.Files.Visibility.CUT) {
            dropdownMenu.append(createMenuItem(dropdownMenu, "files-ui-cut", function () { cutFile(file); }, "scissors-alt", "action_cut"));
        }
        if (visibilities & Picasso.Files.Visibility.RESTORE) {
            dropdownMenu.append(createMenuItem(dropdownMenu, "files-ui-restore", function () { restoreFile(file); }, "refresh", "action_restore"));
        }
        if (visibilities & Picasso.Files.Visibility.DELETE) {
            dropdownMenu.append(createMenuItem(dropdownMenu, "files-ui-delete", function () { DeleteFile([file]); }, "bin", "action_delete"));
        }
        if (visibilities & Picasso.Files.Visibility.LEAVE) {
            dropdownMenu.append(createMenuItem(dropdownMenu, "files-ui-leave", function () { leaveFolder([file]); }, "door", "action_leave"));
        }
        if (visibilities & Picasso.Files.Visibility.SETTINGS) {
            dropdownMenu.append(createMenuItem(dropdownMenu, "files-ui-settings", function () { settings(file); }, "settings", "action_manage"));
        }
    }

    var createMenuItem = function (dropdownMenu, className, handler, glyphIcon, langName) {
        var span = $("<li>", {
            class: "pica-table-selection-single",
        });
        var a = $("<a>", {
            class:className,
            href: "javascript:void(0);",
            role: "button"
        });

        a.append(Picasso.Helper.createGlyph(glyphIcon, "pica-glyph"));
        a.append(Picasso.Helper.createLang(langName));

        span.append(a);
        a.on("click", function (e) {
            e.preventDefault();
            handler();
            dropdownMenu.hide();
        });
        return span;
    };





    /**
     * Render given json object for grid
     *
     * @param {Picasso.File}  file  A #{@link Picasso.File}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderFileGrid = function (file) {
        var div = $("<div>", {
            id: file.getElemId(),
            class: "pica-flex-file pica-borderless text-center",
            "data-search-keys": [file.getName()],
            draggable: !file.is(
                Picasso.File.Flags.FOLDER|Picasso.File.Flags.INVITED_TO)
        });

        var p = $("<p>", {
            class: "pica-well dropdown",
        });

        /* Add icon link */
        var a = $("<a>", {
            "data-serial": file.serialize()
        });

        /* Add icon */
        var icon = file.getIcon();

        icon.addClass("pica-glyph-big");

        a.append(icon);

        /* Add thumbnail if any */
        if (file.isImage()) {
            var img = $("<img>", {
                class: "pica-icon-thumb-gallery",

                onLoad: function () {
                    icon.hide();
                    icon.after(this);
                },

                src: file.getThumbnailUrl()
            });
        }

        a.click(handleClickName);

        p.append(a);

        /* Add name link */
        a = $("<a>", {
            class: "pica-name pica-line-normal pica-overflow center-block dropdown-toggle",
            "data-toggle": "dropdown",
            text: file.getName()
        });

        /* Add handler to set visibilities based on permission */
        a.click(function () {

            /* Toggle visibility of ui actions */
            var visibilities = getVisibilities(file.getPermission(),
                file.getPermission(), (file.getFlags() &
                    (Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY|
                        Picasso.File.Flags.FILE)),
                1);

            toggleVisibility(visibilities);
        });

        /* Add caret and name */
        a.append($("<span>", { class: "caret" }));

        /* Add tooltip with path when searching */
        var isSearching = $("#pica_action_search_input").val().trim().length > 0;
        var tooltipText = isSearching && file.getPath() ? file.getPath() : file.getName();
        Picasso.Helper.addTooltip(a, tooltipText);

        p.append(a);

        /* Add dropdown menu */
        var menu = Picasso.Combo.createDropdownMenu(optionsForGallery,
            handleOptionsForGallery);

        p.append(menu);

        div.append(p);

        return div;
    };

    /**
     * Render given json object for gallery
     *
     * @param {Picasso.File}  file  A {@link Picasso.File}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderFileGallery = function (file) {
        /* Create slide */
        var div = $("<div>", {
            class: "pica-vmtable-gallery-slide"
        });
        //handle browsers not sypporting tiff images
        if (file.isTifImage()) {
            var xhr = new XMLHttpRequest();
            xhr.open('GET', file.getDownloadUrl());
            xhr.responseType = 'arraybuffer';
            xhr.onload = function (e) {
                var buffer = xhr.response;
                var tiff = new Tiff({buffer: buffer});
                var canvas = tiff.toCanvas();
                if (canvas) {
                    div.append(canvas);
                }
            };
            xhr.send();
        }else{
            div.append($("<img>", {
                src: file.getDownloadUrl()
            }));
        }


        /* Whether this image should be the active one */
        if (file.equals(Picasso.get("selectedFile"))) {
            div.addClass("active");
        }

        /* Create caption */
        var caption = $("<div>", {
            class: "pica-vmtable-gallery-caption"
        });

        caption.append($("<h3>", {
            text: file.getName()
        }));

        div.append(caption);

        return div;
    };




    /**
     * Fetch JSON data
     *
     * @param {Picasso.File}  file       A #{@link Picasso.File}
     * @param {Boolean}       append     Whether to append data
     * @param {Function}      onSuccess  Success handler
     * @param {Function}      onError    Error handler
     **/

    Picasso.Files.fetch = function (file, append, onSuccess, onError, query, recursive) {

        Picasso.set("cwd", file);
        var createDropdown = $("#pica-files-create-dropdown");

        createDropdown.show();

        /* Reset page */
        if (true !== append) {
            PAGE = 1;
            showFilesSpinner();
        }

        /* Update file list */
        var updateFiles = function (files, json) {

            /* Set federated services */
            if (json.hasOwnProperty("ResultSet") &&
                json.ResultSet.hasOwnProperty("fedServices")) {
                Picasso.set("fedServices", json.ResultSet.fedServices);
            }

            /* Check for folder permission */
            if (json.hasOwnProperty("ResultSet") &&
                    json.ResultSet.hasOwnProperty("permission"))
            {
                Picasso.set("folderPermission", Picasso.Permission.fromString(
                    json.ResultSet.permission));
            } else {
                Picasso.unset("folderPermission");
            }

            /* Call renderer before infinite scroll */
            if (true === append) {
                VMTABLE.append(files);
            } else {
                VMTABLE.update(files);
            }

            /* Start polling for files with active sync status */
            $.each(files, function(i, file) {
                if (file.getWebSyncStatus() && file.isOODocument()) {
                    setTimeout(function() {
                        refreshFileSyncStatus(file);
                    }, SYNC_STATUS_REFRESH_DELAY);
                }
            });

            if (!Picasso.get("cwd") && account.getMaxFolders() && account.getMaxFolders() > -1 &&
                files.length >= account.getMaxFolders()) {

                warnOveruse(files.length);
            }

            /* Bind infinite scroll */
            Picasso.Helper.bindInfiniteScroll(json, VMTABLE.getTable(), PAGE,
                function () {
                    PAGE++;

                    Picasso.Files.fetch(Picasso.get("cwd"), true,null,null,query);
                }
            );

            handleOnSelection();

            if (onSuccess) onSuccess(files);
        };

        /* Check whether sub or root level */
        if (file) {
            /* Update cwd, store folder ID and federated state */
            Picasso.set("folderId", file.getFolderId());
            Picasso.set("isFederated", file.is(Picasso.File.Flags.FEDERATED));

            /* Hide create dropdown if no permission */
            if (file.getPermission() === Picasso.Permission.READ) {
                createDropdown.hide();
            }

            file.getFiles(updateFiles, onError, PAGE,null,null,query, recursive);
        } else {
            const urlParams = new URLSearchParams(window.location.search);
            const filter = urlParams.get("filter")
            Picasso.File.getAll(updateFiles, onError, PAGE, null, null, query, filter);
        }
    };

    /* Visibility - must be defined after constructor */

    if (!window.Picasso.Files.hasOwnProperty("Visibility")) {
        window.Picasso.Files.Visibility = {};
    }

    Picasso.Files.Visibility.CREATE = (1 << 0);
    Picasso.Files.Visibility.DOWNLOAD = (1 << 1);
    Picasso.Files.Visibility.RENAME = (1 << 2);
    Picasso.Files.Visibility.COPY = (1 << 3);
    Picasso.Files.Visibility.CUT = (1 << 4);
    Picasso.Files.Visibility.RESTORE = (1 << 5);
    Picasso.Files.Visibility.DELETE = (1 << 6);
    Picasso.Files.Visibility.LEAVE = (1 << 7);
    Picasso.Files.Visibility.UPLOAD = (1 << 8);
    Picasso.Files.Visibility.SETTINGS = (1 << 9);

    /* Register view mode table callbacks */
    VMTABLE.getTable().setSelectionHandler(handleOnSelection);
    VMTABLE.setListRenderer(renderFileList);
    VMTABLE.setGridRenderer(renderFileGrid);
    VMTABLE.setGalleryRenderer(renderFileGallery);
    VMTABLE.setGalleryFilter(filterFileGallery);

    VMTABLE.getTable().showEmptyMessage = function() {
        var tbody = this._htmlTable.find("tbody");
        var cwd = Picasso.get("cwd");
        var isTopLevel = !cwd;
        var isSearching = $("#pica_action_search_input").val().trim().length > 0;
        var account = Picasso.get("account");
        var hasStorage = account && account.getQuotaTotal() > 0;
        var hasWritePermission = !cwd || cwd.getPermission() >= Picasso.Permission.READ_WRITE;

        var showEmptyState = (cwd || (isTopLevel && hasStorage)) && !isSearching;
        var showButtons = showEmptyState && hasWritePermission;
        var tr = $("<tr>", { class: showButtons ? "pica-table-empty pica-table-empty-root" : "pica-table-empty" });
        var td = $("<td>", {
            class: "text-center",
            colspan: this.getVisibleColsCount()
        });

        if (showButtons) {
            var container = $("<div>", {
                class: "pica-empty-folder-container"
            });

            var headingText = isTopLevel ? "string_no_folders" : "string_folder_empty";
            var subtextText = isTopLevel ? "string_no_folders_subtext" : "string_folder_empty_subtext";

            var heading = $("<h2>", {
                text: Picasso.Lang.get(headingText)
            });

            var subtext = $("<p>", {
                text: Picasso.Lang.get(subtextText)
            });

            container.append(heading, subtext);

            if (isTopLevel) {
                var createFolderBtn = $("<button>", {
                    class: "btn btn-primary",
                    text: Picasso.Lang.get("tooltip_create_folder")
                }).click(function(e) {
                    e.preventDefault();
                    handleClickCreateDir();
                });
                container.append(createFolderBtn);
            } else {
                var buttonsDiv = $("<div>", {
                    class: "pica-empty-folder-buttons"
                });

                var uploadBtn = $("<button>", {
                    class: "btn btn-primary",
                    text: Picasso.Lang.get("action_upload_files")
                }).click(function(e) {
                    e.preventDefault();
                    $(".files-ui-upload").trigger("click");
                });

                var createDocBtn = $("<button>", {
                    class: "btn btn-default",
                    text: Picasso.Lang.get("action_create_document")
                }).click(function(e) {
                    e.preventDefault();
                    $(".files-ui-create-doc").first().trigger("click");
                });

                var newFolderBtn = $("<button>", {
                    class: "btn btn-default",
                    text: Picasso.Lang.get("action_new_subdirectory")
                }).click(function(e) {
                    e.preventDefault();
                    $(".files-ui-create-dir").trigger("click");
                });

                buttonsDiv.append(uploadBtn, createDocBtn, newFolderBtn);
                container.append(buttonsDiv);
            }

            td.append(container);
        } else if (showEmptyState) {
            td.html(Picasso.Helper.createLang("string_folder_empty"));
        } else {
            td.html(Picasso.Helper.createLang("string_nothing_found"));
        }

        tr.append(td);
        tbody.html(tr);
    };

    /* Bind handlers */
    $("#files_home").click(handleClickCrumb);

    $(".files-ui-create-dir").click(handleClickCreateDir);
    $(".files-ui-create-doc").click(handleClickCreateDoc);
    $(".files-ui-rename").click(handleClickRename);
    $(".files-ui-cut").click(handleClickCut);
    $(".files-ui-copy").click(handleClickCopy);
    $(".files-ui-paste").click(handleClickPaste);
    $(".files-ui-delete").click(handleClickDelete);
    $(".files-ui-restore").click(handleClickRestore);
    $(".files-ui-leave").click(handleClickLeave);
    $(".files-ui-download").click(handleClickDownload);
    $(".files-ui-upload").click(handleClickUpload);
    $(".files-ui-settings").click(handleClickSettings);
    $("#files_folder_share").click(handleClickFolderShare);

    /* Add dropzone handler */
    var content = $("#pica_content");

    content.on("dragover", handleDropzoneDragOver);
    content.on("dragleave", handleDropzoneDragLeave);
    content.on("dragleave", handleDropzoneDragLeave);
    content.on("drop", handleDropzoneDrop);
    content.on("dragend", handleDragEnd);

    Picasso.Actionbar.setFilterHandler(handleFilter);

    $(document).on('click', function (event) {
        if (!$(event.target).closest('.conext-dropdown-menu').length &&
            !$(event.target).closest('.dropdown-toggle').length) {
            $('.conext-dropdown-menu').hide();
            $('.pica-table-selection-disabled').removeClass('pica-table-selection-disabled');
        }
    });

})();
