/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Links")) {
    window.Picasso.Links = {};
}

/* Private scope */
(function () {

    /* Globals */
    var TABLE = new Picasso.Table("#links_table",
        Picasso.Table.Flags.ROWS_SELECTABLE);
    var PAGE = 1;

    /**
     * Open link in new tab
     *
     * @param {Picasso.Link}  link  A #{@link Picasso.Link}
     **/

    var openLink = function (link) {

        /* Create a new anchor and simulate click to force a new window reliably */
        var a = $("<a>", {
            href: link.getLinkUrl(),
            target: "_blank"
        });

        $("body").append(a);
        a[0].click();
        a.remove();
    };

    /**
     * Handle click on visit button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickVisit = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        if (1 === selected.length) {
            var link = new Picasso.Link($(selected[0]).find(
                ".pica-serial").data("serial"));

            openLink(link);
        }
    };

    /**
     * Handle click on settings button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickSettings = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        if (1 === selected.length) {
            var link = new Picasso.Link($(selected[0]).find(
                ".pica-serial").data("serial"));

            /* Show dialog */
            var dialog = new Picasso.Dialog.Link();

            dialog.edit(link,

                /* Success */
                function () {
                    /* Replace row */
                    var row = $("#" + link.getElemId());
                    var rendered = renderLink(link);

                    row.replaceWith(rendered);
                    rendered.addClass("pica-highlight");

                    Picasso.Notification.show(Picasso.Lang.get(
                        "notification_item_updated",
                        Picasso.Lang.get("label_links")), "info");
                }
            );
        }
    };

    /**
     * Handle click on visit globe button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickVisitGlobe = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get link */
        var link = new Picasso.Link(
            $(this).closest("tr").find(".pica-serial").data("serial"));

        if (link) {
            openLink(link);
        }
    };

    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDelete = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Link($(val).find(".pica-serial").data("serial"))
        });

        /* Show dialog */
        var what = (1 == selected.length ? selected[0].getName() :
            Picasso.Lang.get("string_selected_elements", selected.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_delete"),
            Picasso.Lang.get("dialog_body_delete", what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {
                var link = this;

                link.remove(

                    /* Success */
                    function () {
                        Picasso.Helper.fadeOut($("#" + link.getElemId()),
                            function () {
                                this.remove();

                                TABLE.checkIfEmpty();
                            });

                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_deleted",
                                Picasso.Lang.get("label_links")), "info");
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);
                    }
                );
            });
        });

        dialog.show();
    };

    var handleFilter = function (filterString) {
        TABLE.clear();
        TABLE.hideEmptyMessage();
        var tr = $("<tr>", { class: "pica-not-striped" });
        var td = $("<td>", { class: "text-center", colspan: "6" });
        var span = $("<span>", { class: "pica-glyph pica-spinner glyphicons glyphicons-repeat" });
        td.append(span);
        tr.append(td);
        $("#links_table tbody").append(tr);

        if (filterString.length >= 1) {
            Picasso.Links.fetch(false, filterString);
        } else {
            Picasso.Links.fetch();
        }
    };

    /**
     * Render given json object
     *
     * @param {Picasso.Link}  link  A #{@type Picasso.Link}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     */

    var renderLink = function (link) {
        /* FIXME: PF-1347: Check if source file has been deleted */
        if (link.is(Picasso.Link.Flags.DELETED)) return null;

        var tr = $("<tr>", {
            id: link.getElemId(),
            "data-search-keys": [link.getName()],
            draggable: false,
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        var icon = link.getFile().getIcon();

        td.append(icon);

        /* Add thumbnail if found */
        if (link.getFile().isImage()) {
            var img = $("<img>", {
                class: "pica-icon-thumb-list",

                onLoad: function () {
                    icon.hide();
                    icon.after(this);
                },

                src: link.getFile().getThumbnailUrl()
            });
        }

        tr.append(td);

        /* 2a. column: Foldername */
        var td = $("<td>", { class: "pica-overflow pica-serial" });

        td.data("serial", link.serialize());

        var a = $("<a>", {
            class: "pica-name",
            href: link.getLinkUrl(),
            target: "_blank",
            text: link.getFolderName()
        });

        td.append(a);

        /* 2b. column: Filename */
        if (link.getName() != link.getFolderName()) {
        //if (link.getFile().is(Picasso.File.Flags.FILE|Picasso.File.Flags.DIRECTORY)) {
            /* Update previous link */
            a.addClass("pica-slash-after");
            a.removeClass("pica-name");
            // a.prop("href", link.getRelUrl()).removeProp("target");

            a = $("<a>", {
                class: "pica-name",
                href: link.getLinkUrl(),
                target: "_blank",
                text: link.getName(),
            });

            td.append(a);
        }

        tr.append(td);

        /* 3a. column: Downloads */
        td = $("<td>", { class: "hidden-sm hidden-xs" });
        if (link.getMaxDownloads() == 0) {
            td.append($("<span>", {
                text: Picasso.Lang.get("download_not_allowed")
            }));
        } else {
            td.append($("<span>", {
                text: link.getDownloads()
            }));
        }

        /* 3b. column: Max downloads */
        if (link.getMaxDownloads() > 0) {
            td.append($("<span>", {
                class: "pica-slash-before",
                text: link.getMaxDownloads()
            }));
        }

        tr.append(td);

        /* 4. column: Valid till */
        td = $("<td>", { class: "hidden-sm hidden-xs" });
        var validTill = link.getValidTill();

        var timestamp = validTill ? Date.parse(validTill) || 0 : Infinity;

        td.attr("data-sort", timestamp);

        if (-1 !== link.getValidTill()) {
            span = $("<span>", {
                "data-tooltip": "",
                title: Picasso.Helper.formatAge(link.getValidTill()),
                text: Picasso.Helper.formatDate(link.getValidTill(), true)
            });

            span.tooltip({ container: "body" });

            td.append(span);
        }

        tr.append(td);

        /* 5. column: Created by */
        td = $("<td>");

        if (link.is(Picasso.Link.Flags.FROM_ANONYMOUS)) {
            span = $("<span>", {
                "data-tooltip": "",
                "data-placement": "bottom",
                title: Picasso.Lang.get("tooltip_upload_form"),
                text: Picasso.Lang.get("label_anonymous")
            });

            span.tooltip({ container: "body" });
            span.addClass("text-warning");

            td.append(span);
        } else {
            var account = Picasso.get("account");

            td.append((account.isMe(link.getAccountOid()) ?
                Picasso.Lang.get("label_you") :
                link.getAccountName()));
        }

        tr.append(td);

        /* 6. column: Created at */
        td = $("<td>", {
            class: "hidden-sm hidden-xs",
            "data-sort": Date.parse(link.getCreatedAt()) || 0
        });

        span = $("<span>", {
            "data-tooltip": "",
            title: Picasso.Helper.formatDate(link.getCreatedAt()),
            text: Picasso.Helper.formatAge(link.getCreatedAt())
        });

        span.tooltip({ container: "body" });

        td.append(span);
        tr.append(td);

        /* 5. column: Globe */
        td = $("<td>", { class: "pica-icon-column text-right" });

        var a = $("<a>", {
            class: "pica-action-link"
        });

        a.click(handleClickVisitGlobe);

        var span = $("<span>", {
            class: "pica-glyph glyphicons glyphicons-globe",
            "data-tooltip": "",
            "data-placement": "bottom",
            title: Picasso.Lang.get("tooltip_visit")
        });

        span.tooltip({ container: "body" });

        a.append(span);
        td.append(a);
        tr.append(td);

        return tr;
    };

    /**
     * Fetch JSON data
     *
     * @param {Boolean}  append  Whether to append data
     **/

    Picasso.Links.fetch = function (append,query) {

        /* Reset page */
        if (true !== append) {
            PAGE = 1;
        }

        Picasso.Link.getAll(

            /* Success */
            function (links, json) {
                if (true === append) {
                    TABLE.append(links, renderLink);
                } else {
                    TABLE.update(links, renderLink);
                }

                TABLE.checkIfEmpty();

                /* Bind infinite scroll */
                Picasso.Helper.bindInfiniteScroll(json, TABLE, PAGE,
                    function () {
                        PAGE++;

                        Picasso.Links.fetch(true,query);
                    }
                );
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }, PAGE,null,query
        );
    };

    /* Bind handlers */
    $("#links_visit").click(handleClickVisit);
    $("#links_settings").click(handleClickSettings);
    $("#links_delete").click(handleClickDelete);

    Picasso.Actionbar.setFilterHandler(handleFilter);

    /* Init */
    Picasso.Links.fetch();
})();
