/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Organizations")) {
    window.Picasso.Organizations = {};
}

/* Private scope */
(function () {

    /* Globals */
    var TABLE = new Picasso.Table("#organizations_table",
        Picasso.Table.Flags.ROWS_SELECTABLE);
    var PAGE = 1;

    /**
     * Handle click on create button
     *
     * @param {Event}  e  Click event
     **/
    var handleClickBuyInfo = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var url = "https://powerfolder.atlassian.net/wiki/spaces/PF/pages/2176155649/TERRA+Drive+wird+zu+PowerFolder+Drive#Bestellungen";
        window.open(url, '_blank');
    };

    /**
     * Handle click on create button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCreate = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Show dialog */
        var dialog = new Picasso.Dialog.Organization();
        var defaultColor  = "#000;"
        $('#color1').colorpicker('setValue', defaultColor);
        $('#color2').colorpicker('setValue', defaultColor);
        $('#color3').colorpicker('setValue', defaultColor);

        dialog.create(

            /* Success */
            function () {
                Picasso.Organizations.fetch();

                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_created",
                        Picasso.Lang.get("label_organizations")), "info");
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    var handleClickSimplifiedCreate = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var dialog = new Picasso.Dialog.Organization();


        dialog.simpleCreate(

            /* Success */
            function () {
                Picasso.Organizations.fetch();

                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_created",
                        Picasso.Lang.get("label_organizations")), "info");
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Handle click on edit button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickEdit = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        if (1 === selected.length) {
            var organization = new Picasso.Organization(
                $(selected[0]).find(".pica-serial").data("serial"));

            var dialog = new Picasso.Dialog.Organization();

            $('#color1').colorpicker();
            $('#color2').colorpicker();
            $('#color3').colorpicker();

            dialog.edit(organization,

                /* Success */
                function (organization) {
                    /* Replace row */
                    var row = $("#" + organization.getElemId());
                    var rendered = renderOrganization(organization);

                    row.replaceWith(rendered);
                    rendered.addClass("pica-highlight");

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_updated",
                            Picasso.Lang.get("label_organizations")), "info");
                }
            );
        }
    };

    /**
     * Handle click on report button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickReport = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var url = "/wapi/organizations?action=downloadReport";

        var req = new XMLHttpRequest();

        req.open("GET", url, true);
        req.responseType = "blob";
        req.onload = function (event) {
            if (4 === req.readyState && (200 === req.status || 0 === req.status)) {
                var content_type = req.getResponseHeader("Content-Type");
                if (content_type && content_type.indexOf('text/csv') === 0){
                    var blob = req.response;
                    var fileName = Picasso.Helper.getFileNameByContentDisposition(req.getResponseHeader("content-disposition"))
                    var link = document.createElement('a');
                    link.href = window.URL.createObjectURL(blob);
                    link.download = fileName;
                    link.click();
                    Picasso.Notification.show(Picasso.Lang.get(
                            "notification_report_export_complete"), "info");
                }

            } else {
                Picasso.Notification.show(Picasso.Lang.get(
                        "notification_report_export_failed"), "danger");
            }
        };
        req.setRequestHeader("X-CSRFToken", Picasso.get("CSRF"));
        req.send();
    };

    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDelete = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Organization($(val).find(".pica-serial").data("serial"))
        });

        /* Show dialog */
        var what = (1 === selected.length ? selected[0].getName() :
                Picasso.Lang.get("string_selected_elements", selected.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
                Picasso.Lang.get("dialog_title_delete"),
                Picasso.Lang.get("dialog_body_org_delete", what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {
                var organization = this;

                organization.remove(

                        /* Success */
                        function () {
                            Picasso.Helper.fadeOut($("#" + organization.getElemId()),
                                    function () {
                                        this.remove();

                                        TABLE.checkIfEmpty();
                                    });

                            Picasso.Notification.show(
                                    Picasso.Lang.get("notification_item_deleted",
                                            Picasso.Lang.get("label_organizations")), "info");
                        },

                        /* Error */
                        function (e) {
                            Picasso.debugLog(e);
                        }
                );
            });
        });

        dialog.show();
    };

    /**
     * Handle click on copy button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickMigrationCopy = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Organization($(val).find(".pica-serial").data("serial"))
        });

        $.each(selected, function () {
            var organization = this;
            organization.migrationCopy(
                /* Success */
                function () {
                    Picasso.Helper.fadeOut($("#" + organization.getElemId()),
                        function () {
                            this.remove();

                            TABLE.checkIfEmpty();
                        });

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_created",
                            organization.getName()), "info");
                },

                /* Error */
                function (e) {
                    Picasso.Notification.show(Picasso.Lang.get("notification_error_create",
                        organization.getName()), "warning");
                }
            );
        });
    };

    /**
     * Handle click on set migration active button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickMigrationSetActive = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Organization($(val).find(".pica-serial").data("serial"))
        });

        $.each(selected, function () {
            var organization = this;
            organization.migrationSetActive(
                /* Success */
                function () {
                    Picasso.Helper.fadeOut($("#" + organization.getElemId()),
                        function () {
                            this.remove();

                            TABLE.checkIfEmpty();
                        });

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_enabled",
                            organization.getName()), "info");
                },

                /* Error */
                function (e) {
                    Picasso.Notification.show(Picasso.Lang.get("notification_error_update",
                        organization.getName()), "warning");
                }
            );
        });
    };

    var handleFilter = function (filterString) {
        TABLE.clear();
        TABLE.hideEmptyMessage();
        var tr = $("<tr>", { class: "pica-not-striped" });
        var td = $("<td>", { class: "text-center", colspan: "7" });
        var span = $("<span>", { class: "pica-glyph pica-spinner glyphicons glyphicons-repeat" });
        td.append(span);
        tr.append(td);
        $("#organizations_table tbody").append(tr);

        if (filterString.length >= 1) {
            Picasso.Organizations.fetch(false, filterString);
        } else {
            Picasso.Organizations.fetch();
        }
    };


    /**
     * Render given json object
     *
     * @param {Picasso.Organization}  organization  A #{@type Picasso.Organization}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderOrganization = function (organization) {
        var tr = $("<tr>", {
            id: organization.getElemId(),
            "data-search-keys": [organization.getName()],
            draggable: false,
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });


        Picasso.Helper.createGlyphAndAvatar(td, "education", organization);

        tr.append(td);

        /* 2. column: Name */
        td = $("<td>", {
            class: "pica-overflow pica-serial",
        });

        var name = organization.getName();
        if (organization.getMigrationPercentage() === 0 || organization.getMigrationPercentage()) {
            var s = "";
            if (organization.getMigrationStatus()) {
                s = organization.getMigrationStatus().charAt(0) + " ";
            }
            if (organization.getMigrationPercentage() >= 0) {
                name = "(" + s + organization.getMigrationPercentage() + "%) " + name;
            } else {
                name = "(---%) " + name;
            }
        }
        var a = $("<a>", {
            class: "pica-name",
            text: name
        });

        a.click(function (e) {
            tr.addClass(Picasso.Table.Marker.SELECTED);

            handleClickEdit(e);
        });

        td.data("serial", organization.serialize());

        td.append(a);
        tr.append(td);

        /* 3. column: Quota */
        td = $("<td>", {
            "data-sort": organization.getQuotaUsed()
        });

        td.append($("<span>", {
            text: Picasso.Helper.formatSize(organization.getQuotaUsed()),
        }));

        if (0 < organization.getQuotaTotal()) {
            td.append($("<span>", {
                class: "pica-slash-before",
                text: Picasso.Helper.formatSize(organization.getQuotaTotal())
            }));

            td.attr("data-sort", organization.getQuotaTotal());
        }

        tr.append(td);

        /* 4. column: Accounts */
        td = $("<td>");

        td.append($("<span>", {
            text: organization.getAccountsCount()
        }));

        if (0 < organization.getAccountsMax()) {
            td.append($("<span>", {
                class: "pica-slash-before",
                text: organization.getAccountsMax()
            }));
        }

        tr.append(td);

        /* 5. column: Valid from */
        td = $("<td>", {
            class: "hidden-xs",
        });

        if (-1 !== organization.getValidFrom()) {
            td.append(Picasso.Helper.formatDate(
                organization.getValidFrom(), true));
        }

        tr.append(td);

        /* 6. column: Valid till */
        td = $("<td>");

        if (-1 !== organization.getValidTill()) {
            var span = $("<span>", {
                "data-tooltip": "",
                title: Picasso.Helper.formatAge(
                    organization.getValidTillMS()),
                text: Picasso.Helper.formatDate(
                    organization.getValidTill(), true)
            });

            span.tooltip({ container: "body" });

            td.append(span);
        }

        tr.append(td);

        /* 7. column: reseller */
        td = $("<td>", {
            class: "text-right",
        });
        if (organization.getHasReseller()) {
            td.append(Picasso.Helper.createGlyph("crown","pica-glyph"))
        }
        tr.append(td)

        return tr;
    };

    /**
     * Fetch JSON data
     *
     * @param {Boolean}  append  Whether to append data
     **/

    Picasso.Organizations.fetch = function (append, query) {

        /* Reset page */
        if (true !== append) {
            PAGE = 1;
        }

        Picasso.Organization.getAll(

            /* Success */
            function (organizations, json) {
                if (true === append) {
                    TABLE.append(organizations, renderOrganization);
                } else {
                    TABLE.update(organizations, renderOrganization);
                }

                /* Bind infinite scroll */
                Picasso.Helper.bindInfiniteScroll(json, TABLE, PAGE,
                    function () {
                        PAGE++;

                        Picasso.Organizations.fetch(true, query);
                    }
                );
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }, PAGE, null, query
        );
    };

    // Disable create and delete button for restricted organization admins and restricted resellers
    if ((Picasso.get("account").is(Picasso.Account.Flags.RESTRICTED_ORG_ADMIN)) || (Picasso.get("account").is(Picasso.Account.Flags.RESELLER))) {
        $("#organizations_create").click(handleClickBuyInfo);
        $("#organizations_create_simplified").click(handleClickSimplifiedCreate);
        $("#organizations_delete").remove();
    } else {
        $("#organizations_create").click(handleClickCreate);
    }

    /* Bind handlers */
    $("#organizations_edit").click(handleClickEdit);
    $("#organizations_delete").click(handleClickDelete);
    $("#organizations_migration_copy").click(handleClickMigrationCopy);
    $("#organizations_migration_set_active").click(handleClickMigrationSetActive);
    $("#organizations_report").click(handleClickReport);

    Picasso.Actionbar.setFilterHandler(handleFilter);

    /* Init */
    Picasso.Organizations.fetch();
})();

