"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}
if (!window.Picasso.hasOwnProperty("Pricing")) {
    window.Picasso.Pricing = {};
}

(function() {

    Picasso.Pricing.init = function() {
        Picasso.Pricing.fetchPlans(function(plans) {
            Picasso.Pricing.render(plans);
        });
    };

    Picasso.Pricing.render = function(plans) {
        const monthlyContainer = $("#monthlyPlans");
        const yearlyContainer  = $("#yearlyPlans");

        monthlyContainer.empty();
        yearlyContainer.empty();

        plans.forEach(function(plan) {
            const monthlyCard = createMonthlyPlanCard(plan);
            monthlyContainer.append(monthlyCard);
        });

        plans.forEach(function(plan) {
            const yearlyCard = createYearlyPlanCard(plan);
            yearlyContainer.append(yearlyCard);
        });
    };

    const locale = Picasso.Lang.LANG || navigator.language || 'de-DE';

    function createMonthlyPlanCard(plan) {
        let card = $("<div>", {class: "plan-card"});

        let contentDiv = $("<div>", {class: "plan-content"});
        card.append(contentDiv);

        if (plan.getImages() && plan.getImages().length > 0) {
            let imageUrl = plan.getImages()[0];
            contentDiv.append(
                $("<div>", {
                    class: "plan-image",
                }).append(
                    $("<img>", {
                        src: imageUrl,
                        alt: plan.getName(),
                        class: "plan-product-image"
                    })
                )
            );
        }

        contentDiv.append(
            $("<div>", {class: "plan-title"}).text(plan.getName())
        );

        // Wrapper for variable-height content
        let flexibleContent = $("<div>", {class: "plan-flexible-content"});

        flexibleContent.append(
            $("<div>", {class: "plan-desc"}).text(plan.getMonthlyDesc())
        );

        let featuresDiv = $("<div>", {class: "plan-features"});
        let ul = $("<ul>");

        if (plan.getMarketingFeatures() && plan.getMarketingFeatures().length > 0) {
            plan.getMarketingFeatures().forEach(feature => {
                ul.append($("<li>", {class: "marketing-feature"}).text(feature));
            });
        }

        featuresDiv.append(ul);
        flexibleContent.append(featuresDiv);

        contentDiv.append(flexibleContent);

        // Price section - always at bottom
        let priceSection = $("<div>", {class: "plan-price-section"});

        const formatter = new Intl.NumberFormat(locale, {
            style: 'currency',
            currency: plan.getMonthlyCurrency()
        });
        priceSection.append(
            $("<div>", {
                class: "plan-price"
            }).text(formatter.format(plan.getMonthlyPrice()))
        );

        priceSection.append(
            $("<div>", {class: "plan-duration"}).text(Picasso.Lang.get("label_monthly"))
        );

        contentDiv.append(priceSection);

        let ctaOrBadge;
        if (plan.getIsCurrentPlan()) {
            ctaOrBadge = $("<div>", {class: "current-badge"}).text(Picasso.Lang.get("label_current"));
        } else if (plan.getIsDisabled()) {
            ctaOrBadge = $("<div>", {
                class: "disabled-badge",
                title: Picasso.Lang.get("tooltip_plan_unavailable")
            }).text(Picasso.Lang.get("label_unavailable"));
        } else {
            ctaOrBadge = $("<a>", {href: "#", class: "plan-cta"})
                .text(Picasso.Lang.get("label_choose", plan.getName()))
                .data("priceid", plan.getMonthlyPriceId())
                .on("click", function (evt) {
                    evt.preventDefault();
                    const priceId = $(this).data("priceid");
                    Picasso.Pricing.startCheckout(priceId);
                });
        }

        card.append(ctaOrBadge);
        return card;
    }


    function formatStorage(value) {
        let numericValue = (typeof value === "number") ? value : parseFloat(value);

        if (!isNaN(numericValue)) {
            if (numericValue >= 1024) {
                let tb = numericValue / 1024;
                return tb.toFixed(1) + " TB";
            } else {
                return numericValue + " GB";
            }
        }
        return value;
    }


    function createYearlyPlanCard(plan) {
        let card = $("<div>", {class: "plan-card"});

        let contentDiv = $("<div>", {class: "plan-content"});
        card.append(contentDiv);

        if (plan.getImages() && plan.getImages().length > 0) {
            let imageUrl = plan.getImages()[0];
            contentDiv.append(
                $("<div>", {
                    class: "plan-image"
                }).append(
                    $("<img>", {
                        src: imageUrl,
                        alt: plan.getName(),
                        class: "plan-product-image"
                    })
                )
            );
        }

        contentDiv.append(
            $("<div>", {class: "plan-title"}).text(plan.getName())
        );

        // Wrapper for variable-height content
        let flexibleContent = $("<div>", {class: "plan-flexible-content"});

        flexibleContent.append(
            $("<div>", {class: "plan-desc"}).text(plan.getYearlyDesc())
        );


        let featuresDiv = $("<div>", {class: "plan-features"});
        let ul = $("<ul>");

        if (plan.getMarketingFeatures() && plan.getMarketingFeatures().length > 0) {
            plan.getMarketingFeatures().forEach(feature => {
                ul.append($("<li>", {class: "marketing-feature"}).text(feature));
            });
        }

        featuresDiv.append(ul);
        flexibleContent.append(featuresDiv);

        contentDiv.append(flexibleContent);

        // Price section - always at bottom
        let priceSection = $("<div>", {class: "plan-price-section"});

        const formatter = new Intl.NumberFormat(locale, {
            style: 'currency',
            currency: plan.getYearlyCurrency()
        });
        priceSection.append(
            $("<div>", {
                class: "plan-price"
            }).text(formatter.format(plan.getYearlyPrice()))
        );

        priceSection.append(
            $("<div>", {class: "plan-duration"}).text(Picasso.Lang.get("label_yearly"))
        );

        contentDiv.append(priceSection);

        let ctaOrBadge;
        if (plan.getIsCurrentPlan()) {
            ctaOrBadge = $("<div>", {class: "current-badge"}).text(Picasso.Lang.get("label_current"));
        } else if (plan.getIsDisabled()) {
            ctaOrBadge = $("<div>", {
                class: "disabled-badge",
                title: Picasso.Lang.get("tooltip_plan_unavailable")
            }).text(Picasso.Lang.get("label_unavailable"));
        } else {
            ctaOrBadge = $("<a>", {href: "#", class: "plan-cta"})
                .text(Picasso.Lang.get("label_choose", plan.getName()))
                .data("priceid", plan.getYearlyPriceId())
                .on("click", function (evt) {
                    evt.preventDefault();
                    const priceId = $(this).data("priceid");
                    Picasso.Pricing.startCheckout(priceId);
                });
        }

        card.append(ctaOrBadge);

        if (plan.getYearlyPriceIsDefault()) {
            $("#yearlyRadio").prop("checked", true).trigger("change");
            $("#monthlyRadio").prop("checked", false).trigger("change");
            $("#yearlyPlans").removeClass("hidden");
            $("#monthlyPlans").addClass("hidden");
        }

        return card;
    }


    Picasso.Pricing.startCheckout = function(priceId) {
        Picasso.Helper.fireAjax(
            "/wapi/pricing",
            { action: "createCheckout", priceId: priceId },
            // onSuccess
            function (json, status) {
                if (json && json.sessionUrl) {
                    window.location = json.sessionUrl;
                } else if (json && json.redirectUrl) {
                    window.location.href = json.redirectUrl;
                } else {
                    console.error("No sessionUrl from createCheckout", json);
                }
            },
            // onError
            function(error, httpCode, resp) {
                Picasso.Notification.show(resp.message);
            }
        );
    };

    Picasso.Pricing.fetchPlans = function(onSuccess, onError) {
        Picasso.Helper.fireAjax(
            "/wapi/pricing",
            { action: "getAll" },
            function(json, status) {

                let arr = (json && json.ResultSet && json.ResultSet.Result)
                    ? json.ResultSet.Result
                    : [];

                let plans = arr.map(function(pm) {
                    return new Picasso.Plan(pm);
                });

                if (onSuccess) onSuccess(plans);
            },
            function(error, httpCode, resp) {
                if (onError) onError(error);
            }
        );
    };





    $(document).ready(function(){
        Picasso.Pricing.init();
    });


    document.addEventListener("DOMContentLoaded", function () {
        const urlParams = new URLSearchParams(window.location.search);

        if (urlParams.has('session_id')) {
            Picasso.Notification.show(
                Picasso.Lang.get("notification_success_payment_completed"), "success");

            const banner = document.getElementById('upgradeBanner');
            if (banner) banner.remove();

            document.getElementById("pica_menu")?.style.setProperty("top", "0px", "important");
            document.getElementById("pica_pane")?.style.setProperty("top", "0px", "important");



            setTimeout(() => {
                window.location.href = window.location.pathname;
            }, 5000);
        }


        if (urlParams.has('canceled') && urlParams.get('canceled') === 'true') {
            Picasso.Notification.show(
                Picasso.Lang.get("notification_info_payment_canceled"), "info");
        }
        if (urlParams.has('expired') && urlParams.get('expired') === 'true') {
            Picasso.Notification.show(
                Picasso.Lang.get("notification_error_account_expired_buynow"), "warning", true);
        }
    });

})();
