/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Profile")) {
    window.Picasso.Profile = {};
}

(function () {

    /* Globals */
    var ORGANIZATIONS_TABLE = new Picasso.Table("#profile_organizations_table");
    var DEVICES_TABLE = new Picasso.Table("#profile_devices_table", Picasso.Table.Flags.ROWS_SELECTABLE);

    var optionsForDevices = [
        {
            text: Picasso.Lang.get("action_delete"),
            icon: "bin",
            klass: "files-ui-delete"
        }
    ];

    let mnemonicWords = [];
    let validationIndices = [];

    /* Create input list */
    var MAILS_INPUTLIST = new Picasso.InputList("#profile_mails_inputlist",
        Picasso.InputList.Flags.LIST_MODE|Picasso.InputList.Flags.VALIDATE);

    MAILS_INPUTLIST.setSubmitHandler(function (e, userdata) {
        var name = this.getInputValue();

        if(0 < name.length) {
            this.addItem("message-new", name, null, {
                oid: null,
                name: name
            });
        }
    });


    MAILS_INPUTLIST.setInputValidationHandler(function (email){
        var account = Picasso.get("account");
        var valid = true;
        if (!Picasso.Helper.isValidEmail(email)){
            Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_invalid_email"), "warning");
            valid = false;
        }else if (account.getUserName() === email){
            Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_email_match_username"), "warning");
            valid = false;
        }
        return valid;
    })

    /**
     * Update emails
     *
     * @param {Array}  emails  Emails list
     **/

    var updateMails = function (emails) {
        var mails = $("#profile_emails");
        var table = $("<table>", {
            class: "pica-table-list"
        });

        $.each(emails, function () {
            var tr = $("<tr>");

            var td = $("<td>", {
                class: "text-muted",
                text: this
            });

            tr.append(td);
            table.append(tr);
        });

        mails.html(table);
    };

    /**
     * Handle click on change button
     *
     * @param {String}  elemId     Element ID
     * @param {String}  labelName  Name of the label
     * @param {String}  fieldName  Name of the field
     **/

    var handleClickChange = function (elemId, labelName, fieldName) {
        var account = Picasso.get("account");

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_input_dialog",
            Picasso.Lang.get("dialog_title_change"),
            Picasso.Lang.get("dialog_body_change", Picasso.Lang.get(labelName)),
            (Picasso.Dialog.Flags.HIDE_ON_ENTER|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE));

        dialog.setAssignHandler(function (inputs) {
            inputs.pica_input_text.val(
                account["get" + Picasso.Helper.capitalize(fieldName)]);
        });

        dialog.setOkHandler(function (inputs, dialog) {
            var val = inputs.pica_input_text.val();

            account["set" + Picasso.Helper.capitalize(fieldName)](val);

            /* Store details */
            account.setDetails(

                /* Success */
                function () {
                    Picasso.set("account", account);

                    $(elemId).text(val);

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_updated",
                            Picasso.Lang.get("label_accounts")), "info");

                    dialog.hide();
                },

                /* Error */
                function (e) {
                    Picasso.debugLog(e);
                }
            );
        });

        dialog.show();
    };



    var handleClickLanguage = function () {
        var account = Picasso.get("account");

        /* Show dialog */
        var dialog = new Picasso.Dialog("#profile_language_dialog",
                Picasso.Lang.get("dialog_title_change"),
                Picasso.Lang.get("dialog_language_change_language"),
                (Picasso.Dialog.Flags.HIDE_ON_ENTER|Picasso.Dialog.Flags.HIDE_ON_ESC|
                        Picasso.Dialog.Flags.NOf_AUTOCLOSE));

        dialog.setAssignHandler(function () {
           $("#profile_language").val(
                    account["get" + Picasso.Helper.capitalize("language")]);
        });

        dialog.setOkHandler(function (inputs,dialog) {
            var val =  $("#profile_language").val();

            account["set" + Picasso.Helper.capitalize("language")](val);

            /* Store details */
            account.setDetails(

                    /* Success */
                    function () {
                        Picasso.set("account", account);

                        $("#profile_language_text").text(val);

                        Picasso.Notification.show(
                                Picasso.Lang.get("notification_item_updated",
                                        Picasso.Lang.get("label_accounts")), "info");

                        dialog.hide();
                        location.reload();
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);
                    }
            );
        });

        dialog.show();
    };

    var handleClickCountry = function () {
        var customer = Picasso.get("customer");

        /* Show dialog */
        var dialog = new Picasso.Dialog("#stripe_country_dialog",
            Picasso.Lang.get("dialog_title_change"),
            Picasso.Lang.get("dialog_country_change", Picasso.Lang.get("label_country")),
            (Picasso.Dialog.Flags.HIDE_ON_ENTER|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NOf_AUTOCLOSE));

        dialog.setAssignHandler(function () {
            $("#stripe_country").val(
                customer["get" + Picasso.Helper.capitalize("addressCountry")]);
        });

        dialog.setOkHandler(function (inputs,dialog) {
            var val =  $("#stripe_country").val();

            customer["set" + Picasso.Helper.capitalize("addressCountry")](val);

            /* Store details */
            customer.setCustomer(
                /* Success */
                function (data, status) {

                    Picasso.set("customer", customer);

                    $("#pica_customer_address_country").text(Picasso.Lang.get("lang_country_"+val));

                    switch (status) {
                        case 200:
                            Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_updated"),"info");
                            break;
                        case 201:
                            Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_updated"),"info");
                            break
                    }

                    dialog.hide();
                },

                /* Error */
                function (data, status) {
                    switch (status) {
                        case 420:
                            Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_update_failed"),"danger");
                            break
                        case 424:
                            Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_update_failed"),"danger");
                            break
                    }
                    dialog.hide();
                }
            );
        });

        dialog.show();
    };

    /**
     * Handle click on password change
     *
     * @param {Event}  e  Click event
     **/

    var handleClickPassword = function (e) {
        /* Show dialog */
        var dialog = new Picasso.Dialog("#profile_password_dialog",
            Picasso.Lang.get("dialog_title_change"),
            Picasso.Lang.get("dialog_profile_password_body_change"),
            (Picasso.Dialog.Flags.VALIDATE|
                Picasso.Dialog.Flags.HIDE_ON_ESC|Picasso.Dialog.Flags.NO_AUTOCLOSE));

        dialog.setAssignHandler(function (inputs) {
            Picasso.Helper.updatePasswordPolicy(inputs.profile_password_new);

            /* Add change handler to update policy info */
            inputs.profile_password_new.add(inputs.profile_password_repeat).on("keyup", function (e) {
                Picasso.Helper.updatePasswordPolicy(this, e.originalEvent.target.value);
                if (inputs.profile_password_new.val() == inputs.profile_password_repeat.val()) {
                    inputs.profile_password_repeat.closest(".form-group").removeClass("has-error");
                    $("#profile_password_dialog .btn-primary").prop('disabled', false);
                } else {
                    inputs.profile_password_repeat.closest(".form-group").addClass("has-error");
                    $("#profile_password_dialog .btn-primary").prop('disabled', true);
                }
            });

            /* Add click handler to clear button to update policy as well */
            inputs.profile_password_new.siblings(
                ".pica-input-clear").click(function ()
            {
                Picasso.Helper.updatePasswordPolicy(
                    inputs.profile_password_new);
            });
        });

        dialog.setOkHandler(function (inputs, dialog) {
            var account = Picasso.get("account");

            account.changePassword(inputs.profile_password_old.val(),
                inputs.profile_password_new.val(),

                /* Success */
                function () {
                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_updated",
                            Picasso.Lang.get("label_accounts")), "info");

                    dialog.hide();
                },

                /* Error */
                function (e, status) {
                    switch (status) {
                        case 400:
                            Picasso.Notification.show(
                                Picasso.Lang.get(
                                    "notification_error_old_password_wrong"));
                            break;

                        case 406:
                            updatePasswordPolicy();

                            Picasso.Notification.show(
                                Picasso.Lang.get(
                                    "notification_error_password_policy_mismatch"));
                            break;

                        default:
                            Picasso.debugLog(e);
                    }
                }
            );
        });

        dialog.show();
    };

    /**
     * Handle click on manage button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickManageEmails = function (e) {
        var account = Picasso.get("account");

        /* Show dialog */
        var dialog = new Picasso.Dialog("#profile_emails_dialog",
            Picasso.Lang.get("dialog_title_edit"),
            Picasso.Lang.get("dialog_body_edit",
                Picasso.Lang.get(Picasso.Lang.Flags.PLURALIZE, "label_emails")));

        dialog.setAssignHandler(function (inputs) {
            account.getEmails(
                function (emails) {
                    MAILS_INPUTLIST.clear();

                    $.each(emails, function () {
                        MAILS_INPUTLIST.addItem(
                            "envelope", this, null, {
                                oid: null,
                                name: this
                            }
                        )
                    });
                }
            );
        });

        dialog.setOkHandler(function () {
            var items = MAILS_INPUTLIST.getItems();
            var emails = $.map(items, function (val) {
                return val.name
            });

            account.setEmails(emails,

                /* Success */
                function (data, status) {
                    switch (status) {
                        case 202: // Email or merge verification needed
                            var verificationDialog = new Picasso.Dialog(
                                '#profile_verification_dialog',
                                Picasso.Lang.get("dialog_title_verification_needed"),
                                Picasso.Lang.get("dialog_body_verification_needed"));
                            verificationDialog.show();
                            break;
                        case 200: // Only removed emails
                        case 204: // Nothing chagned
                            updateMails(emails);

                            Picasso.Notification.show(Picasso.Lang.get("notification_item_updated",
                                Picasso.Lang.get(Picasso.Lang.Flags.PLURALIZE,
                                    "label_emails")), "info");
                    }
                },

                /* Error */
                function (e) {
                    Picasso.debugLog(e);
                }
            );
        });

        dialog.show();
    };

    /**
     * Handle click on 2fa manage button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickManage2FA = function (e) {
        var account = Picasso.get("account");

        /* Show dialog */
        var dialog = new Picasso.Dialog("#profile_2fa_dialog",
            Picasso.Lang.get("dialog_title_settings"),
            Picasso.Lang.get("dialog_body_settings",
                Picasso.Lang.get(Picasso.Lang.Flags.PLURALIZE, "label_2fa")),Picasso.Dialog.Flags.NO_AUTOCLOSE);

        dialog.setAssignHandler(function (inputs) {

                function displayQr(url){
                    var canvas = document.getElementById("board")
                    var qrcode = new QRious({
                        element: canvas,
                        background: '#ffffff',
                        backgroundAlpha: 1,
                        foreground: '#000000',
                        foregroundAlpha: 1,
                        level: 'H',
                        padding: 0,
                        size: 300,
                        value: url
                    });
                    $("#board").show();
                }

                $("#topt_code").off("keyup").keyup(function (e) {
                    var m = $(this).val().match(/([0-9]{6})/)
                    if (m){
                        $("#enable_2f").removeProp("disabled");
                        $("#enable_2f").prop("disabled",false);
                        $(this).removeClass('has-error');
                        $("#enable_2f").click();
                    }else {
                        $(this).addClass('has-error');
                    }
                })

                account.getDetails(
                    /* Success */
                    function (self) {

                        if (self.getTwoFactorAuthenticationEnabled()) {
                            $("#btn_2fa_toggle").addClass("active");
                            $("#btn_2fa_toggle").find("i").removeClass();
                            $("#btn_2fa_toggle").find("i").addClass("state-icon glyphicons glyphicons-check");
                            $("#pica_2fa_active").trigger("click");
                            displayQr(self.getTwoFactorAuthenticationUrl())
                            $("#board").show();
                        } else {
                            $("#btn_2fa_toggle").removeClass("active");
                            $("#btn_2fa_toggle").find("i").removeClass();
                            $("#btn_2fa_toggle").find("i").addClass("state-icon glyphicons glyphicons-unchecked");
                            $("#topt_code").closest(".row").hide();
                            $("#board").hide();
                        }
                        // reset state
                        $("#recovery_phrase_section").hide();
                        $("#recovery_phrase_buttons").show();
                        $("#recovery_phrase_message").show();
                        $("#copyRecoveryPhrase").show();
                        $("#downloadRecoveryPhrase").show();
                        $("#validationFields").hide();
                        $("#verifyButtonRow").hide();
                        $("#verification_message").hide();
                        $("#qr_code_section").hide();



                        $("#btn_2fa_toggle").bind('click', function (e) {
                            if ($("#btn_2fa_toggle").hasClass("active")) {
                                account.setup2fa(function (result) {
                                    if (result.OtpAuthUrl && result.OtpAuthUrl !== "") {

                                        $("#recovery_phrase_section").show();

                                        const recoveryPhraseDiv = $("#recovery_phrase_buttons");
                                        recoveryPhraseDiv.empty(); // Clear any existing buttons
                                        mnemonicWords = result.recoveryPhrase.split(' ');
                                        mnemonicWords.forEach((word, index) => {
                                            const buttonHtml = `
                                                    <button type="button" class="btn btn-secondary btn-sm mx-1 mb-1" disabled>
                                                        <span class="badge badge-light">${index + 1}</span> ${word}
                                                    </button>
                                            `;
                                            recoveryPhraseDiv.append(buttonHtml);
                                        });

                                        // Copy to clipboard text
                                        $("#recovery_phrase_text").text(result.recoveryPhrase);

                                        // Generate input fields for validation indices
                                        validationIndices = result.validationIndices;
                                        const validationFieldsDiv = $("#validationFields");
                                        validationFieldsDiv.empty(); // Clear any existing fields
                                        validationIndices.forEach(index => {
                                            const fieldHtml = `
                                                    <div class="form-group col-lg-4">
                                                        <label for="word${index + 1}"> ${Picasso.Lang.get("label_word", index + 1)}:</label>
                                                        <input type="text" id="word${index + 1}" name="word${index + 1}" class="form-control validationField">
                                                    </div>
                                                `;
                                            validationFieldsDiv.append(fieldHtml);
                                        });


                                        displayQr(result.OtpAuthUrl)
                                        $("#topt_code").closest(".row").show();
                                        $("#secret_key").val(result.secret);
                                        $("#encodedRecoveryPhrase").val(result.encodedRecoveryPhrase);
                                        $("#enable_2f").prop("disabled",true);
                                    } else {
                                        $("#board").hide();
                                    }

                                }, function () {

                                })
                            }else if (account.getTwoFactorAuthenticationEnabled()) {
                                var confirmation = new Picasso.Dialog("#pica_confirmation_dialog",
                                    Picasso.Lang.get("dialog_title_disable"),
                                    Picasso.Lang.get("dialog_body_disable_2fa"));

                                confirmation.setOkHandler(function () {
                                    account.toggle2fa(false, "", "", "",
                                        /* Success */
                                        function () {
                                            $("#btn_2fa_toggle").removeClass("active");
                                            $("#board").hide();
                                            $("#topt_code").closest(".row").hide();
                                            $("#enable_2f").removeProp("disabled");
                                            $("#enable_2f").prop("disabled", false);

                                            account.setTwoFactorAuthenticationEnabled(false);
                                            Picasso.set("account", account);

                                            Picasso.Notification.show(
                                                Picasso.Lang.get("notification_2fa_disabled"), "info");

                                            confirmation.hide();
                                            dialog.hide();
                                        },
                                        /* Error */
                                        function (e, status) {
                                            switch (status) {
                                                case 400:
                                                    Picasso.Notification.show(
                                                        Picasso.Lang.get("2fa.secret.missing"), "danger");
                                                    break;
                                                case 406:
                                                    Picasso.Notification.show(
                                                        Picasso.Lang.get("2fa.code.mismatch"), "danger");
                                                    break;
                                            }
                                        }
                                    );
                                });

                                confirmation.setCancelHandler(function (){
                                    $("#btn_2fa_toggle").removeClass();
                                    $("#btn_2fa_toggle").addClass("btn btn-primary active");
                                    $("#btn_2fa_toggle").find("i").removeClass();
                                    $("#btn_2fa_toggle").find("i").addClass("state-icon glyphicons glyphicons-check");
                                    dialog.hide()
                                })

                                confirmation.show();

                            }else {
                                $("#topt_code").closest(".row").hide();
                                $("#board").hide();
                            }
                        })

                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);

                    });

            }
        )

        dialog.setOkHandler(function () {
            var state = $("#btn_2fa_toggle").hasClass("active");

            if (state) {
                var secretKey = $("#secret_key").val();
                var encodedRecoveryPhrase = $("#encodedRecoveryPhrase").val();
                var totp = $("#topt_code").val();
                account.toggle2fa(state, secretKey, totp, encodedRecoveryPhrase,

                    /* Success */
                    function () {
                        $("#topt_code").closest(".row").hide();

                        account.setTwoFactorAuthenticationEnabled(true);
                        Picasso.set("account", account);

                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_2fa_enabled"), "info");
                        dialog.hide();
                    },
                    /* Error */
                    function (e, status) {
                        switch (status) {
                            case 400:
                                Picasso.Notification.show(
                                    Picasso.Lang.get("2fa.secret.missing"), "danger");
                                break;
                            case 406:
                                Picasso.Notification.show(
                                    Picasso.Lang.get("2fa.code.mismatch"), "danger");
                        }
                    }
                );
            } else {
                dialog.hide();
            }
        });

        dialog.show();
    };

    function copyToClipboard() {
        const recoveryPhrase = $("#recovery_phrase_text").text();
        const $temp = $("<input>");
        $("body").append($temp);
        $temp.val(recoveryPhrase).select();
        document.execCommand("copy");
        $temp.remove();
        Picasso.Notification.show(Picasso.Lang.get("notification_copy_success"), "info");
        // Hide copy and download buttons
        $("#copyRecoveryPhrase").hide();
        $("#downloadRecoveryPhrase").hide();
        // Hide recovery phrase and show validation fields, verify button, and verification message
        $("#recovery_phrase_buttons").hide();
        $("#recovery_phrase_message").hide();
        $("#validationFields").show();
        $("#verifyButtonRow").show();
        $("#verification_message").show();
    }


    function downloadWords() {
        const recoveryPhrase = $("#recovery_phrase_text").text();
        const blob = new Blob([recoveryPhrase], {type: "text/plain;charset=utf-8"});
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement("a");
        a.style.display = "none";
        a.href = url;
        a.download = "two-factor-recovery-words.txt";
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        // Hide copy and download buttons
        $("#copyRecoveryPhrase").hide();
        $("#downloadRecoveryPhrase").hide();
        // Hide recovery phrase and show validation fields, verify button, and verification message
        $("#recovery_phrase_buttons").hide();
        $("#recovery_phrase_message").hide();
        $("#validationFields").show();
        $("#verifyButtonRow").show();
        $("#verification_message").show();
    }


    function verifyRecoveryPhrase() {
        let isValid = true;
        $(".validationField").each(function () {
            const id = $(this).attr('id');
            const index = parseInt(id.replace('word', ''), 10) - 1;
            const userInput = $(this).val().trim();

            if (mnemonicWords[index] !== userInput) {
                isValid = false;
                $(this).addClass('has-error');
            } else {
                $(this).removeClass('has-error');
            }
        });

        if (isValid) {
            Picasso.Notification.show(Picasso.Lang.get("notification_recovery_phrase_verified"), "info");
            // Show the QR code and TOTP input fields
            $("#qr_code_section").show();
            $("#recovery_phrase_section").hide();
        } else {
            Picasso.Notification.show(Picasso.Lang.get("notification_recovery_phrase_verification_failed"), "warning");
        }
    }


    /**
     * Handle click on close account button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCloseAccount = function (e) {
        var account = Picasso.get("account");

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_close_account_confirmation_dialog",
            Picasso.Lang.get("dialog_title_close_account"),
            Picasso.Lang.get("dialog_body_close_account"),Picasso.Dialog.Flags.NO_AUTOCLOSE);

        dialog.setAssignHandler(function (inputs) {
                account.getDetails(
                    /* Success */
                    function (self) {
                        if (self.getTwoFactorAuthenticationEnabled()) {
                            $("#2fa_code").closest(".row").show();
                            $("#confirm-button").prop("disabled", true);

                            $("#2fa_code").on("input", function () {
                                var inputText = $(this).val().trim();
                                $("#confirm-button").prop("disabled", inputText.length !== 6 || !/^\d+$/.test(inputText));
                            });
                        }
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);
                    });
            }
        )

        dialog.setOkHandler(function () {
            var mfaCode = $("#2fa_code").val();
            account.closeAccount(mfaCode,
                /* Success */
                function () {
                    $("#topt_code").closest(".row").hide();
                    Picasso.Notification.show(Picasso.Lang.get("notification_account_close_request_submitted"), "info");
                    dialog.hide();
                },
                /* Error */
                function (e,status) {
                    switch (status){
                        case 406:
                            Picasso.Notification.show(Picasso.Lang.get("2fa.code.mismatch"), "danger");
                            break
                    }
                }
            );
        });
        dialog.show();
    };

    /**
     * Handle click on navigation tabs
     *
     * @param {Event}  e  Click event
     **/

    var handleClickNav = function (e) {
        var tab = $(this).data("tab");

        /* Add data */
        var account = Picasso.get("account");

        switch (tab) {
            case "organizations":
                account.getOrganizations(
                    function (organizations) {
                        ORGANIZATIONS_TABLE.update(
                            organizations, renderOrganization);
                    }
                );
                break;

            case "devices":
                account.getDevices(
                    function (devices) {
                        DEVICES_TABLE.update(devices, renderDevice);
                    }
                );
                break;
            case "stripe":
                Picasso.Customer.getCustomer(account.getOid(),
                    function (json) {
                        var customer = new Picasso.Customer(json);
                        Picasso.set("customer",customer);
                        renderCustomer(customer, account.getUsername())


                    },function (data, status) {
                        switch (status){
                            case 412: // Not enabled
                                $("#profile_customer_table").hide();
                                break;
                            case 404:
                                Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_not_found"),"warning");
                                break;
                        }
                        var customer = new Picasso.Customer();
                        customer.setAccountOid(account.getOid());
                        Picasso.set("customer", customer);
                    }
                );
                break;

        }
    };



    /**
     * Render given json object for list
     *
     * @param {Picasso.Organization}  organization  A #{Picasso.Organization}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderOrganization = function (organization) {
        var tr = $("<tr>", {
            id: organization.getElemId(),
            class: "pica-table-selection-disabled",
            "data-search-keys": [organization.getName()],
            draggable: false
        });

        var td = $("<td>", { class: "text-center pica-icon-column" });


        Picasso.Helper.createGlyphAndAvatar(td, "education", organization);

        tr.append(td);

        /* 2. column: Name */
        var td = $("<td>", { class: "pica-overflow" });

        var span = $("<span>", {
            text: organization.getName()
        });

        td.append(span);
        tr.append(td);

        /* 3 .column: Permissions */
        td = $("<td>", {
            class: "text-left"
        });

        var button = $("<button>", {
            class: "btn btn-default disabled",
            text: Picasso.Lang.get((organization.is(
                Picasso.Organization.Flags.ORG_ADMIN) ?
                "permission_you_admin" : "permission_you_member"))
        });

        td.append(button);
        tr.append(td);

        /* 4. column: reseller */
        td = $("<td>", {
            class: "text-right",
        });
        if (organization.getHasReseller()) {
            td.append(Picasso.Helper.createGlyph("crown","pica-glyph"))
        }
        tr.append(td)

        tr.addClass("pica-table-selection-disabled");

        return tr;
    };

    /**
     * Render given json object for list
     *
     * @param {Object}  obj  Given json object
     * @returns {String} Rendered json object
     **/

    var renderDevice = function (device) {
        var tr = $("<tr>", {
            id: device.getElemId(),
            draggable: false
        });

        /* 1. column: Icon */
        var td = $("<td>", {class: "text-center pica-icon-column pica-serial"});

        td.data("serial", device.serialize());

        /* Add icon */
        var span = $("<span>", {
            class: "pica-icon pica-glyph glyphicons"
        });

        span.addClass(device.getConnected() ? "glyphicons-ok-circle" : "glyphicons-ban-circle");
        span.tooltip({container: "body"});

        td.append(span);
        tr.append(td);

        /* 2. column: Name */
        var td = $("<td>", {
            class: "pica-overflow",
            text: device.getNick()
        });

        tr.append(td);

        var mapLink = "http://www.openstreetmap.org/";
        mapLink += "?mlat=";
        mapLink += device.getLocationLat();
        mapLink += "&mlon=";
        mapLink += device.getLocationLon();
        mapLink += "&zoom=12";

        /* 3. column: Location */
        var td = $("<td>", {
            text: device.getLocation()
        });

        if (device.getLocation() != null && device.getLocation().length > 0) {
            td.click(function () {
                window.open(mapLink, '_blank');
            });
        }
        tr.append(td);

        /* 3. column: Last seen */
        tr.append($("<td>", {
            text: device.getConnected() ? Picasso.Lang.get("status_connected") : device.getLastConnectionTime() ? Picasso.Helper.formatAge(device.getLastConnectionTime()) : ""
        }));


        /* Add dropdown menu */
        var div = $("<div>", { class: "pica-inline dropdown" });
        var menu = Picasso.Combo.createDropdownMenu(optionsForDevices);

        menu.addClass("dropdown-menu-right");
        div.append(menu);
        td.append(div);

        return tr;
    };

    /**
     * Update fields based on authType
     *
     * @param {Picasso.Account}  account  A #{@link Picasso.Account}
     **/

    var toggleClickability = function (account) {
        if (account.getAuthType() && "database" !== account.getAuthType()) {

            $("#profile_change_title")
                .attr("disabled", true)
                .off("click");

            $("#profile_change_first_name")
                .attr("disabled", true)
                .off("click");

            $("#profile_change_last_name")
                .attr("disabled", true)
                .off("click");

            $("#profile_change_phone")
                .attr("disabled", true)
                .off("click");


            $("#profile_change_password")
                .attr("disabled", true)
                .off("click")


            $("#profile_upgrade_validity,#profile_upgrade_storage")
                .attr("disabled", true)
                .click(function (e) {
                    e.preventDefault();
                })
        }
    };

    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDeleteDevice = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = DEVICES_TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Device($(val).find(".pica-serial").data("serial"))
        });

        /* Show dialog */
        var what = (1 === selected.length ? selected[0].getNick() :
            Picasso.Lang.get("string_selected_elements", selected.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_delete"),
            Picasso.Lang.get("dialog_body_delete", what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {
                var device = this;

                device.remove(

                    /* Success */
                    function () {
                        Picasso.Helper.fadeOut($("#" + device.getElemId()),
                            function () {
                                this.remove();

                                TABLE.checkIfEmpty();
                            });

                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_deleted",
                                Picasso.Lang.get("label_devices")), "info");
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);
                    }
                );
            });
        });

        dialog.show();
    };

    /**
     * Assign data to inputs
     *
     * @param {Picasso.Dialog.Account}  dialog  A #{@link Picasso.Dialog.Account}
     * @param {Object}                  inputs   Input fields ({ "id" = HTMLElement })
     **/

    var renderCustomer = function (customer, accountEmail) {
        $("#pica_customer_id").val(customer.getId());
        if ("" === customer.getEmail) {
            $("#pica_customer_email").text(accountEmail);
        } else {
            $("#pica_customer_email").text(customer.getEmail());
        }
        $("#pica_customer_name").text(customer.getName());
        $("#pica_customer_phone").text(customer.getPhone());
        $("#pica_customer_address_line1").text(customer.getAddressLine1());
        $("#pica_customer_address_line2").text(customer.getAddressLine2());
        $("#pica_customer_address_postal_code").text(customer.getAddressPostalCode());
        $("#pica_customer_address_city").text(customer.getAddressCity());
        $("#pica_customer_vat_id").text(customer.getVatId());
        if (customer.getAddressCountry()) {
            $("#pica_customer_address_country").text(Picasso.Lang.get("lang_country_" + customer.getAddressCountry()));
        }
    }

    /**
     * Handle click on change button
     *
     * @param {String}  elemId     Element ID
     * @param {String}  labelName  Name of the label
     * @param {String}  fieldName  Name of the field
     **/

    var handleClickChangeCustomer = function (elemId, labelName, fieldName) {
        var customer = Picasso.get("customer");

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_input_dialog",
            Picasso.Lang.get("dialog_title_change"),
            Picasso.Lang.get("dialog_body_change", Picasso.Lang.get(labelName)),
            (Picasso.Dialog.Flags.HIDE_ON_ENTER|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE));

        dialog.setAssignHandler(function (inputs) {
            inputs.pica_input_text.val(
                customer["get" + Picasso.Helper.capitalize(fieldName)]);
        });

        dialog.setOkHandler(function (inputs, dialog) {
            var val = inputs.pica_input_text.val();

            customer["set" + Picasso.Helper.capitalize(fieldName)](val);

            /* Store details */
            customer.setCustomer(
                /* Success */
                function (data, status) {

                    Picasso.set("customer", customer);

                    $(elemId).text(val);

                    switch (status) {
                        case 200:
                            Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_updated"),"info");
                            break;
                        case 201:
                            Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_updated"),"info");
                            break
                    }

                    dialog.hide();
                },

                    /* Error */
                    function (data, status) {
                        var oldVal = $(elemId).text();
                        customer["set" + Picasso.Helper.capitalize(fieldName)](oldVal);

                        switch (status) {
                            case 420:
                                Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_update_failed"),"danger");
                                break
                            case 424:
                                Picasso.Notification.show(Picasso.Lang.get("notification_stripe_customer_update_failed"),"danger");
                                break
                        }
                        dialog.hide();
                    }
            );
        });

        dialog.show();
    };


    /**
     * Fetch JSON data
     *
     * @param {Function}  onSuccess  Success handler
     * @param {Function}  onError    Error handler
     **/

    Picasso.Profile.fetch = function (onSuccess, onError) {
        var account = Picasso.get("account");

        /* Fetch account details */
        account.getDetails(

            /* Success */
            function (self) {
                $("#profile_username").text(self.getUserName());
                $("#profile_title").text(self.getTitle());
                $("#profile_first_name").text(self.getFirstName());
                $("#profile_last_name").text(self.getLastName());
                $("#profile_phone").text(self.getPhone());
                $("#profile_eth_address").text(self.getEthAddress());
                $("#profile_created").text(self.getCreationDate());
                $("#profile_language_text").text(Picasso.Lang.get("lang_"+self.getLanguage()));

                /* Add valid till */
                if (-1 !== self.getValidTill() && self.getValidTill()) {
                    var span = $("<span>", {
                        "data-tooltip": "",
                        title: Picasso.Helper.formatAge(self.getValidTill()),
                        text: Picasso.Helper.formatDate(self.getValidTill(), true)
                    });

                    span.tooltip({ container: "body" });

                    $("#profile_valid_till").html(span);
                }

                Picasso.Helper.updateStorageBar(self, "#profile_storage")
                toggleClickability(self);

                if (onSuccess) onSuccess();
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);

                if (onError) onError(e);
            }
        );

        /* Fetch account emails */
        account.getEmails(
            function (emails) {
                updateMails(emails);
            }
        );
    };

    /* Bind handlers */
    Picasso.Helper.bindNav($("#profile_nav"), handleClickNav);
    $("#device_delete").click(handleClickDeleteDevice);

    $(".profile-ui-avatar-change").click(function () {
        var account = Picasso.get("account");

        Picasso.Dialog.Upload.changeAvatar(account);
    });

    $(".profile-ui-avatar-delete").click(function () {
        var account = Picasso.get("account");

        Picasso.Dialog.Upload.deleteAvatar(account);
    });

    $("#profile_change_title").click(function (e) {
        handleClickChange.apply(this, [
            "#profile_title", "label_title", "Title"
        ]);
    });

    $("#profile_change_first_name").click(function (e) {
        handleClickChange.apply(this, [
            "#profile_first_name", "label_first_name", "FirstName"
        ]);
    });

    $("#profile_change_last_name").click(function (e) {
        handleClickChange.apply(this, [
            "#profile_last_name", "label_last_name", "LastName"
        ]);
    });

    $("#profile_change_phone").click(function (e) {
        handleClickChange.apply(this, [
            "#profile_phone", "label_phone", "Phone"
        ]);
    });

    $("#profile_change_eth_address").click(function (e) {
        handleClickChange.apply(this, [
            "#profile_eth_address", "label_eth_address", "ethAddress"
        ]);
    });

    $("#profile_change_email").click(function (e) {
        handleClickChangeCustomer.apply(this, [
            "#pica_customer_email", "label_customer_email", "Email"
        ]);
    });

    $("#profile_change_company_name").click(function (e) {
        handleClickChangeCustomer.apply(this, [
            "#pica_customer_name", "label_customer_name", "name"
        ]);
    });

    $("#profile_change_first_line").click(function (e) {
        handleClickChangeCustomer.apply(this, [
            "#pica_customer_address_line1", "label_customer_address_line1", "addressLine1"
        ]);
    });

    $("#profile_change_second_line").click(function (e) {
        handleClickChangeCustomer.apply(this, [
            "#pica_customer_address_line2", "label_customer_address_line2", "addressLine2"
        ]);
    });
    $("#profile_change_postal_code").click(function (e) {
        handleClickChangeCustomer.apply(this, [
            "#pica_customer_address_postal_code", "label_customer_address_postal_code", "addressPostalCode"
        ]);
    });
    $("#profile_change_location").click(function (e) {
        handleClickChangeCustomer.apply(this, [
            "#pica_customer_address_city", "label_customer_address_city", "addressCity"
        ]);
    });

    $("#profile_change_vat").click(function (e) {
        handleClickChangeCustomer.apply(this, [
            "#pica_customer_vat_id", "label_vat_id", "vatId"
        ]);
    });

    $("#profile_change_stripe_phone").click(function (e) {
        handleClickChangeCustomer.apply(this, [
            "#pica_customer_phone", "label_phone", "phone"
        ]);
    });

    $("#profile_manage_invoices").click(function (e) {
        window.location = '/wapi/customers?action=redirectToPortalSession&CSRFToken=' + Picasso.get("CSRF");
    });

    $("#profile_change_password").click(handleClickPassword);
    $("#profile_manage_language").click(handleClickLanguage);
    $("#profile_change_country").click(handleClickCountry);
    $("#profile_manage_emails").click(handleClickManageEmails);
    $("#profile_change_2f").click(handleClickManage2FA);
    $("#profile_close_account").click(handleClickCloseAccount);
    $("#copyRecoveryPhrase").click(copyToClipboard);
    $("#downloadRecoveryPhrase").click(downloadWords);
    $("#verifyRecoveryPhrase").click(verifyRecoveryPhrase);

    document.addEventListener("DOMContentLoaded", function () {
        const urlParams = new URLSearchParams(window.location.search);
        if (urlParams.has('storageWarning') && urlParams.get('storageWarning') === 'true') {
            Picasso.Notification.show(
                Picasso.Lang.get("notification_warning_storage_exceeded"), "warning");
        }
        if (urlParams.has('adminMFAWarning') && urlParams.get('adminMFAWarning') === 'true') {
            Picasso.Notification.show(
                Picasso.Lang.get("notification_warning_admin_without_mfa"), "warning");
        }

        if (urlParams.has('invoicesDue') && urlParams.get('invoicesDue') === 'true') {
            const dialog = new Picasso.Dialog("#pica_confirmation_dialog",
                Picasso.Lang.get("dialog_title_invoices_due"),
                Picasso.Lang.get("dialog_body_invoices_due"),
                Picasso.Dialog.Flags.NO_AUTOCLOSE);

            // Hide the Cancel button before showing the dialog
            $("#pica_confirmation_dialog .btn-default").hide();

            dialog.setOkHandler(function () {
                window.location = '/wapi/customers?action=redirectToPortalSession&CSRFToken=' + Picasso.get("CSRF");
            });

            dialog.show();
        }

    });


    /* Init */
    Picasso.Profile.fetch();
})();

