/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Servers")) {
    window.Picasso.Servers = {};
}

/* Private scope */
(function () {

    /* Globals */
    var TABLE = new Picasso.ViewModeTable("#table_servers",
        (Picasso.ViewModeTable.Flags.LIST_MODE|
            Picasso.ViewModeTable.Flags.NO_GALLERY_MODE|
            Picasso.ViewModeTable.Flags.ROWS_SELECTABLE));

    var optionsForServers = [
        {
            text: Picasso.Lang.get("action_restart"),
            icon: "restart",
            klass: "files-ui-download"
        },
        {
            text: Picasso.Lang.get("action_delete"),
            icon: "bin",
            klass: "files-ui-delete"
        }
    ];

    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDelete = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getTable().getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Server($(val).find(".pica-serial").data("serial"))
        });

        /* Show dialog */
        var what = (1 === selected.length ? selected[0].getName() :
            Picasso.Lang.get("string_selected_elements", selected.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_delete"),
            Picasso.Lang.get("dialog_body_delete", what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {
                var server = this;

                server.removeNode(

                    /* Success */
                    function () {
                        Picasso.Helper.fadeOut($("#" + server.getElemId()),
                            function () {
                                this.remove();

                                TABLE.checkIfEmpty();
                            });

                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_deleted",
                                Picasso.Lang.get("label_servers")), "info");
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);
                    }
                );
            });
        });

        dialog.show();
    };

    /**
     * Handle click on restart button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickRestart = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getTable().getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Server($(val).find(".pica-serial").data("serial"))
        });

        /* Show dialog */
        var what = (1 === selected.length ? selected[0]._name :
            Picasso.Lang.get("string_selected_elements", selected.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_restart"),
            Picasso.Lang.get("dialog_body_restart", what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {
                var server = this;

                server.restartNode(

                    /* Success */
                    function () {
                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_restarted",
                                Picasso.Lang.get("label_servers")), "info");
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);
                    }
                );
            });
        });

        dialog.show();
    };

    /**
     * Render given json object
     *
     * @param {Object}  server  Given json object
     * @returns {*}
     **/

    var renderServerGrid = function (server) {
        var div = $("<div>", {
            class: "pica-borderless text-center",
            "data-search-keys": [server.getName()]
        });

        var canvas = $("<canvas>", {
            width: 300,
            height: 150
        });

        var colors = Picasso.Helper.getChartColors();

        /* Create chart */
        new Chart(canvas, {
            type: "doughnut",
            data: {
                labels: [
                    Picasso.Lang.get("label_memory"),
                    Picasso.Lang.get("label_cpu"),
                    Picasso.Lang.get("label_folders"),
                    Picasso.Lang.get("label_accounts")
                ],
                datasets: [
                    {
                        data: [server.getMemoryUsage() || 0, 0, 0, 0],
                        backgroundColor: colors,
                        labels: [
                            Picasso.Lang.get("label_memory"),
                        ]
                    },
                    {
                        data: [server.getCpuUsage() || 0, 0, 0, 0],
                        backgroundColor: colors,
                        labels: [
                            Picasso.Lang.get("label_cpu"),
                        ]
                    },
                    {
                        data: [server.getFoldersCount(), server.getAccountsCount(), 0, 0],
                        backgroundColor: [colors[2], colors[3], colors[0], colors[1]],
                        labels: [
                            Picasso.Lang.get("label_folders"),
                            Picasso.Lang.get("label_accounts")
                        ]
                    }
                ]
            },
            options: {
                responsive: false,
                legend: {
                    position: "right",
                    labels: {
                        usePointStyles: true
                    }
                },
                tooltips: {
                    callbacks: {
                        label: function (tooltipItem, data) {
                            var dataset = data.datasets[tooltipItem.datasetIndex];
                            var index = tooltipItem.index;

                            return " " + dataset.labels[index] + ": " +
                                (dataset.formatter ? dataset.formatter(dataset.data[index]) : dataset.data[index]);
                        }
                    }
                }
            }
        });

        div.append($("<h3>", {
            class: "pica-overflow pica-border-bottom",
            text: server.getName()
        }));
        div.append(canvas);

        return div;
    };

    /**
     * Render given json object
     *
     * @param {Object}  server  Given json object
     * @returns {*}
     */

    var renderServerList = function (server) {
        var tr = $("<tr>", {
            id: server.getElemId(),
            draggable: false
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column pica-serial" });

        td.data("serial", server.serialize());

        var span = $("<span>", {
            class: "pica-icon pica-glyph glyphicons",
            "data-tooltip": ""
        });

        if (server.getLastOnlineTime() && 1 === server.getLastOnlineTime()) {
            span.addClass("glyphicons-ok-circle");
        } else {
            span.addClass("glyphicons-ban-circle");
        }

        span.tooltip({ container: "body" });

        td.append(span);
        tr.append(td);

        /* 2. column: Name */
        tr.append($("<td>", {
            class: "pica-name",
            text: server.getName()
        }));

        /* 3. column: Version */
        tr.append($("<td>", {
            text: server.getProgramVersion()
        }));

        /* 4. column: Memory usage */
        tr.append($("<td>", {
            class: "hidden-xs hidden-sm hidden-md hidden-lg-custom",
            text: (server.getMemoryUsageBytes() > 0 ?
                server.getMemoryUsage() + "% (" +
                Picasso.Helper.formatSize(server.getMemoryUsageBytes()) + ")" :
                "-")
        }));

        /* 5. column: Storage */
        td = $("<td>");

        if (server.getSpaceUsageBytes() && server.getSpaceFreeBytes()) {
            var p = Math.round(100 * server.getSpaceUsageBytes() / (server.getSpaceFreeBytes() + server.getSpaceUsageBytes()));
            td.append($("<span>", {
                text: p + "% "
            }));
            td.append($("<span>", {
                text: "(" + Picasso.Helper.formatSize(server.getSpaceUsageBytes()) + ")"
            }));
            td.addClass("hidden-xs hidden-sm hidden-md");
        }

        tr.append(td);

        /* 6. column: Client connections */
        tr.append($("<td>", {
            class: "hidden-xs hidden-sm hidden-md",
            text: server.getActiveSessionsCount()
        }));

        /* 7. column: Accounts */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: server.getAccountsCount()
        }));

        /* 8. column: Folders */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: server.getFoldersCount()
        }));

        /* 9. column: Last connect time */
        tr.append($("<td>", {
            class: "hidden-xs hidden-sm",
            text: server.getLastOnlineTime() !== 1 ?
                (   server.getLastOnlineTime() ?
                    Picasso.Helper.formatDate(server.getLastOnlineTime()) :
                    Picasso.Lang.get("status_unknown")
                ) :
                Picasso.Lang.get("status_connected")
        }));

        /* 10. column */
        tr.append($("<td>", {
            class: "pica-icon-column"
        }));

        /* Add dropdown menu */
        var div = $("<div>", { class: "pica-inline dropdown" });
        var menu = Picasso.Combo.createDropdownMenu(optionsForServers);

        menu.addClass("dropdown-menu-right");
        div.append(menu);
        td.append(div);

        return tr;
    };

    TABLE.setGridRenderer(renderServerGrid);
    TABLE.setListRenderer(renderServerList);

    /**
     * Fetch JSON data
     **/

    Picasso.Servers.fetch = function () {

        Picasso.Server.getAll(

            /* Success */
            function (servers) {
                TABLE.update(servers, renderServerList);
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /* Bind handlers */
    $("#servers_restart").click(handleClickRestart);
    $("#servers_delete").click(handleClickDelete);

    /* Init */
    Picasso.Servers.fetch();
})();

