/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Storage")) {
    window.Picasso.Storage = {};
}

/* Private scope */
(function () {

    /* Globals */
    Picasso.Storage.table = new Picasso.ViewModeTable("#storages_table",
        (Picasso.ViewModeTable.Flags.LIST_MODE|
            Picasso.ViewModeTable.Flags.NO_GALLERY_MODE|Picasso.ViewModeTable.Flags.ROWS_SELECTABLE));




    var optionsForStorages = [
        {
            text: "",
            icon: "",
            klass: "storages_disable"
        },
        {
            text: Picasso.Lang.get("action_delete"),
            icon: "bin",
            klass: "storages_delete"
        }

    ];

    /**
     * Handle options for storage
     *
     * @param {Number}  optIdx  Selected index
     **/

    var handleOptionsForStorage = function (optIdx) {
        var storage =$(this).parents("p").find("[data-serial]").data("serial")

        switch(optIdx) {
            case 0: ///< Enable disable
                ToggleStorage(storage);
                break;
            case 1: ///< Delete
                DeleteStorage([storage]);
                break;
        }
    };


    /**
     * Handle click on account create button
     **/

    var handleClickCreateStorage = function () {
        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_input_dialog",
                Picasso.Lang.get("dialog_title_create"),
                Picasso.Lang.get("dialog_body_create", Picasso.Lang.get("label_storage")));
        dialog.setOkHandler(function(inputs){
            Picasso.Storage.create(inputs.pica_input_text.val())
        });
        dialog.show();
    };

    /**
     * Handle click  delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDelete = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = Picasso.Storage.table.getTable().getSelectedRows();

        selected = $.map(selected, function (val) {
            return $(val).data("serial")
        });

        DeleteStorage(selected);
    };

    /**
     * Handle click  enable /  disable button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickToggle = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = Picasso.Storage.table.getTable().getSelectedRows()[0];

        ToggleStorage($(selected).data("serial"));
    };

    /**
     *  delete given Storage
     *
     * @param paths List of storages
     **/

    var DeleteStorage = function (paths) {

        var title = "dialog_title_delete";
        var body = "dialog_body_delete";
        var notification = "notification_item_deleted";



        /* Show dialog */
        var what = (1 === paths.length ? paths[0].path :
                Picasso.Lang.get("string_selected_elements", paths.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
                Picasso.Lang.get(title), Picasso.Lang.get(body, what));

        dialog.setOkHandler(function () {
            $.each(paths, function () {
                var obj = this;

                var url = "/wapi/storages" ;
                var data = {
                    action: "delete",
                    path: obj.path,
                    ID:obj.ID
                }
                Picasso.Helper.fireAjax(url, data,

                        function () {
                            Picasso.Notification.show(Picasso.Lang.get(notification,
                                    Picasso.Lang.get("label_storage")), "info");
                            Picasso.Storage.fetch();
                        },
                        function (e,status) {
                            switch (status) {
                                case 400:
                                    Picasso.Notification.show(
                                            Picasso.Lang.get(
                                                    "notification_error_delete_primary"),
                                            "danger");
                                    break;
                            }
                        }
                );
            });
        });

        dialog.show();
    };

    /**
     *  Enable / Disable given Storage
     *
     * @param paths List of storages
     **/

    var ToggleStorage = function (storage) {

        var title = "dialog_title_enable";
        var body = "dialog_body_enable";
        var notification = "notification_item_enabled";
        var action = "enable"

        if (storage.enabled) {
             title = "dialog_title_disable";
             body = "dialog_body_disable";
             notification = "notification_item_disabled";
             action = "disable"
        }



        /* Show dialog */
        var what = storage.path;

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
                Picasso.Lang.get(title), Picasso.Lang.get(body, what));

        dialog.setOkHandler(function () {
                var url = "/wapi/storages" ;
                var data = {
                    action: action,
                    path: storage.path,
                    ID:storage.ID
                }
                Picasso.Helper.fireAjax(url, data,

                        function () {
                            Picasso.Notification.show(Picasso.Lang.get(notification,
                                    Picasso.Lang.get("label_storage")), "info");
                            Picasso.Storage.fetch();
                        },
                        function (e) {
                            Picasso.debugLog(e);
                        }
                );
            });
        dialog.show();
    };

    /**
     * Render given json object
     *
     * @param {Object}  obj  Given json object
     * @returns {*}
     */

    var renderStorageGrid = function (obj) {
        var div = $("<div>", {
            id: "storage_" + Picasso.Helper.hashCode(obj.ID),
            class: "pica-borderless text-center",
            "data-search-keys": [obj.path]
        });

        /* Create chart */
        var canvas = $("<canvas>", {
            width: 300,
            height: 150
        });

        var colors = Picasso.Helper.getChartColors();
        var chart = Picasso.Helper.createChart(canvas, colors,
            Picasso.Helper.formatSize);

        Picasso.Helper.updateChart(chart, {
                "chart_remaining": obj.spaceFreeBytes,
                "chart_used": obj.spaceUsedBytes
        });

        /* Assemble */

        var h3 = $("<h3>", {
            class: "pica-overflow pica-border-bottom",
            text: obj.path
        })

        if (obj.primary) {
            h3.append($("<span>", {
                class: "glyphicon glyphicon-ok"
            }));
        }

        div.append(h3);

        div.append(canvas);

        var p = $("<p>", {
            class: "pica-well dropdown",
        });

        var a = $("<a>", {
            class: "pica-name pica-line-normal pica-overflow center-block dropdown-toggle",
            "data-toggle": "dropdown",
            "data-serial": JSON.stringify(obj),
            text: obj.path
        });

        a.append($("<span>", { class: "caret" }));

        p.append(a);

        if (obj.enabled) {
            optionsForStorages[0]["text"]=Picasso.Lang.get("action_disable");
            optionsForStorages[0]["icon"]="ban-circle";
        }else {
            optionsForStorages[0]["text"]=Picasso.Lang.get("action_enable");
            optionsForStorages[0]["icon"]="ok-circle";
        }

        /* Add dropdown menu */
        var menu = Picasso.Combo.createDropdownMenu(optionsForStorages,
                handleOptionsForStorage);

        p.append(menu);

        div.append(p);



        return div;
    };

    /**
     * Render given json object
     *
     * @param {Object}  obj  Given json object
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderStorageList = function (obj) {
        var tr = $("<tr>", {
            id: "storage_" + Picasso.Helper.hashCode(obj.ID),
            "data-search-keys": [obj.path],
            "data-serial":JSON.stringify(obj),
            draggable: false
        });

        var displayEnable = function(){
            $(".storages_enable").removeClass("pica-table-selection-hide");
            $(".storages_enable").show();
            $(".storages_disable").addClass("pica-table-selection-hide");
            $(".storages_disable").hide();
        }
        var displayDisable = function(){
            $(".storages_enable").addClass("pica-table-selection-hide");
            $(".storages_enable").hide();
            $(".storages_disable").removeClass("pica-table-selection-hide");
            $(".storages_disable").show();
        }
        tr.on("click",function(){
            if (obj.enabled) {
                displayDisable()
            }else {
                displayEnable()
            }
        })

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        var span = $("<span>", {
            class: "pica-icon pica-glyph glyphicon"
        });

        if (obj.enabled) {
            span.addClass("glyphicon-ok-circle");
        } else {
            span.addClass("glyphicon-ban-circle");
        }

        td.append(span);
        tr.append(td);

        /* 2. column: Path */
        tr.append($("<td>", {
            class: "pica-overflow",
            text: obj.path
        }));

        /* 2. column: Space */
        td = $("<td>");

        td.append($("<span>", {
            text: Picasso.Helper.formatSize(obj.spaceUsedBytes)
        }));

        td.append($("<span>", {
            class: "pica-slash-before",
            text: Picasso.Helper.formatSize(obj.spaceFreeBytes)
        }));

        tr.append(td);

        /* 3. column: Accounts */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: obj.nAccounts
        }));

        /* 4. column: Folders */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: obj.nFolders
        }));

        /* 5. column: Notes */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: obj.notes
        }));

        /* 6. column */
        var td = $("<td>", { class: "pica-icon-column" })
        if (obj.primary) {
            td.append($("<span>", {
                class: "glyphicon glyphicon-ok"
            }));
        }
        tr.append(td);

        return tr;
    };


    /**
     * Fetch JSON data
     **/

    Picasso.Storage.fetch = function () {
        var data = {
            action: "getAll"
        };

        Picasso.Storage.table.updateFromJSON("/wapi/storages", data);
    };

    Picasso.Storage.create = function (path) {
        var url = "/wapi/storages" ;
        var data = {
            action: "create",
            path: path
        }
        Picasso.Helper.fireAjax(url, data,

                /*  Success */
                function () {
                    Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_created",Picasso.Lang.get("label_storage")), "info");
                    Picasso.Storage.fetch();
                },
                /* Error */
                function (e) {
                    Picasso.debugLog(e);
                }
        );
    };

    Picasso.Storage.table.setGridRenderer(renderStorageGrid);
    Picasso.Storage.table.setListRenderer(renderStorageList);

    /* Bind handlers */
    $("#storages_create").click(handleClickCreateStorage);
    $(".storages_delete").click(handleClickDelete);
    $(".storages_enable").click(handleClickToggle);
    $(".storages_disable").click(handleClickToggle);

    /* Init */
    Picasso.Storage.fetch();
})();
