/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/* Private scope */
(function () {

    /**
     * Constructor for {@link Picasso.Group}
     *
     * @constructor
     **/

    Picasso.Group = function (data) {
        var json = Picasso.Helper.tryJSON(data);

        /* Fetch data: Load from server or restore from stringified object  */
        this._flags = Picasso.Helper.getPropOrDefValue(json, 0, "_flags");
        this._oid = Picasso.Helper.getPropOrDefValue(json, "", "id", "ID", "oid", "OID", "_oid");
        this._name = Picasso.Helper.getPropOrDefValue(json, "", "name", "displayName", "_name");
        this._notes = Picasso.Helper.getPropOrDefValue(json, "", "notes", "_notes");
        this._orgName = Picasso.Helper.getPropOrDefValue(json, "", "orgName", "_orgName");

        this._foldersCount = Picasso.Helper.getPropOrDefValue(json, 0, "nFolder", "_foldersCount");
        this._accountsCount = Picasso.Helper.getPropOrDefValue(json, 0, "nAccounts", "_accountsCount");
        this._adminsCount = Picasso.Helper.getPropOrDefValue(json, 0, "nAdmins", "_adminsCount");

        /* Create proxy to catch getter/setter calls */
        return Picasso.Helper.createProxy(this);
    };

    /**
     * Format folders and accounts
     *
     * @returns {String} Formatted description
     **/

    Picasso.Group.prototype.getDescription = function () {
        /* Assemble description */
        var description = null;

        if (0 === this.getAccountsCount()) {
            description = Picasso.Lang.get("string_members_none");
        } else if (0 === this.getFoldersCount()) {
            description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                "string_members_only", this.getFoldersCount());
        } else {
            description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                "string_members_and_folders",
                this.getAccountsCount(), this.getFoldersCount());
        }

        return description;
    };

    /**
     * Get avatar URL
     *
     * @returns {String} Url for avatar
     **/

    Picasso.Group.prototype.getAvatarUrl = function () {
        if ("" === this._oid) return "";

        return ("/" + Picasso.Helper.joinPaths([ "avatars", this._oid ]));
    };

    /**
     * Set folders for {@link Picasso.Group}
     *
     * @param {Array}     folders    Array of folders
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Group.prototype.setFolders = function (folders, onSuccess, onError) {
        var data = $.extend({
            id: this._oid,
            name: this._name ///< Required for create
        }, folders);

        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues(
            "/wapi/groups", "saveFoldersToGroup", data, onSuccess, onError
        );
    };

    /**
     * Get folders for {@link Picasso.Group}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Group.prototype.getFolders = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/groups";
        var data = {
            action: "getFolders",
            id: this._oid
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                /* Create folder objects */
                ary = $.map(ary, function (val) {
                    var file = new Picasso.File(val);

                    file.setOid(val.folderID); ///< folderID is actually OID

                    return file;
                });

                if (onSuccess) onSuccess(ary);
            }, onError
        );
    };

    /**
     * Set accounts of {@link Picasso.Group}
     *
     * @param {Array}     organizations  Array of organizations
     * @param {Function}  onSuccess      Call on success
     * @param {Function}  onError        Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Group.prototype.setAccounts = function (members, onSuccess, onError) {
        var data = $.extend({
            id: this._oid,
            name: this._name ///< Required for create
        }, members);

        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues(
            "/wapi/groups", "saveAccountsToGroup", data, onSuccess, onError
        );
    };

    /**
     * Get members for {@link Picasso.Group}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Group.prototype.getAccounts = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/groups";
        var data = {
            action: "getAccounts",
            id: this._oid
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                /* Create group objects */
                ary = $.map(ary, function (val) {
                    var account = new Picasso.Account(val);

                    /* Check whether account is group admin */
                    if (val.hasOwnProperty("isGroupAdmin") &&
                        true === val.isGroupAdmin)
                    {
                        account.setFlag(Picasso.Account.Flags.GROUP_ADMIN);
                    }

                    /* Check whether account is organization admin */
                    if (val.hasOwnProperty("isOrgAdmin") &&
                            true === val.isOrgAdmin)
                    {
                        account.setFlag(Picasso.Account.Flags.ORG_ADMIN);
                    }

                    return account;
                });

                if (onSuccess) onSuccess(ary);
            }, onError
        );
    };

    /**
     * Set organizations for {@link Picasso.Group}
     *
     * @param {Array}     organizations  Array of organizations
     * @param {Function}  onSuccess      Call on success
     * @param {Function}  onError        Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Group.prototype.setOrganizations = function (organizations,
                                                         onSuccess, onError)
    {
        var data = $.extend({
            id: this._oid,
            name: this._name ///< Required for create
        }, organizations);

        /* Fire ajax call */
        return Picasso.Helper.saveMultiValues(
            "/wapi/groups", "saveOrganizationsToGroup", data, onSuccess, onError
        );
    };

    /**
     * Get organizations for {@link Picasso.Group}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Group.prototype.getOrganizations = function (onSuccess, onError) {
        if ("" === this._oid) return;

        var url = "/wapi/groups";
        var data = {
            action: "getOrganizations",
            id: this._oid
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                /* Create group objects */
                ary = $.map(ary, function (val) {
                    return new Picasso.Organization(val);
                });

                if (onSuccess) onSuccess(ary);
            }, onError
        );
    };

    /**
     * Grant given permission on given folder
     *
     * @param {Picasso.File}        file        A #{@link Picasso.File}
     * @param {Picasso.Permission}  permission  A #{@link Picasso.Permission}
     * @param {Function}            onSuccess   Call on success
     * @param {Function}            onError     Call on error
     **/

    Picasso.Group.prototype.grant = function (file, permission, onSuccess,
                                              onError)
    {
        var url = "/wapi/groups";
        var data = {
            action: "grant",
            id: this.getOid(),
            folderID: file.getOid(),
            permission: Picasso.Permission.toString(permission)
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Revoke all permissions from given folder
     *
     * @param {Picasso.File}  file        A #{@link Picasso.File}
     * @param {Function}      onSuccess   Call on success
     * @param {Function}      onError     Call on error
     **/

    Picasso.Group.prototype.revokeAll = function (file, onSuccess, onError) {
        var url = "/wapi/groups";
        var data = {
            action: "revokeAll",
            id: this.getOid(),
            folderID: file.getOid()
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Remove {@link Picasso.Group}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Group.prototype.remove = function (onSuccess, onError) {
        var url = "/wapi/groups";
        var data = {
            action: "delete",
            id: this.getOid(),
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Leave {@link Picasso.Group}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Group.prototype.leave = function (onSuccess, onError) {
        var url = "/wapi/groups";
        var data = {
            action: "leave",
            id: this.getOid(),
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Save {@link Picasso.Group} settings
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Group.prototype.save = function (onSuccess, onError) {
        var self = this;
        var url = "/wapi/groups";
        var data = {
            method: "POST",
            action: "store",
            name: this.getName(),
            notes: this.getNotes()
        };

        /* Add oid if any */
        if (this._oid) {
            data.id = this._oid;
        }

        /* Fire ajax call */
        return Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (data, status) {
                if (data.hasOwnProperty("ID")) {
                    self.setOid(data.ID);
                }

                if (onSuccess) onSuccess(self);
            },
            onError
        );
    };

    /* Global - must be defined after constructor */

    /**
     * Get all groups
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     * @param {Number}    pageNum    Page number to load (optional)
     * @param {Number}    pageSize   Number of items per page (optional)
     * @param {String}    query      Query to filter results (optional)
     **/

    Picasso.Group.getAll = function (onSuccess, onError,
                                     pageNum, pageSize, query)
    {
        var url = "/wapi/groups";
        var data = {
            action: "getAll",
            page: pageNum,
            size: pageSize
        };

        /* Add query parameter */
        if (query) {
            data.query = query;
        }

        //check if there is already a calls on the page and cancel them
        if (Picasso.Helper.xhrPool[url] && Picasso.Helper.xhrPool[url].length > 0) {
            Picasso.Helper.abortAll(url)
            $("#groups_table tbody .pica-spinner").hide();
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                ary = $.map(ary, function (val) {
                    var group = new Picasso.Group(val);

                    if (val.hasOwnProperty("isGroupAdmin") &&
                            true === val.isGroupAdmin)
                    {
                        group.setFlag(Picasso.Group.Flags.GROUP_ADMIN);
                    }

                    return group;
                });

                if (onSuccess) onSuccess(ary, json);
            }, onError
        );
    };

    /* Flags - must be defined after constructor */

    if (!window.Picasso.Group.hasOwnProperty("Flags")) {
        window.Picasso.Group.Flags = {};
    }

    /* Flags */
    Picasso.Group.Flags.GROUP_ADMIN = (1 << 0); ///< Account is group admin
})();
