/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/* Private scope */
(function () {

    /**
     * Constructor for {@link Picasso.Link}
     *
     * @constructor
     **/

    Picasso.Invitation = function (data) {
        var json = Picasso.Helper.tryJSON(data);

        /* Fetch data: Load from server or restore from stringified object  */
        this._flags = Picasso.Helper.getPropOrDefValue(json, 0, "_flags");
        this._oid = Picasso.Helper.getPropOrDefValue(json, "", "_oid", "oid", "OID");

        this._folderName = Picasso.Helper.getPropOrDefValue(json, "", "_folderName", "folderName");

        this._recipientOid = Picasso.Helper.getPropOrDefValue(json, "", "_recipientOid");
        this._recipientName = Picasso.Helper.getPropOrDefValue(json, "", "_recipientName", "recipientName");
        this._displayName = Picasso.Helper.getPropOrDefValue(json, null, "_displayName", "displayName");
        this._recipientEmail = Picasso.Helper.getPropOrDefValue(json, "", "_recipientEmail", "recipientEmail");

        /* Parse permissions */
        this._permission = Picasso.Permission.fromValue(
            Picasso.Helper.getPropOrDefValue(json, null, "permission", "_permission"));


        this._file = null;

        /* Create proxy to catch getter/setter calls */
        return Picasso.Helper.createProxy(this);
    };

    /* Prototype methods */

    /**
     * Accept invitation
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Invitation.prototype.accept = function (onSuccess, onError) {
        var url = "/wapi/invitations";
        var data = {
            method: "POST",
            action: "accept",
            invitationID: this._oid
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Decline invitation
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Invitation.prototype.decline = function (onSuccess, onError) {
        var url = "/wapi/invitations";
        var data = {
            method: "POST",
            action: "decline",
            invitationID: this._oid
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Save invitation
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Invitation.prototype.save = function (onSuccess, onError) {
        var url = Picasso.Helper.joinPaths(["/wapi/invitations",
            this._file._url
        ]);
        var data = {
            method: "POST",
            action: "invite",
            permission: Picasso.Permission.toString(this._permission),
            sendInvitationMail: true,
            isGroup: this.is(Picasso.Invitation.Flags.FOR_GROUP),
            FolderID: this._file.getFolderId()
        };
        if (null !== this._recipientEmail) {
            data["recipientEmail"] = this._recipientEmail;
        }
        /* Add parameters (Prefer oid over name) */
        if (null !== this._recipientOid) {
            data["recipientOid"] = this._recipientOid;
        } else if (null !== this._recipientName) {
            data["recipientUsername"] = this._recipientName;///< FIXME: Should be renamed in API
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Remove invitation
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Invitation.prototype.remove = function (onSuccess, onError) {
        var url = "/wapi/invitations";
        var data = {
            action: "delete",
            invitationID: this.getOid()
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /* Global - must be defined after constructor */

    /**
     * Create file invitation
     *
     * @param {Picasso.File}              file        A {@link Picasso.File}
     * @param {String}                    oid         Oid of recipient (user or group)
     * @param {String}                    name        Name of recipient (user or group)
     * @param {Picasso.Permission}        permission  A {@link Picasso.Permission}
     * @param {Picasso.Invitation.Flags}  flags       Invitation flags
     * @param {Function}                  onSuccess   Call on success
     * @param {Function}                  onError     Call on error
     **/

    Picasso.Invitation.create = function (file, oid, email, name, permission, flags, onSuccess, onError) {

        var invitation = new Picasso.Invitation();

        invitation.setFile(file);
        invitation.setRecipientOid(oid);
        invitation.setRecipientName(name);
        invitation.setRecipientEmail(email);
        invitation.setPermission(permission);
        invitation.setFlags(flags);

        invitation.save(
                /* Success */
                function (data, status) {
                    /* Update object */
                    if (data.isInvitation) {
                        /* FIXME: Oid can be either from invitations or recipient */
                        invitation.setOid(data.oid);
                        invitation.setFlags(Picasso.Invitation.Flags.PENDING);
                    } else {
                        invitation.setRecipientOid(data.oid);
                    }

                    switch (status) {
                        case 200:
                            if (onSuccess) onSuccess(invitation);
                            break;
                    }
                }, onError
        );
    };

    /**
     * Get upload form if any
     *
     * @param {Picasso.File}  file       A {@link Picasso.File}
     * @param {Function}      onSuccess  Call on success
     * @param {Function}      onError    Call on error
     **/

    Picasso.Invitation.find = function (file, onSuccess, onError) {

        /* Handle types */
        if (file.is(Picasso.File.Flags.FOLDER)) {
            var url = "/wapi/invitations";
            var data = {
                action: "getByFolder",
                FolderID: file.getFolderId()
            };

            Picasso.Helper.fireAjax(url, data,

                /* Success */
                function (json, status) {
                    switch(status) {
                        case 200:
                            var ary = Picasso.Helper.getResultArray(json);
                            var invitations = [];

                            /* Sanity checks */
                            if (ary && 0 < ary.length) {

                            }
                                /* Create invitation objects  */
                                $.each(ary, function (k, v) {
                                    var invitation = new Picasso.Invitation(v);
                                    invitation.setFile(file);
                                    invitations.push(invitation);
                                });

                            if (onSuccess) {
                                onSuccess(invitations, status);
                            }
                            break;
                    }
                }, onError);

        } else if (onError) {
            onError();
        }
    };

     /* Flags - must be defined after constructor */

    if (!window.Picasso.Invitation.hasOwnProperty("Flags")) {
        window.Picasso.Invitation.Flags = {};
    }

    /* Flags */
    Picasso.Invitation.Flags.FOR_GROUP = (1 << 0);
    Picasso.Invitation.Flags.PENDING = (1 << 1);
})();
