/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/* Private scope */
(function () {

    /**
     * Constructor for {@link Picasso.Uploadform}
     *
     * @constructor
     **/

    Picasso.Uploadform = function (data) {
        var json = Picasso.Helper.tryJSON(data);

        /* Fetch data: Load from server or restore from stringified object  */
        this._flags = Picasso.Helper.getPropOrDefValue(json, 0, "_flags");
        this._folderId = Picasso.Helper.getPropOrDefValue(json, "", "folderID", "_folderId");
        this._description = Picasso.Helper.getPropOrDefValue(json, "", "description", "_description");
        this._heading = Picasso.Helper.getPropOrDefValue(json, "", "heading", "_heading");
        this._mails = Picasso.Helper.getPropOrDefValue(json, "", "mails", "_mails");
        this._url = Picasso.Helper.getPropOrDefValue(json, "", "url", "_url");
        this._validTill = Picasso.Helper.getPropOrDefValue(json, -1, "expiration_date", "_validTill");
        this._password = Picasso.Helper.getPropOrDefValue(json, "", "password", "_password");
        this._notifyOwner = Picasso.Helper.getPropOrDefValue(json,1,"notifyOwner","_notifyOwner")
        this._notifyAdmins = Picasso.Helper.getPropOrDefValue(json,1,"notifyAdmins","_notifyAdmins")
        this._notifyAdditional = Picasso.Helper.getPropOrDefValue(json,1,"notifyAdditional","_notifyAdditional")

        this._file = null;

        /* Update flags if unset */
        if (json && !json.hasOwnProperty("_flags")) {
            if (json.hasOwnProperty("passwordPolicy")) {
                switch (json.passwordPolicy.toLowerCase()) {
                    case "optional":
                        this._flags |=
                            Picasso.Uploadform.Flags.PASSWORD_OPTIONAL;
                        break;

                    case "required":
                        this._flags |=
                            Picasso.Uploadform.Flags.PASSWORD_REQUIRED;
                        break;

                    case "recommended":
                        this._flags |=
                            Picasso.Uploadform.Flags.PASSWORD_RECOMMENDED;
                        break;
                }
            }
        }

        /* Create proxy to catch getter/setter calls */
        return Picasso.Helper.createProxy(this);
    };

    /* Prototype methods */

    /**
     * Remove upload form
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Uploadform.prototype.remove = function (onSuccess, onError) {
        var url = "/wapi/folders";
        var data = {
            action: "removeUploadForm",
            FolderID: this.getFile().getFolderId()
        };

        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Save uploadform settings
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Uploadform.prototype.save = function (onSuccess, onError) {
        var url = "/wapi/folders";
        var data = {
            action: "updateUploadForm",
            method: "POST",
            FolderID: this.getFile().getFolderId(),
            heading: this.getHeading(),
            description: this.getDescription(),
            mails: this.getMails(),
            password: this.getPassword(),
            notifyOwner: this.getNotifyOwner(),
            notifyAdmins: this.getNotifyAdmins(),
            notifyAdditional: this.getNotifyAdditional()
        };

        /* Check values before sending them */
        if (-1 !== this.getValidTill()) {
            data.validTill = Picasso.Helper.formatJSONDate(
                this.getValidTill());
        } else {
            data.validTill = "";
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /* Global - must be defined after constructor */

    /**
     * Find upload form if any
     *
     * @param {Picasso.File}  file       A {@link Picasso.File}
     * @param {Function}      onSuccess  Call on success
     * @param {Function}      onError    Call on error
     **/

    Picasso.Uploadform.find = function (file, onSuccess, onError) {

        var url = "/wapi/folders";
        var data = {
            action: "getUploadForm",
            FolderID: file.getFolderId()
        };

        Picasso.Helper.fireAjax(url, data,

                /* Success */
                function (json, status) {
                    switch (status) {
                        case 200:
                            var ary = Picasso.Helper.getResultArray(json);

                            /* Sanity checks */
                            if (ary && 0 < ary.length) {
                                /* Create uploadform object */
                                ary = new Picasso.Uploadform(ary[0]);

                                ary.setFile(file);
                            }

                            if (onSuccess) {
                                onSuccess(ary, status);
                            }
                    }
                }, onError
        );
    };

    /* Flags - must be defined after constructor */

    if (!window.Picasso.Uploadform.hasOwnProperty("Flags")) {
        window.Picasso.Uploadform.Flags = {};
    }

    /* Flags */
    Picasso.Uploadform.Flags.PASSWORD_OPTIONAL = (1 << 0);
    Picasso.Uploadform.Flags.PASSWORD_REQUIRED = (1 << 1);
    Picasso.Uploadform.Flags.PASSWORD_RECOMMENDED = (1 << 2);
})();
