/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Combo")) {
    window.Picasso.Combo = {};
}

/* Private scope */
(function () {

    /**
     * Helper to create a dropdown menu or a button when there is only one option
     *
     * @param {Array}     options    Array with options ({ text | icon | id | klass | group })
     * @param {Function}  onSelect   Selection callback
     * @param {Number}    startIdx   Index of the visible item (optional; defaults to 0)
     * @param {String}    klass      Additional classes (optional)
     * @param {String}    caption    Caption of toggle button (optional; defaults to 0 option elem)
     *
     * @returns {jQuery|HTMLElement} Assembled dropdown button with menu
     **/

    Picasso.Combo.createDropdownOrButton = function (options, onSelect,
                                                     startIdx, klass, caption) {
        var elem;
        var selected = options[startIdx || 0];

        /* Create a button or a dropdown */
        if (1 === options.length) {
            elem = $("<button>", {
                class: "btn btn-default " + (klass ? klass : ""),
                text: (caption || selected.text)
            });

            /* Add custom classes */
            if (options[0].klass) {
                /* Check whether passed class styles the button */
                if (-1 !== options[0].klass.indexOf("btn-")) {
                    elem.removeClass("btn-default");
                }

                elem.addClass(options[0].klass);
            }

            if (onSelect) {
                elem.click(function () {
                    var that = $(this);

                    /* Call if not disabled */
                    if (!that.hasClass("disabled")) {
                        onSelect.apply(this, [0]);
                    }
                });
            }
        } else {
            /* Create html */
            elem = $("<div>", {
                class: "pica-inline dropdown " + (klass ? klass : ""),
                "data-selected": selected.text
            });

            var toggle = $("<button>", {
                class: "btn btn-default dropdown-toggle",
                text: (caption || selected.text),
                "data-toggle": "dropdown"
            });

            toggle.append($("<span>", { class: "caret" }));

            elem.append(toggle);

            /* Create menu */
            var menu = Picasso.Combo.createDropdownMenu(
                options, onSelect, (startIdx || 0));

            elem.append(menu);
        }

        return elem;
    };

    /**
     * Create dropdown menu
     *
     * The return value of the selection callback determines whether the selection gets set:
     *
     * {@code true} - Selection gets set
     * {@code false} - Selection is unchanged (eg for options)
     *
     * @param {Array}     options    Array with options ({ text | icon | id | klass | group })
     * @param {Function}  onSelect   Selection callback (Return value true uses selected value as caption)
     * @param {Number}    startIdx   Index of the visible item (optional)
     *
     * @returns {jQuery|HTMLElement} Assembled dropdown menu
     **/

    Picasso.Combo.createDropdownMenu = function (options, onSelect, startIdx) {

        /* Create menu */
        var menu = $("<ul>", { class: "dropdown-menu" });

        /* Handle menu click */
        var handleMenuClick = function (e, userdata) {
            var that = $(this);

            /* Call select handler and check result */
            if (onSelect) {
                var ret = onSelect.apply(this, [that.data("dropdown-idx"), userdata]);

                /* The return value can change the caption of the dropdown box */
                if (true === ret || "string" === typeof ret) {
                    var toggle = that.parents(".dropdown").find(".dropdown-toggle");

                    /* Rewrite toggle content */
                    toggle.text((true === ret ? that.text() : ret));
                    toggle.append($("<span>", { class: "caret" }));
                }
            }

            /* Handle type and group */
            var type = that.data("dropdown-type");
            var group = that.data("dropdown-group");

            var span = that.find("span");
            var grouped = menu.find("a[data-dropdown-group=" + group + "] span");

            /* Handle different types:
             *  - Checkbox: Allow selection of all or none; when in a group
             *              behave like radio buttons
             *  - Radio: Allow selection of only one
             **/
            if ("checkbox" === type) {
                var checked = grouped.filter(".glyphicons-check");

                if ((span.hasClass("glyphicons-check") && 1 < checked.length) ||
                    span.hasClass("glyphicons-unchecked")) {
                    span.toggleClass("glyphicons-unchecked")
                        .toggleClass("glyphicons-check");
                }
            } else if ("radio" === type) {
                grouped.removeClass("glyphicons-check")
                    .addClass("glyphicons-unchecked");

                span.removeClass("glyphicons-unchecked")
                    .addClass("glyphicons-check");
            }
        };

        /* Finally add options */
        $.each(options, function (i, option) {
            if (!option) {
                return;
            }
            var li = $("<li>");

            if (null === option) {
                li.addClass("divider");
            } else if (option.hasOwnProperty("type") &&
                "divider" === option.type)
            {
                li.addClass("divider");

                if (option.hasOwnProperty("klass")) {
                    li.addClass(option.klass);
                }
            } else {
                /* Add id if any */
                if (option.hasOwnProperty("id")) {
                    li.attr("id", option.id);
                }

                /* Add class if any */
                if (option.hasOwnProperty("klass")) {
                    li.addClass(option.klass);
                }

                var a = $("<a>");

                /* Add icon if any */
                if (option.hasOwnProperty("icon")) {
                    a.append(Picasso.Helper.createGlyph(option.icon));
                }

                /* Add name if any */
                if (option.hasOwnProperty("name")) {
                    a.attr("name", option.name);
                }

                /* Add value if any */
                if (option.hasOwnProperty("value")) {
                    a.attr("value", option.value);
                }

                /* Add type if any */
                if (option.hasOwnProperty("type")) {
                    a.append(Picasso.Helper.createGlyph(
                        (startIdx === i ? "check" : "unchecked")));
                    a.attr("data-dropdown-type", option.type);
                }

                /* Add group if any */
                if (option.hasOwnProperty("group")) {
                    a.attr("data-dropdown-group", option.group);
                }

                a.data("dropdown-idx", i); ///< Assign index to make handler easier

                /* Add lang if any */
                if (option.hasOwnProperty("lang")) {
                    var lang = $("<lang>", {
                        name: option.lang,
                        text: Picasso.Lang.get(option.lang)
                    });

                    a.append(lang);
                } else {
                    a.append(option.text);
                }

                /* Disable if any */
                if (!option.hasOwnProperty("disabled")) {
                    a.click(handleMenuClick);
                }

                li.append(a);
            }

            menu.append(li);
        });

        return menu;
    };

    /**
     * Fix position of dropdowns in scroller
     *
     * @param {jQuery|HTMLElement}  elem  Parent scroller
     **/

    Picasso.Combo.fixDropdownInScroller = function (elem) {
        /* Fix dropdowns in scroller */
        $(elem).on("show.bs.dropdown", function(e) {
            var toggle = $(e.target).find(".dropdown-toggle");
            var menu = $(e.target).find(".dropdown-menu");

            var leftPos = 0;
            var topPos = toggle.offset().top;

            /* Update position based on browser */
            if (Picasso.Helper.isInternetExplorer()) {
                leftPos = toggle.offset().left;
                topPos += toggle.outerHeight(true);
            } else {
                leftPos = (toggle.offset().left - $(this).offset().left);
            }

            menu.css({
                "top": topPos + "px",
                "left": leftPos + "px",
                "right": "auto"
            });

            menu.data("open", true);
        });
    };
})();