/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/**
 * Some custom classes for specific behavior are used:
 *
 * .pica-taginput - Base class for parent div
 * .pica-taginput-tag - Taginput tag
 * .pica-taginput-input - Input field
 * .pica-taginput-button - Send button
 * .pica-taginput-dropdown - Dropdown for completion
 * .pica-taginput-dropdown-fixed - Dropdown entry that is not removed
 **/

/* Private scope */
(function () {

    var EMAIL_REGEX = /([a-zA-Z0-9!#$%&'*+\/=?^_`{|}~-]+(?:\.[a-zA-Z0-9!#$%&'*+\/=?^_`{|}~-]+)*@(?:[a-zA-Z0-9](?:[a-zA-Z0-9-]*[a-zA-Z0-9])?\.)+[a-zA-Z0-9](?:[a-zA-Z0-9-]*[a-zA-Z0-9])?)/g;
    var OUTLOOK_REGEX = /[0-9A-Za-zÀ-ÖØ-öø-ÿ+,()| ]+<{1}\S+[0-9A-Za-zÀ-ÖØ-öø-ÿ+,()| ]@[a-z0-9-\.]+>{1}/g
    var SEPARATORS = [',',';',':','\t'];


    /**
     * Handle keys on tag input
     *
     * @param {Event}  e  Key down event
     **/

    var handleKeyPress = function (e) {
        var value = e.currentTarget.value;
        var lis = this._dropdown.find("li").not(".divider");



        /* Handle keys */
        switch (e.which) {
            case 8: ///< Backspace

                /* Remove last tag if input is empty */
                if (0 < (this._flags & Picasso.TagInput.Flags.ADD_TAGS) &&
                        0 === value.length)
                {
                    $(".pica-taginput-tag").last().click();
                }

                /* Show completions when editing or filter mode is enabled */
                if (this._onCompletion && (0 < value.length ||
                    0 < (this._flags & Picasso.TagInput.Flags.COMPLETION_FILTER)))
                {
                    this._onCompletion(this, value);
                }
                break;

            case 13: ///< Enter

                /* Don't bubble this up */
                e.stopPropagation();
                e.preventDefault();

                var selected = lis.filter(".selected");

                /* 1) Autocompletion entry selected */
                if (0 !== selected.length) {
                    selected.find("a").click(); ///< Trigger click on selected entry
                /* 2) Enter when just a string is entered */
                } else {
                    this.hideCompletions();

                    if (0 < value.length) {
                        /* Validate input if necessary */
                        if (this.isInputValid()) {
                            if (0 === (this._flags & Picasso.TagInput.Flags.COMPLETION_ONLY)) {
                                if (this._onSubmit) {
                                    this._onSubmit(e, {});
                                    this.clear(); ///< Clear after handler
                                    break;
                                }
                                this.clear(); ///< Clear after handler
                            }

                            if (0 < (this._flags & Picasso.TagInput.Flags.ADD_TAGS) &&
                                (0 === (this._flags & Picasso.TagInput.Flags.COMPLETION_ONLY)))
                            {
                                this.addTag(value, this._newIcon);
                                this.clearInput();
                                break;
                            }
                        } else if(this._onValidationError) {
                            this._onValidationError();
                            break;
                        }
                    }


                }
                break;

            case 27: ///< ESC
                /* Don't bubble this up to dialog, otherwise it gets closed */
                e.stopPropagation();
                e.preventDefault();

                /* Hide dropdown and clear input */
                this.hideCompletions();
                this.clearInput();
                break;

            case 38: ///< Arrow up
                /* Fall through */
            case 40: ///< Arrow down
                if (this._dropdown.is(":visible")) {

                    /* Don't bubble this up */
                    e.stopPropagation();
                    e.preventDefault();

                    var selected = lis.filter(".selected");
                    var nextSelected = (38 === e.which ? selected.prevAll() :
                        selected.nextAll()).not(".divider").first();

                    /* Handle prev and next selection via arrow keys */
                    if (0 !== selected.length && 0 !== nextSelected.length) {
                        lis.removeClass("selected");
                        nextSelected.addClass("selected");
                    } else if (38 === e.which) {
                        lis.removeClass("selected");
                        lis.last().addClass("selected");
                    } else if (40 === e.which) {
                        lis.removeClass("selected");
                        lis.first().addClass("selected")
                    }
                }
                break;

            default: ///< Other keys
                if (0 < value.length && this._onCompletion) {
                    this._onCompletion(this, value);
                }
        }
    };

    /**
     * Prepend or append li into dropdown based on position of fixed entry
     *
     * @param {jQuery|HTMLElement}  dropdown  Dropdown to append to
     * @param {jQuery|HTMLElement}  li        Li to insert
     **/

    var prependOrAppend = function (dropdown, li) {
        /* Prepend or append */
        var fixed = dropdown.find("li.pica-taginput-dropdown-fixed");

        if (0 < fixed.length) {
            li.insertBefore(fixed);
        } else {
            dropdown.append(li);
        }
    };

    /**
     * Constructor for {@link Picasso.TagInput}
     *
     * @param {String}  id       Element Id
     * @param {Number}  options  Options for this element (optional)
     *
     * @constructor
     **/

    Picasso.TagInput = function (id, options) {
        var self = this;
        var container = $(id);

        /* Init */
        this._div = container.find(".pica-taginput-div");
        this._dropdown = container.find("ul.pica-taginput-dropdown");
        this._input = container.find(".pica-taginput-input");

        this._newIcon = "message-new";

        /* Options */
        this._flags = (options || 0);

        /* Callbacks */
        this._onCompletion = null;
        this._onCompletionSel = null;
        this._onValidationError = null;
        this._onValidation = null;
        this._onSubmit = null;

        /* Bind events */
        this._input.off("keyup").on("keyup", $.proxy(handleKeyPress, this));

        if (0 < (self._flags & (Picasso.TagInput.Flags.COMPLETION_FILTER|
            Picasso.TagInput.Flags.COMPLETION_ONLY_ONCE)))
        {
            this._input.off("focus").on("focus", function (e) {
                if (self._onCompletion) {

                    /* Check whether we complete only once */
                    if (0 < (self._flags & Picasso.TagInput.Flags.COMPLETION_ONLY_ONCE)) {
                        if (1 < self._dropdown.children().length) {
                            self.showCompletions();

                            return;
                        }

                    }

                    self._onCompletion(self, "");
                }
            });
        }

        container.find(".pica-taginput-button").click(function (e) {
            e.stopPropagation();
            e.preventDefault();

            /* Validate input if necessary */
            if (self.isInputValid()) {
                self.hideCompletions();

                if (self._onSubmit) {
                    self._onSubmit(e, {});
                }

                self.clear(); ///< Clear after handler
            } else if (self._onValidationError) {
                self._onValidationError();
            }
        });
    };

    /**
     * Check whether input is valid
     *
     * @return {Boolean} Either true when input is valid; otherwise false
     **/

    Picasso.TagInput.prototype.isInputValid = function () {
        //if (0 === (this._flags & Picasso.TagInput.Flags.VALIDATE))
        //    return true;
        if (this._onValidation){
            return this._onValidation(this._input.val());
        }

        function isEmailList(input) {
            return input.match(EMAIL_REGEX) && input.match(EMAIL_REGEX).length;
        }

        function isOutlookList(input) {
            return input.match(OUTLOOK_REGEX) && input.match(OUTLOOK_REGEX).length;
        }

        function isAuthorizedSeparator(input) {
            //check if all used separators are authorized
            var delimiters = input.replace(EMAIL_REGEX,"").trim();
            var matching  = delimiters.match(new RegExp(SEPARATORS.join('|'), 'g'));
            return  matching && matching .length === delimiters.length;
        }


        if (this._input.attr("type") == "email" && !this._input.attr("should-validate"))
            return true;
        else if (this._input.attr("type") == "email" && isOutlookList(this._input.val().trim()))
            return true;
        else if (this._input.attr("type") == "email" && isEmailList(this._input.val().trim().replace(/ /g,';')) && isAuthorizedSeparator(this._input.val().trim().replace(/ /g,';'))) {
            return true;
        } else
            return this._input.get(0).checkValidity();
    };

    /**
     * Add tag
     *
     * @param {String}  name      Name of the tag
     * @param {String}  icon      Name of the icon if any
     * @param {Object}  userdata  Additional userdata (optional)
     *
     * @return {jQuery|HTMLElement} Created tag or null
     **/

    Picasso.TagInput.prototype.addTag = function (name, icon, userdata) {
        var self = this;

        /* Check whether this tag already exists */
        var id = "tag_" + Picasso.Helper.hashCode(name);

        if (0 !== $("#" + id).length) return null;

        /* Check validity */
        if (!this.isInputValid() && this._onValidationError && icon!="group") {
            this._onValidationError();
        }

        /* Create new tag */
        var div = $("<div>", {
            id: id,
            class: "pica-taginput-tag"
        });

        div.click(function () {
            $(this).remove();

            self.recalculate();
        });

        /* Add icon if any */
        if (icon) {
            div.append(Picasso.Helper.createGlyph(icon));
        }

        div.append(name);
        div.append($("<span>", { html: "&times;" }));

        div.data("tag-name", name);
        div.data("tag-userdata", userdata);

        this._input.before(div);

        this.recalculate();

        return div;
    };

    /**
     * Get all tags
     *
     * @return {Array}  Array of tags; might be empty
     **/

    Picasso.TagInput.prototype.getTags = function () {
        var tags = [];

        /* Find tags */
        $.each(this._div.find(".pica-taginput-tag"), function () {
            var that = $(this);

            tags.push({
                name: that.data("tag-name"),
                userdata: that.data("tag-userdata")
            });
        });

        /* Check whether input still has a string */
        var val = this._input.val().trim();

        //check outLook Format
        var outlookEmails = val.match(/[0-9A-Za-zÀ-ÖØ-öø-ÿ+,()| ]+<{1}\S+[0-9A-Za-zÀ-ÖØ-öø-ÿ+,()| ]@[a-z0-9-\.]+>{1}/g);
        if (outlookEmails && 0 !== outlookEmails.length) {
            for (var i = 0 ;i< outlookEmails.length;i++){
                let mail = outlookEmails[i];
                var name = mail.split('<')[0].trim();
                var email = mail.split('<')[1];
                email =email.substring(0, email.length - 1);
                tags.push({
                    name: name,
                    userdata: {"OID" : null,
                        "email": email}
                });
            }

        }else {
            val = val.replace(/ /g,';');
            var recipients = val.split(new RegExp(SEPARATORS.join('|'), 'g')).filter(Boolean);

            if (0 !== recipients.length) {

                for (var i =0;i<recipients.length;i++) {
                    let recipient = recipients[i] ? recipients[i].trim().toLowerCase() : "";
                    tags.push({
                        name: recipient,
                        userdata: null
                    });
                }


            }
        }
        return tags;
    };

    /**
     * Get value of input field
     *
     * @returns {String} Value of input field
     **/

    Picasso.TagInput.prototype.getInputValue = function () {
        return this._input.val();
    };

    /**
     * Get completion entries
     *
     * @returns {Array} Completion entries
     **/

    Picasso.TagInput.prototype.getCompletionEntries = function () {
        return this._dropdown.find("li");
    };

    /**
     * Add completion divider
     *
     * @returns {jQuery|HTMLElement} Rendered element
     **/

    Picasso.TagInput.prototype.addCompletionDivider = function () {
        var li = $("<li>", {
            class: "divider"
        });

        prependOrAppend(this._dropdown, li);

        return li;
    };

    /**
     * Add completion entry
     *
     * @param {String}  name         Name of the entry
     * @param {String}  icon         Name of the icon (optional)
     * @param {String}  description  Description of the entry (optional)
     * @param {Object}  userdata     Additional userdata (optional)
     *
     * @returns {jQuery|HTMLElement} Rendered element
     **/

    Picasso.TagInput.prototype.addCompletionEntry = function (name, icon,
                                                              description, userdata)
    {
        var self = this;

        /* Create entry */
        var li = $("<li>");
        var a = $("<a>");

        a.click(function (e) {
            e.stopPropagation();
            e.preventDefault();

            /* Add only when allowed */
            if (0 < (self._flags & Picasso.TagInput.Flags.ADD_TAGS)) {
                self.addTag(name, icon, userdata);

                self.clearInput();
            } else if (self._onCompletionSel) { ///< Simulate enter

                if ($(self._div).find("button")){
                    $(self._div).find("button").text(userdata.name);
                }

                var ret = self._onCompletionSel(e, userdata);
                if (true === ret) {
                    self._input.val(name);
                } else {
                    self.clearInput();
                }
            }

            /* Check whether dropdown must be closed after selection */
            if (0 < (self._flags & Picasso.TagInput.Flags.COMPLETION_SELECTION_SINGLE)) {
                self.hideCompletions();
            } else {
                /* Reset focus to catch keys like ESC */
                self._input.focus();
            }
        });

        /* Create media */
        var media = $("<div>", { class: "media" });

        /* Add icon if any */
        if (icon) {
            var left = $("<div>", {
                class: "pica-vertical-middle media-left",
                html: Picasso.Helper.createGlyph(icon, "pica-glyph")
            });

            media.append(left);
        }

        /* Create media body */
        var body = $("<div>", { class: "media-body" });

        /* Add heading */
        var heading = $("<h5>", {
            class: "media-heading text-primary",
            text: name
        });

        body.append(heading);

        /* Create description if any */
        if (description) {
            var p = $("<p>", {
                class: "small text-muted",
                text: description
            });

            body.append(p);
        }

        media.append(body);

        a.append(media);
        li.append(a);

        prependOrAppend(this._dropdown, li);

        return li;
    };

    /**
     * Add fixed completion entry
     *
     * @param {String}  name         Name of the entry
     * @param {String}  icon         Name of the icon (optional)
     * @param {String}  description  Description of the entry (optional)
     * @param {Object}  userdata     Additional userdata (optional)
     *
     * @returns {jQuery|HTMLElement} Rendered element
     **/

    Picasso.TagInput.prototype.addFixedCompletionEntry = function (name, icon,
                                                                   description,
                                                                   userdata)
    {
        var li = this.addCompletionEntry(name, icon, description, userdata);

        li.addClass("pica-taginput-dropdown-fixed");

        return li;
    };

     /**
     * Recalculate height of input field
     **/

    Picasso.TagInput.prototype.recalculate = function () {
        var height = this._div.css("height");

        /* Height might be 0 before initial rendering - so we reset
         * (aka unset) the calculated heights to initial value */
        if(0 === parseInt(height)) {
            height = "";
        }

        this._div.siblings(".input-group-btn")
            .children("button").css("height", height);
        this._div.find(".pica-taginput-input").css("height", height);
    };

    /**
     * Whether completions dropdown is visible
     *
     * @returns {Boolean} Either true when it is visible; otherwise false
     **/

    Picasso.TagInput.prototype.isCompletionVisible = function () {
        return this._dropdown.is(":visible");
    };

    /**
     * Show completions
     **/

    Picasso.TagInput.prototype.showCompletions = function () {
        if (!this.isCompletionVisible()) {
            if ($(".dropdown-toggle").length > 0) {
             this._dropdown.dropdown("toggle");
            }
        }
    };

    /**
     * Hide completions
     **/

    Picasso.TagInput.prototype.hideCompletions = function () {
        if (this.isCompletionVisible()) {
            this._dropdown.dropdown("toggle");
        }

        this.clearCompletions();
    };

    /**
     * Clear all tags
     **/

    Picasso.TagInput.prototype.clear = function () {
        this._div.find(".pica-taginput-tag").remove();

        this.clearInput();
        this.recalculate();
    };

    /**
     * Clear fixed elements
     **/

    Picasso.TagInput.prototype.clearFixed = function () {
        this._dropdown.find("li.pica-taginput-dropdown-fixed").remove();
    };

    /**
     * Clear all completions
     **/

    Picasso.TagInput.prototype.clearCompletions = function () {
        this._dropdown.find("li:not(.pica-taginput-dropdown-fixed)").remove();
    };

    /**
     * Clear input
     **/

    Picasso.TagInput.prototype.clearInput = function () {
        this._input.val("");
    };

    /**
     * Set completion handler
     *
     * @param {Function}  onCompletion  Completion handler
     **/

    Picasso.TagInput.prototype.setCompletionHandler = function (onCompletion) {
        if (onCompletion) {
            this._onCompletion = Picasso.Helper.debounce(onCompletion, 500);
        }
    };

    /**
     * Set completion select handler
     *
     * @param {Function}  onComplSel  Selection handler
     **/

    Picasso.TagInput.prototype.setCompletionSelectHandler = function (onComplSel) {
        if (onComplSel) {
            this._onCompletionSel = onComplSel;
        }
    };

    /**
     * Set submit handler
     *
     * @param {Function}  onSubmit  Button press handler (onSubmit(tagInput))
     **/

    Picasso.TagInput.prototype.setSubmitHandler = function (onSubmit) {
        if (onSubmit) {
            this._onSubmit = onSubmit;
        }
    };

    /**
     * Set validation error handler
     *
     * @param {Function}  onValidationError  Validation error handler
     **/

    Picasso.TagInput.prototype.setValidationErrorHandler = function (onValidationError) {
        if (onValidationError) {
            this._onValidationError = onValidationError;
        }
    };
    /**
     * Set input validation  handler
     *
     * @param {Function}  onValidationError  Validation  handler
     **/

    Picasso.TagInput.prototype.setValidationHandler = function (onValidation) {
        if (onValidation) {
            this._onValidation = onValidation;
        }
    };

    /**
     * Set icon for new tags
     *
     * @param {String}  newIcon  Icon for new tags
     **/

    Picasso.TagInput.prototype.setNewIcon = function (newIcon) {
        if (newIcon) {
            this._newIcon = newIcon;
        }
    };

    /* Flags - must be defined after constructor */

    if (!window.Picasso.TagInput.hasOwnProperty("Flags")) {
        window.Picasso.TagInput.Flags = {};
    }

    /* Flags */
    Picasso.TagInput.Flags.ADD_TAGS = (1 << 0); ///< Add tags
    Picasso.TagInput.Flags.VALIDATE = (1 << 1); ///< Validate input before adding tags
    Picasso.TagInput.Flags.COMPLETION_ONLY = (1 << 2); ///< Accept only completed inputs
    Picasso.TagInput.Flags.COMPLETION_ONLY_ONCE = (1 << 3); ///< Complete only once
    Picasso.TagInput.Flags.COMPLETION_FILTER = (1 << 4); ///< Reverse logic and filter set
    Picasso.TagInput.Flags.COMPLETION_SELECTION_SINGLE = (1 << 5); ///< Single selection closes dropdown

    /* Global - must be defined after constructor */

    /**
     * Handle member and group completion
     *
     * @param {Picasso.TagInput}  tagInput  A #{@link Picasso.TagInput}
     * @param {String}            val       Value to load completions for
     **/

    Picasso.TagInput.handleAccountAndGroupCompletion = function (tagInput, val) {
        var url = "/wapi/accounts";
        var data = {
            action: "findAccountsAndGroups",
            query: val
        };

        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                var firstGroup = false;

                tagInput.clearCompletions();

                $.each(ary, function (idx) {
                    var icon = (this.isGroup ? "group" : "user");
                    var description = this.email;

                    /* Add divider between users and groups */
                    if (!firstGroup && this.isGroup) {
                        firstGroup = true;

                        if (0 !== idx) tagInput.addCompletionDivider();
                    }

                    if (this.isGroup) {
                        if (0 === this.nAccounts) {
                            description = Picasso.Lang.get("string_members_none");
                        } else if (0 === this.nFolder) {
                            description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                                "string_members_only", this.nAccounts);
                        } else {
                            description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                                "string_members_and_folders",
                                this.nAccounts, this.nFolder);
                        }
                    }

                    tagInput.addCompletionEntry(this.displayName, icon,
                        description, this);
                });

                /* Only show completions if there are any */
                if(ary && 0 < ary.length) {
                    tagInput.showCompletions();
                }
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Handle folder completion
     *
     * @param {Picasso.TagInput}  tagInput  A #{@link Picasso.TagInput}
     * @param {String}            val       Value to load completions for
     **/

    Picasso.TagInput.handleFolderCompletion = function (tagInput, val) {
        var url = "/wapi/autocomplete";
        var data = {
            action: "folders",
            query: val
        };

        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                tagInput.clearCompletions();

                $.each(ary, function () {
                    var file = new Picasso.File(this);

                    /* Skip invitations */
                    if (file.is(Picasso.File.Flags.INVITED_TO)) return;

                    /* Assemble description */
                    var description = null;

                    if (0 === file.getAccountsCount() &&
                        0 === file.getGroupsCount())
                    {
                        description = Picasso.Lang.get("string_members_none");
                    } else if (0 === file.getGroupsCount()) {
                        description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                            "string_members_only", file.getAccountsCount());
                    } else {
                        description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                            "string_members_and_groups",
                            file.getAccountsCount(), file.getGroupsCount());
                    }

                    tagInput.addCompletionEntry(file.getName(),
                        "folder-closed", description, file);
                });

                /* Only show completions if there are any */
                if(ary && 0 < ary.length) {
                    tagInput.showCompletions();
                }
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Handle account completion
     *
     * @param {Picasso.TagInput}  tagInput  A #{@link Picasso.TagInput}
     * @param {String}            val       Value to load completions for
     **/

    Picasso.TagInput.handleAccountCompletion = function (tagInput, val) {
        var url = "/wapi/autocomplete";
        var data = {
            action: "accounts",
            query: val
        };

        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                tagInput.clearCompletions();

                $.each(ary, function () {
                    var account = new Picasso.Account(this);

                    tagInput.addCompletionEntry(account.getName(),
                        "user", account.getDescription(), account);
                });

                /* Only show completions if there are any */
                if(ary && 0 < ary.length) {
                    tagInput.showCompletions();
                }
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Handle group completion
     *
     * @param {Picasso.TagInput}  tagInput  A #{@link Picasso.TagInput}
     * @param {String}            val       Value to load completions for
     **/

    Picasso.TagInput.handleGroupCompletion = function (tagInput, val) {
        var url = "/wapi/autocomplete";
        var data = {
            action: "groups",
            query: val
        };

        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                tagInput.clearCompletions();

                $.each(ary, function () {
                    var group = new Picasso.Group(this);

                    tagInput.addCompletionEntry(group.getName(),
                        "group", group.getDescription(), group);
                });

                /* Only show completions if there are any */
                if(ary && 0 < ary.length) {
                    tagInput.showCompletions();
                }
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Handle organization completion
     *
     * @param {Picasso.TagInput}  tagInput  A #{@link Picasso.TagInput}
     * @param {String}            val       Value to load completions for
     **/

    Picasso.TagInput.handleOrganizationCompletion = function (tagInput, val) {
        var url = "/wapi/autocomplete";
        var data = {
            action: "organizations",
            query: val
        };

        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                tagInput.clearCompletions();

                $.each(ary, function () {
                    var organization = new Picasso.Organization(this);


                    tagInput.addCompletionEntry(organization.getName(),
                        organization.getIcon(), organization.getDescription(), organization);
                });

                /* Only show completions if there are any */
                if(ary && 0 < ary.length) {
                    tagInput.showCompletions();
                }
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Handle property completion
     *
     * @param {Picasso.TagInput}  tagInput  A #{@link Picasso.TagInput}
     * @param {String}            val       Value to load completions for
     **/

    Picasso.TagInput.handleConfigurationCompletion = function (tagInput, val) {
        Picasso.Configuration.getDefaultConfiguration(

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                tagInput.clearCompletions();

                $.each(ary, function () {
                    var description = (this.value ? this.value :
                        Picasso.Lang.get("string_no_default_value"));

                    tagInput.addCompletionEntry(this.key,
                        "cogwheel", description, this);
                });

                /* Only show completions if there are any */
                if(ary && 0 < ary.length) {
                    tagInput.showCompletions();
                }
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }, val
        );
    };
})();
