/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/**
 * The table uses some custom classes for specific behavior:
 *
 * .pica-vmtable-toggle - Toggle button to select view mode
 * .pica-vmtable-toggle-list - Element to switch to list mode
 * .pica-vmtable-toggle-grid - Element to switch to grid mode
 * .pica-vmtable-toggle-gallery - Element to switch to gallery mode
 *
 * .pica-vmtable-gallery-slide - Slide for gallery item
 * .pica-vmtable-gallery-caption - Caption of gallery item
 *
 * .pica-vmtable-only-list - Show this elements in list mode only
 * .pica-vmtable-only-grid - Show this elements in grid mode only
 * .pica-vmtable-only-gallery - Show this elements in gallery mode only
 **/

/* Private scope */
(function () {

    /** Create view mode menu items
     *
     * @param {String}  linkClass  Link CSS class
     * @param {Array}   linkArgs   Arguments to click callback
     * @param {String}  glyphIcon  Glypicon of the menu item
     * @param {String}  langName   Locale name of the menu item
     *
     * @returns {jQuery|HTMLElement} Created element
     **/

    var createMenuItem = function (linkClass, linkArgs, glyphIcon, langName) {
        var li = $("<li>");

        var a = $("<a>", { class: linkClass });

        a.append(Picasso.Helper.createGlyph(glyphIcon, "pica-glyph"));
        a.append(Picasso.Helper.createLang(langName));

        a.click(linkArgs, handleClickViewMode);

        li.append(a);

        return li;
    };

    /**
     * Create view mode toggle dropdown
     *
     * @returns {jQuery|HTMLElement} Created element
     **/

    var createModeToggle = function () {
        var div = $("<div>", {
            class: "pica-vmtable-toggle pica-inline pica-table-selection-hide dropdown"
        });

        var a = $("<a>", {
            class: "dropdown-toggle",
            "data-toggle": "dropdown",
            "data-placement": "bottom",
            "data-tooltip": "",
            "title": Picasso.Lang.get("tooltip_select_view_mode")
        });

        a.tooltip({ container: "body" });

        /* Create current mode icons */
        a.append(Picasso.Helper.createGlyph("show-thumbnails-with-lines",
            "pica-vmtable-toggle-list pica-glyph",
            (Picasso.ViewModeTable.Flags.LIST_MODE === this._viewMode)));

        a.append(Picasso.Helper.createGlyph("show-thumbnails",
            "pica-vmtable-toggle-gallery pica-glyph",
            (Picasso.ViewModeTable.Flags.GRID_MODE === this._viewMode)));

        a.append($("<span>", { class: "caret" }));

        div.append(a);

        /* Create menu */
        var ul = $("<ul>", {
            class: "pica-arrow-right dropdown-menu dropdown-menu-right"
        });

        ul.append(createMenuItem("pica-vmtable-toggle-list", [ this ],
            "show-thumbnails-with-lines", "label_list"));

        ul.append($("<li>", { class: "divider" }));

        ul.append(createMenuItem("pica-vmtable-toggle-grid", [ this ],
            "show-thumbnails", "label_grid"));

        /* Check whether no gallery mode is set */
        if (0 === (this._flags & Picasso.ViewModeTable.Flags.NO_GALLERY_MODE)) {
            ul.append($("<li>", { class: "divider" }));

            ul.append(createMenuItem("pica-vmtable-toggle-gallery", [this],
                "pictures", "label_gallery"));
        }

        div.append(ul);

        return div;
    };

    /**
     * Handle click on view mode button
     **/

    var handleClickViewMode = function (e) {
        var that = $(this);
        var vmTable = e.data[0];

        var glyph;
        var viewMode;

        /* Toggle mode */
        if (that.hasClass("pica-vmtable-toggle-list")) {
            viewMode = Picasso.ViewModeTable.Flags.LIST_MODE;
            glyph = "show-thumbnails-with-lines";
        } else if (that.hasClass("pica-vmtable-toggle-grid")) {
            viewMode = Picasso.ViewModeTable.Flags.GRID_MODE;
            glyph = "show-thumbnails";
        } else if (that.hasClass("pica-vmtable-toggle-gallery")) {
            viewMode = Picasso.ViewModeTable.Flags.GALLERY_MODE;
            glyph = "pictures";
        }

        /* Toggle visibility of mode buttons in list and grid mode */
        if (Picasso.ViewModeTable.Flags.GALLERY_MODE !== viewMode) {
            var spans = that.parents(
                ".pica-vmtable-toggle").find("> a span.glyphicons");

            spans.hide();
            spans.filter(".glyphicons-" + glyph).show();
        }

        /* Render data */
        vmTable.setViewMode(viewMode);
        vmTable.render();
    };

    /**
     * Render gallery
     *
     * @param {Array}  data  Data array
     *
     * @returns {Number} Number of appended rows
     **/

    var renderGallery = function (data) {
        /* Remove old gallery if any */
        var id = this._id + "_gallery";

        $(id).remove();

        /* Create overlay */
        var gallery = $("<div>", {
            id: id.replace("#", ""),
            class: "modal fade slide in"
        });

        /* Create overlay */
        var dialog = $("<div>", {
            class: "pica-gallery modal-dialog"
        });

        gallery.append(dialog);

        /* Create content */
        var content = $("<div>", {
            class: "modal-content"
        });

        dialog.append(content);

        var body = $("<div>", {
            class: "modal-body"
        });

        content.append(body);

        /* Create carousel */
        var carouselId = this._id + "_carousel";

        var carousel = $("<div>", {
            id: carouselId.replace("#", ""),
            class: "carousel slide",
            "data-ride": "carousel"
        });

        body.append(carousel);

        /* Create inner and populate it */
        var inner = $("<div>", {class: "carousel-inner"});
        var nrows = 0;
        var self = this;

        $.each(data, function (i, elem) {
            /* Call render func and update rendered slide */
            var rendered = self._renderGalleryFunc(elem);

            var slide = rendered.addBack().filter(".pica-vmtable-gallery-slide");
            var caption = rendered.find(".pica-vmtable-gallery-caption");

            slide.addClass("item");
            caption.addClass("carousel-caption").hide();

            inner.append(rendered);

            slide.on("click", function () {
                caption.toggle();
            });

            nrows++;
        });

        carousel.append(inner);

        /* Create controls */
        carousel.append($("<a>", {
            class: "left carousel-control",
            href: carouselId,
            "data-slide": "prev",
            html: $("<div>", {
                class: "pica-table",
                html: Picasso.Helper.createGlyph("chevron-left")
            })
        }));

        carousel.append($("<a>", {
            class: "right carousel-control",
            href: carouselId,
            "data-slide": "next",
            html: $("<div>", {
                class: "pica-table",
                html: Picasso.Helper.createGlyph("chevron-right")
            })
        }));

        carousel.carousel();

        /* Ensure that one element is active */
        var active = inner.find("div.item");

        if (0 === active.filter(".active").length) {
            active.first().addClass("active");
        }

        /* Register closing handler */
        gallery.bind("hide.bs.modal", function () {
            self._viewMode &= ~Picasso.ViewModeTable.Flags.GALLERY_MODE;

            Picasso.unset("selectedFile");
        });

        gallery.modal();

        return nrows;
    };

    /**
     * Constructor for {@link Picasso.ViewModeTable}
     *
     * @param {String}                       id       Table Id
     * @param {Picasso.ViewModeTable.Flags}  options  Options for this element (optional)
     * @constructor
     **/

    Picasso.ViewModeTable = function (id, options) {
        /* Init */
        this._id = id;
        this._table = new Picasso.Table(id, (options &&
            (0 < (options & Picasso.ViewModeTable.Flags.ROWS_SELECTABLE)) ?
                Picasso.Table.Flags.ROWS_SELECTABLE : 0));
        this._cache = [];

        /* Render and filter funcs */
        this._renderListFunc = null;
        this._renderGridFunc = null;
        this._renderGalleryFunc = null;
        this._filterGalleryFunc = null;

        /* Options */
        this._flags = (options || Picasso.ViewModeTable.Flags.LIST_MODE);

        /* Initial view mode */
        this._viewMode = (this._flags & (Picasso.ViewModeTable.Flags.LIST_MODE|
            Picasso.ViewModeTable.Flags.GRID_MODE));

        /* Add mode toggle button */
        if (0 === (this._flags & Picasso.ViewModeTable.Flags.NO_MODE_TOGGLE)) {
            var thead = $(this._id).find("thead").last();
            var target = thead.find("th.pica-vmtable-mode-toggle-target");
            if (0 === target.length) {
                target = thead.find("th:last-child");
            }
            target.append(createModeToggle.apply(this));
        }
    };

    /**
     * Set list renderer
     *
     * @param {Function}  renderFunc  Render function for elements
     **/

    Picasso.ViewModeTable.prototype.setListRenderer = function (renderFunc) {
        if (renderFunc) {
            this._renderListFunc = renderFunc;
        }
    };

    /**
     * Set grid renderer and rows/spans
     *
     * @param {Function}  renderFunc  Render function for elements
     **/

    Picasso.ViewModeTable.prototype.setGridRenderer = function (renderFunc) {
        if (renderFunc) {
            this._renderGridFunc = renderFunc;
        }
    };

    /**
     * Set gallery renderer
     *
     * @param {Function}  renderFunc  Render function for elements
     **/

    Picasso.ViewModeTable.prototype.setGalleryRenderer = function (renderFunc) {
        if (renderFunc) {
            this._flags &= ~Picasso.ViewModeTable.Flags.NO_GALLERY_MODE;

            this._renderGalleryFunc = renderFunc;
        }
    };

    /**
     * Set gallery filter
     *
     * @param {Function}  filterFunc  Filter for gallery elements
     **/

    Picasso.ViewModeTable.prototype.setGalleryFilter = function (filterFunc) {
        if (filterFunc) {
            this._flags &= ~Picasso.ViewModeTable.Flags.NO_GALLERY_MODE;

            this._filterGalleryFunc = filterFunc;
        }
    };

    /**
     * Set table mode
     *
     * @param {Picasso.ViewModeTable.Flags}  viewMode  Table view mode
     **/

    Picasso.ViewModeTable.prototype.setViewMode = function (viewMode) {
        /* Gallery mode is a special mode on top of list and grid */
        if (Picasso.ViewModeTable.Flags.GALLERY_MODE === viewMode) {
            this._viewMode |= (viewMode & Picasso.ViewModeTable.Flags.GALLERY_MODE);
        } else {
            this._viewMode = (viewMode & (Picasso.ViewModeTable.Flags.LIST_MODE|
                Picasso.ViewModeTable.Flags.GRID_MODE));
        }
    };

    /**
     * Get underlying table
     *
     * @return {Picasso.Table} Underlying #{@link Picasso.Table}
     **/

    Picasso.ViewModeTable.prototype.getTable = function () {
        return this._table;
    };

    /**
     * Get underlying cache
     *
     * @returns {Array} Table cache
     **/

    Picasso.ViewModeTable.prototype.getCache = function () {
        return this._cache;
    };

    /**
     * Set underlying cache
     *
     * @param {Array}  cache  New table cache
     **/

    Picasso.ViewModeTable.prototype.setCache = function (cache) {
        this._cache = cache;
    };

    /**
     * Append element to table
     *
     * @param {*}  data  Data to add to table
     **/

    Picasso.ViewModeTable.prototype.append = function (data) {

        /* Update cache based on view mode */
        if (Array.isArray(data)) {
            this._cache = $.merge(this._cache, data);
        } else {
            this._cache.push(data);
        }

        /* Update table now */
        if (0 < (this._viewMode & Picasso.ViewModeTable.Flags.LIST_MODE)) {
            this._table.append(data, this._renderListFunc); ///< Just append row
        } else if (0 < (this._viewMode & Picasso.ViewModeTable.Flags.GRID_MODE)) {
            /* Save scrollTop and restore it after rendering */
            var oldScroll = $(this._id).find("tbody").scrollTop();

            this.render(); ///< Render a-new; we might have to re-order

            $(this._id).find("tbody").scrollTop(oldScroll);
        }
    };

    /**
     * Render elements from cache if any
     **/

    Picasso.ViewModeTable.prototype.render = function () {
        if (0 < this._cache.length) {
            this.update(this._cache);
        } else {
            this.checkIfEmpty();
        }
    };

    /**
     * Clear data table and show empty message
     **/

    Picasso.ViewModeTable.prototype.clear = function () {
        this._cache = [];

        this._table.clear();
    };

     /**
     * Check if table is empty and add message
     *
     * @return {Boolean}  Either true when table is empty; otherwise false
     **/

    Picasso.ViewModeTable.prototype.checkIfEmpty = function () {
        /* Special handling when in grid mode */
        if (Picasso.ViewModeTable.Flags.GRID_MODE &&
            this._viewMode && 0 === this._cache.length) {
            this.getTable().showEmptyMessage();

            return true;
        } else {
            return this.getTable().checkIfEmpty();
        }

        return false;
    };

    /**
     * Show gallery
     *
     * @param {Number}  idx  Index to set as active (optional)
     **/

    Picasso.ViewModeTable.prototype.showGallery = function () {
        var filtered = this._filterGalleryFunc(this._cache);
        var nrows = 0;

        if (0 < filtered.length) {
            nrows = renderGallery.apply(this, [ filtered ]);
        } else {
            this._viewMode &= ~Picasso.ViewModeTable.Flags.GALLERY_MODE;
        }

        return nrows;
    };

    /**
     * Update table with data
     *
     * @param {*}  data  Data for update
     **/

    Picasso.ViewModeTable.prototype.update = function (data) {
        var htmlTable = $(this._id);
        var onlyList = htmlTable.find(".pica-vmtable-only-list");
        var onlyGrid = htmlTable.find(".pica-vmtable-only-grid");
        var onlyGallery = htmlTable.find(".pica-vmtable-only-gallery");
        var nrows = 0;

        /* Cache data for quicker view mode changes */
        this.setCache(data);

        /* Handle view mode */
        if (0 < (this._viewMode & Picasso.ViewModeTable.Flags.GALLERY_MODE)) {
            this.showGallery();

            this._viewMode &= ~Picasso.ViewModeTable.Flags.GALLERY_MODE;
        } else if (0 < (this._viewMode & Picasso.ViewModeTable.Flags.LIST_MODE)) {
            onlyList.show();
            onlyGrid.hide();
            onlyGallery.hide();

            htmlTable.addClass("table-striped table-hover");

            nrows = this._table.update(data, this._renderListFunc);
        } else if (0 < (this._viewMode & Picasso.ViewModeTable.Flags.GRID_MODE)) {
            onlyList.hide();
            onlyGrid.show();
            onlyGallery.hide();

            htmlTable.removeClass("table-striped table-hover");

            var tbody = $(htmlTable.find("tbody"));

            /* Store handler if any */
            var events = tbody[0].events || $.data(tbody[0], "events") ||
                $._data(tbody[0], "events");

            nrows = this._table.update(data, this._renderGridFunc);

            /* Wrap rendered rows in flex box */
            tbody = htmlTable.find("tbody");

            var divs = tbody.find("div");

            var tr = $("<tr>", { class: "pica-table-selection-disabled" });
            var td = $("<td>", { colspan: this._table.getVisibleColsCount() });
            var div = $("<div>", { class: "pica-flex" });

            div.append(divs);
            td.append(div);
            tr.append(td);

            tbody.html(tr);

            /* Copy scroll handler if any */
            if (events && events.scroll) {
                tbody.bind("scroll", events.scroll[0].handler);
            }

        }

        this.checkIfEmpty();

        return nrows;
    };

    /**
     * Update table with data
     *
     * @param {String}   url   Url to load data from
     * @param {Object}   data  Url parameters
     **/

    Picasso.ViewModeTable.prototype.updateFromJSON = function (url, data) {
        var self = this;

        /* Fire ajax */
        Picasso.Helper.fireAjax(url, data,

            /*  Success */
            function (json) {
                try {
                    var ary = Picasso.Helper.getResultArray(json);

                    self.update(ary);
                } catch (e) {
                    Picasso.debugLog(e);

                    return;
                }
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /* Flags - must be defined after constructor */

    if (!window.Picasso.ViewModeTable.hasOwnProperty("Flags")) {
        window.Picasso.ViewModeTable.Flags = {};
    }

    /* Flags */
    Picasso.ViewModeTable.Flags.LIST_MODE = (1 << 0); ///< Table is in list mode
    Picasso.ViewModeTable.Flags.GRID_MODE = (1 << 1); ///< Table is in grid mode
    Picasso.ViewModeTable.Flags.GALLERY_MODE = (1 << 2); ///< Table is in gallery mode
    Picasso.ViewModeTable.Flags.ROWS_SELECTABLE = (1 << 3); ///< Table rows are selectable
    Picasso.ViewModeTable.Flags.NO_MODE_TOGGLE = (1 << 4); ///< Don't show mode toggle buttons
    Picasso.ViewModeTable.Flags.NO_GALLERY_MODE = (1 << 5); ///< Disable gallery mode
})();
