/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Dialog")) {
    window.Picasso.Dialog = {};
}

/* Private scope */
(function () {

    /* Options for property dropdown menus */
    var optionsForLists = [
        { text: Picasso.Lang.get("action_edit"), icon: "edit" },
        null,
        { text: Picasso.Lang.get("action_remove"), icon: "bin" }
    ];

    /* Default folders */
    var defaultFolders = [
      { name: "default_folder_personal", directory: "$personal_files" },
      { name: "default_folder_public", directory: "$apps.dir" },
      { name: "default_folder_contacts", directory: "$user.dir.contacts" },
      { name: "default_folder_desktop", directory: "$user.dir.desktop" },
      { name: "default_folder_favorites", directory: "$user.dir.favorites" },
      { name: "default_folder_links", directory: "$user.dir.links" },
      { name: "default_folder_movies", directory: "$apps.dir.movies" },
      { name: "default_folder_downloads", directory: "$user.dir.downloads" },
      { name: "default_folder_dropbox", directory: "$user.dir.dropbox" },
      { name: "default_folder_documents", directory: "$user.dir.documents" },
      { name: "default_folder_music", directory: "$user.dir.music" },
      { name: "default_folder_pictures", directory: "$user.dir.pictures" },
      { name: "default_folder_videos", directory: "$user.dir.videos" },
      { name: "default_folder_app_data", directory: "$apps.dir" },
      { name: "default_folder_firefox", directory: "$apps.dir.firefox" },
      { name: "default_folder_sunbird", directory: "$aps.dir.sunbird" },
      { name: "default_folder_thunderbird", directory: "$apps.dir.thunderbird" },
      { name: "default_folder_outlook", directory: "$apps.dir_outlook" },
      { name: "default_folder_windows_mail", directory: "$apps.dir_windows_mail" }
    ];

    /**
     * Handle default folder completion
     *
     * @param {Picasso.TagInput}  tagInput  A #{@link Picasso.TagInput}
     * @param {String}            val       Value to load completions for
     **/

    var handleDefaultFolderCompletion = function (tagInput, val) {
        tagInput.clearCompletions();

        /* Either filter or return all */
        var query = (val || "").toLowerCase();
        var found = 0;

        $.each(defaultFolders, function () {
            if (0 !== val.length &&-1 === this.directory.indexOf(query) &&
                    -1 === this.name.indexOf(query))
            {
                return;
            }

            tagInput.addCompletionEntry(Picasso.Lang.get(this.name),
                "folder-new", this.directory, this);
            found++;
        });

        /* Only show completions if there are any */
        if(0 < found) {
            tagInput.showCompletions();
        }
    };

    /**
     * Copy data from inputs to link
     *
     * @param {Picasso.Dialog.Configuration}  dialog  A #{@link Picasso.Dialog.Configuration}
     * @param {Object}                        inputs  Input fields ({ "id" = HTMLElement })
     **/

    var copyData = function (dialog, inputs) {
        dialog._configuration.setName(inputs.pica_configuration_name.val());
        dialog._configuration.setAssignByIp(inputs.pica_configuration_assign_by_ip.val());
        dialog._configuration.setAssignByLdap(inputs.pica_configuration_assign_by_ldap.val());

        /* Quota */
        var val = inputs.pica_configuration_storage.val();

        if (-1 !== val) {
            val = Picasso.Helper.parseNumber(val, 0);

            dialog._configuration.setQuota(Math.round(val * 1024 * 1024 * 1024));
        }
    };

    /**
     * Assign data to inputs
     *
     * @param {Picasso.Dialog.Configuration}  dialog  A #{@link Picasso.Dialog.Configuration}
     * @param {Object}                        inputs  Input fields ({ "id" = HTMLElement })
     **/

    var assignData = function (dialog, inputs) {
        inputs.pica_configuration_name.val(dialog._configuration.getName());
        inputs.pica_configuration_default_versions.val(
            dialog._configuration.getDefaultVersions());

        if (-1 !== dialog._configuration.getQuota()) {
            inputs.pica_configuration_storage.val(
                Math.round(dialog._configuration.getQuota() / 1024 / 1024 / 1024));
        }


        /* Load properties */
        dialog._configuration.getProperties(

            /* Success */
            function (properties) {
                $.each(properties, function () {
                    dialog._properties_inputList.addItem(
                        "cogwheel", this.key, this.value, {
                            oid: null,
                            name: this.key,
                            value: this.value
                        }
                    )
                });
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );

        /* Load folders */
        dialog._configuration.getFolders(

            /* Success */
            function (folders) {
                $.each(folders, function () {
                    dialog._folders_inputList.addItem(
                        "folder-flag", this.folderName, this.directory, {
                            oid: this.folderID,
                            name: this.folderName,
                            entryID: this.entryID,
                            directory: this.directory,
                            versions: this.versions,
                            excludes: this.excludes
                        }
                    )
                });
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );

        /* Load folders */
        dialog._configuration.getOrganizations(

            /* Success */
            function (organizations) {
                $.each(organizations, function () {
                    dialog._organizations_inputList.addItem(
                        this.getIcon(), this.getName(), null, {
                            oid: this.getOid(),
                            name: this.getName()
                        }
                    )
                });
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /**
     * Save {@link Picasso.Configuration} data
     *
     * @param {Picasso.Dialog.Configuration}  dialog     A #{@link Picasso.Dialog.Configuration}
     * @param {Function}                      onSuccess  Call on success
     * @param {Function}                      onError    Call on error
     **/

    var saveData = function (dialog, onSuccess, onError) {

         /**
         * WARNING: Apparently, we cannot start all these async requests
         *          at once, because some synchronizing timing issue
         *          raises database exceptions.
         *
         *          So we basically wrap all requests in functions (to prevent
         *          starting them immediately) and chain them with promises..
         **/

        /* Store data */
        var defer_save = function () {
            return dialog._configuration.save(null, onError);
        };

        /* Store properties */
        var properties = dialog._properties_inputList.getItems();

        var defer_properties = function () {
            var data = {
                "entryID[]": [],
                "folderID[]": [],
                "folderName[]": [],
                "directory[]": [],
                "versions[]": [],
                "excludes[]": []
            };

            /* Collect data */
            $.each(properties, function () {
                data["folderName[]"].push(this.name);
            });

            return dialog._configuration.setProperties(data, null,

                /* Error */
                function (e, status) {
                    /* Handle folder errors */
                    switch (status) {
                        case 409:
                            Picasso.Notification.show(Picasso.Lang.get(
                                "notification_error_duplicate",
                                Picasso.Lang.get("label_folders")), "warning");
                            break;
                    }

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
        };

        /* Store folders */
        var folders = dialog._folders_inputList.getItems();

        dialog._configuration._folderscount = folders.length;

        var defer_folders = function () {
            var data = {
                "entryID[]": [],
                "folderID[]": [],
                "folderName[]": [],
                "directory[]": [],
                "versions[]": [],
                "excludes[]": []
            };

            /* Collect data */
            $.each(folders, function () {
                data["entryID[]"].push(this.entryID);
                data["folderID[]"].push(this.oid);
                data["folderName[]"].push(this.name);
                data["directory[]"].push(this.directory);
                data["versions[]"].push(this.versions);
                data["excludes[]"].push(this.excludes);
            });

            return dialog._configuration.setFolders(data, null,

                /* Error */
                function (e, status) {
                    /* Handle folder errors */
                    switch (status) {
                        case 409:
                            Picasso.Notification.show(Picasso.Lang.get(
                                "notification_error_duplicate",
                                Picasso.Lang.get("label_folders")), "warning");
                            break;
                    }

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
        };

        /* Store organizations */
        var organizations = dialog._organizations_inputList.getItems();

        var defer_organizations = function () {
            var data = {
                "organizationID[]": []
            };

            /* Collect data */
            $.each(organizations, function () {
                if (this.hasOwnProperty("oid") && this.oid) {
                    data["organizationID[]"].push(this.oid);
                }
            });

            return dialog._configuration.setOrganizations(data, null,

                /* Error */
                function (e, status) {
                    Picasso.Dialog.handleOrgErrors(status);

                    if (onError) {
                        onError(e, status);
                    }
                }
            )
        };

        /* Future and broken promises */
        $.when(defer_save()).then(function () {
            $.when(defer_properties()).then(function () {
                $.when(defer_folders()).then(function () {
                    $.when(defer_organizations()).then(function () {
                        if (onSuccess) {
                            onSuccess(dialog._configuration);
                        }
                    }, onError)
                }, onError)
            }, onError)
        });
    };

    /**
     * Constructor for {@link Picasso.Dialog.Configuration}
     *
     * @param {Picasso.Dialog.Flags}  options  Options for this element (optional)
     * @constructor
     **/

    Picasso.Dialog.Configuration = function (options) {

        /* Init */
        this._configuration = null;
        this._flags = (options || 0);

        /* Sanity check */
        var dialog = $("#pica_configuration_dialog");

        if (0 === dialog.length) {
            throw "Dialog not found";
        }

        /* Check immutable flag */
        var inputFlags = 0;

        if (0 < (this._flags & Picasso.Dialog.Flags.IMMUTABLE)) {
            inputFlags = Picasso.InputList.Flags.IMMUTABLE;
        }

        /* Init dialog */
        Picasso.Helper.bindNav(dialog.find(".nav"));

        /* Create property list */
        this._properties_inputList =
            new Picasso.InputList("#pica_configuration_properties",
                (inputFlags|Picasso.InputList.Flags.LIST_MODE|
                    Picasso.InputList.Flags.COMPLETION_ONLY|
                    Picasso.InputList.Flags.COMPLETION_NEW|
                    Picasso.InputList.Flags.COMPLETION_FILTER));

        this._properties_inputList.setCompletionHandler(
            Picasso.TagInput.handleConfigurationCompletion);

        this._properties_inputList.setCompletionSelectHandler(
            function (e, userdata) {
                this.addItem("asterisk", userdata.key, userdata.value, {
                        oid: null,
                        name: userdata.key,
                        value: userdata.value

                    }
                );
                this._tagInput.hideCompletions();
            }
        );

        this._properties_inputList.setCompletionNewHandler(
            function () {
                var name = this.getInputValue();
                var inputList = this;

                if(0 < name.length) {

                    /* Show dialog */
                    var editDialog = new Picasso.Dialog(
                        "#pica_configuration_properties_edit_dialog",
                        Picasso.Lang.get("dialog_title_create"),
                        Picasso.Lang.get("dialog_body_create",
                            Picasso.Lang.get("label_properties")));

                    editDialog.setAssignHandler(function (inputs) {
                        inputs.pica_configuration_properties_edit_key.val(name);
                    });

                    editDialog.setOkHandler(function (inputs) {
                        var key = inputs.pica_configuration_properties_edit_key.val();
                        var value = inputs.pica_configuration_properties_edit_value.val();

                        inputList.addItem("asterisk", key, value, {
                                oid: null,
                                name: key,
                                value: value

                            }
                        );
                    });

                    editDialog.show();
                }
            }
        );

        this._properties_inputList.setEditHandler(function (dialog, idx, userdata) {
        });

        /* Create folders list */
        this._folders_inputList =
            new Picasso.InputList("#pica_configuration_folders",
                (inputFlags|Picasso.InputList.Flags.LIST_MODE|
                    Picasso.InputList.Flags.COMPLETION_ONLY|
                    Picasso.InputList.Flags.COMPLETION_NEW|
                    Picasso.InputList.Flags.COMPLETION_FILTER));

        this._folders_inputList.setCompletionHandler(
            handleDefaultFolderCompletion);

        this._folders_inputList.setCompletionSelectHandler(
            function (e, userdata) {
                this.addItem("folder-plus", Picasso.Lang.get(userdata.name),
                    userdata.directory, {
                        oid: null,
                        name: Picasso.Lang.get(userdata.name),
                        directory: userdata.directory,
                        entryID: null,
                        versions: 25,
                        excludes: null
                    }
                );
                this._tagInput.hideCompletions();
            }
        );

        this._folders_inputList.setCompletionNewHandler(
            function () {
                var name = this.getInputValue();

                if(0 < name.length) {
                    this.addItem("folder-plus", name, "", {
                            oid: null,
                            name: name
                        }
                    );
                }
            }
        );

        this._folders_inputList.setEditHandler(function (dialog, idx, userdata) {
            console.dir(arguments);

            /* Show dialog */
            var editDialog = new Picasso.Dialog("#pica_configuration_folders_edit_dialog");

            editDialog.setAssignHandler(function (inputs) {
                inputs.pica_configuration_folders_edit_entry_id.val(userdata.entryID);
                inputs.pica_configuration_folders_edit_folder_id.val(userdata.oid);
                inputs.pica_configuration_folders_edit_folder_name.val(userdata.name);
                inputs.pica_configuration_folders_edit_directory.val(userdata.directory);
                inputs.pica_configuration_folders_edit_versions.val(userdata.versions);
                inputs.pica_configuration_folders_edit_excludes.val(userdata.excludes);
            });

            editDialog.setOkHandler(function (inputs) {
            });

            editDialog.show();
        });

        /* Create organizations list */
        this._organizations_inputList =
            new Picasso.InputList("#pica_configuration_organizations",
                (inputFlags|Picasso.InputList.Flags.LIST_MODE|
                    Picasso.InputList.Flags.COMPLETION_ONLY));

        this._organizations_inputList.setCompletionHandler(
            Picasso.TagInput.handleOrganizationCompletion
        );

        this._organizations_inputList.setCompletionSelectHandler(
            function (e, userdata) {
                this.addItem(userdata.getIcon(), userdata.getName(), null, {
                    oid: userdata.getOid(),
                    name: userdata.getName()
                });
                this._tagInput.hideCompletions();
            }
    );

        this._organizations_inputList.setSubmitHandler(
            function (e, userdata) {
                var name = this.getInputValue();

                if(0 < name.length) {
                    this.addItem("education", name, null, {
                        oid: "",
                        name: name
                    });
                }
            }
        );
    };

    /**
     * Create {@link Picasso.Configuration} via dialog
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Dialog.Configuration.prototype.create = function (onSuccess,
                                                              onError)
    {
        var self = this;

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_configuration_dialog",
            Picasso.Lang.get("dialog_title_create"),
            Picasso.Lang.get("dialog_body_create",
                Picasso.Lang.get("label_configurations")),
            (this._flags|Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE|Picasso.Dialog.Flags.SPINNER));

        dialog.setAssignHandler(function (inputs) {
            $("#pica_configuration_dialog .pica-hide-on-create").hide();
            $("#pica_configuration_name").attr("disabled", false);
        });

        dialog.setOkHandler(function (inputs) {
            self._configuration = new Picasso.Configuration();

            copyData(self, inputs);
            saveData(self,

                /* Success */
                function () {
                    dialog.hide();

                    if (onSuccess) {
                        onSuccess.apply(null, arguments);
                    }
                },

                /* Error */
                function (e, status) {
                    dialog.showSpinner(false);

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
        });

        dialog.show();
    };

    /**
     * Edit {@link Picasso.Configuration} via dialog
     *
     * @param {Picasso.Configuration}  configuration  A #{@link Picasso.Configuration}
     * @param {Function}               onSuccess      Call on success
     * @param {Function}               onError        Call on error
     **/

    Picasso.Dialog.Configuration.prototype.edit = function (configuration,
                                                            onSuccess, onError)
    {
        var self = this;

        this._configuration = configuration;

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_configuration_dialog",
            Picasso.Lang.get("dialog_title_edit"),
            Picasso.Lang.get("dialog_body_edit",
                Picasso.Lang.get("label_profiles")),
            (this._flags|Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.HIDE_ON_ESC|
                Picasso.Dialog.Flags.NO_AUTOCLOSE|Picasso.Dialog.Flags.SPINNER));

        dialog.setAssignHandler(function (inputs) {
            $("#pica_configuration_dialog .pica-hide-on-create").show();

            /* Check whether config is default config */
            if (self._configuration.is(Picasso.Configuration.Flags.DEFAULT_CONFIG)) {
                $("#pica_configuration_name").attr("disabled", true);
            } else {
                $("#pica_configuration_name").attr("disabled", false);
            }

            assignData(self, inputs);
        });

        dialog.setOkHandler(function (inputs) {
            copyData(self, inputs);
            saveData(self,

                /* Success */
                function () {
                    dialog.hide();

                    if (onSuccess) {
                        onSuccess.apply(null, arguments);
                    }
                },

                /* Error */
                function (e, status) {
                    dialog.showSpinner(false);

                    if (onError) {
                        onError(e, status);
                    }
                }
            );
        });

        dialog.show();
    };
})();
