/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Dialog")) {
    window.Picasso.Dialog = {};
}

/* Private scope */
(function () {

    /**
     * Restore {@link Picasso.File} version
     *
     * @param {Picasso.Dialog.Restore}  dialog   A #{@link Picasso.Dialog.Restore}
     * @param {Picasso.File}            file     A #{@link Picasso.File}
     * @param {Number}                  version  Version to restore
     **/

    var restoreVersion = function (dialog, file, version) {
            file.restore(version,

                /* Success */
                function () {
                    /* Update file for display only */
                    if (dialog._file.is(Picasso.File.Flags.DELETED)) {
                        dialog._file.unsetFlag(Picasso.File.Flags.DELETED);
                    }

                    dialog._file.setVersion(dialog._file.getVersion() + 1);

                    renderVersions(dialog);

                    Picasso.Notification.show(
                        Picasso.Lang.get("notification_item_restored",
                            Picasso.Lang.get("label_versions")), "info");
                },

                /* Error */
                function (e,status) {
                    if (405 === status) {
                            Picasso.Notification.show(
                                    Picasso.Lang.get("notification_error_file_under_edit"),
                                    "warning");
                    }
                    else if (420 === status) {
                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_error_storage_exceeded"), "warning");
                    }
                        else {
                        Picasso.debugLog(e);
                    }
                }
            );

    };

    /**
     * Render given json object
     *
     * @param {Picasso.File}  file  A {@type Picasso.File}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderVersion = function (file) {
        var dialog = this;

        var tr = $("<tr>", {
            class: "pica-table-selection-disabled",
            draggable: false
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "pica-icon-column" });

        var icon = file.getIcon();

        td.append(icon);

        /* Add thumbnail if found */
        if (file.isImage()) {
            var img = $("<img>", {
                class: "pica-icon-thumb-list",

                onLoad: function () {
                    icon.hide();
                    icon.after(this);
                },

                src: file.getThumbnailUrl()
            });

            img.on("error", function () {
                icon.show();
                this.remove();

                return true;
            });
        }

        tr.append(td);

        /* 2. column: Name */
        td = $("<td>", {
            colspan:2,
            class: "pica-name pica-overflow pica-vertical-middle text-left",
            "data-sort": file.getVersion(),
            "data-serial": file.serialize()
        });


        var span = $("<span>", {
            text: file.getName(),
            title: file.getName(),
            "data-tooltip": "",
            "data-placement": "bottom"
        });

        span.tooltip({container: "body"});


        /* Set labels based state and version */
        if (file.is(Picasso.File.Flags.DELETED)) {
            span.addClass("label label-danger");
        }

        if (file.getVersion() === dialog._file.getVersion()) {
            span.addClass("label label-success");
        }

        td.append(span);
        tr.append(td);

        /* 3. column: Versions */
        td = $("<td>", {
            class: "pica-vertical-middle text-left",
            html: Picasso.Helper.createLang("string_version", file.getVersion())
        });

        tr.append(td);

        /* 4. column: Last modified */

        td = $("<td>", {
            class: "pica-vertical-middle text-left",
        });

        /* Create tooltip */
        var span = $("<span>", {
            text: Picasso.Helper.formatAge(file.getModifiedDateMS()),
            title: Picasso.Helper.formatDate(file.getModifiedDateMS()),
            "data-tooltip": "",
            "data-placement": "bottom"
        });

        if (file.getModifiedDate()) {
            td.attr("data-sort", new Date(file.getModifiedDate()).getTime());
        }

        td.append(span);

        span.tooltip({container: "body"});


        tr.append(td);

        /* 4. column: Size */
        td = $("<td>", {
            class: "pica-vertical-middle text-left",
        });

        if (file.is(Picasso.File.Flags.FILE)) {
            td.append(Picasso.Helper.formatSize(file.getSize()));
        }

        tr.append(td);

        /* 5. column: Button */
        td = $("<td>", {
            class: "text-right"
        });

        if (file.getVersion() !== dialog._file.getVersion() || file.is(Picasso.File.Flags.DIRECTORY)) {
            var button = $("<button>", {
                class: "btn btn-default pull-right",
                text: Picasso.Lang.get("action_restore")
            });

            button.click(function () {
                restoreVersion(dialog, file, file.getVersion());
            });

            td.append(button);
        } else {
            var button = $("<button>", {
                class: "btn btn-success disabled pull-right",
                text: Picasso.Lang.get("label_current")
            });

            td.append(button);
        }

        tr.append(td);

        return tr;
    };

    /**
     * Assign data to inputs
     *
     * @param {Picasso.Dialog.Restore}  dialog  A #{@link Picasso.Dialog.Restore}
     **/

    var renderVersions = function (dialog) {
        /* Get file versions */
        if (dialog._file.is(Picasso.File.Flags.FILE|
            Picasso.File.Flags.DIRECTORY))
        {

            /* Add version data */
            dialog._file.getVersions(

                /* Success */
                function (versions) {
                    var files = $.merge([ dialog._file ], versions);

                    dialog._table.update(files, $.proxy(renderVersion, dialog));
                    dialog._table.sort(2, false);
                    dialog._table.checkIfEmpty();
                },

                /* Error */
                function (e) {
                    Picasso.debugLog(e);

                    dialog._table.checkIfEmpty();
                }
            );
        }
    };

    /**
     * Constructor for {@link Picasso.Dialog.Restore}
     *
     * @param {Picasso.File}  file  A #{@link Picasso.File}
     * @constructor
     **/

    Picasso.Dialog.Restore = function (file) {
        var self = this;

        /* Init */
        this._file = file;

        /* Sanity check */
        var dialog = $("#pica_restore_dialog");

        if (0 === dialog.length) {
            throw "Dialog not found";
        }

        /* Create versions list */
        this._table = new Picasso.Table("#pica_restore_versions");

        /* Clear on close */
        dialog.bind("hidden.bs.modal", function () {
            self._table.clear();
        });
    };

    /**
     * Show restore dialog
     **/

    Picasso.Dialog.Restore.prototype.show = function () {
        var self = this;

        /* Create dialog */
        var dialog = new Picasso.Dialog("#pica_restore_dialog",
            Picasso.Lang.get("dialog_title_restore"),
            Picasso.Lang.get("dialog_body_restore"));

        dialog.setAssignHandler(function () {
            renderVersions(self);
        });

        /* Finally show dialog */
        dialog.show();
    };

    /* Fix dropdowns in scroller */
    Picasso.Combo.fixDropdownInScroller(".pica-inputlist-scroller");
})();
