/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Files")) {
    window.Picasso.Files = {};
}

/* Private scope */
(function () {

    var VMTABLE = new Picasso.ViewModeTable("#filelink_table",
            (Picasso.ViewModeTable.Flags.LIST_MODE|Picasso.ViewModeTable.Flags.NO_MODE_TOGGLE|
                    Picasso.ViewModeTable.Flags.ROWS_SELECTABLE));
    var PAGE = 1;

    var ID = $("#filelink_id").val();
    var baseurl = window.location.protocol + "//" + window.location.host + "/";



    Picasso.Files.fetch = function (file, append, onSuccess, onError) {
        Picasso.set("cwd", file);



        /* Reset page */
        if (true !== append) {
            PAGE = 1;
        }
        var updateFiles = function (files, json) {
            /* Set federated services */
            if (json.hasOwnProperty("ResultSet") &&
                    json.ResultSet.hasOwnProperty("fedServices")) {
                Picasso.set("fedServices", json.ResultSet.fedServices);
            }

            /* Check for folder permission */
            if (json.hasOwnProperty("ResultSet") &&
                    json.ResultSet.hasOwnProperty("permission"))
            {
                Picasso.set("folderPermission", Picasso.Permission.fromString(
                        json.ResultSet.permission));
            } else {
                Picasso.unset("folderPermission");
            }

            /* Call renderer before infinite scroll */
            if (true === append) {
                VMTABLE.append(files);
            } else {
                VMTABLE.update(files);
            }



            /* Bind infinite scroll */
            Picasso.Helper.bindInfiniteScroll(json, VMTABLE.getTable(), PAGE,
                    function () {
                        PAGE++;

                        Picasso.Files.fetch(Picasso.get("cwd"), true);
                    }
            );

            if (VMTABLE.getTable().getRows().length === 1 && !Picasso.get("cwd")) {
                $(VMTABLE.getTable().getRows()[0]).find("a").first().click();
            }
            if (onSuccess) onSuccess();
        };
        var ID = $("#filelink_id").val();
        if (file) {
            Picasso.set("folderId", file.getFolderId());
            Picasso.set("isFederated", file.is(Picasso.File.Flags.FEDERATED));
            file.getLinkFiles(updateFiles, onError, PAGE,null,ID);
        } else if (ID) {
            Picasso.File.getLinkFolder(updateFiles, onError, PAGE,null,ID)

        }
    };

    /**
     * Render given json object for list
     *
     * @param {Picasso.File}  file  A {@link Picasso.File}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderFileList = function (file) {
        var tr = $("<tr>", {
            id: file.getElemId(),
            "data-search-keys": [file.getName(),file.getOid()],
            draggable: (!file.is(Picasso.File.Flags.FOLDER) &&
                    !file.is(Picasso.File.Flags.INVITED_TO))
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "pica-icon-column" });

        var icon = file.getIcon();

        td.append(icon);

        /* Add thumbnail if found */
        if (file.isImage()) {
            var img = $("<img>", {
                class: "pica-icon-thumb-list",

                onLoad: function () {
                    icon.hide();
                    icon.after(this);
                },

                src: getThumbnailUrl(file)
            });

            img.on("error", function () {
                icon.show();
                this.remove();

                return true;
            });
            if ($('#img-panel').length) {
                var img = $('#img-panel');
                img.attr('src', getDownloadUrl(file));
            }
        }else if (file.isVideo() || file.isAudio()) {
            if ($('#page-player').length) {
                var player = videojs('page-player');
                player.src({type: 'video/mp4', src: getDownloadUrl(file)});
                player.autoplay('muted');
            }
        } else if (file.isInline()) {
            let currentPage = 1;
            let totalPages = 0;
            let pdfDoc = null;

            const canvas = document.getElementById('pdf-viewer');
            if (canvas) {
                const ctx = canvas.getContext('2d');

                const pdfUrl = getDownloadUrl(file, true);

                pdfjsLib.getDocument(pdfUrl).promise.then(doc => {
                    pdfDoc = doc;
                    totalPages = pdfDoc.numPages;
                    updatePageNumber(currentPage);
                    renderPage(currentPage);
                });

                document.getElementById('prev-page').addEventListener('click', () => {
                    if (currentPage > 1) {
                        currentPage--;
                        updatePageNumber(currentPage);
                        renderPage(currentPage);
                    }
                });

                document.getElementById('next-page').addEventListener('click', () => {
                    if (currentPage < totalPages) {
                        currentPage++;
                        updatePageNumber(currentPage);
                        renderPage(currentPage);
                    }
                });

                function updatePageNumber(pageNum) {
                    document.getElementById('page-num').textContent = 'Page ' + pageNum + ' of ' + totalPages;
                }

                function renderPage(pageNum) {
                    pdfDoc.getPage(pageNum).then(page => {
                        const viewport = page.getViewport({scale: 1.5});
                        canvas.width = viewport.width;
                        canvas.height = viewport.height;

                        const renderContext = {
                            canvasContext: ctx,
                            viewport: viewport
                        };

                        page.render(renderContext);
                    });
                }
            }
        }

        tr.append(td);

        /* 2. column: Name */
        td = $("<td>", { class: "pica-overflow",colspan:"10" });

        var a = $("<a>", {
            class: "pica-name",
            text: file.getName(),
            href: file.getOpenUrl() ? file.getOpenUrl() : getDownloadUrl(file),
            "data-serial": file.serialize()
        });

        /* Bind listener */
        if (!file.is(Picasso.File.Flags.INVITED_TO)) {
            a.click(handleClickName);
        }
        td.append(a);
        tr.append(td);
        return tr;
    };

    function addCrumb(name, onClick) {
        var a = $("<a>", {
            class: "pica-crumb"
        });

        var span = $("<span>", {
            text: name,
        });

        /* Handle click handler */
        if (onClick) a.click(onClick);
        a.click(removeSiblings);

        a.append(span);

        $("#files_home").parent().append(a);


        return a;
    }

    /**
     * Remove all elements right of
     *
     * @param {Event}  e  Click event
     **/

    var removeSiblings = function (e) {
        $(this).nextAll("a.pica-crumb").remove();
    };

    /**
     * Handle click on element name
     *
     * @param {Event}  e  Click event
     **/

    var handleClickName = function (e) {
        e.preventDefault();
        e.stopPropagation();

        function isSupported(ext) {
            var json = sessionStorage.getItem("openConfig");
             var config = JSON.parse(json);
             if (config.DOCUMENT.extensions.includes(ext)){
                 return config.DOCUMENT.select.ONLYOFFICE;
             }
            if (config.PDF.extensions.includes(ext)){
                return config.PDF.select.ONLYOFFICE;
            }
            if (config.PRESENTATION.extensions.includes(ext)){
                return config.PRESENTATION.select.ONLYOFFICE;
            }
            if (config.SPREADSHEET.extensions.includes(ext)){
                return config.SPREADSHEET.select.ONLYOFFICE;
            }
            if (config.TEXT.extensions.includes(ext)){
                return config.TEXT.select.ONLYOFFICE;
            }
            return false;
        }

        var file = new Picasso.File($(this).data("serial"));

        /* Check type */
        if (file.is(Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY)) {
            VMTABLE.getTable().deselectAll();

            Picasso.Files.fetch(file, false,

                    /* Success */
                    function () {
                        var a = addCrumb(
                                file.getName(), handleClickCrumb);

                        a.data("file", file.serialize());
                    },

                    /* Error */
                    function () {
                        if (file.is(Picasso.File.Flags.FEDERATED)) {
                            Picasso.Notification.show(
                                    Picasso.Lang.get("notification_federated_service_not_reachable"), "warning");
                        } else {
                            Picasso.debugLog(e);
                        }
                    }
            );
        } else if (file.isImage()) { ///< Image for gallery
            Picasso.set("selectedFile", file);

            VMTABLE.showGallery();
        }else if (file.isVideo()) {
                var player;
            var dialog = new Picasso.Dialog("#pica_video_dialog",
                    Picasso.Lang.get("dialog_title_video"));

                dialog.setAssignHandler(function () {
                    player = videojs('video-player');

                    var filtred = filterFileVideoPlayer(VMTABLE.getCache()) ;

                    var playlist = buildplayList(filtred,file)
                    console.log(playlist);

                    player.playlist(playlist);

                    player.playlist.autoadvance(0);

                    player.playlistUi();

                });

                dialog.setCloseHandler(function () {
                    player.reset();
                });

                $('#pica_video_dialog').on('hidden.bs.modal', function () {
                    player.reset();
                });

                dialog.show()

        } else if (file.isAudio()) {

            var audioPlayer ;
            var dialog = new Picasso.Dialog("#pica_audio_dialog",
                    Picasso.Lang.get("dialog_title_audio",file.getName()));

            dialog.setAssignHandler(function () {
                audioPlayer = videojs('audio-player');
                audioPlayer.src({type: 'audio/mp3', src: getDownloadUrl(file)});

            });

            dialog.setCloseHandler(function () {
                audioPlayer.reset();
            });

            $('#pica_audio_dialog').on('hidden.bs.modal', function () {
                player.reset();
            });

            dialog.show()
        } else { ///< Open
            var url = file.getOpenUrl() ? file.getOpenUrl() : getDownloadUrl(file);
            var ext = file.getName().toLowerCase().substr(file.getName().lastIndexOf(".") + 1);

            if (Picasso.get("isFederated")) {
                url += "?CSRFToken=" + Picasso.get("CSRF");
            }

            if (file.isOODocument() && isSupported(ext)){
                if (-1 === url.indexOf("?")) {
                    url += "?filelink_id=" + ID;
                } else {
                    url += "&filelink_id=" + ID;
                }
            }else if ( file.isInline()){
                url = getDownloadUrl(file);
            }


            /* Create a new anchor and simulate click to download */
            var a = $("<a>", {
                href:   url,
                target: "_blank"
            });

            $("body").append(a);
            a[0].click();
            a.remove();
            Picasso.Files.fetch(Picasso.get("cwd"));
        }
    };

    /**
     * Handle click on bread bar
     **/

    var handleClickCrumb = function () {

        var file = $(this).data("file");

        if (file) {
            file = new Picasso.File(file);
        }

        Picasso.Files.fetch(file);

    };

    /**
     * Handle click on download buttons
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDownload = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selectedFiles = VMTABLE.getTable().getSelectedRows();

        selectedFiles = $.map(selectedFiles, function (val) {
            return getFileFromElem(val);
        });

        if (0 < selectedFiles.length) {
            downloadFilesInBatches(selectedFiles, 5);
        }
    };

    function downloadFilesInBatches(files, batchSize) {
        var index = 0;

        function downloadBatch() {
            var batch = files.slice(index, index + batchSize);
            index += batchSize;

            if (batch.length > 0) {
                $.each(batch, function () {
                    downloadFile(this);
                });

                setTimeout(downloadBatch, 1000); // Wait 1 second before downloading the next batch
            }
        }

        downloadBatch();
    }


    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDelete = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = VMTABLE.getTable().getSelectedRows();

        selected = $.map(selected, function (val) {
            return getFileFromElem(val);
        });

        DeleteFile(selected);
    };


    /**
     * Handle click on rename buttons
     *
     * @param {Event}  e  Click event
     **/

    var handleClickRename = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var selected = VMTABLE.getTable().getSelectedRows();

        if (1 === selected.length) {
            var file = getFileFromElem(selected[0]);

            renameFile(file);
        }
    };

    /**
     * Handle click on copy buttons
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCopy = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var selected = VMTABLE.getTable().getSelectedRows();

        if (1 === selected.length) {
            var file = getFileFromElem(selected[0]);

            var oldName = (file.getIntName() || file.getName());

            /* Show dialog */
            var dialog = new Picasso.Dialog("#pica_input_dialog",
                    Picasso.Lang.get("dialog_title_copy"),
                    Picasso.Lang.get("dialog_body_copy", Picasso.Lang.get("label_name")),
                    Picasso.Dialog.Flags.VALIDATE);

            dialog.setAssignHandler(function (inputs) {
                if (file.is(Picasso.File.Flags.FILE)) {
                    var nameOnly = oldName.replace(/\.[^/.]+$/, "");
                    var ext = oldName.slice(nameOnly.length);
                    inputs.pica_input_text.val(Picasso.Lang.get("string_copy", nameOnly));
                    inputs.pica_input_text.data("original-extension", ext);
                } else {
                    inputs.pica_input_text.val(Picasso.Lang.get("string_copy", oldName));
                    inputs.pica_input_text.removeData("original-extension");
                }
            });

            dialog.setOkHandler(function (inputs) {
                var baseName = inputs.pica_input_text.val();
                var ext = inputs.pica_input_text.data("original-extension") || "";
                var newName = baseName + ext;

                var cwd = Picasso.get("cwd");

                file.moveFileLink(cwd, newName,

                        /* Success */
                        function () {
                            Picasso.Notification.show(Picasso.Lang.get(
                                    "notification_move_copied",
                                    oldName, newName), "info");

                            Picasso.Files.fetch(cwd);
                        },

                        /* Error */
                        function (e, status) {
                            switch (status) {

                                case 403:
                                    Picasso.Notification.show(
                                            Picasso.Lang.get("notification_copy_error_invalid_path",
                                                    newName, cwd.getName()), "warning");
                                    break;

                                case 404:
                                    Picasso.Notification.show(
                                            Picasso.Lang.get("notification_copy_error_directory_not_found",
                                                    newName, cwd.getName()), "warning");
                                    break;

                                case 409:
                                    Picasso.Notification.show(
                                            Picasso.Lang.get("notification_move_error",
                                                    newName, cwd.getName()), "warning");
                                    break;
                            }

                            Picasso.debugLog(e);
                        }
                );
            });

            dialog.show();
        }
    };


    /**
     * Handle click on paste buttons
     *
     * @param {Event}  e  Click event
     **/

    var handleClickPaste = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var cutFiles = Picasso.get("cutFiles");

        if (0 < cutFiles.length) {
            var cwd = Picasso.get("cwd");

            $.each(cutFiles, function (idx, file) {
                file.moveFileLink(cwd, null,

                        /* Success */
                        function () {
                            Picasso.Notification.show(
                                    Picasso.Lang.get("notification_move_success",
                                            file.getName(), cwd.getName()), "info");

                            Picasso.Files.fetch(cwd);
                        },

                        /* Error */
                        function (e, status) {
                            if (file.is(Picasso.File.Flags.FEDERATED)) {
                                Picasso.Notification.show(
                                        Picasso.Lang.get("notification_federated_service_not_reachable"), "warning");
                            } else {
                                switch (status) {
                                    case 405:
                                        Picasso.Notification.show(
                                                Picasso.Lang.get("notification_error_file_under_edit"),
                                                "warning");
                                        break;
                                    case 409:
                                        Picasso.Notification.show(
                                                Picasso.Lang.get("notification_move_error",
                                                        file.getName(), cwd.getName()), "warning");
                                        break;
                                }
                                Picasso.debugLog(e);
                            }
                        }
                );
            });

            Picasso.unset("cutFiles");
            handleOnSelection(0, null);
        }
    };

    /**
     * Get selected file
     *
     * @param {jQuery|HTMLElement}  elem  Element to check
     *
     * @returns {*} Either found #{@link Picasso.File}; otherwise null
     **/

    var getFileFromElem = function (elem) {
        var file = null;
        var serial = $(elem).find("[data-serial]");

        if (0 < serial.length) {
            file = new Picasso.File(serial.data("serial"));
        }

        return file;
    };

    /**
     * Download given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@link Picasso.File}
     **/

    var downloadFile = function (file) {
        /* Create a new anchor and simulate click to download */

        var a = $("<a>", {
            href:getDownloadUrl(file),
            download: file.getName()
        });
        $("body").append(a);
        a[0].click();
        a.remove();


    };

    /**
     * Rename given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@Picasso.File}
     **/

    var renameFile = function (file) {
        var elem = $("#" + file.getElemId());

        var oldName = file.getIntName();

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_input_dialog",
                Picasso.Lang.get("dialog_title_change"),
                Picasso.Lang.get("dialog_body_change", Picasso.Lang.get("label_name")),
                (Picasso.Dialog.Flags.HIDE_ON_ENTER|Picasso.Dialog.Flags.HIDE_ON_ESC|
                        Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.NO_AUTOCLOSE));

        dialog.setAssignHandler(function (inputs) {
            var fullName = file.getIntName() || file.getName();
            if (file.is(Picasso.File.Flags.FILE)) {
                var nameOnly = fullName.replace(/\.[^/.]+$/, "");
                inputs.pica_input_text.val(nameOnly);
                inputs.pica_input_text.data("original-extension", fullName.slice(nameOnly.length));
            } else {
                inputs.pica_input_text.val(fullName);
                inputs.pica_input_text.removeData("original-extension");
            }

        });

        dialog.setOkHandler(function (inputs) {
            var baseName = inputs.pica_input_text.val();
            var ext = inputs.pica_input_text.data("original-extension") || "";
            var newName = baseName + ext;


            file.renameFileLink(newName,

                    /* Success */
                    function () {
                        var a = elem.find(".pica-name");

                        /* Update for new name */
                        a.text(file.getName());
                        a.attr("data-serial", "");
                        a.data("serial", file.serialize());

                        elem.addClass("pica-highlight");

                        Picasso.Notification.show(Picasso.Lang.get(
                                "notification_item_renamed",
                                oldName, newName), "info");

                        dialog.hide();
                        Picasso.Files.fetch(Picasso.get("cwd"));
                    },

                    /* Error */
                    function (e, status) {
                        switch (status) {
                            case 405:
                                Picasso.Notification.show(
                                        Picasso.Lang.get("notification_error_file_under_edit"),
                                        "warning");
                                break;
                            case 406:
                                Picasso.Notification.show(
                                        Picasso.Lang.get("notification_error_illegal_chars"),
                                        "warning");
                                break;

                            case 409:
                                Picasso.Notification.show(
                                        Picasso.Lang.get("notification_error_duplicate", name),
                                        "warning");

                            case 400:
                                /* Falls through */
                            case 500:
                                Picasso.Notification.show(
                                        Picasso.Lang.get("notification_error_rename", name),
                                        "danger");
                                break;
                        }

                        Picasso.debugLog(e);
                    }
            );
        });

        dialog.show();
    };


    /**
     * Handle click on copy buttons
     *
     * @param {Event}  e  Click event
     **/

    var handleClickCut = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var cutFiles = VMTABLE.getTable().getSelectedRows();

        cutFiles = $.map(cutFiles, function (val) {
            return getFileFromElem(val);
        });

        if (cutFiles && 0 < cutFiles.length) {

            //resend folder stable to home
            $("lang[name='"+"label_folders"+"']").parent().on('click', function(event) {
                event.preventDefault();
                $("#files_home").click();
            });

            Picasso.set("cutFiles", cutFiles);

            var header = $("#files_paste_header");
            var iconTh = header.find("th.pica-icon-column");
            var nameTh = header.find("th.pica-table-selection-title");

            if (1 === cutFiles.length) {
                iconTh.html(cutFiles[0].getIcon());
                nameTh.html($("<span>", {
                    text: cutFiles[0].getName()
                }));
            } else {
                iconTh.html(Picasso.Helper.createGlyph(
                        "more-items", "pica-glyph"));
                nameTh.html($("<span>", {
                    text: Picasso.Lang.get(
                            "string_selected_elements", cutFiles.length)
                }));
            }

            /* Add cancel button */
            var a = $("<a>", {
                class: "hidden-xs",
                text: Picasso.Lang.get("action_cancel")
            });

            a.click(function () {
                Picasso.unset("cutFiles");

                handleOnSelection(0, null);

                Picasso.Notification.show(
                        Picasso.Lang.get("notification_cut_stop"), "info");
            });

            nameTh.append(a)

            Picasso.Notification.show(
                    Picasso.Lang.get("notification_cut_start",
                            cutFiles[0].getName()), "info");

            /* Stop selection now */
            VMTABLE.getTable().deselectAll();
        }
    };

    /**
     * Update table header for selection mode
     *
     * @param {Number}  nselected    Number of selected rows
     * @param {Object}  selectedRow  Selected row object
     **/

    var handleOnSelection = function (nselected, selectedRow) {
        var cwd = Picasso.get("cwd");


        if (cwd) {
            var headers = $("#filelink_table thead");

            if (Picasso.hasKey("cutFiles")) {
                headers.hide();
                $(headers[1]).show();
            } else if ($(headers[1]).is(":visible")) {
                headers.hide();
                $(headers[2]).show();
            }

        }
    };

    /**
     * delete given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@Picasso.File}
     **/

    var DeleteFile = function (file) {

        var title = "dialog_title_delete";
        var body = "dialog_body_delete";
        var notification = "notification_item_deleted";

        /* Show dialog */
        var what = (1 === file.length ? file[0].getName() :
                Picasso.Lang.get("string_selected_elements", file.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
                Picasso.Lang.get(title), Picasso.Lang.get(body, what));

        dialog.setOkHandler(function () {
            $.each(file, function () {
                var file = this;

                file.removeFileLink(

                        /* Success */
                        function () {
                            Picasso.Helper.fadeOut($("#" + file.getElemId()),
                                    function () {
                                        this.remove();

                                        VMTABLE.checkIfEmpty();
                                    });

                            Picasso.Notification.show(
                                    Picasso.Helper.capitalize(
                                            Picasso.Lang.get(notification, Picasso.Lang.get(
                                                    file.is(Picasso.File.Flags.FOLDER) ?
                                                            "label_folders" :
                                                            (file.is(Picasso.File.Flags.DIRECTORY) ?
                                                                    "label_directories" : "label_files")))), "info");
                        },

                        /* Error */
                        function (e,status) {

                            if (405===status) {
                                Picasso.Notification.show(
                                        Picasso.Lang.get("notification_error_file_under_edit"),
                                        "warning");
                            }
                            else if (file.is(Picasso.File.Flags.FEDERATED)) {
                                Picasso.Notification.show(
                                        Picasso.Lang.get("notification_federated_service_not_reachable"), "warning");
                            } else {
                                Picasso.debugLog(e);
                            }
                        }
                );
            });
        });

        dialog.show();
    };

    /**
     * Build link file download url given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@link Picasso.File}
     **/
    var getDownloadUrl = function (file, skipCounter) {
        var downloadUrl = "dl/" + ID +"/"+ file.getPath();

        var token = $("#filelink_token").val();

        if (file.isInline()) {
            downloadUrl += "?inline"
        }

        if (0 < token.length) {
            if (-1 === downloadUrl.indexOf("?")) {
                downloadUrl += "?token=" + token;
            } else {
                downloadUrl += "&token=" + token;
            }
        }

        if (skipCounter) {
            if (-1 === downloadUrl.indexOf("?")) {
                downloadUrl += "?skipCounter=true";
            } else {
                downloadUrl += "&skipCounter=true";
            }
        }


        return baseurl + downloadUrl ;
    }
    /**
     * Build link thumbbnail url given {@link Picasso.File}
     *
     * @param {Picasso.File}  file  A #{@link Picasso.File}
     **/
    var getThumbnailUrl = function (file) {
        var thumbnailUrl = "thumb/" + ID +"/"+ file.getPath();
        var token = $("#filelink_token").val();
        if (0 < token.length) {
            if (-1 === thumbnailUrl.indexOf("?")) {
                thumbnailUrl += "?token=" + token;
            } else {
                thumbnailUrl += "&token=" + token;
            }
        }
        if (-1 === thumbnailUrl.indexOf("?")) {
            thumbnailUrl += "?filelink_id=" + ID;
        } else {
            thumbnailUrl += "&filelink_id=" + ID;
        }
        return baseurl + thumbnailUrl ;
    }

    /**
     * Filter files for gallery view
     *
     * @param {Array}  files  Array of {@link Picasso.File}
     *
     * @returns {Array} Filtered list; might be empty
     **/

    var filterFileGallery = function (files) {
        var filtered = files.filter(function (file) {
            return (file instanceof Picasso.File &&
                    file.is(Picasso.File.Flags.FILE) && file.isImage());
        });

        /* Check whether images can be found */
        if (0 === filtered.length) {
            Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_no_images_found"), "warning")
        }

        return filtered;
    };

    /**
     * Filter files for video player
     *
     * @param {Array}  files  Array of {@link Picasso.File}
     *
     * @returns {Array} Filtered list; might be empty
     **/

    var filterFileVideoPlayer = function (files) {
        var filtered = files.filter(function (file) {
            return (file instanceof Picasso.File &&
                    file.is(Picasso.File.Flags.FILE) && file.isVideo());
        });

        /* Check whether videos can be found */
        if (0 === filtered.length) {
            Picasso.Notification.show(
                    Picasso.Lang.get("notification_error_no_videos_found"), "warning")
        }

        return filtered;
    };


    function processItem(file) {
        var data = {
            name:file.getName(),
            sources: [
                {
                    src:getDownloadUrl(file),
                    type: 'video/mp4'
                }
            ]
        }
        return data;
    }

    /**
     * Build playlist compatible with player
     *
     * @param {Array}  files  Array of {@link Picasso.File}
     * @param {File}  file to be a the beggining of the list
     *
     * @returns {Array}  Playlist format
     **/

    var buildplayList = function (files,file) {
        var playlist = [];

        files.forEach(function (file) {
            playlist.push(processItem(file));
        });

        playlist.sort(function (x, y) {
            return x.name == file.getName() ? -1 : y.name == file.getName() ? 1 : 0;
        });

        return playlist;
    };

    /**
     * Render given json object for gallery
     *
     * @param {Picasso.File}  file  A {@link Picasso.File}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderFileGallery = function (file) {
        /* Create slide */
        var div = $("<div>", {
            class: "pica-vmtable-gallery-slide"
        });

        if (file.isTifImage()) {
            var xhr = new XMLHttpRequest();
            xhr.open('GET', getDownloadUrl(file));
            xhr.responseType = 'arraybuffer';
            xhr.onload = function (e) {
                var buffer = xhr.response;
                var tiff = new Tiff({buffer: buffer});
                var canvas = tiff.toCanvas();
                if (canvas) {
                    div.append(canvas);
                }
            };
            xhr.send();
        }else{
            div.append($("<img>", {
                src: getDownloadUrl(file)
            }));
        }

        /* Whether this image should be the active one */
        if (file.equals(Picasso.get("selectedFile"))) {
            div.addClass("active");
        }

        /* Create caption */
        var caption = $("<div>", {
            class: "pica-vmtable-gallery-caption"
        });

        caption.append($("<h3>", {
            text: file.getName()
        }));

        div.append(caption);

        return div;
    };

    $("#files_home").click(handleClickCrumb);
    $("#files_home").click(removeSiblings);
    $(".files-ui-download").click(handleClickDownload);
    $(".files-ui-delete").click(handleClickDelete);
    $(".files-ui-rename").click(handleClickRename);
    $(".files-ui-copy").click(handleClickCopy);
    $(".files-ui-paste").click(handleClickPaste);
    $(".files-ui-cut").click(handleClickCut);
    VMTABLE.setListRenderer(renderFileList);
    VMTABLE.setGalleryRenderer(renderFileGallery);
    VMTABLE.setGalleryFilter(filterFileGallery);
    VMTABLE.getTable().setSelectionHandler(handleOnSelection);



    //Init
    Picasso.Files.fetch();

})();
