/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Login")) {
    window.Picasso.Login = {};
}

/* Private scope */
(function () {

    /* Automatically set password input field focus */
    var username = $("#login_username").val();

    if (username) {
        $("#login_username").val(username.replace(/\s/g, ''));
        $("#login_password").focus();
    }

    /**
     * Recover password for given username or otherwise warn
     *
     * @param {Event}  e  Click event
     **/

    var handleRecoverPassword = function (e) {
        e.stopPropagation();
        e.preventDefault();

        var field = $("#login_username");
        var username = field.val();

        if (0 === username.length) {
            Picasso.Notification.show(
                Picasso.Lang.get("notification_error_missing_fields"),
                "warning");

            /* Add marker and focus */
            field.parent("div").addClass("has-error").addClass("has-danger");
            field.focus();
        } else {
            var url = "/changepassword";
            var data = {
                action: "recover",
                username: username
            };

            /* Fire ajax */
            Picasso.Helper.fireAjax(url, data,

                /* Success */
                function (data, status) {
                    switch (status) {
                        case 200:
                            Picasso.Notification.show(
                                Picasso.Lang.get("notification_recover_password"),
                                "info");
                    }
                },

                /* Error */
                function (e, status) {
                    switch (status) {
                        case 400:
                            Picasso.Notification.show(
                                Picasso.Lang.get(
                                    "notification_error_missing_fields"),
                                "warning");
                            break;

                        case 404:
                            /* Falls through */
                        case 405:
                            Picasso.Notification.show(
                                Picasso.Lang.get(
                                    "notification_error_unknown_login_or_wrong_password",
                                    $("#login_username").attr("placeholder")),
                                "danger");
                            break;

                        default:
                            Picasso.Notification.show(
                                Picasso.Lang.get(
                                    "notification_error_server_error"),
                                "danger");

                            Picasso.debugLog(e);
                    }
                }
            );
        }

        return false;
    };

    /**
     * Handle shibboleth completion
     *
     * @param {Picasso.TagInput}  tagInput  A #{@link Picasso.TagInput}
     * @param {String}            val       Value to load completions for
     **/

    var handleShibbolethCompletion = function (tagInput, val) {
        var url = "/wapi/idpd";
        var data = { action: "getIDPs" };

        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                if (ary) {
                    var shib = ary[0];
                    var query = (val || "").toLowerCase();
                    var found = 0;

                    tagInput.clearCompletions();

                    /* Add external if necessary */
                    if (shib.isExternalEnabled) {
                        $.each(shib.externalSorted, function () {
                            if (0 !== val.length && -1 === (this.name+this.type).indexOf(query)) {
                                return;
                            }
                            if (this.name.indexOf("!")===0) {
                                this.name = this.name.substr(1);
                            }
                            tagInput.addCompletionEntry(this.name, "database", "", {
                                    name: this.name,
                                    type: this.type
                                }
                            );

                            found++;
                        });
                    }

                    /* Add institutions */
                    if (shib.institutions) {
                        /* Add divider when necessary */
                        if (0 < found) {
                            tagInput.addCompletionDivider();
                        }

                        $.each(shib.institutions, function (k, v) {
                            if (0 !== val.length && -1 === (k+v).indexOf(query))
                            {
                                return;
                            }

                            tagInput.addCompletionEntry(k, "education", "", {
                                    name: k,
                                    type: "shib",
                                    url: v
                                }
                            );

                            found++;
                        });
                    }
                }

                /* Only show completions if there are any */
                if(0 < found) {
                    tagInput.showCompletions();
                }
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }, val
        );
    };

    /**
     * Handle shibboleth completion selection
     *
     * @param {Event}   e         Click event
     * @param {Object}  userdata  Passed userdata
     *
     * @returns {boolean}
     */

    var handleShibbolethCompletionSelection = function (e, userdata) {
        /* Handle types */
        switch (userdata.type) {
            case "database":
            /* Falls through */
            case "db":
                /* Falls through */
            case "ldap":
                /* Falls through */
            case "ad":
            /* Falls through */
            case "activedirectory":
                $("#login_submit").removeAttr("disabled");

                $("#login_username").parents(".form-group").show();
                $("#login_password").parents(".form-group").show();
                return true;
            case "shibboleth":
            /* Falls through */
            case "shib":
                $("#login_submit").attr("disabled", true);

                $("#login_username").parents(".form-group").hide();
                $("#login_password").parents(".form-group").hide();

                /* Just forward here */
                window.location.href = userdata.url;
                return true;
        }

        return false;
    };

    /* Global - must be defined after constructor */

    var taginput = $("#login_taginput");

    if (0 !== taginput.length) {
        Picasso.Login.SHIBBOLETH = new Picasso.TagInput("#login_taginput",
            Picasso.TagInput.Flags.COMPLETION_ONLY|
            Picasso.TagInput.Flags.COMPLETION_ONLY_ONCE|
            Picasso.TagInput.Flags.COMPLETION_FILTER|
            Picasso.TagInput.Flags.COMPLETION_SELECTION_SINGLE);

        Picasso.Login.SHIBBOLETH.setCompletionHandler(handleShibbolethCompletion);
        Picasso.Login.SHIBBOLETH.setCompletionSelectHandler(
            handleShibbolethCompletionSelection);
    }

    /* Bind handlers */
    $("#login_recover_password").click(handleRecoverPassword);


    $("#login_form").on("submit", function (e) {

        /* Check validation result */
        if (!Picasso.Helper.validateInputs(this)) {
            e.preventDefault();
            e.stopPropagation();

            Picasso.Notification.show(
                Picasso.Lang.get("notification_error_missing_fields",
                    "warning"));

            return false;
        }
        if ($('#idpButton').length) {
            var buttonValue = $('#idpButton').val();
            $('#login_idpSelect').val(buttonValue);
        }
    });

    $("#login_totp").off("keyup").keyup(function (e) {
        var m = $(this).val().match(/([0-9]{6})/)
        var password = $('#login_password').val();
        if (m && password.trim() !== ''){
            $(this).removeClass('has-error');
            $("#login_form").submit();
        }else {
            $(this).addClass('has-error');
        }
    })

})();
