/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Federation")) {
    window.Picasso.Federation = {};
}

/* Private scope */
(function () {

    /* Globals */
    var TABLE = new Picasso.Table("#federations_table",
        (Picasso.Table.Flags.LIST_MODE | Picasso.Table.Flags.ROWS_SELECTABLE));

    var optionsForServices = [
        {
            text: Picasso.Lang.get("action_delete"),
            icon: "bin",
            klass: "files-ui-delete"
        },
        {
            text: Picasso.Lang.get("action_validate"),
            icon: "validate",
            klass: "files-ui-validate"
        }
    ];

    /**
     * Handle click on account create button
     **/

    var handleClickCreateFederation = function () {

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_input_dialog",
            Picasso.Lang.get("dialog_title_create"),
            Picasso.Lang.get("dialog_body_create", Picasso.Lang.get("label_services")));

        /* Assign on input handler for validating service URL input */
        dialog.setAssignHandler(handleServiceURLValidation);

        dialog.setOkHandler(function (inputs) {

            var url = inputs.pica_input_text.val();

            /* Success */
            var onSuccess = function () {

                Picasso.Federation.fetch();

                Picasso.Notification.show(
                    Picasso.Lang.get("notification_item_created", url), "info");

                dialog.hide();
            };

            /* Error */
            var onError = function (e, status) {
                switch (status) {
                    case 406:
                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_error_illegal_chars"),
                            "warning");
                        break;

                    case 409:
                        Picasso.Notification.show(Picasso.Lang.get(
                            "notification_error_duplicate", url), "warning");
                        break;

                    case 400:
                    /* Falls through */
                    case 500:
                        Picasso.Notification.show(Picasso.Lang.get(
                            "notification_error_create", url), "danger");
                }

                Picasso.debugLog(e);
            };

            Picasso.Server.createFederatedService(url, onSuccess, onError);
        });

        dialog.show();
    };

    /**
     * Validate service URL input.
     *
     * @param inputs
     */

    var handleServiceURLValidation = function (inputs) {

        var okButton = this._dialog.find(
            ".modal-footer button[data-dismiss].btn-primary");
        var createServiceInput = $(inputs.pica_input_text);

        createServiceInput.prop("title", "Enter a valid service URL");
        createServiceInput.prop("placeholder", "Enter a valid service URL");

        createServiceInput.on("input", function () {

            var that = $(this);
            var isValid = true;

            if (!Picasso.Helper.isValidURL(that.val())) {
                okButton.addClass("disabled");
                okButton.prop("disabled", true);
                that.closest(".form-group").addClass("has-error");
                isValid = false;
            } else {
                okButton.removeClass("disabled");
                okButton.prop("disabled", false);
                that.closest(".form-group").removeClass("has-error");
            }

            return isValid;
        });
    };

    /**
     * Handle click on delete button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickDeleteFederation = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Server($(val).find(".pica-serial").data("serial"))
        });

        /* Show dialog */
        var what = (1 === selected.length ? selected[0].getWebUrl() :
            Picasso.Lang.get("string_selected_elements", selected.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_delete"),
            Picasso.Lang.get("dialog_body_delete", what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {
                var server = this;

                server.removeFederatedService(
                    /* Success */
                    function () {
                        Picasso.Helper.fadeOut($("#" + server.getElemId()),
                            function () {
                                this.remove();

                                TABLE.checkIfEmpty();
                            });

                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_deleted",
                                Picasso.Lang.get("label_services")), "info");
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);
                    }
                );
            });
        });

        dialog.show();
    };

    /**
     * Handle click on validate button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickValidateFederation = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Server($(val).find(".pica-serial").data("serial"))
        });

        /* Show dialog */
        var what = (1 === selected.length ? selected[0].getWebUrl() :
            Picasso.Lang.get("string_selected_elements", selected.length));

        var dialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_validate"),
            Picasso.Lang.get("dialog_body_validate", what));

        dialog.setOkHandler(function () {
            $.each(selected, function () {

                var server = this;
                server.validateFederatedService(

                    /* Success */
                    function () {

                        Picasso.Federation.fetch();

                        Picasso.Notification.show(
                            Picasso.Lang.get("notification_item_validated",
                                Picasso.Lang.get("label_services")), "info");

                        dialog.hide();
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);
                    }
                );
            });
        });

        dialog.show();
    };

    /**
     * Update table header for selection mode
     *
     * @param {Integer}  nselected    Number of selected rows
     * @param {Object}   selectedRow  Selected row object
     **/

    var handleOnSelection = function (nselected, selectedRow) {

        if (nselected === 0) {
            TABLE.deselectAll();
            return;
        }

        /* Get selected elements */
        var selected = TABLE.getSelectedRows();

        selected = $.map(selected, function (val) {
            return new Picasso.Server($(val).find(".pica-serial").data("serial"))
        });

        $.each(selected, function () {
            var server = this;
            if (server.getStatus() !== "local") {
                $("#services_validate").hide();
            }
        });
    };

    /**
     * Render given json object
     *
     * @param {Object}  server  Given json object
     * @returns {*}
     **/

    var renderFederationList = function (server) {
        var tr = $("<tr>", {
            "data-search-keys": [server.getWebUrl()],
            id: server.getElemId(),
            draggable: false
        });

        /* 1. column: Icon */
        var td = $("<td>", {class: "text-center pica-icon-column pica-serial"});

        td.data("serial", server.serialize());

        var span;
        if (server.getValidationDate()) {
            span = $("<span>", {
                class: "pica-icon pica-glyph glyphicons glyphicons-engineering-networks"
            });
        } else {
            span = $("<span>", {
                class: "pica-icon pica-glyph glyphicons glyphicons-warning-sign"
            });
        }

        span.tooltip({container: "body"});

        td.append(span);
        tr.append(td);

        /* 2. column: Service */
        td = $("<td>", {class: "pica-overflow"});

        var a = $("<a>", {
            href: server.getWebUrl(),
            text: server.getWebUrl()
        });

        td.append(a);
        tr.append(td);

        /* 3. column: Members */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: server.getAccountsCount()
        }));

        /* 4. column: Folders */
        tr.append($("<td>", {
            class: "hidden-xs",
            text: server.getFoldersCount()
        }));

        /* 5. column: Validation date */
        tr.append($("<td>", {
            class: "hidden-xs hidden-sm",
            text: server.getValidationDate() ? server.getValidationDate() : "-"
        }));

        /* 6. column: Status */
        var status;
        switch (server.getStatus()) {
            case "remote":
                status = Picasso.Lang.get("status_remote_validation");
                break;
            case "local":
                status = Picasso.Lang.get("status_local_validation");
                break;
            case "dissolved":
                status = Picasso.Lang.get("status_dissolved");
                break;
            case "online":
                status = Picasso.Lang.get("status_online");
                break;
            default:
                status = Picasso.Lang.get("status_offline")
        }

        tr.append($("<td>", {
            class: "hidden-xs",
            text: status
        }));

        /* 7. column */
        tr.append($("<td>", {class: "pica-icon-column"}));

        return tr;
    };

    /**
     * Fetch JSON data
     **/

    Picasso.Federation.fetch = function () {

        Picasso.Server.getAllFederatedServices(
            /* Success */
            function (services) {
                TABLE.update(services, renderFederationList);
            },

            /* Error */
            function (e) {
                Picasso.debugLog(e);
            }
        );
    };

    /* Bind handlers */
    TABLE.setSelectionHandler(handleOnSelection);

    $("#federations_create").click(handleClickCreateFederation);
    $("#services_delete").click(handleClickDeleteFederation);
    $("#services_validate").click(handleClickValidateFederation);

    /* Init */
    $(function () {
        Picasso.Federation.fetch();
    });
})();